<?php
/**
* Example codes
* Any of these can be copied and pasted to in Tools > Custom Functions
* Then they will be automatically executed
*
*/

/**
* Accessing booking variables after a new booking
* @param	$id		integer		Booking ID, a.k.a $app_id throughout the codes
* @param	$data	array		Sanitized user submitted data, e.g. name, email, address, udf
* @return none
*/
add_action( 'app_new_appointment', 'app_new_booking', 100, 2 ); # Take priority as minimum 100, so addons can finish their jobs.
function app_new_booking( $id, $data ) {
	$booking	= new WpB_Booking( $id );
	$client_id	= $booking->get_user();
	$worker_id	= $booking->get_worker(); # a.k.a provider
	$start		= $booking->get_start();
	$address	= $booking->get_meta( 'address' );
	// etc...
	// For more variables and details, check class.booking.php
	// Do something with these
}

/**
* Force Payment Required on a particular page
*/
add_filter( 'app_options', 'app_force_payment' );
function app_force_payment( $options ){
    if ( wpb_find_post_id() != 123 ) { # Change 123 with your post ID
        return $options;
	}

    $options['payment_required'] = 'yes';
    return $options;
}


/**
* Don't require payment when admin is making on behalf booking
*/
add_filter( 'app_options', 'app_remove_payment' );
function app_remove_payment( $options ){
    if ( ! current_user_can( WPB_ADMIN_CAP ) || empty( $_POST['app_user_id'] ) || get_current_user_id() != $_POST['app_user_id'] ) {
        return $options;
	}

    $options['payment_required'] = 'no';
    return $options;
}

/**
* Fill calendar cell with available worker count
* Applicable when service duration equals time base
* Return array
*/
add_filter( 'app_weekly_calendar_cell_fill', 'app_calendar_cell_fill', 10, 2 );
add_filter( 'app_monthly_calendar_cell_fill', 'app_calendar_cell_fill', 10, 2 );
function app_calendar_cell_fill( $arr, $slot ) {
    if ( $slot->reason || is_admin() && !(defined( 'WPB_AJAX' ) && WPB_AJAX ) ) {
        return $arr;
	}

   if ( $slot->calendar->is_admin() ) {
     return $arr;
   }
   
   $args = array(
		'start'		=> $slot->get_start(),
		'end'		=> $slot->get_end(),
		'service'	=> $slot->get_service(),
	);
	
	$stats		= array( 'confirmed', 'paid', 'pending', 'test' );
	$busy		= wpb_nof_bookings( $args, $stats );
	$workers	= (array)BASE()->get_worker_ids_by_service( $slot->get_service() );
	$avail_workers = (array)BASE('WH')->available_workers( $slot );
    $avail		= count( array_intersect( $workers, $avail_workers ) );
	$result		= max( 0, $avail - $busy );
    $fill		= (! empty($arr['fill']) ? $arr['fill'] : '').'<span>'.$result.' Free</span>';

    return array( 'class_name' => 'app-cell-centered', 'fill' => $fill );
}

/**
* Add total and available seats count to monthly calendar date
* Only works for an All Day service
*/
add_filter( 'app_monthly_calendar_html_after_td', 'app_monthly_calendar_html_after_td', 10, 2 );
function app_monthly_calendar_html_after_td( $html, $slot ) {
	$start = strtotime( date( 'Y-m-d', $slot->get_start() ) );
	
	if ( ! $slot->is_daily() || $start != $slot->get_start() || ! $slot->is_free() ) {
		return $html;
	}
 
	$args = array(
		'range'		=> 'custom',
		'start'		=> $start,
		'end'		=> $start + 84399,
		'location'	=> $slot->get_location(),
		'service'	=> $slot->get_service(),
		'worker'	=> $slot->get_worker(),
	);
	
	$stats = array( 'confirmed', 'paid', 'pending', 'running', 'test' );
	
	$busy = wpb_nof_booked_seats( $args, $stats );
	$total = $slot->available_workforce();
	$free = max( 0, $total - $busy );
	
	$html .= '<p>'. $free .'/'. $total . '</p>';
	
	return $html;
}

/**
* Validation for zip submission for UDF Addon
* Returning false means value validates
* Returning a string rejects submission and outputs string value as an error message
* Returning true rejects the submission with default error message
*/
add_filter( 'app_zip_validation', 'app_zip_validation', 10, 4 );
function app_zip_validation( $result, $value, $values_arr, $field ) {
    $zips = array( 83000, 83150 ); # List of valid zips

    if ( !in_array( $value, $zips ) ) {
        return "We do not serve to that area!";
    } else {
    	return false;
	}
}

/**
* Validation for udf_4 submission for UDF Addon
* Accepts values 3...10
* Returning false means value validates.
*/
add_filter( 'app_udf_4_validation', 'app_udf_4_validation', 10, 4 );
function app_udf_4_validation( $result, $value, $values_arr, $field ) {
    if ( $value < 3 ) {
        return "Submitted number ({$value}) is too small";
    } else if ( $value > 10 ) {
        return "Submitted number ({$value}) is too large";
	} else {
    	return false;
	}
}

/**
*
* Hiding price field of service setting for non-admin users
*/
add_filter( 'app_services_add_service', 'app_hide_service_price', 10, 3 );
function app_hide_service_price( $html, $no, $service ){
    if ( current_user_can( WPB_ADMIN_CAP ) ) {
        return $html;
	}

	$html = preg_replace( '%<input class="app-service-price"(.*?)/>%',
                         '<input class="app-service-price"$1 style="visibility:hidden"/>',
                         $html );
    return $html;
}

/**
* Example of custom time table display for monthly calendar for a 12-hours service (ID:17)
* You can wrap the output with span tag for easier styling, e.g. return '<span>Morning Session</span>'
*/
add_filter( 'app_timetable_cell_fill', 'app_custom_time_display', 10, 2 );
function app_custom_time_display( $display, $slot ) {
    $service = $slot->get_service();
    if ( $service != 17 || BASE()->is_package( $service )  ) # Replace 17 with your service ID
        return $display;

    if ( date( 'G', $slot->get_start() ) < 12 ) {
        return 'Morning Session';
    } else {
        return 'Afternoon Session';
	}
}

/**
* Remove time display (i.e. show only date) for emails and confirmation form for a 12-hours service (ID:8)
* In other words, assume a service as lasting all day
* Note that working hours of the service should be set accordingly
*/
add_filter( 'app_is_daily', 'app_is_daily', 10, 2 );
function app_is_daily( $result, $ID ) {
	# Replace 8 with your service ID
	if ( 8 == $ID ) {
        return true;
    } else {
        return $result;
	}
}

/**
* Use service filter in Services dropdown
* Service options are filtered out as client types some letters
* Useful if you have many services
*/
add_filter( 'app_js_data', 'app_use_service_filter' );
function app_use_service_filter( $param ) {
	$param['service_filter'] = 1;
    $param['filter_label'] = "Search:";	// Optional, if omitted "Filter:" is used
    $param['filter_placeholder'] = "Type some letters";	// Optional, if omitted "Enter keywords" is used
    return $param;
}

/**
* Override user field values (userdata) by url $_GET values
* These fields are: name, first_name, last_name, email, phone, address, zip, city
* Values should be urlencoded
* e.g. http://example.com/make-a-booking/?name=Hakan+Ozevin&email=example@mail.com
* will prepopulate name and email fields as given
*/
add_filter( 'app_userdata', 'app_userdata', 10, 2 );
function app_userdata( $data, $fields ) {
    foreach ( $fields as $f ) {
        if ( isset( $_GET[$f] ) ) {
            $data[$f] = wpb_clean( urldecode($_GET[$f]) );
		}
    }

    return $data;
}

/**
* Override UDF values by url $_GET values
* Values should be urlencoded
* e.g. http://example.com/make-a-booking/?udf_4=Free+Gift&udf_5=My+best+friend
* OR
* http://example.com/make-a-booking/?gift=Free+Gift&for=My+best+friend
* will prepopulate UDF_4 field as "Free Gift" and UDF_5 field as "My best friend"
*/
add_filter( 'app_udf_value', 'app_udf_value', 10, 3 );
function app_udf_value( $value, $udf, $app_id ) {
    $udf_id = $udf['ID'];

    if ( isset( $_GET['udf_'.$udf_id] ) ) {
        $value = urldecode($_GET['udf_'.$udf_id]);
	}
	// OR (Use only one)
    // If you already know that udf_4 is for gift name and udf_5 is for the recipient:
     if ( $udf_id == 4 && isset( $_GET['gift'] ) ) {
         $value = urldecode($_GET['gift']);
     } else if ( $udf_id == 5 && isset( $_GET['for'] ) ) {
         $value = urldecode($_GET['for']);
	 }

    return wpb_clean( $value );
}

/**
* Prefill UDF_1 with a preset value
* If client is a user and submitted a value before, skip this
* @return string
*/
add_filter( 'app_udf_value', 'app_udf_value_prefill', 10, 3 );
function app_udf_value_prefill( $value, $udf, $app_id ) {
	if ( $value || $app_id ) {
		return $value;
	}

	if ( !empty( $udf['ID'] ) && 1 == $udf['ID'] ) {
		return 'R18.6'; # Change text as required
	}

	return $value;
}

/**
* Create a user and make him Service Provider (worker)
* If user is already WP user, assign him as worker
* Call this only once!
* @return integer|false
*/
// add_action( 'admin_init', 'app_add_worker' );
function app_add_worker(){
    if ( $id = wpb_create_user( array('email' => 'hakan+test4@wp-base.com','name' => 'Hakan Ozevin') ) ) {
        wpb_add_worker( array( 'ID' => $id, 'price' => '$1.234,56','service' => '1,2, 3,5, 6,7,8, 10 ,' ) );
	}

}

/**
* Show coupon field to first time customer only
* @param $result	bool		Result of previous operation
* @param $id		integer		ID of the coupon record (not code)
* @param $slot		object		WpB_Slot object
* @return bool		When false, coupon field is not displayed
*/
add_filter( 'app_show_coupon_field', 'app_show_coupon_once', 10, 3 );
function app_show_coupon_once( $result, $id, $slot ) {
	# Already not shown. Nothing to do
	if ( !$result ) {
		return $result;
	}

	# Client has bookings in their cookie. Do not show coupon field
	if ( BASE()->get_apps_from_cookie() ) {
		return false;
	}

	# No need to check further for non logged in users
	if ( ! $user_id = get_current_user_id() ) {
		return $result;
	}

	# Check if current user has any bookings. Status conditions can be changed as required.
	if ( BASE()->db->get_var( BASE()->db->prepare( "SELECT COUNT(ID) FROM ". BASE()->app_table.
		" WHERE (status<>'removed' AND status<>'reserved') AND user=%d", $user_id ) ) ) {
		return false;
	}

	return $result;
}

/**
* Track Google Ads Conversion
* Don't forget to add gtag_report_conversion snippet that you will get from Google
* @return none
*/
add_action( 'wp_footer', 'app_track_conversion', 10000 );
function app_track_conversion() {
?>
<script type="text/javascript">
jQuery(document).ready(function($) {
	if (typeof gtag_report_conversion === "function") {
		$(document).on("app-checkout", gtag_report_conversion());
	}
});
</script>
<?php
}

/**
* Make service with ID 3 available only if
* workers 1 and 127 working and available at the same time
* @return null|integer	The output of this function is reversed: An integer means not available, null means available
*/
add_filter( 'app_is_unavailable', 'app_is_unavailable', 10, 3 );
function app_is_unavailable( $null, $slot, $irrelevant ) {
	# Not desired service or already unavailable
	if ( 3 != $slot->get_service() || $null ) {
		return $null;
	}

	$_slot	= clone $slot;

	switch ( $slot->get_worker() ) {
		case 1:		$_slot->set_worker( 127 ); break;
		case 127:	$_slot->set_worker( 1 ); break;
		default:	return $null;
	}

	remove_filter( 'app_is_unavailable', __FUNCTION__ );
	$result = $_slot->why_not_free();
	add_filter( 'app_is_unavailable', __FUNCTION__, 10, 3 );

	return $result ? 18 : $null; # Optional: Change integer number as required. see wpb_code2reason function
}

/**
* More general solution for the above
* Make service with ID 3 available only if
* More than two particular workers available at the same time
* @return null|integer	The output of this function is reversed: An integer means not available, null means available
*/
add_filter( 'app_is_unavailable', 'app_is_unavailable2', 10, 3 );
function app_is_unavailable2( $null, $slot, $irrelevant ) {
	# Not desired service or already unavailable
	if ( 3 != $slot->get_service() || $null ) {
		return $null;
	}

	$worker 	= $slot->get_worker();
	$workers	= array( 1, 85, 127 ); # Workers to test

	if ( in_array( $worker, $workers ) ) {
		return $null;
	}

	$reason = null;
	$_slot	= clone $slot;

	remove_filter( 'app_is_unavailable', __FUNCTION__ );

	foreach ( $workers as $wid ) {
		if ( $wid == $worker ) {
			continue;
		}

		$_slot->set_worker( $wid );

		if ( $reason = $_slot->why_not_free() ) {
			break;
		}
	}

	add_filter( 'app_is_unavailable', __FUNCTION__, 10, 3 );

	return $reason ? 18 : $null; # Optional: Change integer number as required. see wpb_code2reason function
}

/**
* Subtract 5 minutes from the duration *display* in Confirmation Form
* If service ID is 1
* @return integer
*/
add_filter( 'app_pre_confirmation_duration', 'app_pre_confirmation_duration', 10, 2 );
function app_pre_confirmation_duration( $duration, $val ) {
	$slot = new WpB_Slot( $val );

	if ( 1 == $slot->get_service() ) {
		$duration = $duration - 5;
	}

	return $duration;
}

/**
* Subtract 5 minutes from end time display in Confirmation Form when service ID is 1
* This can be used as a practical solution instead of paddings
* Works for both single and multiple bookings (Shopping cart)
* @return array
*/
add_filter( 'app_pre_confirmation_reply', 'app_pre_confirmation_reply', 10, 3 );
function app_pre_confirmation_reply( $reply_array, $items, $obj ) {
	if ( empty( $items ) ) {
		return $reply_array;
	}

	$end_max = 0;
	foreach ( $items as $item ) {
		$slot = new WpB_Slot( $item );
		$end = $slot->get_end();

		if ( 1 == $slot->get_service() ) {
			$end = $end - 300;
		}

		$end_max = max( $end_max, BASE()->client_time( $end ) );
	}

	if ( $end_max > 0 ) {
		$reply_array['end'] = BASE()->conf_line_html( 'end_date_time', date_i18n( BASE()->dt_format, $end_max ) );
	}

	return $reply_array;
}

/**
 * Customize confirmation form display for half day services (Longer than 6, shorter than 24 hours)
 * Shows duration as "Half day" and removes time value from start field
 * @return array
 */
add_filter( 'app_pre_confirmation_reply', 'app_pre_confirmation_reply2', 10, 3 );
function app_pre_confirmation_reply2( $arr, $items, $obj ) {
	if ( !$items || count( $items ) > 1 ) {
		return $arr;
	}

	$slot = new WpB_Slot( current($items) );
	$dur_hours = absint(($slot->get_end() - $slot->get_start())/3600);

	if ( $dur_hours < 24 && $dur_hours >= 6 ) { # You can change 6 with, e.g. 8
		$arr['lasts'] = BASE()->conf_line_html( 'lasts', 'Half day' );
		$arr['start'] = BASE()->conf_line_html( 'date_time', date_i18n( BASE()->date_format, $slot->get_start() ) );
	}

	return $arr;
}

/**
* In weekly calendar when user clicks on table header, select all free slots for that day
* Requires Shopping Cart addon
*
* @return none
*/
add_action( 'wp_footer', 'app_footer_pick_all' );
function app_footer_pick_all(){
?>
<script type="text/javascript">
jQuery(document).ready(function($){
	function pickAll(){
		var tbl = $(this).parents("table").first();
		var index = $(this).index()+1;
      	var tds = tbl.find("td.free:nth-child("+index+")");
      	var $td = '';
		var len = tds.length;
		var app_value = [];

		if ( !len ){
			init();
			return false;
		}

		$.each(tds, function(i){
			$td = $(this);
			if ( i == len - 1 ) {
				$(document).data("app_value", app_value);
				$td.click();
			} else {
				$td.addClass("app-selected");
				app_value.push($td.find(".app-packed").val());
			}
		});
	}

  	function init(){
  		$(document).one("click", ".app-schedule-wrapper th", pickAll);
    }

	init();
});
</script>

<?php
}

/**
* When saving the post, set owner of the service to the author of the post which is set as description page of the service
* This example uses "rental" CPT. Change it as required
*
* @return none
*/
add_action( 'save_post_rental', 'app_set_service_owner', 20, 3 );
function app_set_service_owner( $post_id, $post, $update ) {
	if ( empty( $post->post_author ) || !function_exists( 'BASE' ) ) {
		return;
	}

	# Don't set if author is not a worker
	if ( !wpb_is_worker( $post->post_author ) ) {
		return;
	}

	# Is this a description page of a service?
	if ( !$service_id = BASE()->find_service_for_page( $post ) ) {
		return;
	}

	wpb_update_service_meta( $service_id, 'created_by', $post->post_author );
}

/***** Add $10 to the total price when client checks a particular checkbox at checkout ******************/

/**
* Add app-update-price class to UDF
* Adding this class will make price updated on the fly at confirmation form as UDF_3 checkbox is checked/unchecked
* For text and textarea fields use "app-price-field-entry" instead
* For a select field, this function is not required; price update is already applied upon change
* @return array
*/
add_filter( 'app_udf_class', 'app_udf_class', 10, 4 );
function app_udf_class( $class, $udf, $value, $is_readonly ) {
	# Replace 3 with your UDF ID
	if ( $is_readonly || 3 != $udf['ID'] ) {
		return $class;
	}

	# For text and textarea fields use "app-price-field-entry" instead
	$class[] = 'app-update-price';

	return $class;
}

/**
* Add $10 to the total price if UDF_3 is set
* Selecting priority determines at what point price will be applied:
* Extras 100, Coupons 200, Payment Gateway fee has priority 500
* So with priority 400, $10 will be added after coupon discounts applied. On top of $10, gateway fees will be added
* @return string
*/
add_filter( 'app_confirmation_total_price', 'app_confirmation_total_price', 400, 2 );
function app_confirmation_total_price( $price, $items ) {
	# Replace 3 with your UDF ID
	if ( empty( $_POST['app_udf_3'] ) ) {
		return $price;
	}

	# Replace 10 with the desired additional price
	return $price + 10;
}

/**
 * Sets amount (money to be paid at checkout) to zero when location is 1
 * If there are several items in the cart, one of them being in location 1 is sufficient
 * Booking will be saved as pending or confirmed depending on "Auto Confirm" setting
 * @return string|integer
 */
add_filter( 'app_confirmation_total_amount', 'app_confirmation_total_amount', 10, 2 );
function app_confirmation_total_amount( $amount, $items ) {
	foreach ( $items as $item ) {
		$slot = new WpB_Slot( $item );
		if ( 1 == $slot->get_location() ) {
			return 0;
		}
	}
	
	return $amount;
}

/***** END OF Add $10 to the total price when client checks a particular checkbox at checkout ******************/


/**
* Add VAT tax if client is from EU
* Country is selected at UDF_1 whose type is "function" and options is "wpb_countries"
* @return string
*/
add_filter( 'app_confirmation_total_price', 'app_add_tax', 400, 2 );
function app_add_tax( $price, $items ) {
	# Replace 1 with your UDF ID
	if ( empty( $_POST['app_udf_1'] ) ) {
		return $price;
	}

	$taxes = array(
		'AT'	=> 20,
		'BE'	=> 21,
		'BG'	=> 20,
		'CY'	=> 19,
		'CZ'	=> 21,
		'DE'	=> 19,
		'DK'	=> 25,
		'EE'	=> 20,
		'EL'	=> 24,
		'ES'	=> 21,
		'FI'	=> 24,
		'FR'	=> 20,
		'GB'	=> 20,
		'HR'	=> 25,
		'HU'	=> 27,
		'IE'	=> 23,
		'IT'	=> 22,
		'LT'	=> 21,
		'LU'	=> 17,
		'LV'	=> 21,
		'MT'	=> 18,
		'NL'	=> 21,
		'PL'	=> 23,
		'PT'	=> 23,
		'RO'	=> 19,
		'SE'	=> 25,
		'SI'	=> 22,
		'SK'	=> 20,
	);

	if ( !array_key_exists( $_POST['app_udf_1'], $taxes ) ) {
		return $price;
	}

	return wpb_round( $price + $taxes[$_POST['app_udf_1']] * $price / 100 );
}

/**
* Create a custom PDF for user #999
* Requires PDF addon
*/
add_filter( 'app_options', 'app_custom_pdf' );
function app_custom_pdf( $options ){

$template = "
TITLE
LABEL 1: UDF_1
LABEL 2: UDF_2
";

    $options['confirmation_attachment'] = $template;
    return $options;
}

/**
 * Add service price and location price in emails and dialogs
 */
add_filter( 'app_email_replace_pre', 'app_service_location_price_replace', 10, 3 );
function app_service_location_price_replace( $text, $r, $context ) {
	if ( empty( $r->ID ) ) {
		return $text;
	}

	$slot = new WpB_Slot( $r->ID );

	$text = str_replace( 'SERVICE_PRICE', wpb_format_currency( $slot->service_price(), false, true ), $text );
	$text = str_replace( 'LOCATION_PRICE', wpb_format_currency( $slot->location_price(), false, true ), $text );

	return $text;
}

/**
 * Change admin email subject
 */
add_filter( 'app_email_subject_admin_copy', 'app_email_subject_admin_copy', 10, 2 );
function app_email_subject_admin_copy( $text, $r ) {
	$new_text = 'Your new email subject with placeholders';
	return wpb_replace( $new_text, $r, 'confirmation_subject_admin' );
}

/**
 * Change admin email body
 */
add_filter( 'app_email_body_admin_copy', 'app_email_body_admin_copy', 10, 2 );
function app_email_body_admin_copy( $text, $r ) {
	$new_text = wp_texturize( wp_autop( 'Your new email body with placeholders' ) );
	return wpb_replace( $new_text, $r, 'confirmation_message_admin' );
}

/**
 * Apply a discount of $5 when gender is selected as Female at checkout
 * UDF 2 is setup as Gender selection field in User Defined Fields addon
 */
add_filter( 'app_confirmation_total_price', 'app_ladies_discount' );
add_filter( 'app_confirmation_total_amount', 'app_ladies_discount' );
function app_ladies_discount( $price ) {
	if ( !empty( $_POST['app_udf_2'] ) && 'female' == strtolower( $_POST['app_udf_2'] ) ) {
		$price = $price - 5;
	}
	return $price;
}

/**
 * Limits providers set working hours only with 1 hour intervals on their account page
 * @return array
 */
add_filter( 'app_options', 'app_provider_min_time_on_account' );
function app_provider_min_time_on_account( $options ) {
	if ( !( wpb_is_account_page() && !empty ( $_GET['tab'] ) && 'working_hours' == $_GET['tab'] ) ) {
		return $options;
	}
	
	if ( isset( $_POST['action_app'] ) && $_POST['action_app'] == 'save_working_hours' ) {
		return $options;
	}
	$options['min_time'] = 60;
	return $options;
}

/**
 * Controlling quantity of Extras with UDF
 * Extras Multiplied with Pax settings should be set as "No"
 * Change udf_1 as required
 */
add_filter( 'app_extras_quantity', 'app_extra_qty' );
function app_extra_qty( $qty ) {
	$submit = !empty( $_POST['app_udf_1'] ) ? $_POST['app_udf_1'] : false;

	if ( $submit ) {
		$qty = $submit;
	}

	return $qty;
}

/**
 * When Extras addon is used, add label(s) before desired Extras in order to group them (optgroup)
 * Styling between <style> tags is optional. You can move css to "additional css rules" setting
 */
add_action( 'wp_footer', 'app_extras_add_label', 100 );
function app_extras_add_label() {
	?>
<style>
.ui-multiselect-checkboxes .ui-state-disabled {
    font-weight: 700;
    opacity: 1;
}
</style>
<script type="text/javascript">
jQuery(document).ready( function($){
	// extraID: ID of the Extra that label to be inserted before
	// labelText: Title of the label
	function insertLabel( extraID, labelText ) {
		var par = $(document).find(".app_extras .ui-multiselect-checkboxes");
		var checkBox = par.find("input[value='"+extraID+"']");
		var myli = checkBox.parents("li").first();

		if (!myli.prev(".app-extras-optgroup").length){
			var lbl = $('<li class="app-extras-optgroup"><label class="ui-corner-all ui-state-disabled"><span class="app-extras-label app-b">'+labelText+'</span></label></li>');
			lbl.insertBefore(myli);
		}
	}

	function insertAll(){
		insertLabel( 1, "Main Extras" ); // Adds before Extra #1
		insertLabel( 3, "Recommended Extras" ); // Adds before Extra #3
		// Add other lines here, e.g. insertLabel( 6, 'Other Extras' );
	}

    $(document).on("elements_updated", function(){
		$("#app_select_extras").on("multiselectbeforeopen multiselectrefresh", function(){
            insertAll();
        });
    });
});
</script>
<?php
}

/**
 * In Group Bookings, prepopulate 1st Guest name as current user name
 * app_nop_1 means "name of participant" 1, similarly
 * app_eop_3 means "email of participant" 3
 * @return array
 */
add_filter( 'app_pre_confirmation_reply', 'app_populate_name', 11 );
function app_populate_name( $reply ) {
	if ( empty( $_POST['app_nop_1'] ) && $user = get_user_by( 'ID', get_current_user_id() ) ) {
		$_POST['app_nop_1'] = $user->display_name;
	}

	return $reply;
}

/**
 * Prevent self booking
 * Provider cannot be his/her client in any booking in the cart
 * In such a case, an error message is displayed and checkout is not completed
 * @return none
 */
add_action( 'app_post_confirmation_check', 'app_prevent_self_booking', 1 );
function app_prevent_self_booking( $val_arr ) {
	foreach ( $val_arr as $val ) {
		$slot = new WpB_Slot( $val );
		if ( $slot->get_worker() == get_current_user_id() ) {
			die( json_encode( array( 'error' => 'You cannot book yourself!' ) ) );
		}
	}
}

/**
 * Use only  credits payment method on a certain page
 * @return array
 */
add_filter( 'app_confirmation_active_gateways', 'app_only_credits' );
function app_only_credits( $gateways ) {
	$post = get_post();
	# Replace 123 with your post/page ID
	if ( !empty( $post->ID ) && 123 == $post->ID ) {
		foreach ( $gateways as $key => $gateway ) {
			if ( 'credits' == $gateway->plugin_name ) {
				continue;
			}

			unset( $gateways[ $key ] );
		}
	}

	return $gateways;
}

/**
 * In BuddyPress hide Book Me tab for everyone except friends
 * @return bool
 */
add_filter( 'app_bp_is_booking_tab_allowed', 'maybe_hide_bookme_tab', 10, 2 );
function maybe_hide_bookme_tab( $show, $user_id ) {
	if ( ! function_exists( 'friends_check_friendship' ) ) {
		return $show;
	}
  
    $me = bp_loggedin_user_id();
	
	if ( !$me || ! friends_check_friendship( $user_id, $me ) ) {
		return false;
	}
	
	return $show;
}

/**
 * Initiate a custom function when a booking is completed
 */
add_action( 'app_status_changed', 'send_survey_upon_completion', $status, $app_id );
function send_survey_upon_completion( $status, $app_id ) {
	if ( 'completed' != $status ) {
		return;
	}
	
	$booking = new WpB_Booking( $app_id );
	// Send survey for the booking
	// All booking variables are accessible in $booking object
	// See /wp-base-dev/includes/class.booking.php
}

/**
 * Customize timetable title for monthly calendar
 * @param	$title		string		Previous title
 * @param	$day_start	integer		Timestamp of the day
 * @param	$calendar	object		WpB_Calendar object	
 */
add_filter( 'app_timetable_title', 'app_timetable_title', 10, 3 );
function app_timetable_title( $title, $day_start, $calendar ) {
	return 'Select a time for '. $title;
}

/**
 * Create a random user and assign him as vendor after a CPT 'law_firm' has created
 */
add_action( 'wp_insert_post', 'app_insert_post', 10, 3 );
function app_insert_post( $post_ID, $post, $update ) {
	if ( $update || 'law_firm' != $post->post_type ) {
		return;
	}
	
	# See wpb_create_user in functions.booking.php
	$data = array(
		'name'	=> $post->post_title,
		'email' => 'myemail+'. $post_ID .'@gmail.com',
	);
	
	if ( $user_id = wpb_create_user( $data ) ) {
		# See wpb_add_worker
		$args = array(
			'ID'	=> $user_id,
			'page'	=> $post_ID,
		);
		
		wpb_add_worker( $args );
		
		$user = new WP_User( $user_id );
		$user->add_role( 'wpb_worker' );
		$user->add_role( 'wpb_vendor' );
	}
}

/**
 * Bookings can only be made for up to January 1, 2023
 */
add_filter( 'app_upper_limit', 'app_upper_limit' );
function app_upper_limit( $limit ) {
	$start = strtotime( 'today' );
	$end = strtotime( 'January 1, 2023' );
	
	return intval( ($end - $start - 1)/DAY_IN_SECONDS );
}

/**
 * Group Bookings ask List of Participants even there is only one person to join
 */
add_filter( 'app_seats_always_ask_lop', '__return_true' );

/**
 * Group Bookings admin edit and add List of Participants in emails even there is only one person to join
 */
add_filter( 'app_seats_inline_edit_always_ask_lop', '__return_true' );

/**
 * Disable MultiSelect JS Widget at the front end
 */
add_filter( 'app_js_data', function( $data ) {
  $data['is']['noMS'] = true;
  return $data;
});

/**
 * Add 5 MyCred points to the client's account upon new booking
 * See: https://codex.mycred.me/chapter-v/playing-with-balances/
 * @param		$ID_booking		integer|object			Booking ID or WpB_Booking object
 * @return none
 */
add_action( 'app_new_appointment', 'app_mycred_add' ); # Booking from front end
add_action( 'app_inline_edit_new_booking', 'app_mycred_add' ); # Booking from admin
function app_mycred_add( $ID_booking ) {
	$booking = $ID_booking instanceof WpB_Booking ? $ID_booking : new WpB_Booking( $ID_booking );
	
	if ( ! function_exists( mycred_add ) || ! $user_id = $booking->get_user() ) {
		return;
	}
	
	mycred_add( 'reference', $user_id, 5, 'For booking #'. $booking->get_ID() );
}

/**
 * Hides free slots (minimum 2) in Monthly Calendar Timetable
 * EXCEPT the one in the middle 
 * @return string
 */
add_action( 'app_scripts_enqueued', 'app_scripts_enqueued' );
function app_scripts_enqueued(){
	add_action( 'wp_footer', 'app_footer' );
}
function app_footer(){
 ?>
	<script type="text/javascript">
		jQuery(document).ready( function($){
			$(document).on("elements_updated", function() {
				var kids = $(document).find("div.app-timetable-cell.free:visible");
				var count = kids.length;
				if (count > 1) {
					var show_no = parseInt( count/2 );
					$.each(kids, function(i) {
						if ( show_no == i + 1 ) {
							$(this).css("display", "table");
						} else {
							$(this).css("display", "none");
						}
					});
				}
			});
		});
	
	</script>
<?php
}

/**
 * Modify WC Order status as cancelled in case booking cancelled
 * @return none
 */
add_action( 'app_booking_cancelled', 'app_booking_cancelled' ); 
function app_booking_cancelled( $app_id ) {

	$booking = new WpB_Booking( $app_id );

	if ( ! $order_id = $booking->get_wc_order_id() ) {
		return;
	}
	
	if ( ! function_exists( 'wc_get_order' ) ) {
		return;
	}

	if ( ! $order = wc_get_order( $order_id ) ) {
		return;
	}
	
	$note = sprintf( __( 'Order cancelled due to related booking #%1$d being cancelled.', 'wp-base' ), $app_id );

	$order->update_status( 'cancelled', $note, true );
}

/**
 * Provider can select only own previous customers in Manage Bookings
 * Or in app_users shortcode
 * @return array
 */
add_filter( 'app_inline_edit_users_args', 'app_inline_edit_users_args' );
add_filter( 'app_users_dropdown_args', 'app_inline_edit_users_args' );
function app_inline_edit_users_args( $args ) {
	if ( current_user_can( WPB_ADMIN_CAP ) ) {
		return $args;
	}
	
	$user_ids = BASE()->db->get_col( BASE()->db->prepare("SELECT user FROM ". BASE()->app_table ." WHERE worker=%d", get_current_user_id() ) );
	
	$args['include'] = $user_ids ?: array( -1 );
	
	return $args;
}

/**
 * In Recurring Appointments addon, allow only certain selections for service #2
 * @return array
 */
add_filter( 'app_recurring_allowed_repeats', 'app_recurring_allowed_repeats', 10, 2 );
function app_recurring_allowed_repeats( $repeats, $service ) {
	if ( $service == 2 ) {
		$repeats = array( 1,3,5,10);
	}
	
	return $repeats;
}

/**
 * Client cannot remove items from WP BASE Shopping Cart
 */
add_filter( 'app_cart_allow_remove', '__return_false' );

/**
 * In BuddyPress, new child booking does not create notification
 * @return bool
 */
add_filter( 'app_bp_skip_add_notification', 'app_bp_skip_notification', 10, 2 );
function app_bp_skip_notification( $result, $booking ) {
	if ( $booking->is_child() ) {
		$result = false;
	}
	
	return $result;
}

/**
 * In WooCommerce Integration, set price as prepayment amount
 * i.e. make deposit payment
 * @return string (price)
 */
add_filter( 'app_woocommerce_price', 'app_wc_price', 10, 2 );
function app_wc_price( $price, $slot ) {
  return BASE()->calc_downpayment( $slot->get_price() );
}

/**
 * Provider (non-admin user) cannot view pages in service description page setting
 * @return array
 */
add_filter( 'app_pages_filter', 'app_pages_filter' );
function app_pages_filter( $pages ) {
	if ( current_user_can( WPB_ADMIN_CAP ) ) {
		return $pages;
	} else {
		return array();
	}
}								