<?php
/**
 * WPB Working Hours
 *
 * Methods for Working Hours
 *
 * The power of this class comes from its scalability:
 * Execution time will not increase with increasing number of workers or services
 *
 * A brief explanation about principle of operation:
 * If you are familiar with electronics, this is a kind of A/D conversion and multiple bit digital value processing.
 * Schema of DB table of working hours has been designed such that each column represents a 5 minute interval of the day.
 * Also 7 days for each 5 minute interval are packed in the same column (Actually I wanted to have a separate column for each 5 minute in the *week* but mySQL does not allow so many columns: max 1017 for innoDB - see note).
 * During saving of settings, working hours of services/providers (subjects) are divided into 5 minute fragments and written into the correct column of the DB.
 * Then querying for working time of a subject (or multiple subjects) is just checking if its ID is in the matching column or not,
 * after A/D converting time value to be checked to its column number (I call this number system "Wh domain").
 * Since one column can include thousands of subject IDs, and querying for one variable or for an array with thousand keys is almost the same thing,
 * thousands of subject Wh results become available at the same time, without much increase in total processing time (memory usage will increase by number of subjects obviously).
 * What is actually done here is optimization during writing to DB, so that you can readily have all Wh values instead of one.
 *
 * Note: Altough myISAM is the default engine for working hour tables, I chose to keep compatibility with InnoDB engine. In other words, as mySQL engine innoDB can be used instead.
 * Update: If WPB_USE_INNODB constant is defined and set as yes, InnoDB can also be used. In some hosting platforms, myISAM engine is not available.
 *
 * Some parts of this class may be quite difficult to follow.
 * That is not intentional but a result of trying to keep codes optimal in speed
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( !class_exists( 'WpBWH' ) ) {

class WpBWH {

	/**
     * WP BASE instance
     */
	protected $a = null;

	/**
     * Name of the tab/title
     */
	private static $name;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
		self::$name = array( 'working_hours' => __( 'Working Hours', 'wp-base' ) );
	}

	public function add_hooks(){
		add_action( 'app_installed', array( $this, 'new_install' ) );
		add_action( 'app_activated', array( $this, 'new_install' ) );
		add_action( 'app_new_worker_added', array( $this, 'new_worker' ), 14, 1 );
		add_action( 'app_new_service_added', array( $this, 'new_service' ), 14, 1 );

		add_action( 'app_save_settings', array( $this, 'save_wh' ), 12 );				// Save settings
		add_action( 'app_save_account_settings', array( $this, 'save_wh' ), 14 );
		add_filter( 'admin_title', array( $this, 'admin_title' ), 10, 2 );
		add_filter( 'appointments_business_tabs', array( $this, 'add_tab' ), 12 ); 		// Add tab
		add_action( 'app_business_working_hours_tab', array( $this, 'render_tab' ) );	// Display HTML settings on Business Settings
	}

	/**
     * Get current or selected location ID
	 * @since 3.0
	 * @return integer
     */
	public function get_location( $slot = null ) {
		return apply_filters( 'app_wh_location', 0, $slot );
	}

	/**
     * Get location IDs to which new service/worker Wh will be added
	 * @since 3.5.8
	 * @return array
     */
	public function get_location_ids_for_default( ) {
		return apply_filters( 'app_wh_location_ids_for_default', array( 0 ) );
	}

	/**
     * Add default working hours to the DB: For worker and service
     */
	public function new_install( ) {

		foreach( array('worker','service') as $subject ) {

			if ( 'worker' == $subject ) {
				$who = $this->a->get_def_wid();
			} else {
				$who = $this->a->get_first_service_id();
			}

			if ( $this->get( $who, $subject ) ) {
				continue;
			}

			$this->add_default( $who, $subject );
		}
	}

	/**
     * Add default working hours to DB for new service
	 * @param $who		integer		ID of the subject
	 * @return none
     */
	public function new_service( $who ) {
		foreach( $this->get_location_ids_for_default() as $loc_id ) {
			$this->add_default( $who, 'service', $loc_id );
		}
	}

	/**
     * Add default working hours to DB for new worker
	 * @param $who		integer		ID of the subject
	 * @return none
     */
	public function new_worker( $who ) {
		foreach( $this->get_location_ids_for_default() as $loc_id ) {
			$this->add_default( $who, 'worker', $loc_id );
		}
	}

	/**
     * Add default working hours to DB
	 * @param	$who		integer			ID of the subject
	 * @param	$subject	string			'worker', 'service' or 'alt'
	 * @param	$loc_id		integer|null	Optional location ID to write to
	 * @return bool
     */
	public function add_default( $who, $subject, $loc_id = 0 ){

		switch( $subject ) {
			case 'worker':	$table = $this->a->wh_w_table; break;
			case 'service':	$table = $this->a->wh_s_table; break;
			case 'alt':		$table = $this->a->wh_a_table; break;
		}

		$wh[$subject] = $this->get( 'all', $subject );
		if ( !is_array( $wh[$subject] ) )
			$wh[$subject] = array_fill(0,2016,'');

		$d_worker = $this->a->get_def_wid();

		if ( ($who == $d_worker && 'worker' == $subject) || !$this->get( $d_worker, 'worker' ) ) {
			$vals = array_fill( 0, 288, '' ); // Sunday
			foreach ( range(1,5) as $day ) {
				$vals = array_merge( $vals, array_fill( 0, 96, '' ), array_fill( 0, 108, 1 ), array_fill( 0, 84, '' ) );
			}
			$vals = array_merge( $vals, array_fill( 0, 288, '' ) ); // Saturday
		} else {
			$vals = $this->get( $d_worker, 'worker' );
		}

		// Combine vals with wh
		$new_vals = array();
		foreach ( $vals as $key => $val ) {
			$wh_val = isset( $wh[$subject][$key] ) ? $wh[$subject][$key] : '';
			if ( $val ) {
				$new_vals[$key]	= $this->add_item( $wh_val, $who );
			} else if ( $wh_val ) {
				$new_vals[$key]	= $wh_val;
			} else {
				$new_vals[$key] = '';
			}
		}

		ksort( $new_vals );

		if ( $this->pack_n_save( $new_vals, $table, $loc_id ) ) {
			wpb_flush_cache();
			return true;
		}

		return false;
	}

	/**
     * Reset wh of a subject and write to DB, i.e. not working any time
	 * @param	$who		integer		ID of the subject
	 * @param	$subject	string		'worker', 'service' or 'alt'
	 * @since 3.5.8
	 * @return bool			true if some changes made, false if no change
     */
	public function clear( $who, $subject ){
		return $this->fill( $who, $subject, false );
	}

	/**
     * Fully set wh of a subject and write to DB, i.e. working all time
	 * @param	$who		integer		ID of the subject
	 * @param	$subject	string		'worker', 'service' or 'alt'
	 * @param	$fill		bool		If true fill, if false clear
	 * @since 3.5.8
	 * @return bool			true if some changes made, false if no change
     */
	public function fill( $who, $subject, $fill = true ){

		switch( $subject ) {
			case 'worker':	$table = $this->a->wh_w_table; break;
			case 'service':	$table = $this->a->wh_s_table; break;
			case 'alt':		$table = $this->a->wh_a_table; break;
		}

		$wh[$subject] = $this->get( 'all', $subject );

		if ( !is_array( $wh[$subject] ) ) {
			$wh[$subject] = array_fill( 0, 2016, '' );
		}

		$vals = array_fill( 0, 2016, ( $fill ? 1 : '' ) );

		$new_vals = array();
		foreach ( $vals as $key => $val ) {
			$wh_val = isset( $wh[$subject][$key] ) ? $wh[$subject][$key] : '';
			if ( $val ) {
				$new_vals[$key]	= $this->add_item( $wh_val, $who );
			} else if ( $fill && $wh_val ) {
				$new_vals[$key]	= $wh_val;
			} else {
				$new_vals[$key] = '';
			}
		}

		ksort( $new_vals );

		if ( $this->pack_n_save( $new_vals, $table ) ) {
			wpb_flush_cache();
			return true;
		}

		return false;
	}

	/**
     * Return a Wh domain array where key is Wh no and value is who is working in that slot
	 * @param $who				integer|string|array	'all' for all elements of the subject, ID for a particular element, an array or a list of elements
	 * @param $subject			string					'worker', 'service' or 'alt'
	 * @param $location			location				location ID
	 * @param $replace_with		integer					Replace $who with what ID. e.g. Replace Alt ID with worker ID
	 * @return array
     */
	public function get( $who = 'all', $subject = 'worker', $location = 0, $replace_with = false ) {

		# Special case: If alt_id:-1, this is holiday
		if ( 'alt' === $subject && -1 == $who ) {
			return array_fill( 0, 2016, '' );
		}

		$who_text = $who;
		if ( is_array( $who ) ) {
			if ( empty( $who ) ) {
				return array_fill( 0, 2016, '' );
			}

			$who_text = implode( '_', $who );
		}

		$replace_text	= (false === $replace_with) ? 'none' : $replace_with;
		$identifier		= wpb_cache_prefix() . 'wh_'. $who_text . '_' . $subject . '_' . $location. '_' . $replace_text;

		$wh = wp_cache_get( $identifier );

		if ( false === $wh ) {
			# Can be used for external optimize, e.g. when subjects are numerous
			$wh = apply_filters( 'app_wh_optimize', false, $who, $subject, $replace_with, $location );

			if ( false === $wh ) {

				switch( $subject ) {
					case 'worker':	$table = $this->a->wh_w_table; break;
					case 'service':	$table = $this->a->wh_s_table; break;
					case 'alt':		$table = $this->a->wh_a_table; break;
				}

				$wh		= array();
				$id2	= wpb_cache_prefix() . 'wh_raw_'. $subject . '_' . $location;
				$wh_raw	= wp_cache_get( $id2 );

				if ( false === $wh_raw ) {

					$wh_result = $this->a->db->get_row( $this->a->db->prepare( "SELECT * FROM " . $table . " WHERE ID=%d", $location ), ARRAY_A );

					if ( defined( 'WPB_USE_INNODB' ) && WPB_USE_INNODB ) {

						$wh_raw = array();

						if ( isset( $wh_result['ID'] ) ) {
							$wh_raw['ID'] = $wh_result['ID'];
						}

						if ( ! empty( $wh_result['avail_data'] ) && $maybe_data = json_decode( $wh_result['avail_data'], true ) ) {
							$wh_raw = array_merge( $wh_raw, $maybe_data );
						}

					} else {
						$wh_raw = $wh_result;
					}

					$wh_raw = $wh_raw ?: null;

					wp_cache_set( $id2, $wh_raw );
				}

				if ( is_array( $wh_raw ) ) {

					for ( $k = 0; $k < 288; $k++ ) {
						if ( isset( $wh_raw['c'.$k] ) ) {
							$temp = explode( '|', $wh_raw['c'.$k] );
							foreach( $temp as $day => $ids ) {
								if ( $day >=7 ) {
									break;
								}

								if ( is_array( $who ) ) {
									$wh[$day*288+$k] = implode(',', array_intersect( $who, explode( ',', $ids ) ));
								} else if ( 'all' == $who ) {
									$wh[$day*288+$k]= $ids;
								} else {
									$wh[$day*288+$k] = in_array( $who, explode( ',', $ids ) ) ? ($replace_with ? $replace_with : $who) : '';
								}
							}
						}
					}

					$wh = array_replace( array_fill( 0, 2016, '' ), $wh );
					ksort( $wh );
				}

				wp_cache_set( $identifier, $wh );
			}
		}

		return $wh;
	}

	/**
     * Combine alternative schedules and regular ones according to assignments
	 * @param $who				integer|string|array	'all' for all elements of the subject, ID for a particular element, an array or a list of elements
	 * @param $subject			string					'worker', 'service' or 'alt'
	 * @param $location			location				location ID
	 * @param $year_week_no		string					year + week no
	 * @return array
     */
	public function combine_alt( $who, $subject, $year_week_no, $location ) {
		return apply_filters( 'app_combine_alt', $this->get( $who, $subject, $location ), $who, $subject, $year_week_no, $location );
	}

	/**
     * Convert timestamp to Wh domain
	 * @param $slot_start	integer		Timestamp
	 * @return integer					A number showing $slot_start resides in which 5th min of the week
     */
	public function to( $slot_start ) {
		return intval((( $slot_start - wpb_sunday()) % 604800)/300);
	}

	/**
     * Convert Wh number to timestamp
	 * @param $no		integer			A number showing $slot_start resides in which 5th min of the week
	 * @param $ts		integer|false	Timestamp to be taken account for week calculation. If left empty, current week
	 * @return integer					Timestamp
     */
	public function from( $no, $ts = false ) {
		if ( false === $ts )
			$ts = $this->a->_time;

		$day_no = date( 'w', $ts );
		if ( $no < $this->a->start_of_week * 288 ) {
			$no = $no + 2016;
		}

		return wpb_sunday( $ts ) + $no *300;
	}

	/**
     * Find all available workers in a time slot
	 * @param $slot			WpB_Slot object
	 * @return array|null
     */
	public function available_workers( $slot ) {
		$slot_start	= $slot->get_start();
		$slot_end	= $slot->get_end();
		$identifier = wpb_cache_prefix() . 'avail_workers_'. $slot_start. '_' . $slot_end;

		$out = wp_cache_get( $identifier );

		if ( false === $out ) {
			$out = $this->available_subjects( $slot, false, 'worker' );

			wp_cache_set( $identifier, $out );
		}

		return $out;
	}

	/**
     * Find all available services in a time slot
	 * @param $slot			WpB_Slot object
	 * @return array|null
     */
	public function available_services( $slot, $who = false ) {
		$slot_start	= $slot->get_start();
		$slot_end	= $slot->get_end();
		$who_text	= false === $who ? 'all' : $who;
		$identifier	= wpb_cache_prefix() . 'avail_services_'. $slot_start. '_' . $slot_end .'_'. $who_text;

		$out = wp_cache_get( $identifier );

		if ( false === $out ) {
			$out = $this->available_subjects( $slot, $who, 'service' );

			wp_cache_set( $identifier, $out );
		}

		return $out;
	}

	/**
     * Find all available subjects in an interval in Wh domain
	 * @param $slot				object|array	WpB_Slot object or an array in the form array( 'start' => wh_start, 'end' => wh_end, 'year_week_no' => false )
	 * @param $subject			string			'worker', 'service' or 'alt'
	 * @return array|null
     */
	public function available_subjects( $slot, $who = false, $subject = 'worker' ) {

		if ( $slot instanceof WpB_Slot ) {
			$location	= $this->get_location( $slot );
			$start		= $this->to( $slot->get_start() );
			$end		= $this->to( $slot->get_end() );
			$slot_start	= $slot->get_start();
			$year_week_no = date( "Y", $slot_start ). wpb_time2week( $slot_start );
		} else {
			$location	= $this->get_location();
			$start		= $slot['start'];
			$end		= $slot['end'];
			$slot_start	= false;
			$year_week_no = $slot['year_week_no'];
		}

		# If Annual Schedules are not active, result is independent of week number
		$week_text 	= (false === $year_week_no) || !class_exists('WpBAnnual') ? 'none' : $year_week_no;
		$who_text 	= (false === $who) ? 'all' : $who;
		$identifier = wpb_cache_prefix() . 'avail_subj_'. $start .'_'. $end .'_'. $who_text .'_'. $subject .'_'. $week_text;

		$out = wp_cache_get( $identifier );

		if ( false === $out ) {

			if ( false === $year_week_no || !$this->get( 'all', 'alt' ) )
				$wh = $this->get( 'all', $subject, $location );
			else
				$wh = $this->combine_alt( 'all', $subject, $year_week_no, $location );

			if ( 0 === ($end - $start) ) {
				$out = null;
			} else if ( 1 == ($end - $start) || -2015 == ($end - $start) ) {
				if ( isset( $wh[$start] ) && trim( $wh[$start] ) ) {
					$out = explode( ',', $wh[$start] );
				} else {
					$out = null;
				}
			} else {
				if ( $who ) {
					$ids = array( $who );
				} else {
					if ( 'worker' == $subject ) {
						$ids = (array)$this->a->get_worker_ids();
						$ids[] = $this->a->get_def_wid();
						$ids = array_unique( $ids );
					} else {
						$ids = $this->a->get_service_ids();
					}
				}

				if ( ($end - $start) < 0 ) {
					$end = $end + 2016;
				}

				$in = $out = array();

				for ( $k = $start; $k < $end; $k++ ) {
					$mod_k = wpb_mod( $k, 2016 );
					if ( isset( $wh[$mod_k] ) && trim($wh[$mod_k]) ) {
						$in[] = explode( ',', $wh[$mod_k] );
					} else {
						$out = 0;
					}
				}
				$in = array_filter( $in );

				if ( is_array( $out ) && sizeof( $in ) >= 2 ) {
					// Foreach is faster than array_reduce
					$acc = $ids;
					foreach ( $in as $a ) {
						$acc = array_intersect( $acc, $a );
					}
					$out = $acc;
				} else if ( empty( $in ) ) {
					$out = null; // Complete break for anyone of the subject
				}

				if ( null !== $out && $who ) {
					$found_one = false;
					foreach ( $in as $a ) {
						if ( in_array( $who, $a ) ) {
							$found_one = true;
							break;
						}
					}
					if ( !$found_one ) {
						$out = null; // Complete break for tested subject
					}
				}
			}

			wp_cache_set( $identifier, $out );
		}

		return $out;
	}

	/**
     * Return all turning points for a given service or worker
	 * A turning point is a value in minutes where subject starts working either at day start or after a break
	 * Example: 9am-1pm, 2pm-6pm work gives array( 540, 840 )
	 * @param $who			Integer			ID of subject
	 * @param $subject		string			'worker', 'service' or 'alt'
	 * @since 3.0
	 * @return array
     */
	public function find_turning_points( $who, $subject, $location ) {
		$avails = $this->get( $who, $subject, $location );

		$cells = preg_grep( '/(^'.$who.'$|^'.$who.',|,'.$who.',|,'.$who.'$)/', $avails );

		$net_cells = array_keys( array_filter( $cells ) );
		if ( empty( $net_cells ) || 2016 == count( $net_cells ) ) { // All working or all holiday
			return array();
		}

		$initial_el = array_shift( $net_cells );
		$initial 	= array( 0 => $initial_el, 1 => array($initial_el) );

		$res = array_reduce( $net_cells,
			function( $carry, $item ) {
				if ( $item > $carry[0] + 1 ) {
					$carry[1][] = $item;
				}
				$carry[0] = $item;
				return $carry;
			},
			$initial
		);

		$func1 = function( $value ) {
			return wpb_mod( $value, 288 );
		};

		$func2 = function( $value ) {
			return $value * 5;
		};

		return array_map( $func2, array_unique( array_map( $func1, $res[1] ) ) );
	}

	/**
     * Check if a certain time base value divides all wh table turning points
	 * @param $tb	integer		Time base value in minutes
	 * @since 3.0
	 * @return bool
     */
	public function maybe_divide( $tb ){

		$saw = array( 	'service'	=> $this->a->get_services(),
						'worker'	=> $this->a->get_workers(),
						'alt'		=> $this->get_schedules(),
					);

		foreach ( $saw as $what => $subjects ) {
			if ( empty( $subjects ) )
				continue;

			foreach ( $subjects as $subject ) {

				$subject_id = isset( $subject->ID ) ? $subject->ID : key($subject);
				$arr = $this->find_turning_points( $subject_id, $what, 0 );
				if ( empty( $arr ) )
					continue;

				foreach( $arr as $el ) {
					if ( 0 != wpb_mod( $el, $tb ) )
						return false;
				}
			}
		}

		return true;
	}

	/**
     * Find maximum time that a worker or service can work without interruption
	 * @param $who			Integer			ID of subject
	 * @param $subject		string			'worker', 'service' or 'alt'
	 * @since 3.5.6
	 * @return integer	in minutes
     */
	public function find_max_cont_wh( $who, $subject, $location ) {
		$avails = $this->get( $who, $subject, $location );

		$cells = preg_grep( '/(^'.$who.'$|^'.$who.',|,'.$who.',|,'.$who.'$)/', $avails );

		$net_cells = array_keys( array_filter( $cells ) );
		if ( empty( $net_cells ) || 2016 == count( $net_cells ) ) { // All working or all holiday
			return 0;
		}

		$max = 24*60;
		$first = current( $net_cells );

		foreach ( $net_cells as $i => $val ) {
			if ( !isset( $net_cells[$i-1] ) ) {
				continue;
			}

			if ( $val == $net_cells[$i-1] + 1 ) {
				continue;
			}

			$max = min( $max, ($net_cells[$i-1] + 1 - $first) );
			$first = $val;
		}

		return $max*5;
	}

	/**
     * Find working hour start and end times of a worker or unassigned worker
	 * This is used as an optimization by reducing looped time slots
	 * @param $slot		object	WpB_Slot object
	 * @return array (in minutes)
     */
	public function find_limits( $slot ) {
		$location	= $this->get_location( $slot );
		$service	= $slot->get_service();
		$worker		= $slot->get_worker();

		if ( $worker ) {
			$who = $worker;
			$subject = 'worker';
			$who_text = 'worker'. $worker;
		} else if ( $service ) {
			$who = $service;
			$subject = 'service';
			$who_text = 'service'. $service;
		} else {
			return array( 'first' => 0, 'last' => 24*60 );
		}

		$day_start		= strtotime( 'midnight', $slot->get_start() );
		$year_week_no	= date( 'Y', $day_start ) . wpb_time2week( $day_start );
		$identifier		= wpb_cache_prefix() . 'daily_limits_'. $who_text . '_' . $location . '_'. $day_start;
		$limits			= wp_cache_get( $identifier );

		if ( false === $limits ) {
			$no = $this->to( $day_start );

			if ( false === $year_week_no || !$this->get( 'all', 'alt', $location ) )
				$avails = $this->get( $who, $subject, $location );
			else
				$avails = $this->combine_alt( $who, $subject, $year_week_no, $location );

			$avails_day = array_slice( $avails, $no, 288 );

			$slots = preg_grep( '/(^'.$who.'$|^'.$who.',|,'.$who.',|,'.$who.'$)/', $avails_day );

			$first= max( 0, 5*key( $slots ) );
			end( $slots );
			$last = min( 60*24, 5*key( $slots ) );

			$limits = array( 'first' => $first, 'last' => $last );

			wp_cache_set( $identifier, $limits );
		}

		return $limits;
	}

	/**
     * Find available weekly wh cells of a worker or if worker_id=0, that of $service
	 * @param $day_start: Start timestamp of the day within the week to be retreived
	 * @return array or false
     */
	public function find_cells( $slot, $year_week_no ) {
		$location	= $this->get_location( $slot );
		$service	= $slot->get_service();
		$worker		= $slot->get_worker();

		if ( $worker ) {
			$who = $worker;
			$subject = 'worker';
			$who_text = 'worker'. $worker;
		} else if ( $service ) {
			$who = $service;
			$subject = 'service';
			$who_text = 'service'. $service;
		} else {
			return false;
		}

		$identifier = wpb_cache_prefix() . 'daily_slots_'. $who_text . '_' . $location . '_'. $year_week_no;
		$slots = wp_cache_get( $identifier );

		if ( false === $slots ) {

			if ( false === $year_week_no || !$this->get( 'all', 'alt' ) )
				$avails = $this->get( $who, $subject, $location );
			else
				$avails = $this->combine_alt( $who, $subject, $year_week_no, $location );

			$slots = preg_grep( '/(^'.$who.'$|^'.$who.',|,'.$who.',|,'.$who.'$)/', $avails );

			wp_cache_set( $identifier, $slots );
		}

		return $slots;
	}

	/**
     * Find number of available wh cells of a worker or unassigned worker for a given time
	 * and recurring $nof_days times at the same time of the day. There may be gaps in between.
	 * @param $day_start 	integer		Timestamp to be checked
	 * @param $nof_days		integer		Recurring for how many days?
	 * @param $limit		integer		Optional limit. If number of available slots reach this value, counting will be terminated
	 * @return integer
     */
	public function count_lateral_cells( $slot, $nof_days, $limit = null ) {
		$day_start	= $slot->get_start();
		$count		= $missed = 0;

		for ( $d = 0; $d < $nof_days; $d++ ) {
			$slot_start 	= $day_start + $d*DAY_IN_SECONDS;
			$no 			= $this->to( $slot_start );
			$year_week_no 	= date( 'Y', $slot_start) . wpb_time2week( $slot_start );
			$slots 			= $this->find_cells( $slot, $year_week_no );

			if ( isset( $slots[$no] ) )
				$count++;
			else
				$missed++;

			if ( $missed > $nof_days - $limit )
				break;
			if ( $limit && $count >= $limit )
				break;
		}

		return $count;
	}

	/**
     * Quick check if worker is available in ANY part of the day (acc to server)
	 * @param $slot: WpB_Slot object
	 * @return array or false
     */
	public function is_working_day( $slot ) {
		if ( wpb_ignore_bis_rep_wh() ) {
			return $this->is_working_day_for_service( $slot );
		}

		$location	= $this->get_location( $slot );
		$service	= $slot->get_service();
		$who		= $slot->get_worker();

		if ( !$who ){ $who = 'all'; }

		// Even workers on different timezone, server time is taken into account
		$day_start	= strtotime( date("Y-m-d", $slot->get_start() ) );
		$identifier = wpb_cache_prefix() . 'avails_by_day_'. $who . '_' . $location . '_'. $service .'_' . $day_start;
		$nof_avail	= wp_cache_get( $identifier );

		if ( false === $nof_avail ) {
			$year_week_no = date( 'Y', $day_start ) . wpb_time2week( $day_start );
			$no = $this->to( $day_start );

			if ( false === $year_week_no || !$this->get( 'all', 'alt', $location ) ) {
				$avails = $this->get( $who, 'worker', $location );
			} else {
				$avails = $this->combine_alt( 'all', 'worker', $year_week_no, $location );
			}

			$avails_day = array_slice( $avails, $no, 288 );
			$avails_day = array_filter( array_unique( explode( ',', implode( ',', $avails_day ) ) ) );

			if ( 'all' === $service ) {
				$workers_by_service = $this->a->get_worker_ids( );
			} else {
				$workers_by_service = $this->a->get_worker_ids_by_service( $service );
			}

			$nof_avail = count( array_intersect( (array)$workers_by_service, $avails_day ) );

			wp_cache_set( $identifier, $nof_avail );
		}

		return ( $nof_avail > 0 );
	}

	/**
     * Quick check if current service is available in ANY part of the day
	 * @param $slot		object		WpB_Slot object
	 * @return array or false
     */
	public function is_working_day_for_service( $slot ) {
		$location = $this->get_location( $slot );

		if ( !$who = $slot->get_service() ) {
			return false;
		}

		$day_start	= strtotime( date("Y-m-d", $slot->get_start() ) );
		$identifier = wpb_cache_prefix() . 'avails_by_day_for_service_'. $who . '_' . $location . '_'. $day_start;
		$nof_avail	= wp_cache_get( $identifier );

		if ( false === $nof_avail ) {
			$year_week_no = date( 'Y', $day_start ) . wpb_time2week( $day_start );
			$no = $this->to( $day_start );

			if ( false === $year_week_no || !$this->get( 'all', 'alt', $location ) ) {
				$avails = $this->get( $who, 'service' );
			} else {
				$avails = $this->combine_alt( $who, 'service', $year_week_no, $location );
			}

			$avails_day = array_slice( $avails, $no, 288 );
			$avails_day = array_filter( array_unique( explode( ',', implode( ',', $avails_day ) ) ) );
			$nof_avail	= count( $avails_day );

			wp_cache_set( $identifier, $nof_avail );
		}

		return $nof_avail;
	}

	/**
     * Check if subject is NOT working in given interval, except holidays
	 * @param $slot	object|array	WpB_Slot object or an array in the form array( 'start' => wh_start, 'end' => wh_end )
	 * @return bool
     */
	public function is_break( $slot, $who, $subject = 'worker'  ) {
		if ( 'worker' == $subject && $slot instanceof WpB_Slot && wpb_ignore_bis_rep_wh() ) {
			return $this->is_break( $slot, $slot->get_service(), 'service' );
		}

		$subjects = $this->available_subjects( $slot, $who, $subject );
		if ( !$subjects ) {
			if ( null === $subjects ) {
				return 5; # complete_brk: Break until end - can be used to skip the day for optimization
			} else {
				return 2; # no_workers
			}
		}

		return in_array( $who, $subjects ) ? false : 4; # Break
	}

	/**
     * DEPRECATED - Check if subject is NOT working in given interval in wh domain, except holidays
	 * @param start, end	Integer		Start and end wh:no values
	 * @until 3.4.5
	 * @return false
     */
	public function is_break_wh( $start, $end, $who, $subject, $year_week_no ) {
		if ( WpBDebug::is_debug() ) {
			trigger_error( __( 'is_break_wh method of WpBWH class is deprecated.', 'wp-base' ) );
		}

		return false;
	}

	/**
     * Draw working hours table
	 * @param wh_selected: an array of subject|who values
	 * @param subject: Worker, service or alternative
	 * @param bp: false or BuddyPress user ID
	 * @return string (html)
     */
	public function draw( $wh_selected, $bp = false ) {
		$r  = '';
		$r .= apply_filters( 'app_admin_wh_before', $r, $wh_selected );

		$r .= '<div class="postbox">';
		$r .= '<h3 class="hndle"><span>'. __('Working Hour Schedules', 'wp-base'). '</span></h3>';
		$r .= '<div class="inside" id="app-wh">';

		foreach( $wh_selected as $val ) {
			list( $subject_s, $who_s ) = explode( '|', $val );
			$r .= $this->draw_s( $who_s, $subject_s, '25', $bp );
		}

		$r .= "<div style='clear:both'></div>";

		$r .= "</div></div>";


		return $r;
	}

	/**
     * Draw working hours table helper
	 * @param who: ID of the subject
	 * @param subject: Worker, service or alternative
	 * @param width: Width of the calendar in%. If given, it also adds copy/paste buttons
	 * @return string
     */
	public function draw_s( $who, $subject = 'worker', $width = '', $bp = false ) {

		switch( $subject ) {
			case 'worker':	$pre_text = ($who == $this->a->get_def_wid()) ? __('Business Rep','wp-base') : $this->a->get_text('provider');
							$whose =  $pre_text .': '. $this->a->get_worker_name( $who );
							break;
			case 'service':	$whose = $this->a->get_text('service') .': '. $this->a->get_service_name( $who );
							break;
			case 'alt':		$whose = __('Schedule','wp-base') .': ' . ($this->get_schedule_name( $who )
							?: '<input type="hidden" name="year_week_no" value=""><input type="hidden" name="alt_for" value=""><span><input type="text" class="alt-name"></span>');
							break;
							break;
		}

		global $is_gecko;

		$r  = '';
		if ( $width ) {
			$r .= '<div class="app-wrap app-wrap-admin '.( $bp ? 'app-2col' : 'app-4col' ).( $is_gecko ? ' app-firefox' : '' ).'">';
			$r .= '<div class="app-title app-c">' . $whose . '</div>';
		} else if ( null === $width ) {
			$r .= '<div>';
		} else {
			$r .= '<div class="app-wrap">';
			$r .= '<div class="app-title app-c">' . $whose . '</div>';
		}

        $r .= '<div class="app-list">';
		$r .= '<div class="app-schedule-wrapper app-weekly-admin">';
		$r .= '<table class="app-wh app-wh-'.$subject.'|'.$who.' fixed" width="100%">';
		$r .= "<tbody>";
		$r .= $this->a->_get_table_meta_row('thead', null);

		$step 		= $this->a->get_min_time( 'wh' )/5;
		$days 		= wpb_arrange( array(0,1,2,3,4,5,6), -1, false );
		$copy_text 	= __('Copy to clipboard','wp-base');
		$vals 		= array();
		$cl_offset	= $this->a->get_client_offset( ); # TODO: DST?

		$no_offset 	= intval($cl_offset/300);

		$start_set	= wpb_setting( 'wh_starts', '00:00' );
		$end_set	= wpb_setting( 'wh_ends', '23:59' );
		$all_day	= 'all_day' == $start_set || 'all_day' == $end_set || ('00:00' == $start_set && '23:59' == $end_set) 
					  || ('00:00' == $start_set && '00:00' == $end_set) || intval( $start_set ) >= intval( $end_set );
		$starts		= date( 'Hi', strtotime( '1970-01-01 ' . ( $all_day ? '00:00' : $start_set ) ) );
		$ends		= date( 'Hi', strtotime( '1970-01-01 ' . ( $all_day ? '23:59' : $end_set ) ) );
		$kstart		= apply_filters( 'app_wh_kstart', 0, $who, $subject );
		$kend		= apply_filters( 'app_wh_kend', 288, $who, $subject );

		for ( $k = $kstart; $k < $kend; $k = $k + $step ) {
			$hide_row	= false;
			$hours_mins = date( 'Hi', $this->from( $k, 0 ) );

			if (  $hours_mins < $starts || $hours_mins >= $ends ) {
				$hide_row = true;
			}

			$cl_row = 'app_row'. (intval($k/$step) + 1);
			$r  .='<tr '.( $hide_row ? 'style="display:none"' : '').'>';

			foreach ( $days as $day ) {
				$no = wpb_mod( ( $day*288 + $k ), 2016 );
				if ( -1 == $day ) {
					$text = ($step == 288) ? __('All Day','wp-base') : date_i18n( $this->a->time_format, $this->from( wpb_mod($no, 2016) ) );
					$r .= "<td class='app-weekly-hours-mins ui-state-default ".$cl_row."'>".$text."</td>";
				} else {
					$no = $no - $no_offset;
					$cl_column = ' app_'.strtolower( date('l', $this->from( wpb_mod($no, 2016) ) ) );

					if ( $this->is_break(
						array( 'start' => wpb_mod($no, 2016), 'end' => wpb_mod($no, 2016)+$step, 'year_week_no' => false ),
						'new' == $who ? $this->a->get_def_wid() : $who,
						'new' == $who ? 'worker' : $subject
					) ) {
						$vals[$no] = 0;
						$r .= "<td class='notpossible app_select_wh ".$cl_row.$cl_column."'>
								<input type='hidden' class='app_wh_value' data-no='".wpb_mod($no, 2016). "' value='0' /></td>";
					} else {
						$vals[$no] = 1;
						$r .= "<td class='free app_select_wh ".$cl_row.$cl_column."'>
								<input type='hidden' class='app_wh_value' data-no='".wpb_mod($no, 2016). "' value='1' /></td>";
					}
				}
			}
			$r  .='</tr>';
		}

		for ( $no = 0; $no < 2016; $no++ ) {
			if ( !isset( $vals[$no] ) ) {
				$vals[$no] = $last_val;
			} else {
				$last_val = $vals[$no];
			}
		}
		ksort( $vals );

		$app_wh_vals = '';
		foreach( $vals as $no => $val ) {
			$app_wh_vals = $app_wh_vals . $val;
		}

		$r .= "<input type='hidden' name='app_wh[$subject][$who]' class='app_coded_val' value='$app_wh_vals' />";
		$r .= "</tbody></table>";
		if ( $width && -1 != $width ) {
			$r .= '<div class="app-mt app-c"><button data-status="copy_ready" class="app-copy-wh ui-button ui-state-default ui-corner-all ui-shadow">'.$copy_text.'</button></div>';
		}
		$r .= "</div></div></div>";
		$r .= '<input type="hidden" name="app_select_wh[]" class="app-select-wh-val" value="'.$subject.'|'.$who.'" />';

		return $r;
	}

	/**
	 * Get Alt schedules
	 * @return array
     */
	public function get_schedules(){
		return apply_filters( 'app_get_schedules', array() );
	}

	/**
	 * Get name of the schedule given ID
	 * @param $ID	integer		Schedule ID
	 * @return string
     */
	public function get_schedule_name( $ID ){
		$s = $this->get_schedules();
		return (isset( $s[$ID]['name'] ) ? stripslashes( $s[$ID]['name'] ) : '');
	}

/****************************************************
* Methods for admin
*****************************************************
*/

	/**
	 * Change admin SEO title
	 * @since 3.8.0
	 */
	public function admin_title( $admin_title, $title ) {
		if ( ! empty( $_GET['tab'] ) && key( self::$name ) == $_GET['tab'] ) {
			return str_replace( $title, current( self::$name ), $admin_title );
		} else {
			return $admin_title;
		}
	}
	
	/**
	 * Add tab
	 */
	public function add_tab( $tabs ) {
		if ( wpb_admin_access_check( 'manage_working_hours', false ) ) {
			$tabs = array_merge( $tabs, self::$name );
		}

		return $tabs;
	}

	/**
	 * Display HTML codes
	 * @param $profileuser_id: If this is called by a user from his profile
	 */
	public function render_tab( $profileuser_id = false, $bp = false ) {

		wpb_admin_access_check( ($profileuser_id ? 'manage_own_work_hours' : 'manage_working_hours') );
?>
<div id="poststuff" class="metabox-holder">
		<?php
		wpb_desc_infobox( 'wh' );

		$workers 			= $this->a->get_workers();
		$def_worker 		= $this->a->get_def_wid();
		$def_worker_name 	= $this->a->get_worker_name( $def_worker, false );
		$incl_def_worker	= current_user_can( WPB_ADMIN_CAP ) || ( $profileuser_id == $def_worker );

		if ( isset($_POST['app_select_wh']) && is_array( $_POST['app_select_wh'] ) ) {
			$wh_selected = wpb_clean( $_POST['app_select_wh'] );
		} else if ( !empty( $_GET['app_select_whs'] ) ) {
			$wh_selected = explode( ',', wpb_clean( $_GET['app_select_whs'] ) );
		} else if ( $incl_def_worker ) {
			$wh_selected = array( 'worker|'.$def_worker );
		} else {
			$wh_selected = array();
		}

		if ( $incl_def_worker ) {
			$wh_selected[] 	= 'worker|'.$def_worker;
		}

		$wh_selected = apply_filters( 'app_wh_default_selected', array_unique( $wh_selected ), $profileuser_id );

	?>
	<div class="postbox app-prpl">
		<div class="app-submit app-2col">
			<div class="app-mt">
				<form class="app-flex" method="post">
					<span class="app-list-for"><?php _e('List for', 'wp-base')?></span>
					<select multiple data-buttonwidth="250" data-noneselectedtext="<?php _e( 'Select service/provider', 'wp-base' ) ?>" class="app_ms" name="app_select_wh[]" size="10">
					<?php if ( class_exists( 'WpBSP' ) ) { ?>
						<optgroup label="<?php _e('Service Providers','wp-base') ?>" class="optgroup_worker">
					<?php }
						if ( !in_array( $def_worker, (array)$this->a->get_worker_ids() ) && $incl_def_worker ) {
					?>
						<option value="worker|<?php echo $def_worker ?>"><?php printf( __('Business Rep. (%s)', 'wp-base'), $def_worker_name) ?></option>
					<?php
						}
						if ( $workers ) {
							if ( $profileuser_id ) {
								$s = in_array( 'worker|'.$profileuser_id, $wh_selected ) ? " selected='selected'" : '';
								echo '<option value="worker|'.$profileuser_id.'"'.$s.'>' . $this->a->get_worker_name( $profileuser_id, false ) . '</option>';
							} else {
								foreach ( $workers as $worker ) {
									$s = in_array( 'worker|'.$worker->ID, $wh_selected ) ? " selected='selected'" : '';
									echo '<option value="worker|'.$worker->ID.'"'.$s.'>' . $this->a->get_worker_name( $worker->ID, false ) . '</option>';
								}
							}
						}
						if ( class_exists( 'WpBSP' ) ) { ?>
						</optgroup>
						<?php }	 ?>

						<optgroup label="<?php _e('Services','wp-base') ?>" class="optgroup_service">
						<?php
						if ( $profileuser_id ) {
							$services = apply_filters( 'app_wh_get_services_owned_by', $this->a->get_services_owned_by( $profileuser_id, -1 ), $profileuser_id );	
						} else {
							$services = $this->a->get_services();
						}

						foreach ( (array)$services as $service ) {
							$s = in_array( 'service|'. $service->ID, $wh_selected ) ? " selected='selected'" : '';
							echo '<option value="service|'.$service->ID.'"'. $s.'>' . $this->a->get_service_name( $service->ID ) . '</option>';
						}
						?>
						</optgroup>

						<?php if ( class_exists( 'WpBAnnual' ) && !$profileuser_id ) { ?>
						<optgroup label="<?php _e('Custom Schedules','wp-base') ?>" class="optgroup_alt">
						<?php
						foreach ( (array)$this->get_schedules() as $ID => $schedule ) {
							$s = in_array( 'alt|'. $ID, $wh_selected ) ? " selected='selected'" : '';
							echo '<option value="alt|'.$ID.'"'. $s .'>' . stripslashes( $schedule["name"] ) . '</option>';
						}
						?>
						</optgroup>
						<?php } ?>
					</select>
					<?php do_action( 'app_wh_admin_list_for' ) ?>
					<button id="app_sel_wh_options_btn" class="ui-widget ui-button ui-state-default ui-corner-all ui-shadow app-ml10"><?php _e('Show','wp-base') ?></button>
				</form>
			</div>
		</div>
		<div class="app-submit app-4col app-flex">
		<?php do_action( 'app_wh_admin_submit_mid' ) ?>
		</div>
		<div class="app-submit app-4col app-flex">
		<?php do_action( 'app_wh_admin_submit' ) ?>
		</div>
		<div style="clear:both"></div>
	</div>

	<form class="app-form" method="post"  action="<?php echo wpb_add_query_arg( null, null )?>">

		<p class="submit">
			<input type="submit" class="button-primary" value="<?php echo __('Save Working Hours', 'wp-base') ?>" />
		</p>

		<div id="tabs" class="app-tabs">
			<ul></ul>
			<?php
				$bp_user_id = $bp ? $profileuser_id : false;
				echo $this->draw( $wh_selected, $bp );

				$disabled = $this->a->get_nof_workers() ? '' : '';

				if ( !$profileuser_id ) {
			?>

			<div class="postbox">
			<h3 class="hndle"><span><?php echo __('Advanced', 'wp-base') ?></span></h3>
				<div class="inside">
					<table class="form-table">
						<?php wpb_setting_yn( 'ignore_bis_rep_wh' ) ?>
						<?php wpb_setting_yn( 'service_wh_check' ) ?>
						<?php wpb_setting_yn( 'service_wh_covers' ) ?>
						<tr>
							<th scope="row" ><?php WpBConstant::echo_setting_name('wh_starts') ?></th>
							<td>
								<select name="wh_starts">
								<?php echo wpb_time_selection( wpb_setting( 'wh_starts' ) ) ?>
								</select>
								<span class="description app-btm"><?php WpBConstant::echo_setting_desc('wh_starts') ?></span>
							</td>
						</tr>
						<tr>
							<th scope="row" ><?php WpBConstant::echo_setting_name('wh_ends') ?></th>
							<td>
								<select name="wh_ends">
								<?php echo wpb_time_selection( wpb_setting( 'wh_ends' ) ) ?>
								</select>
								<span class="description app-btm"><?php WpBConstant::echo_setting_desc('wh_ends') ?></span>
							</td>
						</tr>
					</table>
				</div>
			</div>
			<?php } ?>
		</div><!-- Tabs -->

		<p class="submit">
			<input type="hidden" name="app_bp_settings_user" value="<?php echo $bp_user_id ?>" />
			<input type="hidden" name="app_location_id" value="<?php echo $this->get_location() ?>" />
			<input type="hidden" name="action_app" value="save_working_hours" />
			<input type="hidden" name="app_nonce" value="<?php echo wp_create_nonce( 'update_app_settings' ) ?>" />
			<input type="submit" class="button-primary" value="<?php echo  __('Save Working Hours', 'wp-base') ?>" />
		</p>
	</form>
</div>
	<?php
	}

	/**
     * Remove service from wh table
	 * @param $ID		integer		ID of the service
	 * @return bool
     */
	public function remove_service( $ID ) {
		$this->remove( $ID, 'service' );
	}

	/**
     * Remove worker from wh table
	 * @param $ID		integer		ID of the worker
	 * @return bool
     */
	public function remove_worker( $ID ) {
		$this->remove( $ID, 'worker' );
	}

	/**
     * Remove a certain element from wh table
	 * @param $who		integer		ID of the subject
	 * @param $subject	string		'worker', 'service' or 'alt'
	 * @return bool
     */
	public function remove( $who, $subject ) {
		switch( $subject ) {
			case 'worker':	$table = $this->a->wh_w_table; break;
			case 'service':	$table = $this->a->wh_s_table; break;
			case 'alt':		$table = $this->a->wh_a_table; break;
		}

		$result = false;
		$new_vals = array();
		foreach ( $this->get( 'all', $subject ) as $key => $val ) {
			$new_vals[$key]	= $this->remove_item( $val, $who );
		}
		ksort( $new_vals );

		$result1 = $this->pack_n_save( $new_vals, $table );

		// Also remove holidays and annual schedules
		$options = $this->a->get_business_options();

		if ( isset( $options['holidays'][$subject] ) ) {
			foreach ( (array)$options['holidays'][$subject] as $who_id => $val ) {
				if ( $who_id == $who )
					unset( $options['holidays'][$subject][$who] );
			}
		}
		if ( isset( $options['alt_schedule_pref'][$subject] ) ) {
			foreach ( (array)$options['alt_schedule_pref'][$subject] as $year_week_no => $who_id ) {
				if ( $who_id == $who )
					unset( $options['alt_schedule_pref'][$subject][$year_week_no][$who] );
			}
		}

		$result2 = $this->a->update_business_options( $options );

		if ( $result1 || $result2 ) {
			wpb_flush_cache();
			return true;
		}

		return false;
	}

	/**
     * Replace a certain element with another element at wh table + Holidays + Alts
	 * @param $who			integer		Find (worker id, service id, alt id)
	 * @param $with_whom	integer		Replace (worker id, service id, alt id)
	 * @return bool
     */
	public function replace( $who, $with_whom, $subject ) {
		switch( $subject ) {
			case 'worker':	$table = $this->a->wh_w_table; break;
			case 'service':	$table = $this->a->wh_s_table; break;
			case 'alt':		$table = $this->a->wh_a_table; break;
		}

		$result = false;
		$new_vals = array();
		foreach ( $this->get( 'all', $subject ) as $key => $val ) {
			if ( in_array( $who, explode( ',', $val ) ) ) {
				$new_vals[$key]	= $this->remove_item( $val, $who );
				$new_vals[$key]	= $this->add_item( $val, $with_whom );
			} else {
				$new_vals[$key] = $val;
			}
		}
		ksort( $new_vals );

		$result1 = $this->pack_n_save( $new_vals, $table );

		// Also remove holidays and annual schedules
		$options = $this->a->get_business_options();

		if ( isset( $options['holidays'][$subject] ) ) {
			foreach ( (array)$options['holidays'][$subject] as $who_id => $val ) {
				if ( $who_id == $who && isset( $options['holidays'][$subject][$who] ) ) {
					$options['holidays'][$subject][$with_whom] = $options['holidays'][$subject][$who];
					unset( $options['holidays'][$subject][$who] );
				}
			}
		}

		if ( isset( $options['alt_schedule_pref'][$subject] ) ) {
			foreach ( (array)$options['alt_schedule_pref'][$subject] as $year_week_no => $who_id ) {
				if ( $who_id == $who && isset($options['alt_schedule_pref'][$subject][$year_week_no][$who]) ) {
					$options['alt_schedule_pref'][$subject][$year_week_no][$with_whom] = $options['alt_schedule_pref'][$subject][$year_week_no][$who];
					unset( $options['alt_schedule_pref'][$subject][$year_week_no][$who] );
				}
			}
		}

		$result2 = $this->a->update_business_options( $options );

		if ( $result1 || $result2 ) {
			wpb_flush_cache();
			return true;
		}

		return false;
	}

	/**
     * Save Wh settings: Decode submit value, pack all subject items together, encode and save
     */
	public function save_wh( $profileuser_id = false ) {

		if ( isset( $_POST['app_nonce'] ) && !wp_verify_nonce( $_POST['app_nonce'],'update_app_settings') ) {
			wpb_notice( 'unauthorised', 'error' );
			return;
		}

		if ( empty( $_POST['action_app'] ) || ( 'save_working_hours' != $_POST["action_app"] && 'save_annual' != $_POST["action_app"] ) || ! isset( $_POST['app_wh'] ) || ! is_array( $_POST['app_wh'] ) ) {
			return;
		}

		$result 	= $result2 = false;
		$wh 		= $updated_workers = array();
		$options 	= wpb_setting();
		$location	= $this->get_location();

		if ( isset( $_POST['service_wh_covers'] ) )
			$options['service_wh_covers']			= wpb_clean( $_POST['service_wh_covers'] );
		if ( isset( $_POST['service_wh_check'] ) )
			$options['service_wh_check']			= wpb_clean( $_POST['service_wh_check'] );
		if ( isset( $_POST['ignore_bis_rep_wh'] ) )
			$options['ignore_bis_rep_wh']			= wpb_clean( $_POST['ignore_bis_rep_wh'] );
		if ( isset( $_POST['wh_starts'] ) )
			$options['wh_starts']					= wpb_clean( $_POST['wh_starts'] );
		if ( isset( $_POST['wh_ends'] ) )
			$options['wh_ends']						= wpb_clean( $_POST['wh_ends'] );

		# Check if there is any setting less than wh_starts or greater than wh_ends
		$this->check_start( $options['wh_starts'] );
		$this->check_end( $options['wh_ends'] );

		if ( $this->a->update_options( $options ) ) {
			$result2 = true;
		}

		// Save sort order
		$sort = '';
		foreach( $_POST['app_wh'] as $subject => $who_data ) {
			foreach( $who_data as $who => $vals ) {
				$sort .= $subject.'|'.$who.',';
			}
		}

		foreach( array('worker','service','alt') as $subject ) {
			if ( empty( $_POST['app_wh'][$subject] ) ) {
				continue;
			}

			switch( $subject ) {
				case 'worker':	$table = $this->a->wh_w_table; break;
				case 'service':	$table = $this->a->wh_s_table; break;
				case 'alt':		$table = $this->a->wh_a_table; break;
			}

			$wh[$subject] = $this->get( 'all', $subject, $location );

			foreach( $_POST['app_wh'][$subject] as $who => $coded_val ) {

				$vals = str_split( wpb_clean( $coded_val ) );

				// Combine vals with wh
				$new_vals = array();
				foreach ( $vals as $key => $val ) {
					$wh_val = isset( $wh[$subject][$key] ) ? $wh[$subject][$key] : '';
					if ( $val )
						$new_vals[$key]	= $this->add_item( $wh_val, $who );
					else if ( $wh_val )
						$new_vals[$key]	= $this->remove_item( $wh_val, $who );
					else
						$new_vals[$key] = '';
				}

				$wh[$subject] = $new_vals;
				if ( 'worker' == $subject )
					$updated_workers[]= $who;
			}

			ksort( $new_vals );

			if ( $this->pack_n_save( $new_vals, $table ) ) {
				wpb_flush_cache();
				$result = true;
			}
		}

		if ( $result ) {
			wpb_flush_cache();

			if ( $this->a->get_nof_workers() && !empty( $updated_workers ) && 'no' != $options["service_wh_covers"] ) {
				$wh_services = $this->get( 'all','service', $location );
				foreach ( $updated_workers as $w_id ) {
					$serv = $this->a->get_services_by_worker( $w_id );
					if ( empty( $serv ) )
						continue;

					$serv_ids = implode( ',', array_keys( $serv ) );
					$wh_worker = $this->get( $w_id, 'worker', $location );
					foreach( $wh_worker as $key => $val ) {
						if ( !$val )
							continue;

						$wh_services_val = isset( $wh_services[$key] ) ? $wh_services[$key] : '';
						$wh_services[$key]= $this->add_item( $wh_services_val, $serv_ids );
					}
				}
				$this->pack_n_save( $wh_services, $this->a->wh_s_table );
				wpb_flush_cache();
			}
		}

		if ( $result || $result2 ) {
			wpb_notice( 'saved' );
			do_action( 'app_wh_saved', $location, $profileuser_id );
		}
	}

	/**
     * Check if any wh setting starts earlier than wh_starts setting
	 * @param $value	string		Current wh_starts value
	 * @since 3.6.2
	 * @return none
     */
	public function check_start( $value ){
		
		if ( ! is_admin() ) {
			return;
		}

		if ( 'all_day' == $value || '00:00' == $value ) {
			return;
		}

		list( $hours, $mins ) = explode( ':', $value );
		$mins = $hours *60 + $mins;
		if ( !$mins ) {
			return;
		}

		$saw = array( 	'service'	=> $this->a->get_services(),
						'worker'	=> $this->a->get_workers(),
						'alt'		=> $this->get_schedules(),
					);

		foreach ( $saw as $what => $subjects ) {
			if ( empty( $subjects ) ) {
				continue;
			}

			foreach ( $subjects as $subject ) {

				$subject_id = isset( $subject->ID ) ? $subject->ID : key( $subject );
				$arr = $this->find_turning_points( $subject_id, $what, 0 );
				if ( empty( $arr ) ) {
					continue;
				}

				$min = min( $arr );

				if ( $min < $mins ) {
					wpb_notice(
						sprintf( __( 'It looks like %1$s is set as %2$s, however minimum value for %3$s is %4$s. Note that this %5$s will start working at %4$s, NOT at %2$s. If you are aware of this fact, you can ignore this message.', 'wp-base' ),
							WpBConstant::get_setting_name( 'wh_starts' ),
							wpb_secs2hours( $mins * 60 ),
							$this->translate( $what, $subject_id ),
							wpb_secs2hours( $min * 60 ),
							$this->translate( $what )
					), 'error' );
					return;
				}
			}
		}
	}

	/**
     * Check if any wh setting ends later than wh_ends setting
	 * @param $value	string		Current wh_ends value
	 * @since 3.6.2
	 * @return none
     */
	public function check_end( $value ){

		if ( ! is_admin() ) {
			return;
		}

		if ( 'all_day' == $value || '00:00' == $value ) {
			return;
		}

		list( $hours, $mins ) = explode( ':', $value );
		$mins = $hours *60 + $mins;
		if ( !$mins ) {
			return;
		}

		$saw = array( 	'service'	=> $this->a->get_services(),
						'worker'	=> $this->a->get_workers(),
						'alt'		=> $this->get_schedules(),
					);

		foreach ( $saw as $what => $subjects ) {
			if ( empty( $subjects ) ) {
				continue;
			}

			foreach ( $subjects as $subject ) {

				$subject_id = isset( $subject->ID ) ? $subject->ID : key( $subject );
				$arr = $this->find_turning_points( $subject_id, $what, 0 );
				if ( empty( $arr ) ) {
					continue;
				}

				$max = max( $arr );

				if ( $max > $mins ) {
					wpb_notice(
						sprintf( __( 'It looks like %1$s is set as %2$s, however maximum value for %3$s is %4$s. Note that this %5$s will end working at %4$s, NOT at %2$s. If you are aware of this fact, you can ignore this message.', 'wp-base' ),
							WpBConstant::get_setting_name( 'wh_ends' ),
							wpb_secs2hours( $mins * 60 ),
							$this->translate( $what, $subject_id ),
							wpb_secs2hours( $max * 60 ),
							$this->translate( $what )
					), 'error' );
					return;
				}
			}
		}
	}

	/**
     * Make a variable translatable
	 * @param $context		string		Variable type
	 * @since 3.6.2
	 * @return string
     */
	private function translate( $context, $who = null ) {
		switch( $context ) {
			case 'service':		return $this->a->get_text( 'service' ) . ($who ? ': '. $this->a->get_service_name( $who ) : '');
			case 'worker':		$pre_text = ($who == $this->a->get_def_wid()) ? __('Business Rep','wp-base') : $this->a->get_text('provider');
								return $pre_text . ($who ? ': '. $this->a->get_worker_name( $who ) : '');
			case 'alt':			return __( 'Custom Schedule', 'wp-base' ) . ($who ? ': '. $this->get_schedule_name( $who ) : '');
		}
	}

	/**
     * Add a new value to a comma delimited string
	 * @param $wh_val	string		Comma delimited string
	 * @param $who		integer		ID of subject
	 * @return string				Comma delimited string
     */
	public function add_item( $wh_val, $who ) {
		return wpb_sanitize_commas( $wh_val .','. $who, true );
	}

	/**
     * Remove a value from a comma delimited string
	 * @param $wh_val	string		Comma delimited string
	 * @param $who		integer		Subject Id to be removed
	 * @return string				Comma delimited string
     */
	public function remove_item( $wh_val, $who ) {
		$arr = explode( ',', $wh_val );
		$arr = array_flip( $arr );
		unset( $arr[$who] );
		return implode( ',', array_flip( $arr ) );
	}

	/**
     * Pack a 2016-element $new_vals array into 288 and actually save
	 * Save data in an easily usable way (not easily readable, though)
	 * @param	$new_vals		array			Encoded working hours array
	 * @param	$table			string			mySQL table name
	 * @param	$loc_id			integer|null	Optional location ID to write to
	 * @return integer|bool		Insert query result
     */
	 public function pack_n_save( $new_vals, $table, $loc_id = null ) {
		$location = $loc_id === null ? $this->get_location() : $loc_id;

		$col_vals = array();
		foreach ( range( 0, 6 ) as $day ) {
			for ( $k = 0; $k < 288; $k++ ) {
				$no = $day*288 + $k;
				$new_val = isset( $new_vals[$no] ) ? $new_vals[$no] : '';
				$col_val = isset( $col_vals[$k] ) ? $col_vals[$k] : '';
				$col_vals[$k] = $col_val . $new_val .'|';
			}
		}
		unset( $new_vals );
		ksort( $col_vals );

		if ( defined( 'WPB_USE_INNODB' ) && WPB_USE_INNODB ) {
			return $this->do_save_inno( $col_vals, $table, $location );
		} else {
			return $this->do_save( $col_vals, $table, $location );
		}
	 }

	/**
     * Save encoded data in DB with engine InnoDB
	 * @param	$col_vals		array			Encoded working hours array
	 * @param	$table			string			mySQL table name
	 * @param	$location		integer			Optional location ID to write to
	 * @since 3.7.5
	 * @return integer|bool		Insert query result
     */
	public function do_save_inno( $col_vals, $table, $location ) {

		$avail_data = array();

		foreach ( $col_vals as $key => $val ) {

			if ( ! str_replace( '|', '', $val ) ) {
				$val = '';
			}

			$avail_data[ 'c'.$key ] = $val;
		}

		unset( $col_vals );

		$avail_data_json = wp_json_encode( $avail_data );

		return $this->a->db->query( "INSERT INTO ". $table . " (ID,avail_data) VALUES ($location,'$avail_data_json')
			ON DUPLICATE KEY UPDATE avail_data = '$avail_data_json' ");
	}

	/**
     * Save encoded data in DB with engine MyISAM
	 * @param	$col_vals		array			Encoded working hours array
	 * @param	$table			string			mySQL table name
	 * @param	$location		integer			Optional location ID to write to
	 * @since 3.7.5
	 * @return integer|bool		Insert query result
     */
	public function do_save( $col_vals, $table, $location ) {

		$columns = $values = $update = '';

		foreach ( $col_vals as $key => $val ) {
			if ( ! str_replace( '|', '', $val ) ) {
				$val = '';
			}

			$columns	.= '`c'.( $key ).'`,';
			$values		.= "'".esc_sql( $val )."',";
			$update		.= "`c".$key."`='".esc_sql( $val )."',";
		}

		unset( $col_vals );

		$location	= absint( $location );
		$columns	= rtrim( $columns, ',' );
		$values		= rtrim( $values, ',' );
		$update		= rtrim( $update, ',');

		return $this->a->db->query( "INSERT INTO ". $table . " (ID,$columns) VALUES ($location,$values)
			ON DUPLICATE KEY UPDATE $update ");
	}

}

	BASE('WH')->add_hooks();
}
