<?php
/**
 * WPB Users
 *
 * Methods for Clients (WP Users + Non logged in users)
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBUser' ) ) {

class WpBUser {

	/**
     * WP BASE instance
     */
	protected $a = null;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}

	public function add_hooks(){
		add_action( 'app_save_settings', array( $this, 'save_settings' ), 12 );
		add_action( 'admin_menu', array( $this, 'admin_menu' ) );
		add_action( 'deleted_user', array( $this, 'user_deleted' ), 12, 2 );
		add_filter( 'app_pre_confirmation_reply', array( $this, 'pre_confirmation_reply' ), 10, 3 );	// Control visibility of fields
		add_action( 'app_new_appointment', array( $this, 'save_meta' ), 10, 3 );						// Write user data to DB
		add_filter( 'app_edit_maybe_updated', array( $this, 'maybe_save_meta' ), 10, 3 );				// Write user data to DB for FEE
		add_filter( 'appointments_tabs', array( $this, 'add_tab' ), 16 ); 								// Add tab
		add_action( 'app_user_tab', array( $this, 'settings' ) );										// Display HTML settings on Business Settings
		add_action( 'add_user_role', array( $this, 'add_user_role' ), 10, 2 );

		add_action( 'app_exim_db_imported', array( $this, 'check_worker_roles' ) );
		add_action( 'app_db_reset', array( $this, 'check_worker_roles' ) );
	}

	/**
	 * Remove/add roles for workers who are changed in bulk
	 * @since 3.7.7
	 */
	public function check_worker_roles() {

		$worker_ids = array();

		foreach( $this->a->get_workers() as $worker ) {
			$user = new WP_User( $worker->ID );
			$user->add_role( 'wpb_worker' );
			$worker_ids[] = $worker->ID;
		}

		foreach( (array)get_users( array( 'role'=> 'wpb_worker' ) ) as $user ) {
			if ( in_array( $user->ID, $worker_ids ) ) {
				continue;
			}

			$user->remove_role( 'wpb_worker' );
		}
	}

	/**
	 * Find client phone given his appointment
	 * @param app_id: Appointment ID
	 * @since 2.0
	 * @return string
	 */
	public function get_client_phone( $app_id ) {
		if ( ! $phone = wpb_get_app_meta( $app_id, 'phone' ) ) {

			$app = wpb_get_app( $app_id );

			if ( ! empty( $app->user ) ) {
				$phone = get_user_meta( $app->user, 'app_phone', true );
			}
		}

		return apply_filters( 'app_client_phone', $this->add_dial_code( $phone ), $app_id );
	}

	/**
	 * Find client phone given his ID
	 * @param $ID: user ID
	 * @since 2.0
	 * @return string
	 */
	public function get_worker_phone( $ID ) {
		$phone = get_user_meta( $ID, 'app_phone', true );
		return apply_filters( 'app_worker_phone', $this->add_dial_code( $phone ), $ID );
	}

	/**
     * Get admin phones or first admin phone
	 * @since 2.0
	 * @return mixed	array ($first:false) or string ($first:true)
     */
	public function get_admin_phone( $first = false ) {
		$admin_phones = array();

		if ( $_phone = wpb_setting('admin_phone') ) {
			$temp = array_filter( array_map( 'trim', explode( ',', $_phone ) ) );
			foreach( $temp as $phone ) {
				$admin_phones[] = $this->add_dial_code( $phone );
			}
		}

		if ( empty( $admin_phones ) ) {
			$admin_email = get_option('admin_email');
			if ( $admin_email ) {
				$user = get_user_by( 'email', $admin_email );
				if ( ! empty( $user->ID ) && $phone = get_user_meta( $user->ID, 'app_phone', true ) ) {
					$admin_phones[] = $this->add_dial_code( $phone );
				}
			}
		}

		$admin_phones = apply_filters( 'app_get_admin_phones', $admin_phones );

		if ( $first ) {
			return apply_filters( 'app_get_admin_phone', current( $admin_phones ) );
		} else {
			return $admin_phones;
		}
	}

	/**
	 * Add the default dialing code to a phone when required
	 * @since 2.0
	 * @return string
	 */
	public function add_dial_code( $phone ) {
		if ( ! $phone ) {
			return $phone;
		}

		$phone = preg_replace( '/[^0-9]\+/', '', $phone );

		if ( substr( $phone, 0, 1 ) == '+' ) {
			$phone = '+' . str_replace( '+', '', $phone );
		} else if ( substr( $phone, 0, 2 ) == '00' ) {
			$phone = '+' . str_replace( '+', '', ltrim( $phone, '0' ) );
		} else {
			$phone = '+' . preg_replace( '/[^0-9]/', '', wpb_setting( 'phone_code' ) ) . str_replace( '+', '', ltrim( $phone, '0' ) );
		}

		return $phone;
	}

	/**
	 * Check if current user has one of the capabilities
	 * @param $cap 	array|string	List of required capabilities separated by comma or array. "none" also returns true
	 * @since 2.0
	 */
	public function is_capable( $cap ) {

		$is_capable = false;
		$caps = is_array( $cap ) ? $cap : explode( ",", $cap );

		foreach ( $caps as $c ) {
			if ( 'none' == strtolower( $c ) || current_user_can( $c ) ) {
				$is_capable = true;
				break;
			}
		}

		return $is_capable;
	}

	/**
	 * Check if user is a WP BASE Client
	 * @since 3.7.8
	 * @return bool
	 */
	public function is_client( $user_id = null ) {
		$user_id	= $user_id ?: get_current_user_id();
		$user		= new WP_User( $user_id );

		return $user->has_cap( 'wpb_client' );
	}

	/**
	 * Override WP capability check (Only for debug texts)
	 * @param cap: Capability
	 * @since 2.0
	 * @return bool
	 */
	public function _current_user_can( $cap ) {
		return apply_filters( 'app_current_user_can', current_user_can( $cap ) );
	}

	/**
	 * Return admin email(s)
	 * since 1.2.7
	 * @return mixed	returns string (the first email) if $first is true, returns array if $first is false
	 */
	public function get_admin_email( $first = false ) {
		$admin_emails = array();

		if ( wpb_setting('admin_email') ) {
			$temp = explode( ',', wpb_setting('admin_email') );
			foreach( $temp as $email ) {
				$e = trim( $email );
				if ( is_email( $e ) ) {
					$admin_emails[] = $e;
				}
			}
		}

		if ( empty( $admin_emails ) ) {
			global $current_site;
			$admin_email = get_option('admin_email');
			if ( $admin_email ) {
				$admin_emails[] = $admin_email;
			} else {
				$admin_emails[] = 'admin@' . $current_site->domain;
			}
		}

		$admin_emails = apply_filters( 'app_get_admin_emails', $admin_emails );
		if ( $first ) {
			return apply_filters( 'app_get_admin_email', current( $admin_emails ) );
		} else {
			return $admin_emails;
		}
	}

	/**
	 * Find client name given his appointment, together with a link to his profile page if current user can edit users
	 * @param $r			object		Appointment object
	 * @param $add_link		bool		Add link of the user. Disable this in emails
	 * @param $limit		integer 	Character limit for name
	 * @param $prime_cache	bool		Also load 100 subsequent apps to the cache
	 * @return string
	 */
	public function get_client_name( $app_id, $r = null, $add_link = true, $limit = 22, $prime_cache = false ) {
		$name	= '';
		$limit	= apply_filters( 'app_chr_limit', $limit );
		$result = (null !== $r) ? $r : wpb_get_app( $app_id, $prime_cache );

		if ( $result ) {
			// Client can be a user
			if ( $result->user ) {
				$name = $this->get_name( $result->user  );
			}

			if ( ! $name ) {
				$name = wpb_get_app_meta( $app_id, 'name' );
			}

			if ( ! $name ) {
				$name = trim( wpb_get_app_meta( $app_id, 'first_name' ) . ' '. wpb_get_app_meta( $app_id, 'last_name' ) );
			}

			if ( ! trim( $name ) && $maybe_email = wpb_get_app_meta( $app_id, 'email' ) ) {
				$name = $maybe_email;
			}

			$name = stripslashes( $name );

			$full_name = $name;

			if ( $limit && strlen($name) > $limit + 3 ) {
				$name = mb_substr( $name, 0, $limit, 'UTF-8' ) . '...';
			}

			# Also add other details of client
			$userdata = array_merge(
				array_filter( $this->get_app_userdata( 0, $result->user ) ),
				array_filter( $this->get_app_userdata( $app_id, 0, $prime_cache ) )
			);

			$tt = $full_name;

			if ( isset($userdata['email']) ) {
				$tt = $tt . WPB_TT_SEPARATOR . $userdata['email'];
			}

			if ( isset($userdata['phone']) ) {
				$tt = $tt . WPB_TT_SEPARATOR . $userdata['phone'];
			}

			$tt = apply_filters( 'app_client_name_tooltip', $tt, $result );

			if ( $name && $add_link ) {
				if ( current_user_can( 'edit_users' ) && $result->user ) {
					$name = '<a href="'. admin_url("user-edit.php?user_id="). $result->user . '" target="_blank" title="'.$tt.'">'. $name . '</a>';
				} else if ( $tt ) {
					$name = '<abbr title="'.$tt.'">'. $name . '</abbr>';
				}
			}
		}

		# Fallback
		if ( ! $name ) {
			$name = wp_unslash( $this->a->get_text( ! empty( $r->status ) && 'reserved' == $r->status ? 'gcal' : 'client' ) );
		}

		return apply_filters( 'app_get_client_name', $name, $app_id, $r );
	}

	/**
	 * Find user name given his ID. User does not need to be a client
	 * User names are called from user cache. If not in cache, user cache is updated for that user.
	 * @return string
	 * @since 2.0
	 */
	public function get_name( $user_id = 0 ) {
		$user_id = $user_id ?: get_current_user_id();

		$name = '';

		$name_meta = get_user_meta( $user_id, 'app_name', true );
		if ( $name_meta ) {
			$name = $name_meta;
		} else {
			$user_info = $this->_get_userdata( $user_id );
			if ( is_object( $user_info ) ) {
				if ( $user_info->display_name ) {
					$name = $user_info->display_name;
				} else if ( $user_info->user_nicename ) {
					$name = $user_info->user_nicename;
				} else if ( $user_info->user_login ) {
					$name = $user_info->user_login;
				}
			}
		}

		$name = apply_filters( 'app_get_user_name', $name, $user_id );

		return stripslashes( $name );
	}

	/**
	 * Find client email given his appointment ID with email check
	 * @param r: Appointment object
	 * @return string
	 */
	public function get_client_email( $app_id ) {
		$userdata = $this->get_app_userdata( $app_id );
		if ( ! empty( $userdata['email'] ) ) {
			if ( is_email( $userdata['email'] ) ) {
				return $userdata['email'];
			}
		}

		return '';
	}

	/**
	 * Get userdata from cache
	 * @return false if not found, WP user object if found
	 * @since 2.0
	 */
	public function _get_userdata( $user_id ) {

		$userdata = wp_cache_get($user_id, 'users', false, $found);

		if ( ! $found ) {
			$userdata = get_user_by( 'id', $user_id );
			wp_cache_set($user_id, $userdata, 'users');
		}

		return $userdata;
	}

	/**
	 * Find user data given his appointment or user ID
	 * @param app_id: App ID
	 * @param $prime_cache: Cache all apps because there is a loop with several calls, e.g on admin-bookings
	 * @return array
	 * @since 2.0
	 */
	 function get_app_userdata( $app_id, $user_id = 0, $prime_cache = false ) {
		$all_fields = $this->get_fields();

		foreach( $all_fields as $f ) {
			${$f} = '';
		}
		$note = $remember = '';

		// If app_id is defined (editing), bring data of the owner of the app
		if ( $app_id > 0 && $app = wpb_get_app( $app_id, $prime_cache ) ) {
			foreach ( $all_fields as $f ) {
				${$f} = wpb_get_app_meta( $app_id, $f );
			}

			$note = wpb_get_app_meta( $app_id, 'note' );
			$remember = '';
		} else if ( $user_id ) {
			foreach( $all_fields as $f ) {
				${$f} = get_user_meta( $user_id, 'app_'.$f, true );
			}

			$user_info = $this->_get_userdata( $user_id );

			if ( empty( $name ) ) {
				if ( ! $name = $this->get_name( $user_id ) ) {
					$name = isset( $user_info->display_name ) ? $user_info->display_name : '';
				}
			}

			if ( empty( $email ) ) {
				$email = isset( $user_info->user_email ) ? $user_info->user_email : '';
			}

			if ( empty( $first_name  ) ) {
				$first_name = isset( $user_info->first_name ) ? $user_info->first_name : '';
			}

			if ( empty( $last_name ) ) {
				$last_name = isset( $user_info->last_name ) ? $user_info->last_name : '';
			}

		} else if ( ! $user_id ) {
			// Get user form data from his cookie
			$data = $this->get_userdata_from_cookie();

			// First try to read full field names from cookie
			foreach( $all_fields as $f ) {
				if ( isset( $data[ $f ] ) ) {
					${$f} = sanitize_text_field( $data[ $f ] );
				}
			}

			// User may have already saved his data before
			if ( is_user_logged_in() ) {
				$cuid = get_current_user_id();
				$user_info = $this->_get_userdata( $cuid );

				foreach( $all_fields as $f ) {
					$temp = get_user_meta( $cuid, 'app_'.$f, true );
					${$f} = $temp ? $temp : ${$f};
				}

				if ( empty( $name ) ) {
					$name = $this->get_name( $cuid );
				}

				if ( empty( $email ) ) {
					$email = $user_info->user_email;
				}

			}
		}
		# Note: $user_id == 0 is accepted, e.g. for non logged in users

		$out				= array();
		$out['remember']	= $remember;
		$out['note']		= $note;

		foreach( $all_fields as $f ) {
			$out[ $f ] = ${$f};
		}

		// User data can be overridden, e.g. with $_GET['name']
		return apply_filters( 'app_userdata', $out, $all_fields );
	}

	/**
	 * Get users with caching
	 * @return array
	 * @since 3.6.7
	 */
	private static function get_users_cached( $query_args ) {
		$id		= wpb_cache_prefix() . 'users_'. md5( serialize( $query_args ) );
		$users	= wp_cache_get( $id );

		if ( false === $users ) {
			$users = get_users( $query_args );
			wp_cache_set( $id, $users );
		}

		return $users;
	}

	/**
	 * Modified version of wp_dropdown_users
	 * https://codex.wordpress.org/Function_Reference/wp_dropdown_users
	 * @return string
	 * @since 2.0
	 */
	public function app_dropdown_users( $args = '' ) {
		// Width parameter is new
        $defaults = array(
                'show_option_all' => '', 'show_option_none' => '', 'hide_if_only_one_author' => '',
                'orderby' => 'display_name', 'order' => 'ASC',
                'include' => '', 'exclude' => '', 'multi' => 0,
                'show' => 'display_name', 'echo' => 1,
                'selected' => 0, 'name' => 'user', 'class' => '', 'id' => '', 'width' => '',
                'blog_id' => $GLOBALS['blog_id'], 'who' => '', 'include_selected' => false,
                'option_none_value' => -1,
				'disabled' => array(), # Array of user IDS to disable
				'add_email' => false, # whether add email to display
				'data' => array(), # data attribute for select
        );
        $defaults['selected'] = is_author() ? get_query_var( 'author' ) : 0;
        $r = wp_parse_args( $args, $defaults );
		$add_email = $r['add_email'];
        $show = $r['show'];
        $show_option_all = $r['show_option_all'];
        $show_option_none = $r['show_option_none'];
        $option_none_value = $r['option_none_value'];
        $query_args = wp_array_slice_assoc( $r, array( 'blog_id', 'include', 'exclude', 'orderby', 'order', 'who', 'role' ) );
        $query_args['fields'] = array( 'ID', 'user_login', 'user_email', $show );
        $users = self::get_users_cached( $query_args );
        $output = '';
        if ( ! empty( $users ) && ( empty( $r['hide_if_only_one_author'] ) || count( (array)$users ) > 1 ) ) {
			# This condition is new
			if ( $r['width'] )
				$style = " style='width:".$r['width']."'";
			else
				$style = '';

			$name = esc_attr( $r['name'] );
			if ( $r['multi'] && ! $r['id'] ) {
					$id = '';
			} else {
					$id = $r['id'] ? " id='" . esc_attr( $r['id'] ) . "'" : " id='$name'";
			}

			$datas = '';

			if ( ! empty( $r['data'] ) && is_array( $r['data'] ) ) {
				foreach( $r['data'] as $key => $val ) {
					$datas .= 'data-' . sanitize_key( $key ) .'="'. sanitize_text_field( $val ) .'" ';
				}
			}

			$output = "<select $datas name='{$name}'{$id} class='" . $r['class'] . "' {$style} >\n";
			if ( $show_option_all ) {
					$output .= "\t<option value='0'>$show_option_all</option>\n";
			}
			if ( $show_option_none ) {
					$_selected = selected( $option_none_value, $r['selected'], false );
					$output .= "\t<option value='" . esc_attr( $option_none_value ) . "'$_selected>$show_option_none</option>\n";
			}
			$found_selected = false;

			foreach ( (array) $users as $user ) {
				$user->ID = (int) $user->ID;
				$_selected = selected( $user->ID, $r['selected'], false );
				$_disabled = !$_selected && in_array( $user->ID, $r['disabled'] ) ? " disabled='disabled'" : "";
				if ( $_selected ) {
						$found_selected = true;
				}
				# New: If $show = display_name, replace it with get_name
				if ( 'display_name' == $show && ! empty( $user->display_name ) ) {
					$display = $user->display_name;
				} else {
					$display = ! empty( $user->$show ) ? $user->$show : $this->get_name( $user->ID );
				}

				if ( $add_email ) {
					$display = $display .' '. '('. $user->user_email . ')';
				}

				$display = apply_filters( 'app_dropdown_users_display', $display, $user );
				$output .= "\t<option value='$user->ID'$_selected$_disabled>" . esc_html( $display ) . "</option>\n";
			}

			if ( $r['include_selected'] && ! $found_selected && ( $r['selected'] > 0 ) ) {
				$user = get_userdata( $r['selected'] );
				$_selected = selected( $user->ID, $r['selected'], false );
				$display = ! empty( $user->$show ) ? $user->$show : '('. $user->user_login . ')';
				$output .= "\t<option value='$user->ID'$_selected>" . esc_html( $display ) . "</option>\n";
			}

			$output .= "</select>";
        }
        /**
         * Filter the app_dropdown_users() HTML output.
         *
         * @param string $output HTML output generated by wp_dropdown_users().
         */
        $html = apply_filters( 'app_dropdown_users', $output );

        if ( $r['echo'] ) {
           echo $html;
        }

        return $html;
	}

	/**
	 * Get user ID from submitted user form
	 * @since 2.0
	 */
	public function read_user_id() {
		return isset( $_REQUEST["app_user_id"] ) ? wpb_clean( $_REQUEST["app_user_id"] ) : get_current_user_id();
	}

	/**
	 * Sanitize and apply filter to submitted data
	 * May die if required fields are empty
	 * @param $validate		string	Validate only if set to 'validate'
	 * @since 3.0
	 * @return array|none	If used as an action hook, returns none. If used as a method call, returns array (sanitized user submit)
	 */
	public function sanitize_submitted_userdata( $validate = 'validate' ){

		$sub_udata = array();

		$app_user_data = isset($_POST['app_user_data']) ? json_decode( wp_unslash($_POST['app_user_data']), true ) : array();

		// Sanitize and Apply check filter
		foreach( (array)$app_user_data as $f => $value ) {

			if ( strpos( $f, 'udf_' ) !== false ) {
				continue;
			}

			$sub_udata[ $f ] = ! empty( $value ) ? trim( sanitize_text_field( $value ) ) : '';

			if ( 'validate' === $validate ) {
				/**
				 * Filter should return false if submitted value validates.
				 * If not validates, a string value will be regarded as reason of not validation (error)
				 * goTO: class name to which screen will be scrolled to
				 */
				 if ( $error = apply_filters( "app_".$f."_validation", empty( $sub_udata[ $f ] ), $sub_udata[ $f ], $app_user_data, $f ) ) {
					if ( is_string( $error ) ) {
						die( json_encode( array( 'error' => $error, "goTo"=>"app-".$f."-field-entry" ) ) );
					} else {
						wpb_json_die( $f );
					}
				 }
			}
		}

		return $sub_udata;
	}

	/**
	 * Control visibility of required fields
	 * @Since 3.0
	 * @return array
	 */
	public function pre_confirmation_reply( $reply_array, $items, $obj ) {

		if ( empty( $_POST['app_user_fields'] ) ) {
			return $reply_array;
		}

		// Check which user fields will be asked from the client (visible ones are always required)
		// Filtered by the shortcode attr
		$user_fields = json_decode( wp_unslash( $_POST['app_user_fields'] ), true );

		if ( ! is_array( $user_fields ) ) {
			return $reply_array;
		}

		$ask = array();
		$ask['note']		= wpb_setting('ask_note') ? 'ask' : '';
		$ask['remember']	= wpb_setting('ask_remember') ? 'ask' : '';

		foreach( $this->a->get_user_fields() as $f ) {
			$ask[ $f ] = in_array( $f, array_values( $user_fields ) ) ? 'ask' : '';
		}

		return array_merge( $reply_array, $ask );
	}

	/**
	 * Handle/Check submitted user data at post confirmation
	 * @param $app_id_edit:	Integer		Appt ID only if editing
	 * @Since 3.0
	 * @return none if check fails or array
	 */
	public function post_confirmation_userdata( $app_id_edit ) {
		$current_user_id = get_current_user_id();

		$sub_udata = $this->sanitize_submitted_userdata( );

		if ( ! defined( 'WPB_SPAM_PROTECTION' ) ) {
			define( 'WPB_SPAM_PROTECTION', true ); # This can be set to false earlier
		}

		if ( ! $app_id_edit && WPB_SPAM_PROTECTION ) {
			if ( ! empty( $_POST['app_confirm_email_username'] ) ) {

				$msg = sprintf(
					__( 'Spam attack from IP %1$s has been avoided. User Agent: %2$s. User ID: %3$s. Submitted data: %4$s', 'wp-base' ),
					$_SERVER['REMOTE_ADDR'],
					$_SERVER['HTTP_USER_AGENT'],
					$current_user_id ? $current_user_id : __( 'None','wp-base' ),
					print_r( $sub_udata, true )
				);

				$this->a->log( $msg );
				die();
			}
		}

		if ( $app_id_edit ) {
			$app = wpb_get_app( $app_id_edit );

			if ( empty( $app->user ) ) {
				die( json_encode( array( 'error' => $this->a->get_text( 'invalid_booking' ) ) ) );
			}

			$user_id 	= $app->user;
			$userdata 	= $this->_get_userdata( $user_id );
			$user_email = $userdata->user_email;
			$user_name 	= $this->get_name( $user_id );
			$_user		= new WP_User( $user_id );
			$old_roles	= $_user->get_role_caps();
		} else if ( $current_user_id ) {
			// On behalf booking
			$user_id 	= ! empty( $_POST['app_user_id'] ) ? sanitize_text_field( $_POST['app_user_id'] ) : $current_user_id;
			$userdata 	= $this->_get_userdata( $user_id );
			$user_email = $userdata->user_email;
			$user_name 	= $this->get_name( $user_id );
			$_user		= new WP_User( $user_id );
			$old_roles	= $_user->get_role_caps();
		} else {
			$user_id 	= 0;
			$user_email = '';
			$user_name 	= '';
			$old_roles	= array();
		}

		$sub_udata['name'] 	= ! empty( $sub_udata['name'] ) ? $sub_udata['name'] : $user_name;
		$sub_udata['email'] = ! empty( $sub_udata['email'] ) ? $sub_udata['email'] : $user_email;
		$note 				= isset( $_POST["app_note"] ) ? sanitize_text_field( $_POST["app_note"] ) : '';
		$sub_udata['note'] 	= apply_filters( 'app_note_field', $note );

		# Disable remember only if selected so
		if ( ! wpb_setting('ask_remember') || ! empty( $_POST["app_remember"] ) ) {
			$sub_udata['remember'] = 1;
		} else {
			$sub_udata['remember'] = '';
		}

		# If editing and email not changed, don't go further
		if ( $app_id_edit && $sub_udata['email'] && $sub_udata['email'] == $user_email ) {
			return array( $user_id => $sub_udata );
		}

		/* Additional checks for email */
		if ( wpb_setting( 'ask_email' ) && ! is_email( $sub_udata['email'] ) ) {
			wpb_json_die( 'email', 'app-email-field-entry' );
		}

		# Email blacklist check
		$mod_keys = trim( get_option('disallowed_keys') );
		if ( wpb_setting( 'ask_email' ) && $sub_udata['email'] && $mod_keys ) {
			$words = explode("\n", $mod_keys );

			foreach ( (array) $words as $word ) {
				$word = trim($word);

				if ( empty($word) ) { continue; }

				// Do some escaping magic so that '#' chars in the
				// spam words don't break things:
				$word = preg_quote($word, '#');

				$pattern = "#$word#i";
				if ( preg_match($pattern, $sub_udata['email']) ) {
					die( json_encode( array( 'error' => $this->a->get_text('blacklisted') ) ) );
				}
			}

			// Multisite email check
			if ( is_multisite() && is_email_address_unsafe( $user_email ) ) {
				die( json_encode( array( 'error' => $this->a->get_text('blacklisted') ) ) );
			}
		}

		# Check if user is not logged in and submitted email is registered
		# Without this check, someone may login instead of another person using his email
		if ( ! is_user_logged_in() && wpb_setting( 'ask_email' ) && $sub_udata['email'] ) {
			$maybe_user = get_user_by( 'email', $sub_udata['email'] );
			if ( $maybe_user ) {
				die( json_encode( array( 'error' => $this->a->get_text('login_required') ) ) );
			}
		} else if ( is_user_logged_in() && $sub_udata['email'] && !current_user_can( WPB_ADMIN_CAP ) ) {
			# Logged in client tries to use another registered user's email
			$maybe_user = get_user_by( 'email', $sub_udata['email'] );
			if ( ! empty( $maybe_user->ID ) && $maybe_user->ID != $current_user_id ) {
				die( json_encode( array( 'error' => $this->a->get_text('email_unmatch') ) ) );
			}
		}

		if ( ! $user_id && $sub_udata['email'] && 'yes' == wpb_setting('auto_register_client') ) {
			if ( $maybe_user_id = $this->create_user( $sub_udata ) ) {
				$user_id = $maybe_user_id;
				/* Auto login
				 * https://codex.wordpress.org/Function_Reference/wp_set_current_user#Examples
				 * For WC and EDD, don't wait for setting
				 */
				if ( ! is_user_logged_in() && ('yes' == wpb_setting('auto_register_login')
					|| wpb_is_product_page()
					|| ( BASE('WooCommerce') && BASE('WooCommerce')->is_app_wc_page() )
					|| ( BASE('EDD') && BASE('EDD')->is_app_edd_page() ) ) ) {

					$user = get_user_by( 'id', $user_id );
					if ( $user ) {
						wp_set_current_user( $user_id, $user->user_login );
						wp_set_auth_cookie( $user_id );
						try {
							do_action( 'wp_login', $user->user_login, $user );
						} catch ( Exception $e ) {
							$this->a->log( $e->getMessage() );
						}
					}
				}
			}
		}

		if ( ! empty( $user_id ) && ! array_key_exists( 'wpb_client', $old_roles ) ) {
			$new_user = new WP_User( $user_id );
			$new_user->add_role( 'wpb_client' );

			do_action( 'app_new_client', $user_id );
		}

		return apply_filters( 'app_post_confirmation_userdata', array( $user_id => $sub_udata ) );
	}

	/**
	 * Set "client since" meta value
	 * @uses add_user_role action hook
	 * @since 3.9.0
	 * @return none
	 */
	 public function add_user_role( $user_id, $role ) {
		if ( ! $user_id || 'wpb_client' != $role ) {
			return;
		}

		add_user_meta( $user_id, 'app_client_since', time(), true );
	}

	public function maybe_save_meta( $result, $booking_id, $sub_udata ) {

		if ( $this->save_meta( $booking_id, $sub_udata ) ) {
			$result = true;
		}

		return $result;
	}

	/**
	 * Save user submitted data to app meta and save cookie
	 * @param $booking_id	integer		ID of the new booking
	 * @param $sub_udata	array		Submitted user data
	 * @param $parent_id	integer		ID of booking's parent. If falsy or equal to $booking_id, booking is the parent itself
	 * @return bool			True if something changed, false if nothing changed
	 */
	public function save_meta( $booking_id, $sub_udata, $parent_id = null ) {
		$result 	= false;
		$app 		= wpb_get_app( $booking_id );
		$user_id 	= ! empty( $app->user ) ? $app->user : 0;

		// Booking on Behalf
		$cuid = get_current_user_id();

		if ( $cuid && ( $cuid != $user_id || current_user_can( WPB_ADMIN_CAP ) ) && wpb_update_app_meta( $booking_id, 'booking_on_behalf', $cuid ) ) {
			$result = true;
		}

		# Save user data even in on behalf case ($user_id is not necessarily current user id)
		# To save server resources, do this only once for the cart, just for the parent
		if ( $user_id && ( ! $parent_id || $parent_id == $booking_id ) ) {

			$umeta_fields = (array)apply_filters( 'app_save_user_meta', $this->get_fields(), $user_id, $booking_id, $sub_udata );

			foreach( $umeta_fields as $f ) {
				if ( isset( $sub_udata[ $f ] ) && update_user_meta( $user_id, 'app_'.$f, $sub_udata[ $f ] ) ) {
					$result = true;
				}
			}
		}

		foreach ( $sub_udata as $f => $value ) {
			if ( wpb_update_app_meta( $booking_id, $f, $value ) ) {
				$result = true;
			}
		}

		if ( $this->save_cookie( $booking_id, $user_id, false ) ) {
			$result = true;
		}

		return $result;
	}

	/**
	 * Save a cookie so that not logged in client can follow his appointments
	 */
	public function save_cookie( $app_id, $user_id, $userdata ) {

		# DO NOT SAVE data in COOKIE if user is logged in
		if ( is_user_logged_in() ) {
			return;
		}

		# It should come here with ajax request, so headers should not be sent, but just in case
		if ( headers_sent() ) {
			return;
		}

		$cookiepath 	= defined('COOKIEPATH') ? COOKIEPATH :  "/";
		$cookiedomain 	= defined('COOKIEDOMAIN') ? COOKIEDOMAIN : '';
		$expire 		= $this->a->_time + 3600 * 24 * ( $this->a->get_upper_limit() + 365 ); # Add 365 days grace time
		$expire 		= apply_filters( 'app_cookie_time', $expire );

		if ( ! empty( $userdata['remember'] ) ) {

			unset( $userdata['remember'] );

			$userdata = apply_filters( 'app_cookie_userdata', $userdata, $user_id );
			$userdata['timestamp'] = $this->a->_time;

			$ser_data = wp_json_encode( array(
					'hash'		=> $this->a->create_hash( $userdata, 'userdata_cookie', $this->get_anon_id( $userdata ) ),
					'userdata'	=> $userdata,
			) );

			if ( setcookie( 'wpb_userdata', $ser_data, $expire, $cookiepath, $cookiedomain ) ) {
				$_COOKIE['wpb_userdata'] = $ser_data;
			}
		} else {
			# Do not remember selected - Delete cookie if exists
			setcookie( 'wpb_userdata', '', time() - 24*3600, $cookiepath, $cookiedomain );
			unset( $_COOKIE['wpb_userdata'] );
		}

		if ( $app_id ) {
			$apps = $this->a->get_apps_from_cookie();

			$apps[] = $app_id;
			$apps = array_filter( array_unique( $apps ) );

			$ser_data = wp_json_encode( array(
				'hash'		=> $this->a->create_hash( $apps, 'bookings_cookie', $this->get_anon_id( $userdata ) ),
				'bookings'	=> $apps,
			) );

			if ( setcookie( 'wpb_bookings', $ser_data, $expire, $cookiepath, $cookiedomain ) ) {
				$_COOKIE['wpb_bookings'] = $ser_data;
			}
		}

		return true;
	}

	/**
	 * Get user data from cookie by checking hash
	 * To prevent tampering with cookie
	 * @return array
	 * @since 3.0
	 */
	public function get_userdata_from_cookie() {
		if ( empty( $_COOKIE['wpb_userdata'] ) ) {
			return array();
		}

		$data = json_decode( wp_unslash( $_COOKIE['wpb_userdata'] ), true );

		# Corrupted cookie
		if ( empty( $data['userdata'] ) ) {
			return array();
		}

		$udata = wpb_clean( $data['userdata'] );

		# Tampered cookie
		if ( empty( $data['hash'] ) || ! is_array( $udata )
			|| $data['hash'] != $this->a->create_hash( $udata, 'userdata_cookie', $this->get_anon_id( $udata ) ) ) {

			return array();
		}

		return $udata;
	}

	/**
	 * Find a string from user data to identify a returning anon user
	 * @param $udata		array		User data
	 * @return string
	 * @since 3.0
	 */
	public function get_anon_id( $udata ) {
		if ( ! empty( $udata['email'] ) ) {
			return $udata['email'];
		}

		if ( ! empty( $udata['name'] ) ) {
			return $udata['name'];
		}

		$first_name = ! empty( $udata['first_name'] ) ? $udata['first_name'] : '';
		$last_name	= ! empty( $udata['last_name'] ) ? $udata['last_name'] : '';

		if ( '' != $first_name . $last_name ) {
			return $first_name .' '. $last_name;
		}

		if ( ! empty( $udata['phone'] ) ) {
			return $udata['phone'];
		}

		if ( ! empty( $udata['timestamp'] ) ) {
			return $udata['timestamp'];
		}

		# If no userdata provided salt will also work
		return wpb_get_salt();
	}

	/**
	 * Add tab
	 */
	public function add_tab( $tabs ) {
		$tabs['user'] = __('Users', 'wp-base');

		return $tabs;
	}

	/**
	 * Save user related admin settings
	 */
	public function save_settings() {
		if ( isset( $_POST['profileuser_id'] ) ) {
			$result = $this->save_profile();
		}

		if ( 'save_user' == $_POST['action_app'] ) {
			$options = wpb_setting();

			if ( isset( $_POST['login_required'] ) ) {
				$options['login_required']			= $_POST['login_required'];
			}

			$options['multitasking']				= $_POST['multitasking'];
			$options['auto_register_client']		= $_POST['auto_register_client'];
			$options['auto_register_client_notify']	= $_POST['auto_register_client_notify'];
			$options['auto_register_login']			= $_POST['auto_register_login'];
			$options['default_worker']				= ! empty( $_POST['default_worker'] ) && $_POST['default_worker'] > 0
													  ? $_POST['default_worker']
													  : $this->a->get_def_wid();

			if ( $this->a->update_options( $options ) ) {
				wpb_notice( 'saved' );
			}
		}
	}

	/**
	 * Display HTML codes
	 */
	public function settings() {
	?>
		<div id="poststuff" class="metabox-holder">
		<form class="app-form" method="post" action="<?php echo wpb_add_query_arg( null, null ) ?>" >
			<div class="postbox">
			<h3 class="hndle"><span class="dashicons dashicons-admin-users"></span><span><?php _e('Users', 'wp-base' ) ?></span></h3>
				<div class="inside">

				<table class="form-table">
					<?php
						if ( ! BASE('Login') ) {
							wpb_setting_yn( 'login_required' );
						}
					?>
					<tr id="default-worker">
						<th scope="row" ><?php WpBConstant::echo_setting_name('default_worker') ?></th>
						<td>
						<?php
						$set = wpb_setting( 'default_worker' );
						$this->app_dropdown_users( array(
							'show'				=> 'display_name',
							'add_email' 		=> true,
							'selected'			=> $set,
							'class'				=> $set && get_user_by( 'ID', $set ) ? '' : 'error',
							'name'				=> 'default_worker',
							'show_option_none'	=> __( 'Not selected', 'wp-base' ),
						) );

						?>
						<span class="description app-btm"><?php WpBConstant::echo_setting_desc('default_worker') ?></span>
						</td>
					</tr>

					<?php wpb_setting_yn( 'multitasking' ) ?>
					<?php wpb_setting_yn( 'auto_register_client' ) ?>
					<?php wpb_setting_yn( 'auto_register_login' ) ?>
					<?php wpb_setting_yn( 'auto_register_client_notify' ) ?>

				</table>
				</div>
			</div>
			<input type="hidden" name="action_app" value="save_user" />
			<?php wp_nonce_field( 'update_app_settings', 'app_nonce' ); ?>
			<p class="submit">
			<input class="button-primary app-save-payment-btn" type="submit" name="submit_settings" value="<?php _e('Save Settings', 'wp-base' ) ?>" />
			</p>
		</form>
		</div>
	<?php
	}

	private function first_user(){
		return (int)$this->a->db->get_var( "SELECT MIN(ID) FROM ".$this->a->db->users );
	}

	public function admin_menu(){
		add_users_page( 'Your Bookings', __('Your Bookings','wp-base'), 'read',  'your_bookings', array( $this,'show_profile') );
	}

	/**
	 * Deletes a client/worker's database records in case he is deleted
	 * @param int|null $reassign ID of the user to reassign posts and links to.
	 * @since 1.0.4
	 */
	public function user_deleted( $ID, $reassign = null ) {
		if ( ! $ID ) {
			return;
		}

		$changed = false;

		if ( BASE('WH')->remove( $ID, 'worker' ) ) {
			$changed = true;
		}

		$assign_to = $reassign && wpb_is_worker( $reassign ) ? $reassign : 0;

		if ( $this->a->db->update( $this->a->app_table,
			array( 'worker'	=> $assign_to ),
			array( 'worker'	=> $ID )
		) ) {
			$changed = true;
		}

		if ( $this->a->db->update( $this->a->app_table,
			array( 'user'	=> $reassign ),
			array( 'user'	=> $ID )
		) ) {
			$changed = true;
		}

		if ( $this->a->db->update( $this->a->meta_table,
			array( 'meta_value'	=> $reassign ),
			array( 'meta_type' => 'app', 'meta_key' => 'created_by', 'meta_value' => $ID )
		) ) {
			$changed = true;
		}

		if ( $this->a->db->delete( $this->a->meta_table,
			array( 'meta_type' => 'app', 'meta_key' => 'booking_on_behalf', 'meta_value' => $ID )
		) ) {
			$changed = true;
		}

		if ( $this->a->db->update( $this->a->meta_table,
			array( 'meta_value'	=> $reassign ),
			array( 'meta_type' => 'service', 'meta_key' => 'created_by', 'meta_value' => $ID )
		) ) {
			$changed = true;
		}

		if ( $this->a->db->update( $this->a->meta_table,
			array( 'meta_value'	=> $reassign ),
			array( 'meta_type' => 'service', 'meta_key' => 'owner', 'meta_value' => $ID )
		) ) {
			$changed = true;
		}

		if ( wpb_is_worker( $ID ) && wpb_delete_worker( $ID ) ) {
			do_action( 'app_worker_removed', $ID );
			$changed = true;
		}

		if ( $changed ) {
			wpb_flush_cache();
		}
	}

	/**
	 * Saves changes in user profile
	 */
	public function save_profile( $bp_user_id = false ) {
		$current_user_id = get_current_user_id();

		if ( $bp_user_id ) {
			$profileuser_id = $bp_user_id;
		} else {
			if ( isset( $_POST['profileuser_id'] ) ) {
				$profileuser_id = wpb_clean( $_POST['profileuser_id'] );
			} else if ( $_POST['app_bp_settings_user'] ) {
				$profileuser_id = wpb_clean( $_POST['app_bp_settings_user'] );
			}
		}

		// Only user himself can save his data
		if ( $current_user_id != $profileuser_id ) {
			return;
		}

		if ( wpb_is_demo() ) {
			wpb_notice( 'demo' );
			return;
		}

		$r = false;
		// Save user meta
		foreach ( $this->get_fields() as $f ) {

			if ( 'email' == $f ) {
				if ( ! empty( $_POST['app_email'] ) && is_email( $_POST['app_email'] ) ) {
					if ( wp_update_user( array ('ID' => $profileuser_id, 'user_email' => $_POST['app_email'] ) ) ) {
						$r = true;
					}
				} else {
					$error = true;
					wpb_notice( $this->a->get_text( 'invalid_email' ) );
				}
			}

			if ( 'name' == $f ) {
				if ( ! empty( $_POST['app_name'] ) ) {
					if ( wp_update_user( array ('ID' => $profileuser_id, 'display_name' => $_POST['app_name'] ) ) ) {
						$r = true;
					}
				}
			}

			if ( isset( $_POST['app_'.$f] ) ) {
				if ( update_user_meta( $profileuser_id, 'app_'.$f, wpb_clean( $_POST['app_'.$f] ) ) ) {
					$r = true;
				}
			}
		}

		if ( $r ) {
			wpb_notice( 'saved' );
		}

		// Client or worker cancels own appointment
		if ( 'yes' == wpb_setting('allow_cancel') && ! empty( $_POST['app_cancel'] ) && is_array( $_POST['app_cancel'] ) ) {

			foreach ( $_POST['app_cancel'] as $app_id => $value ) {

				if ( $this->a->change_status( 'removed', $app_id, false ) ) {

					$app = wpb_get_app( $app_id );
					if ( $app->worker && $app->worker == get_current_user_id() ) {
						$meta_val = 'worker_cancelled';
						$text = sprintf( __('Provider %1$s cancelled booking with ID: %2$s','wp-base'), $this->a->get_worker_name( $app->worker ), $app_id );
					} else {
						$meta_val = 'cancelled';
						$text = sprintf( __('Client %1$s cancelled booking with ID: %2$s','wp-base'), $this->get_client_name( $app_id, null, false ), $app_id );
					}

					wpb_update_app_meta( $app_id, 'abandoned', $meta_val );
					$this->a->log( $text );
					$this->a->send_notification( $app_id, true ); // This is always sent regardless of email settings
				}
			}

			wp_redirect( wpb_add_query_arg( 'app_cancelled', $app_id ) );
			exit;
		}

		do_action( 'app_save_profile', $profileuser_id );

		// Only user who is a worker can save the rest
		if ( ! wpb_is_worker( $profileuser_id ) ) {
			return $r;
		}

		// Confirm an appointment using profile page
		if ( 'yes' == wpb_setting('allow_worker_confirm') && isset( $_POST['app_confirm'] ) && is_array( $_POST['app_confirm'] ) && ! empty( $_POST['app_confirm'] ) ) {

			foreach ( $_POST['app_confirm'] as $app_id => $value ) {

				if ( $this->a->change_status( 'confirmed', $app_id, false ) ) {

					$this->a->log( sprintf(
						__('Service Provider %1$s manually confirmed appointment with ID: %2$s','wp-base'),
						$this->a->get_worker_name( $current_user_id ),
						$app_id
					));

					$this->a->maybe_send_message( $app_id, 'confirmation' );

					wpb_notice( __( 'Selected appointment has been confirmed', 'wp-base' ) );
				}
			}

		}

		// Save working hours table
		if ( 'save_working_hours' == $_POST['action_app'] && 'yes' == wpb_setting('allow_worker_wh') ) {
			$r = BASE('WH')->save_wh( $profileuser_id );
		}

		return $r;
	}

	/**
	 * Displays appointment schedule on the user profile
	 */
	public function show_profile( $profileuser = false ) {
		$current_user_id = get_current_user_id();
		$profileuser_id = isset( $_GET['user_id'] ) ? absint( $_GET['user_id'] ) : $current_user_id;

		// Only user or admin can see his data
		if ( $current_user_id != $profileuser_id && !current_user_can('list_users') )
			return;

		// For other than user himself, display data as readonly
		$is_readonly = $current_user_id != $profileuser_id ? ' readonly="readonly"' : '';
		$is_readonly = apply_filters( 'app_show_profile_readonly', $is_readonly, $profileuser_id );
		$gcal = 'yes' == wpb_setting("gcal") ? '' : ' gcal="0"'; // Default is already enabled
		$title = apply_filters( 'app_account_title', sprintf( __('%s Bookings & Settings', 'wp-base'), WPB_NAME ) );
	?>
	<div class="wrap">
	<?php if ( $title ) : ?>
		<h2 class="<?php echo (is_admin() ? 'app-dashicons-before dashicons-id-alt' : '') ?>"><?php echo $title ?></h2>
	<?php endif; ?>
		<h3 class="nav-tab-wrapper">
			<?php

			if ( wpb_is_worker( $profileuser_id ) ) {
				$tabs = array(
					'bookings'		=> $this->a->get_text('bp_bookings'),
					'weekly'      	=> $this->a->get_text('bp_weekly'),
					'4weeks' 	    => $this->a->get_text('bp_4weeks'),
					'monthly' 	    => $this->a->get_text('bp_monthly'),
					'3months'		=> $this->a->get_text('bp_3months'),
					'services'		=> $this->a->get_text('bp_services'),
					'working_hours'	=> $this->a->get_text('bp_wh'),
					'holidays'		=> $this->a->get_text('bp_holidays'),
					'seasonal'		=> $this->a->get_text('bp_annual'),
					'payments'		=> $this->a->get_text('bp_payments'),
					'settings'		=> $this->a->get_text('bp_settings'),
				);
				if ( !( 'yes'== wpb_setting('allow_worker_create_service') && wpb_admin_access_check( 'manage_own_services', false ) ) ) {
					unset( $tabs['services'] );
				}
				if ( !( 'yes' == wpb_setting("allow_worker_wh") && wpb_admin_access_check( 'manage_own_work_hours', false ) ) ) {
					unset( $tabs['working_hours'] );
					unset( $tabs['holidays'] );
				}

				if ( ! wpb_user_can_view_payments( $profileuser_id ) ) {
					unset( $tabs['payments'] );
				}

				if ( !( BASE('Annual') && 'yes' == wpb_setting('allow_worker_annual') && wpb_admin_access_check( 'manage_own_work_hours', false ) ) ) {
					unset( $tabs['seasonal'] );
				}
			} else {
				$tabs = array(
					'bookings'	=> $this->a->get_text('bp_bookings'),
					'settings'	=> $this->a->get_text('bp_settings'),
				);
			}

			$tabhtml = array();

			$tabs		= apply_filters( 'app_user_profile_tabs', $tabs, $profileuser_id );
			$first_tab	= apply_filters( 'app_user_profile_first_tab', (array_key_exists( 'bookings', $tabs ) ? 'bookings' : key( $tabs )), $tabs );
			$tab		= ! empty( $_GET['tab'] ) ? wpb_clean( $_GET['tab'] ) : $first_tab;
			$args		= apply_filters( 'app_user_tab_query_args', array(
							'add_new'		=> false,
							'app_id'		=> false,
							'app_worker'	=> false,
							'app_timestamp'	=> false,
							'cpy_from'		=> false,
							'type'			=> false,
							'stype'			=> false,
							'app_s'			=> false,
							'app_gw_success'=> false,
							'app_hash'		=> false,
							'export-failed'	=> false,
						 ), $profileuser_id );

			foreach ( $tabs as $stub => $title ) {
				$args['tab'] = $stub;
				$class = $stub == $tab ? ' nav-tab-active' : '';
				$href = is_admin()
						? admin_url( 'users.php?page=your_bookings&tab=' . $stub )
						: add_query_arg( array( 'tab' => $stub ) );

				$tabhtml[] = '	<a href="' . esc_attr($href) . '" class="app-tab nav-tab'.$class.'" id="app_tab_'.$stub.'">'.$title.'</a>';
			}

			echo implode( "\n", $tabhtml );
			?>
		</h3>
		<div class="clear"></div>

		<?php
		$user_id = $profileuser_id;

		switch( $tab ) {
			case 'bookings':
				$this->show_profile_bookings( $profileuser_id, $is_readonly );
				break;

			case 'weekly'	:
				if ( ! wpb_is_worker( $profileuser_id ) ) {
					break;
				}
				?>
				<div id="poststuff" class="metabox-holder">
					<div class="postbox">
						<div class="inside">
						<?php
						echo $this->a->pagination( array( 'select_date' => 1, 'disable_legend' => 1 ) );
						$wscodes = BASE('Schedules')->weekly_shortcodes( 1, $profileuser_id );
						echo current( $wscodes );
						echo $this->a->pagination( array( 'select_date' => 0, ) );
						?>
						</div>
					</div>
				</div>
				<?php
				break;

			case '4weeks':
				if( ! wpb_is_worker( $profileuser_id ) ) {
					break;
				}
				?>
				<div id="poststuff" class="metabox-holder">
					<div class="postbox">
						<div class="inside">
						<?php
						echo $this->a->pagination( array( 'select_date' => 1, 'disable_legend' => 1, 'step' => 4 ) );
						foreach( BASE('Schedules')->weekly_shortcodes( 4, $profileuser_id) as $scode ) {
							echo $scode;
						}
						echo '<div style="clear:both"></div>';
						echo $this->a->pagination( array( 'select_date' => 0, 'step' => 4 ) );
						?>
						</div>
					</div>
				</div>
				<?php
				break;

			case 'monthly':
				if( ! wpb_is_worker( $profileuser_id ) ) {
					break;
				}
				?>
				<div id="poststuff" class="metabox-holder">
					<div class="postbox">
						<div class="inside">
						<?php
						echo $this->a->pagination( array( 'select_date' => 1, 'disable_legend' => 1, 'unit' => 'month','step' => 1 ) );
						$mscodes = BASE('Schedules')->monthly_shortcodes( 1, $profileuser_id );
						echo $mscodes[0];
						echo $this->a->pagination( array( 'select_date' => 0, 'unit'=> 'month','step' => 1 ) );
						?>
						</div>
					</div>
				</div>
				<?php
				break;

			case '3months':
				if ( ! wpb_is_worker( $profileuser_id ) ) {
					break;
				}
				?>
				<div id="poststuff" class="metabox-holder">
					<div class="postbox">
						<div class="inside">
						<?php
						echo $this->a->pagination( array( 'select_date' => 1, 'disable_legend' => 1, 'unit'=> 'month','step' => 3 ) );
						foreach( BASE('Schedules')->monthly_shortcodes( 3, $profileuser_id) as $scode ) {
							echo $scode;
						}
						echo '<div style="clear:both"></div>';
						echo $this->a->pagination( array( 'select_date' => 0, 'unit' => 'month','step' => 3 ) );
						?>
						</div>
					</div>
				</div>
				<?php
				break;

			case 'services':
				if ( 'yes' != wpb_setting('allow_worker_create_service') ) {
					break;
				}

				include_once( WPBASE_PLUGIN_DIR . '/includes/admin/base-admin.php' );

				?><div class="wp-core-ui"><?php

				BASE('AdminServices')->listing( $profileuser_id );

				?></div><?php

				break;

			case 'working_hours':
				if ( ! wpb_is_worker( $profileuser_id ) || 'yes' != wpb_setting("allow_worker_wh") ) {
					break;
				}

				BASE('WH')->render_tab( $profileuser_id );

				break;

			case 'holidays':
				if ( ! wpb_is_worker( $profileuser_id ) || 'yes' != wpb_setting("allow_worker_wh") ) {
					break;
				}

				BASE('Holidays')->render_tab( $profileuser_id );

				break;

			case 'seasonal':
				if ( ! wpb_is_worker( $profileuser_id ) || ! class_exists('WpBAnnual') || 'yes' != wpb_setting("allow_worker_annual") ) {
					break;
				}

				BASE('Annual')->render_tab( $profileuser_id );

				break;

			case 'payments':
				include_once( WPBASE_PLUGIN_DIR . '/includes/admin/transactions.php' );

				?><div id="wpbody-content" class="app-sc wp-admin wp-core-ui app-fem app-clearfix"><?php

				BASE('Transactions')->transaction_list( $profileuser_id );

				?></div><?php

				break;

			case 'settings':
				$this->show_profile_settings( $profileuser_id, $is_readonly );
			break;

			case $tab:	do_action( 'app_user_profile_'.$tab.'_tab', $profileuser_id );
			break;
		}
?>
	</div>
	<?php
	}

	/**
	 * Helper to display bookings
	 */
	private function show_profile_bookings( $profileuser_id, $is_readonly ) {
	?><div id="poststuff" class="metabox-holder">
		<div class="postbox">
			<div class="inside">
				<form id="app-your-profile" class="app-form" method="post"><?php
					$columns = array('id','service','worker','date_time','end_date_time','status','cancel','confirm','edit','pay','zoom','jitsi','hangout');
					$columns_w = array('id','service','client','date_time','end_date_time','status','cancel','confirm','edit','pay','zoom','jitsi','hangout');
					$columns = apply_filters( 'app_user_listing_columns', (wpb_is_worker( $profileuser_id ) ? $columns_w : $columns), $profileuser_id );

					$status = apply_filters( 'app_user_listing_status', array('paid','confirmed','pending','running'), $profileuser_id );

					echo BASE('Listing')->display( array( '_user_page' => 1, 'status' => implode( ',', $status ), 'columns' => implode( ',', $columns ), 'columns_mobile' => implode( ',', $columns ) ) );
				?>
					<input type="hidden" name="action_app" value="save_profile" />
					<input type="hidden" name="profileuser_id" value="<?php echo $profileuser_id ?>" />
					<?php wp_nonce_field( 'update_app_settings', 'app_nonce' ); ?>
				</form>
			</div>
		</div>
	</div>
	<?php
	}

	/**
	 * Helper to display settings form
	 */
	private function show_profile_settings( $profileuser_id, $is_readonly ) {
	?>
	<form id="app-your-profile" class="app-form" method="post">
		<div id="poststuff" class="metabox-holder">
			<div class="postbox">
				<div class="inside">
					<table class="form-table app-profile-settings">

					<?php do_action( 'app_show_profile_before_fields', $profileuser_id ) ?>

					<?php foreach( $this->get_fields() as $f ) {
						# Do not let name and email fields to be empty
						if ( 'name' == $f || 'email' == $f ) {
							if ( ! $meta =  get_user_meta( $profileuser_id, 'app_'.$f, true ) ) {
								$user = new WP_User( $profileuser_id );
								if ( 'name' == $f ) {
									$meta = ! empty( $user->display_name ) ? $user->display_name : '';
								} else {
									$meta = ! empty( $user->user_email ) ? $user->user_email : '';
								}
							}
						} else {
							$meta = get_user_meta( $profileuser_id, 'app_'.$f, true );
						}

					?>
						<tr>
						<th><label><?php echo $this->a->get_text($f) ?></label></th>
						<td>
						<input type="text" name="app_<?php echo $f?>" value="<?php echo $meta ?>" <?php echo $is_readonly ?> />
						</td>
						</tr>
					<?php } ?>

					<?php do_action( 'app_show_profile', $profileuser_id ) ?>

					</table>
				</div>
			</div>
			<?php do_action( 'app_show_profile_outer', $profileuser_id ) ?>
			<input type="hidden" name="action_app" value="save_profile" />
			<input type="hidden" name="profileuser_id" value="<?php echo $profileuser_id ?>" />
			<?php wp_nonce_field( 'update_app_settings', 'app_nonce' ); ?>
			<p class="submit">
				<div class="app_rates_hidden" style="display:none"></div>
				<input type="submit" class="button-primary app_save_profile_submit" value="<?php _e('Save Profile Settings', 'wp-base' ) ?>" />
			</p>
		</div>
	</form>
	<?php
	}

	public function get_fields(){
		return apply_filters( 'app_user_fields', array('name', 'first_name', 'last_name', 'email', 'phone', 'address', 'city', 'zip') );
	}

	/**
	 * Create a user from an array of fields ($data['email'], $data['name'], etc ).
	 * If user already exists, returns his ID
	 * Also save user meta
	 * @param $notify_admin	bool		Whether notify admin
	 * @param $notify_user	bool|null	Whether notify user (If null, auto_register_client_notify is in effect. If false, user not notified. If true, user is notified)
	 * @return mix		false|integer	User ID on success, false on failure
	 */
	public function create_user( $data, $notify_admin = true, $notify_user = null ) {

		foreach( $this->get_fields() as $f ) {
			${$f} = ! empty( $data[ $f ] ) ? $data[ $f ] : '';
		}

		$user_id = false;
		$wp_user = get_user_by( 'email', $email );

		if ( ! empty( $wp_user->ID ) ) {
			$user_id = $wp_user->ID;
		} else {

			if ( apply_filters( 'app_create_user_email_required', false ) ) {
				if ( empty( $email ) || ! is_email( $email ) ) {
					return false;
				}
			}

			$password = empty( $data['password'] ) ? wp_generate_password(12, false) : $data['password'];

			if ( $name || $first_name || $last_name ) {
				$username = $name
					? preg_replace('/[^_0-9a-z]/i', '_', strtolower($name))
					: trim( preg_replace('/[^_0-9a-z]/i', '_', strtolower($first_name)) . '_' . preg_replace('/[^_0-9a-z]/i', '_', strtolower($last_name)), '_' )
				;
			} else if ( $email ) {
				$arr = explode( '@', $email );
				$username = ! empty( $arr[0] ) ? $arr[0] : 'guest'. rand( 0, 100 );
			} else {
				$username = 'guest'. rand( 0, 100 );
			}

			$count = 0;
			while (username_exists($username)) {
				$username .= rand(0,9);
				if (++$count > 10) break;
			}

			$user_id = wp_insert_user( array(
				'user_login'	=> $username,
				'user_pass'		=> $password,
				'user_email'	=> $email,
				'first_name'	=> $first_name,
				'last_name'		=> $last_name,
				) );

			if ( is_wp_error( $user_id ) ) {
				$this->a->log( $user_id->get_error_message() );
				return false;
			}

			if ( function_exists( 'add_new_user_to_blog' ) ) {
				add_new_user_to_blog( $user_id, $email, get_option('default_role') );
			}

			$notify = ('yes' == wpb_setting('auto_register_client_notify') && false !== $notify_user ) || true === $notify_user ? ( $notify_admin ? 'both' : 'user' ) : ( $notify_admin ? 'admin' : 'none' );

			if ( ! empty( $user_id ) && 'none' != $notify ) {
				wp_new_user_notification( $user_id, null, $notify );
			}

			if ( ! empty( $user_id ) ) {
				$new_user = new WP_user( $user_id );
				$new_user->add_role( 'wpb_client' );
				do_action( 'app_new_client', $user_id );
			}
		}

		if ( empty( $user_id ) ) {
			return false;
		}

		foreach ( $this->get_fields() as $f ) {
			if ( ! empty( $data[ $f ] ) ) {
				update_user_meta( $user_id, 'app_'.$f, $data[ $f ] );
			}
		}

		return $user_id;
	}

}
	BASE('User')->add_hooks();
}