<?php
/**
 * WPB Pay Later
 *
 * Allows clients to pay after they make booking
 *
 * @author		Hakan Ozevin
 * @package     WP BASE Manual Payments
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

 if ( ! defined( 'ABSPATH' ) ) exit;

if ( !class_exists('WpB_Gateway_Pay_Later') ) {
class WpB_Gateway_Pay_Later extends WpB_Gateway_API {

	# Private gateway slug. Lowercase alpha (a-z) and dashes (-) only please!
	public $plugin_name = 'pay-later';

	# Name of the gateway, for the admin side.
	public $admin_name = '';

	# Public name of the gateway, for lists and such.
	public $public_name = '';

	# Url for an image for checkout method. Displayed on method form
	public $method_img_url = '';
	
	# Header (Legend tag) of payment form
	public $form_header = '';

	# Note that will be placed in the email when payment method is this one
	public $email_note;

	/**
	* Runs when gateway class is instantiated.
	*/
	public function on_creation() {
		$this->admin_name = __('Pay Later', 'wp-base');

		add_filter( 'app_shortcode_desc', array( $this, 'shortcode_desc' ) );

		$this->apply_settings();
	}

	/**
	* Load latest settings
	*/
	public function apply_settings(){
		parent::apply_settings();
		
		$this->method_img_url	= $this->a->plugin_url . '/images/pay-later.png';
		$this->public_name		= wpb_gateway_setting( $this->plugin_name, 'name', __('Pay Later', 'wp-base' ) );
		$this->email_note		= wpb_gateway_setting( $this->plugin_name, 'email-note', __('Your account has a balance of $ABS_BALANCE. Please click this link to make a payment before PENDING_UNTIL: MAKE_PAYMENT', 'wp-base' ) );
		$this->form_header		= wpb_gateway_setting( $this->plugin_name, 'form-header', __('Please confirm payment for Booking APP_ID', 'wp-base' ) );
	}

	/**
	* Echo a settings meta box with whatever settings you need for you gateway.
	* Form field names should be prefixed with mp[gateways][plugin_name], like "mp[gateways][plugin_name][mysetting]".
	* You can access saved settings via $settings array.
	*/
	public function gateway_settings_box($settings) {
		$this->apply_settings();
?>
<div id="mp_pay_later" class="postbox mp-pages-msgs">
	<h3 class="hndle"><span><?php _e('Pay Later', 'wp-base'); ?></span></h3>
	<div class="inside">
		<span class="description"><?php _e('Allows clients make payment after they make booking. At least another payment method other than Manual Payments should be active, e.g. PayPal, Stripe, Credits, etc.', 'wp-base' ) ?></span>
		<table class="form-table">
		
			<?php $this->common_settings( ) ?>
			
			<tr>
				<th scope="row"><label for="pay_later_disable_checkout"><?php _e('Disable Pay Later for Checkout', 'wp-base' ) ?></label></th>
				<td>
					<?php $set = wpb_gateway_setting( $this->plugin_name, 'disable-checkout' ) ?>
					<select name="mp[gateways][pay-later][disable-checkout]" id="pay_later_disable_checkout">
						<option value="no" <?php selected( 'no', $set ) ?>><?php _e( 'No', 'wp-base' ) ?></option>
						<option value="yes" <?php selected( 'yes', $set ) ?>><?php _e( 'Yes', 'wp-base' ) ?></option>
					</select>
					<span class="description app-btm"><?php _e('You can disable Pay Later selection option at checkout, and use it to collect payment from clients for their negative balances after checkout. As long as PayPal or another credit card processor gateway, e.g. Stripe, is active, clients can pay for their negative balance on Bookings tab of their Account page, or link created by MAKE_PAYMENT placeholder in emails they receive. For example you can add the placeholder in reminder email template.', 'wp-base') ?></span>
				</td>
			</tr>
			<tr>
				<th scope="row"><label for="pay_later_payment_page"><?php _e('Payment Page', 'wp-base' ) ?></label></th>
				<td>
					<?php wp_dropdown_pages( array(
							'name'				=> 'mp[gateways][pay-later][payment-page]',
							'selected'			=> wpb_gateway_setting( $this->plugin_name, 'payment-page' ),
							'id'				=> 'pay_later_payment_page',
							'show_option_none'	=> __( 'Home page', 'wp-base' ),
						) );
					?>
					<span class="description app-btm"><?php _e('The page which <code>[app_pay]</code> resides. When client clicks "Pay" button on their account page, or the link generated by MAKE_PAYMENT placeholder in the email, they will be redirected to this page. This page should not be a booking page! If you don\'t add the shortcode to the page, same functionality will be automatically created.', 'wp-base') ?></span>
				</td>
			</tr>
			<tr>
				<th scope="row"><label for="pay_later_form_header"><?php _e('Form Header', 'wp-base' ) ?></label></th>
				<td>
					<input value="<?php echo esc_attr( $this->form_header ); ?>" class="app-1" name="mp[gateways][<?php echo $this->plugin_name ?>][form-header]" type="text" />
					<?php do_action( 'app_admin_gateway_settings_after_form_header', $this ) ?>
					<span class="description"><?php printf( __('Payment form header. Keep it short. <abbr title="%s">Email placeholders</abbr> can be used.', 'wp-base'), WpBConstant::email_desc(1) ) ?></span>
				</td>
			</tr>			
			<tr>
				<th scope="row"><label for="pay_later_email_note"><?php _e('Email Note', 'wp-base' ) ?></label></th>
				<td>
					<?php wp_editor( $this->email_note, 'pay_later_email_note', array('textarea_name'=>'mp[gateways][pay-later][email-note]', 'editor_height'=> WPB_EDITOR_HEIGHT ) ); ?>
					<?php do_action( 'app_admin_gateway_settings_after_email_note', $this ) ?>
					<span class="description"><?php printf( __('Upon submission for an appointment, an email using "Pending" template will be sent to the client, even if "Send Pending email" setting is No. You can use placeholder PAY_LATER_NOTE in this template (or other templates) to inform client how to proceed to make a payment - HTML allowed if email uses HTML. <abbr title="%s">Email placeholders</abbr> can be used.', 'wp-base'), WpBConstant::email_desc(1) ) ?></span>
				</td>
			</tr>

		</table>
	</div>
</div>
<?php
	}

	/**
     * Adds help content to description array
     */
	public function shortcode_desc( $desc ) {
		return array_merge( $desc, array(
			'app_pay'	=> array(
				'name'			=> 'Make Payment',
				'description'	=>  __('Creates a payment form to be displayed when Pay button on List of Bookings or Booking tab of Account Page or link created by MAKE_PAYMENT placeholder in client emails is clicked. This shortcode DOES NOT show anything if page is visited directly. User should always be redirected here either with the button or link in the email.', 'wp-base' ),
				'parameters'	=> array(
					'fields'		=> __('Fields that client needs to fill to submit the form. By default, fields in <b>Display Settings > General > Default User Info Fields in the confirmation form</b> setting are used, but that can be overridden with this attribute. Selectable fields: "name, first_name, last_name, email, phone, address, city, zip"', 'wp-base' ),
				)),
		));
	}
}


add_action( 'plugins_loaded', function( ){
	wpb_register_gateway_plugin( 'WpB_Gateway_Pay_Later', 'pay-later', __('Pay Later', 'wp-base' ) );
	wpb_load_main( __FILE__ );}, 20
);

}