<?php
/**
 * WPB Multiple
 *
 * Common methods for Multiple Appointments Addons (Packages, Recurring, Shopping Cart, Woocommerce, Extras)
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBMultiple' ) ) {

class WpBMultiple {

	/**
     * An array of virtual bookings
     */
	private $virtual = array();

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}

	public function add_hooks(){

		add_action( 'init', array( $this, 'init' ) );												// Check if cart will be emptied
		add_action( 'app_pre_confirmation_check', array( $this, 'check_pre' ), 10, 1 );				// Check submitted vars in pre confirmation
		add_action( 'app_post_confirmation_check', array( $this, 'check_post' ), 10, 1 );			// Check submitted vars in post confirmation
		add_filter( 'app_reserved_status_query', array( $this, 'status_query' ), 16, 2 );			// Modify reserved statuses query
		add_action( 'app_inline_edit_updated', array( $this, 'inline_edit_update' ), 10, 2 );		// Modify user data after inline edit update
		add_action( 'app_change_status_children', array( $this, 'change_status' ), 10, 2 );			// Change status of childs
		add_action( 'app_deleted', array( $this, 'delete_h' ), 10, 2 );								// Delete childs
		add_action( 'app_bulk_status_change', array( $this, 'bulk_status_change' ), 10, 2 );		// Bulk change status of childs
		add_filter( 'app_statuses', array( $this, 'get_statuses' ), 18 );
		add_filter( 'app_pre_confirmation_reply', array( $this, 'pre_confirmation' ), 200, 2 );

		# Email
		add_filter( 'app_email_replace_pre', array( $this, 'email_replace' ), 100, 3 );				// Change several values in case of MA

		# List table
		add_filter( 'app_list_add_cell', array( $this, 'add_cell' ), 12, 4 );						// Add cell for List shortcode

		# Admin
		add_filter( 'app_id_email', array( $this, 'app_id_email' ) );								// Modify app id in admin emails
		add_filter( 'app_search', array( $this, 'app_search' ), 10, 2 );							// Modify search
		add_filter( 'app_bookings_add_cell', array( $this, 'add_cell' ), 12, 4 );					// Add cell for List shortcode
	}

	/**
	 * Check if there is an cart clear request
	 * @since 3.4.5.2
	 * @return none
	 */
	public function init() {
		if ( ! empty( $_GET['app_empty_cart'] ) && check_ajax_referer( 'front', 'ajax_nonce', false ) ) {

			$this->empty_cart();

			$this->a->update_appointments();

			wp_redirect( wpb_add_query_arg( array( 'app_empty_cart' => false, 'ajax_nonce' => false ) ) );
			exit;
		}
	}

	/**
	 * Check if any multiple appt addon is active
	 * @since 2.0
	 * @return bool
	 */
	public function is_active() {
		if ( class_exists( 'WpBShoppingCart' ) || class_exists( 'WpBPackages' ) || class_exists( 'WpBRecurring' ) || class_exists( 'WpBWooCommerce' ) ) {
			return true;
		}

		return false;
	}

	/**
     * Modify statuses array and add cart status inside, only for admin
	 * @return array
     */
	public function get_statuses( $s ) {
		if ( ! $this->is_active() || ! wpb_is_admin() ) {
			return $s;
		}

		if ( ! isset( $s['cart'] ) ) {
			$s['cart'] = $this->a->get_text('cart');
		}

		if ( ! isset( $s['hold'] ) ) {
			if ( ! isset( $this->hold_count ) ) {
				$this->hold_count = $this->a->db->get_var( "SELECT COUNT(*) FROM " . $this->a->app_table . " WHERE status='hold' " );
			}

			if ( $this->hold_count ) {
				$s['hold'] = $this->a->get_text('hold');
			}
		}

		return $s;
	}

	/**
     * Modify the query that defines reserved
	 * Do not auto remove temp appointments
	 * But, directly delete expired temp appointments
	 * @param scope: Which method calls the filter
     */
	public function status_query( $q, $context ) {
		if ( ! $this->is_active() ) {
			return $q;
		}

		$q_add = " OR status='cart'";

		if ( strpos( $q, $q_add ) !== false ) {
			return $q;
		}

		return $q. $q_add;
	}

	/**
	 * Return session ID to mark virtual appts
	 * @since 2.0
	 * @return array
	 */
	public function get_session_id(){
		return $this->a->session()->get_id();
	}

	/**
	 * Get virtual appt values
	 * @since 2.0
	 * @return array
	 */
	public function get_virtual() {
		$id = $this->get_session_id();
		return isset( $this->virtual[$id] ) ? $this->virtual[$id] : array();
	}

	/**
	 * Set virtual appt values
	 * @param $v: array of appt objects
	 * @since 2.0
	 * @return none
	 */
	public function set_virtual( $v ) {
		$id = $this->get_session_id();
		$this->virtual[$id] = $v;
	}

	/**
	 * Reset virtual appt values
	 * @since 2.0
	 * @return none
	 */
	public function reset_virtual() {
		$id = $this->get_session_id();
		$this->virtual[$id] = array();
	}

	/**
	 * Delete a set of virtuals
	 * @since 2.0
	 * @return none
	 */
	public function delete_virtual( $ids ) {
		if ( empty( $ids ) || ! is_array( $ids ) ) {
			return;
		}

		$id = $this->get_session_id();

		foreach ( $ids as $vid ) {
			unset( $this->virtual[$id][$vid] );
		}
	}

	/**
     * Create a virtual app to see what could be happened if such an app existed
	 * @return object
     */
	public function create_virtual( $ID, $slot ) {
		// Account for on behalf booking
		$user_id = empty( $_POST['app_user_id'] ) ? 0 : wpb_clean( $_POST['app_user_id'] );
		$user_id = ! $user_id && get_current_user_id() ? get_current_user_id() : 0;

		$v = new StdClass;
		$v->ID = $ID;
		$v->created	= date ("Y-m-d H:i:s", $this->a->_time );
		$v->user = $user_id;
		$v->location = $slot->get_location();
		$v->service = $slot->get_service();
		$v->worker = $slot->get_worker();
		$v->start = date( "Y-m-d H:i:s", $slot->get_start() );
		$v->end = date( "Y-m-d H:i:s", $slot->get_end() );
		$v->seats = $slot->get_unfiltered_pax() ?: 1;

		return $v;
	}

	/**
     * Save app data after inline edit update
	 * @param $booking		object		WpB_Booking instance for current booking values
	 * @param $old_booking	object		WpB_Booking instance for previous booking values
     */
	public function inline_edit_update( $booking, $old_booking ) {

		$data = $booking->get_save_data();

		if ( ! ( is_array( $data ) && $booking->get_ID() ) ) {
			return;
		}

		if ( ! $kids = $this->get_children( $booking->get_ID() ) ) {
			return;
		}

		# If app was a parent and became a child now, assign children to new parent
		$new_parent_arr = $data['parent_id'] ? array( 'parent_id' => $data['parent_id'] ) : array();

		foreach ( $kids as $child ) {
			# If old child is new parent now, set accordingly
			if ( $data['parent_id'] == $child->ID ) {
				$new_parent_arr['parent_id'] = 0;
			}

			# If status changed, match children to the new status
			if ( ! apply_filters( 'app_skip_child_status_change', false, $child, $data['status'], $booking->get_ID() ) ) {

				if ( in_array( $data['status'], array_keys( $this->a->get_statuses() ) ) ) {
					$new_parent_arr['status'] = $data['status'];
				}
			}

			$this->a->db->update(
				$this->a->app_table,
				array_merge( $new_parent_arr, array( 'user' => $data['user'] ) ),
				array('ID' => $child->ID)
			);
		}
	}

	/**
     * Match status of child to parent
	 * @param app_id: Parent ID
     */
	public function change_status( $stat, $app_id ) {
		$app 	= wpb_get_app( $app_id );
		$kids	= $this->get_children( $app_id );
		$result = false;

		if ( $kids && array_key_exists( $app->status, $this->a->get_statuses() ) ) {

			foreach ( $kids as $child ) {

				if ( $stat == $child->status ) {
					continue;
				}

				if ( apply_filters( 'app_skip_child_status_change', false, $child, $stat, $app_id ) ) {
					continue;
				}

				if ( $this->a->db->update( $this->a->app_table, array( 'status' => $app->status ), array( 'ID' => $child->ID ) ) ) {
					$result = true;
					do_action( 'app_child_status_changed', $app->status, $child->ID );
				}
			}
		}

		if ( $result ) {
			wpb_flush_cache();
		}
	}

	/**
     * Bulk change status of childs
	 * @param $stat: New status
	 * @param $post_vars: $_POST variable in array form or an array of ids
     */
	public function bulk_status_change( $stat, $post_vars ) {
		if ( is_array( $post_vars ) && array_key_exists( $stat, $this->a->get_statuses() ) ) {
			$result = false;
			foreach ( $post_vars as $app_id ) {
				foreach ( (array)$this->get_children( $app_id ) as $child ) {

					if ( apply_filters( 'app_skip_child_status_change', false, $child, $stat, $app_id ) ) {
						continue;
					}

					if ( $this->a->change_status( $stat, $child, false ) ) {
						$result = true;
						do_action( 'app_child_status_changed', $stat, $child->ID );
					}
				}
			}

			if ( $result ) {
				wpb_flush_cache();
			}
		}
	}

	/**
     * Delete children handler
	 * @param $post_vars: $_POST variable in array form (app IDs array which are deleted)
     */
	public function delete_h( $context, $app_ids=null ) {
		if ( 'child' == $context || 'children' == $context ) {
			return;
		}

		return $this->delete_children( $app_ids );
	}

	/**
     * Delete childs (In case of expiration)
	 * @param $post_vars: $_POST variable in array form (app IDs array which are deleted) or an array of ids of parents
     */
	public function delete_children( $post_vars ) {
		$result = false;

		if ( ! empty( $post_vars ) && is_array( $post_vars ) ) {

			$children = array();

			do_action( 'app_delete_pre', 'children', $post_vars );

			foreach ( $post_vars as $app_id ) {
				foreach ( (array)$this->get_children( $app_id ) as $child ) {
					do_action( 'app_delete_pre', 'child', $child->ID );
					if ( wpb_delete_booking( $child->ID ) ) {
						$result = true;
						$children[] = $child->ID;
						do_action( 'app_deleted_single', 'child', $child->ID );
					}
				}
			}

			if ( $result ) {
				wpb_flush_cache();
				do_action( 'app_deleted', 'children', $children );
			}
		}

		return $result;
	}

	/**
     * Get children given a parent appointment
	 * @param $app_id: Parent ID
	 * @param $apps: Optionally a list of app objects to get children from
	 * @return array of objects
     */
	public function get_children( $app_id, $apps = null, $order_by = 'start' ) {
		if ( ! $app_id ) {
			return array();
		}

		$app_text	= ! empty( $apps ) ? implode( '_', array_keys( $apps ) ) .'_'. $app_id .'_'. $order_by: $app_id;
		$id			= wpb_cache_prefix() . 'child_apps_'. $app_text;
		$kids		= wp_cache_get( $id );

		if ( false === $kids ) {
			if ( ! empty( $apps ) ) {
				$kids = (array)$this->get_children_from_apps( $app_id, $apps );
			} else {
				$kids = (array)$this->a->db->get_results( $this->a->db->prepare(
							"SELECT * FROM " . $this->a->app_table .
							" WHERE parent_id=%d AND parent_id>0 ORDER BY %s", $app_id, $order_by ),
							OBJECT_K );
			}

			wp_cache_set( $id, $kids );
		}

		return $kids;
	}

	/**
     * Get children from a known list of objects given a parent appointment
	 * Adapted from WP function get_page_children
	 * @param $app_id: Parent id
	 * @param $apps: A list of app objects to get children from
	 * @return array of objects
     */
	public function get_children_from_apps( $app_id, $apps ) {
		// Build a hash of ID -> children.
		$children = array();
		foreach ( (array) $apps as $app ) {
			$children[ intval( $app->parent_id ) ][] = $app;
		}

		$app_list = array();

		// Start the search by looking at immediate children.
		if ( isset( $children[ $app_id ] ) ) {
			// Always start at the end of the stack in order to preserve original `$apps` order.
			$to_look = array_reverse( $children[ $app_id ] );

			while ( $to_look ) {
				$p = array_pop( $to_look );
				$app_list[] = $p;
				if ( isset( $children[ $p->ID ] ) ) {
					foreach ( array_reverse( $children[ $p->ID ] ) as $child ) {
						// Append to the `$to_look` stack to descend the tree.
						$to_look[] = $child;
					}
				}
			}
		}

		return $app_list;
	}

	/**
     * Find other children of a child's parent
	 * @param app_id: ID of the Child
	 * @return array of objects
     */
	public function get_siblings( $app_id ) {
		if ( ! $app_id ) {
			return false;
		}

		$identifier = wpb_cache_prefix() . 'sibling_apps_'. $app_id;
		$siblings = wp_cache_get( $identifier );

		if ( false === $siblings ) {
			$child = wpb_get_app( $app_id );
			if ( empty( $child->parent_id ) ) {
				$siblings = null;
			} else {
				$siblings = $this->a->db->get_results( $this->a->db->prepare(
					"SELECT * FROM " . $this->a->app_table .
					" WHERE parent_id=%d AND parent_id>0 AND ID<>%d",
					$child->parent_id, $app_id ), OBJECT_K );
			}

			wp_cache_set( $identifier, $siblings );
		}

		return $siblings;
	}

	/**
     * Read cart items from session
	 * @return array
     */
	public function get_cart_items( ) {
		return array_filter( array_map( 'intval', (array)wpb_get_session_val('app_cart', array()) ) );
	}

	/**
     * Get all appts in a cart in a certain status
	 * @return array of objects
     */
	public function get_apps( $status = 'hold' ) {
		if ( ! $ids = $this->get_cart_items() ) {
			return false;
		}

		$ids_str = implode("','", array_map( 'esc_sql', $ids ) );

		return $this->a->db->get_results( $this->a->db->prepare(
			"SELECT * FROM " . $this->a->app_table .
			" WHERE ID IN ('". $ids_str ."') AND status=%s ORDER BY start", $status  ), OBJECT_K
		);
	}

	/**
     * Check if apps in the SESSION are still in the DB and in cart status
	 * If not, clear them
	 * @return none
     */
	public function check_cart() {

		// When cart is not enabled, or emptied there must not be any appt in the DB related to the session
		if ( ! $this->get_cart_items( ) ) {
			$ids = $this->a->db->get_col( $this->a->db->prepare(
				"SELECT ID FROM " . $this->a->app_table.
				" WHERE (status='cart' OR status='hold') AND ID IN(
				SELECT object_id FROM ". $this->a->meta_table."
				WHERE meta_type='app' AND meta_key='session_id'
				AND meta_value=%s )", $this->get_session_id()
			));

			if ( ! empty( $ids ) ) {
				wpb_delete_booking( $ids );
			}
		}

		$ids = $this->get_cart_items();
		if ( empty( $ids ) )
			return;

		if ( ! $results = $this->get_apps( 'cart' ) ) {
			wpb_set_session_val('app_cart', null);
			return;
		}

		$exists = array();
		foreach( $results as $r ) {
			$exists[] = $r->ID;
		}

		foreach ( $ids as $key => $id ) {
			if ( ! in_array( $id, $exists ) ) {
				unset( $ids[ $key ] );
			}
		}

		wpb_set_session_val('app_cart', $ids);
	}

	 /**
     * Create an "app_value" result using app ids in session variable
	 * @return array if success, false if failure
     */
	public function values() {
		if ( ! $results = $this->get_apps( 'cart' ) ) {
			wpb_set_session_val( 'app_cart', null );
			return false;
		}

		$out = array();
		foreach( $results as $r ) {
			$r = apply_filters( 'app_cart_values', $r, $results );

			if ( empty( $r->service ) ){
				continue;
			}

			$slot = new WpB_Slot( new WpB_Booking( $r->ID ) );
			$out[] = $slot->pack( );
		}

		return $out;
	}

	 /**
     * Adds a booking to DB in cart status, saves app_id to the cart
	 * This is "lazy load", so not all the properties are complete
	 * @param $context:	package or recurring
	 * @return $app_id (integer) if success, false if failure
     */
	public function add( $val, $main = 0, $parent_id = 0, $context = '', $key = null ) {

		$slot		= new WpB_Slot( $val );
		$pax		= 'package' == $context ? apply_filters( 'app_pax', 1, $main, $slot ): $slot->get_pax();
		$category	= $slot->get_category();
		$service	= $slot->get_service();
		$start		= $slot->get_start();
		$end		= $slot->get_end();
		$format 	= $slot->is_daily() ? 'Y-m-d' : 'Y-m-d H:i:s';

		if ( $main && ! $parent_id ) {
			$main_service = $this->a->get_service( $main );
			$uprice =  apply_filters( 'app_get_price', (! empty( $main_service->price ) ? $main_service->price : 0), $slot, false );
			if ( BASE('Recurring') && $repeat = BASE('Recurring')->get_repeat() ) {
				$uprice = $uprice * $repeat;
			}
		} else {
			$uprice = 0;
		}

		$data = array(
			'parent_id'	=>	$parent_id,
			'created'	=>	date( 'Y-m-d H:i:s', $this->a->_time ),
			'user'		=>	! empty( $_POST['app_user_id'] ) ? $_POST['app_user_id'] : get_current_user_id(),
			'location'	=>	$slot->get_location(),
			'service'	=>	$service,
			'worker'	=> 	$slot->get_worker(),
			'status'	=>	'cart',
			'start'		=>	date( $format, $start ),
			'end'		=>	date( $format, $end ),
			'seats'		=>	$pax,
			'price'		=>	$pax * $uprice,
		);

		$result = $this->a->db->insert( $this->a->app_table, $data );

		if ( $result ) {
			$app_id = $this->a->db->insert_id;

			$booking = new WpB_Booking( $app_id );

			$booking->update_session_id( $this->get_session_id() );
			$booking->update_category( $category );

			if ( $main && $context ) {
				# Mark booking as package or recurring
				if ( 'recurring' == $context ) {
					$booking->set_as_recurring( $main, $key );
				} else {
					$booking->set_as_package( $main, $key );
					if ( $parent_id ) {
						wpb_flush_cache();
						$parent = new WpB_Booking( $parent_id );
						if ( $parent->is_recurring() ) {
							$booking->set_as_recurring( $main, $key );
						}
					}
				}
			}

			if ( ! empty( $_POST['has_cart'] ) ) {
				$temp = $this->get_cart_items();

				$temp[] = $app_id;
				wpb_set_session_val( 'app_cart', array_filter( array_unique( $temp ) ) );
			}

			wpb_flush_cache();
			return $app_id;
		}

		return false;
	}

	/**
     * Take a booking to hold status
	 * @return db result
     */
	public function hold( $app_id ) {
		if ( ! $app_id ) {
			return false;
		}

		if ( $this->a->db->update( $this->a->app_table, array( 'status'=>'hold' ), array( 'ID'=>$app_id ) ) ) {
			wpb_flush_cache();
			return true;
		}

		return false;
	}

	/**
     * Take a booking back to cart status
	 * Also parent and siblings returned back to cart
	 * @return db result
     */
	public function unhold( $app_id ) {
		if ( ! $app_id ) {
			return false;
		}

		$result = 	false;
		$app 	= 	wpb_get_app( $app_id );
		$parent = 	! empty( $app->parent_id )
					? array( $app->parent_id => wpb_get_app( $app->parent_id ) )
					: array();
		$me			= array( $app_id => $app );
		$siblings 	= $this->get_siblings( $app_id );
		$children 	= $this->get_children( $app_id );
		$friends 	= $this->get_apps(); // Apps in cart + in hold
		$related 	= array_merge( (array)$siblings, (array)$children, (array)$friends, $parent, $me );

		foreach ( $related as $rel ) {
			if ( empty( $rel->ID ) ) {
				continue;
			}

			if ( $this->a->db->update( $this->a->app_table, array( 'status' => 'cart'), array( 'ID' => $rel->ID ) ) ) {
				$result = true;
			}
		}

		if ( $result ) {
			wpb_flush_cache();
		}

		return $result;
	}

	/**
     * Removes a cart item from cart
	 * @return $val string if success, false if failure
     */
	public function remove( $app_id ) {
		if ( ! $ids = $this->get_cart_items() ) {
			return false;
		}

		$key = array_search( $app_id, $temp );

		if ( false !== $key ) {
			unset( $temp[ $key ] );
			wpb_set_session_val('app_cart', $temp);
			return true;
		}

		return false;
	}

	/**
     * Deletes cart contents and related apps from DB
	 * @return none
     */
	public function empty_cart() {
		if( ! $ids = $this->get_cart_items() ) {
			$this->check_cart();
			return false;
		}

		$result = false;
		foreach ( $ids as $app_id ) {
			if ( $this->delete_item( $app_id ) ) {
				$result = true;
			}
		}

		do_action( 'app_cart_emptied', wpb_get_session_val('app_cart') );

		wpb_set_session_val('app_cart', null);

		return $result;
	}

	/**
     * Removes cart item from cart and related app from DB
	 * @param $val:
	 * @return array
     */
	public function remove_item( $val ) {
		$slot	= new WpB_Slot( $val );
		$app_id	= $slot->get_app_id();

		if ( ! $app_id ) {
			return array();
		}

		$save_app = wpb_get_app( $app_id );

		if ( $this->delete_item( $app_id ) ) {
			$reply_array['unblocked'] = $this->block_unblock( array( $save_app ), false );
			return $reply_array;
		} else {
			return $array();
		}
	}

	/**
     * Removes cart item from cart and related app from DB
	 * @return string if success, false if failure
     */
	public function delete_item( $app_id ) {
		if ( wpb_delete_booking( $app_id, 'cart' ) ) {
			if ( $temp = $this->get_cart_items( ) ) {
				if ( is_array( $temp ) ) {
					$key = array_search( $app_id, $temp );
					if ( false !== $key ) {
						unset( $temp[ $key ] );
						wpb_set_session_val('app_cart', $temp);
					}
				}
			}

			return true;
		}

		return false;
	}

	/**
	 * Minimum number of appointments that can be booked
	 * @param $value_arr	null|array		null or an array "packed" strings which can be directly used to instantiate WpB_Slot class
	 * @return integer
	 */
	public function get_apt_count_min( $value_arr=null ) {
		return apply_filters( 'app_apt_count_min', wpb_setting( 'apt_count_min', 1 ), $value_arr );
	}

	/**
	 * Maximum number of appointments that can be booked
	 * @param $value_arr	null|array		null or an array of "packed" strings which can be directly used to instantiate WpB_Slot class
	 *										for an example usage see cart_contents_html method
	 * @since 3.0
	 * @return integer
	 */
	public function get_apt_count_max( $value_arr ) {
		return apply_filters( 'app_apt_count_max', wpb_setting( 'apt_count_max', 0 ), $value_arr );
	}

	/**
     * Check max limit before continuing any further
	 *
     */
	public function check_pre( $value_arr ) {

		$app_count_max = $this->get_apt_count_max( $value_arr );

		if ( $app_count_max && apply_filters( 'app_multiple_app_count', count( $value_arr ), $value_arr ) > $app_count_max ) {
			die( json_encode( array(
				'error' 	  => sprintf( $this->a->get_text('limit_exceeded'), $app_count_max ),
				'remove_last' => 1 )
			));
		}
	}

	/**
     * Check min and max limit before save
     */
	public function check_post( $value_arr ) {

		$this->check_pre( $value_arr );

		$app_count_min = $this->get_apt_count_min( $value_arr );
		if ( apply_filters( 'app_multiple_app_count', count( $value_arr ), $value_arr ) < $app_count_min ) {
			die( json_encode( array( 'error' => sprintf( $this->a->get_text('too_less'), $app_count_min ) ) ) );
		}
	}

	/**
     * Get remaining time (Time left to checkout a cart before it is deleted)
	 * @param $ids: An array of app_id's in the cart
	 * @return integer (time in seconds)
     */
	public function remaining_time( $ids = null, $user_id = 0 ) {
		$ids = null === $ids ? $this->get_cart_items() : $ids;

		if ( ( empty( $ids ) || ! is_array( $ids ) ) &&  ! $user_id ) {
			return 0;
		}

		$cdown = wpb_setting("countdown_time");
		if ( $cdown > 0 ) {
			$clear_secs = (int)$cdown * 60;
		} else {
			return 0;
		}

		if ( $ids ) {
			$ids_str = implode("','", array_map( 'esc_sql', $ids ) );
		} else {
			$ids_str = uniqid().WPB_HUGE_NUMBER;
		}

		$query = $this->a->db->prepare(
			"SELECT created
			FROM {$this->a->app_table}
			WHERE created>%s
			AND ( ID IN ('". $ids_str ."') OR (user<>0 AND user=%d) )
			AND (status='cart' OR status='hold')
			ORDER BY created ASC
			LIMIT 1", date ("Y-m-d H:i:s", $this->a->_time - $clear_secs ), $user_id );

		$created = $this->a->db->get_var( $query );

		if ( ! $created ) {
			return 0;
		} else {
			return strtotime( $created ) - current_time('timestamp') + $clear_secs;
		}
	}

	/**
     * Add remaining time and cart contents to pre confirmation output array
	 * @since 2.0
	 * @return array
     */
	public function pre_confirmation( $reply_array, $value_arr ) {
		if ( ! isset( $reply_array['remaining_time'] ) ) {
			$reply_array['remaining_time'] = $this->remaining_time();
		}

		if ( ! empty( $value_arr ) && count( $value_arr ) > 1 ) {
			$reply_array['cart_contents'] = $this->cart_contents_html( $value_arr );
		}

		if ( ! $apps = BASE('Multiple')->get_cart_items() ) {
			return $reply_array;
		}

		$block = $this->block_unblock( $apps );

		if ( ! empty( $block ) ) {
			$reply_array['blocked'] = array_unique( $block );
		}

		return $reply_array;
	}

	/**
     * Helper function to block/unblock slots on the front end
	 * @param	$apps		array		Array of StdClass app objects
	 * @param	$block		bool		If true block, if false, unblock
	 * @since 3.5.7.6
	 * @return string
     */
	private function block_unblock( $apps, $block = true ) {
		$out			= array();
		$tb_sec			= 60 * $this->a->get_min_time();
		$true_if_block	= $block ? true : false;

		foreach( $apps as $app ) {

			$slot	= new WpB_Slot( new WpB_Booking( $app ) );
			$rep	= absint( ( $slot->get_end() - $slot->get_start() ) / $tb_sec );

			foreach ( range( 1, max( 1, $rep ) ) as $k ) {

				if ( $true_if_block === (bool)$slot->why_not_free( ) ) {
					if ( $main = $this->a->is_app_package( $slot->get_app_id() ) ) {
						$slot->set_service( $main );
						$out[] = $slot->pack( true );
					} else {
						$out[] = $slot->pack( true );
					}
				}

				$slot->set_start( $slot->get_start() + $tb_sec );
				$slot->set_end( $slot->get_end() + $tb_sec );
			}
		}

		return $out;
	}

	/**
     * Generate html for cart contents
	 * @since 3.0
	 * @return string
     */
	public function cart_contents_html( $value_arr ) {
		$value_arr = apply_filters( 'app_cart_content_items', $value_arr );

		if ( wpb_is_hidden('details') ) {
			return '';
		}

		$cart = array();

		foreach ( $value_arr as $val  ) {
			$cart[] = $val instanceof WpB_Slot ? $val : new WpB_Slot( $val );
		}

		$cart_keys = array_keys( $cart );

		if ( count( $cart_keys ) > 1 ) {
			$starts = $services = array();
			foreach ( $cart as $key => $r ) {
				$starts[ $key ] = $r->get_start();
				$services[ $key ] = $r->get_service();
			}

			if ( defined( 'WPB_CART_CONTENTS_SORT_BY_DATE' ) && WPB_CART_CONTENTS_SORT_BY_DATE ) {
				array_multisort( $starts, $services, $cart );
			} else {
				array_multisort( $services, $starts, $cart );
			}
		}

		$is_next_day = false;

		$html  = '<label><span class="app-conf-title">'.$this->a->get_text('details'). '</span>';
		$html .= '<dl>';

		foreach ( $cart as $key => $r ) {

			$pax = apply_filters( 'app_pax', 1, $r->get_service(), $r );
			$pax_text = $pax > 1 ? ' ('. $pax. ' '. $this->a->get_text('pax') . ')' : '';

			$start_end = $this->format_start_end( $r ) . $pax_text;

			if ( strpos( $start_end, '<sup' ) !== false ) {
				$is_next_day = true;
			}

			if ( ! isset( $services[ ($key-1) ] ) || $services[ $key ] != $services[ ($key-1) ] ) {
				$html .= '<dt>' . $this->a->get_service_name( $r->get_service() ) . '</dt>';
			}

			$html .= '<dd  data-value="'.$r->pack().'">';

			# Make this false to disable remove from cart/details
			if ( apply_filters( 'app_cart_allow_remove', true, $r, $cart ) ) {
				$html .= '<a class="app-remove-cart-item" data-value="'.$r->pack().'" data-app_id="'.$r->get_app_id().'" href="javascript:void(0)" title="'.esc_attr($this->a->get_text('click_to_remove')).'">';
				$html .= '<em class="app-icon icon-trash"></em></a>';
			} else {
				$html .= '<em class="app-icon icon-clock"></em></a>';
			}
			$html .= $start_end;
			$html .= '</dd>';
		}

		if ( $is_next_day ) {
			$html .= '<dd class="app-next-day-note"><sup> *</sup>'.$this->a->get_text('next_day').'</dd>';
		}

		$html .= '</dl>';
		$html .= '</label>';

		return $html;
	}

	/**
     * Show both start and end date/times in a space saving way
	 * @param $r	object		WpB_Slot object
	 * @since 3.0
	 * @return string
     */
	public function format_start_end( $r ) {
		$hours = intval( ceil( ($r->get_end() - $r->get_start())/9600 ) );

		$cl_start	= $this->a->client_time( $r->get_start() );
		$cl_end 	= $this->a->client_time( $r->get_end() );

		if ( $r->is_daily() || $hours >= 24 ) {
			$result = $this->a->format_start_end( $cl_start, $cl_end - 1 );
		} else {
			$is_next_day = date( 'Y-m-d', $cl_start ) == date( 'Y-m-d', $cl_end -1 ) ? false : true;
			$end_hour = $r->client_end_time();
			if ( '00:00' === $end_hour && 'H:i' == $this->a->time_format )
				$end_hour = '24:00';
			$result = $r->client_dt() .' - '. $end_hour;

			if ( $is_next_day ) {
				$result = $result . '<sup class="app-next-day"> *</sup>';
			}
		}

		return apply_filters( 'app_multiple_format_start_end', $result, $r );
	}

	/**
     * Find min start timestamp including children (latest appt)
	 * @since 3.0
	 * @return integer
     */
	public function find_start_min( $r ) {
		$child_start_min = strtotime( $r->start );

		if ( ! empty( $r->parent_id ) || ! $kids = $this->get_children( $r->ID ) ) {
			return $child_start_min;
		}

		foreach ( $kids as $child ) {
			$child_start = strtotime( $child->start );
			if ( $child_start < $child_start_min ) {
				$child_start_min = $child_start;
			}
		}

		return $child_start_min;
	}

	/**
     * Find max end timestamp including children (latest appt)
	 * @since 3.0
	 * @return integer
     */
	public function find_end_max( $r ) {
		$child_end_max = strtotime( $r->end );

		if ( ! empty( $r->parent_id ) || ! $kids = $this->get_children( $r->ID ) ) {
			return $child_end_max;
		}

		foreach ( $kids as $child ) {
			$child_end = strtotime( $child->end );
			if ( $child_end > $child_end_max ) {
				$child_end_max = $child_end;
			}
		}

		return $child_end_max;
	}

	/**
     * Replace email placeholders
	 * @return string
     */
	public function email_replace( $text, $r, $context ) {

		if ( ! ( $this->is_active() && $kids = $this->get_children( $r->ID ) ) ) {
			return $text;
		}

		$nof_apps	= count( $kids ) + 1;
		$text 		= str_replace( 'NOF_APPS', $nof_apps, $text );

		if ( apply_filters( 'app_multiple_email_replace_skip_subject', true, $r, $context ) ) {

			if ( preg_match( '/(reminder|follow_up|subject)/', $context ) ) {
				return $text;
			}
		}

		if ( strpos( $context, 'cancellation' ) !== false ) {
			return $text;
		}

		$booking	= new WpB_Booking( $r->ID );
		$format		= $booking->is_daily() ? $this->a->date_format : $this->a->dt_format;
		$duration	= $parent_start_ts = $parent_end_ts = $child_end_max = 0;
		$serv_ids 	= $worker_names = $start_ts_arr = array();

		foreach ( $kids as $child ) {

			if ( apply_filters( 'app_multiple_email_replace_skip_child', false, $child, $r, $context ) ) {
				continue;
			}

			$_booking		= new WpB_Booking( $child->ID );
			$child_start	= strtotime( $_booking->get_start() );
			$child_end		= strtotime( $_booking->get_end() );

			if ( $child_end > $child_end_max ) {
				$child_end_max = $child_end;
			}

			$duration += $child_end - $child_start;

			if ( $this->a->is_internal( $_booking->get_service() ) ) {
				continue;
			}

			$serv_ids[]		 = $_booking->get_service();
			$worker_names[]  = $this->a->get_worker_name( $_booking->get_worker() );
			$format_c		 = $_booking->is_daily( ) ? $this->a->date_format : $this->a->dt_format;
			$start_ts_arr[]	 = $_booking->get_start();
		}

		if ( ! apply_filters( 'app_multiple_email_replace_skip_parent', false, $r, $context ) ) {
			$serv_ids[] 		= $booking->get_service();
			$worker_names[] 	= $this->a->get_worker_name( $booking->get_worker() );
			$start_ts_arr[]	    = $booking->get_start();
			$parent_start_ts	= strtotime( $booking->get_start() );
			$parent_end_ts 		= strtotime( $booking->get_end() );
			$duration 		   += $parent_end_ts - $parent_start_ts;
		}

		$end 		= date_i18n( $format, max( $parent_end_ts, $child_end_max ) + $booking->client_offset() );
		$server_end = date_i18n( $format, max( $parent_end_ts, $child_end_max ) );
		$serv_ids 	= apply_filters( 'app_multi_service_ids', array_unique( $serv_ids ), $r );

		$service_names = array_unique( array_map( array( $this->a, 'get_service_name' ), $serv_ids ) );
		sort( $service_names );
		$service_name = implode( ", ", $service_names );

		$worker_names = array_unique( $worker_names );
		sort( $worker_names );
		$worker_name = implode( ", ", $worker_names );

		sort( $start_ts_arr );

		$start_arr = $server_start_arr = array();
		foreach ( $start_ts_arr as $ts ) {
			$start_arr[] = date_i18n( $format, $this->a->client_time( wpb_strtotime( $ts ) ) );
			$server_start_arr[] = date_i18n( $format, wpb_strtotime( $ts ) );
		}

		if ( apply_filters( 'app_multiple_email_replace_skip_time_change', false, $r, $context ) ) {
			$text = str_replace(
				array('WORKER', 'SERVICE'),
				array($worker_name, $service_name),
				$text
			);
		} else {
			$text = str_replace(
				array('WORKER', 'SERVICE', 'SERVER_END_DATE_TIME', 'SERVER_DATE_TIME', 'END_DATE_TIME', 'DATE_TIME', 'DURATION'),
				array($worker_name, $service_name, $server_end, implode( ' / ', $server_start_arr ), $end, implode( ' / ', $start_arr ), wpb_format_duration( intval($duration/60) )),
				$text
			);
		}

		return $text;
	}

/****************************************************
* Methods for admin
*****************************************************
*/

	/**
	 * Add "MA" tab
	 */
	public function add_tab( $tabs ) {
		if ( ! $this->is_active() ) {
			return $tabs;
		}

		$temp['multiple'] = __('Multiple Appointments', 'wp-base');
		$tabs = array_merge( $temp, $tabs );

		return $tabs;
	}

	/**
     * Get number and start dates of jobs_total, etc for an app
	 * @return array
     */
	public function get_app_job_details( $app ) {
		$app_id = empty( $app->ID ) ? 0 : $app->ID;
		$identifier = wpb_cache_prefix() . 'app_job_details_' . $app_id;
		$details = wp_cache_get( $identifier );

		if ( $details === false ) {
			$details = array();
			# Main is only set for package parent - But what if package parent is a child of a multi?
			if ( $this->a->is_app_package( $app_id ) || $this->a->is_recurring( $app_id ) || wpb_get_app_meta( $app_id, 'woocommerce') ) {
				$completed = $cancelled = $remaining = array();
				$parent = ! empty( $app->parent_id ) ? array( $app->parent_id => wpb_get_app( $app->parent_id ) ) : array();

				$me = array( $app_id => wpb_get_app( $app_id, true ) );

				if ( empty( $parent ) ) {
					# This is a parent itself
					$children = $this->get_children( $app_id );
					$related = array_merge( (array)$children, $me  );
				} else {
					# This is a child
					$siblings = $this->get_siblings( $app_id );
					$children = $this->get_children( $app_id );
					$related = array_merge( (array)$siblings, (array)$children, $parent, $me );
				}

				$related = array_filter( $related );

				if ( ! empty( $related ) ) {
					foreach ( $related as $r ) {
						// What if this is a multiple appointment of packages...
						if ( 'completed' == $r->status )
							$completed[] = $r->ID;
						else if ( 'removed' == $r->status )
							$cancelled[] = $r->ID;
						else
							$remaining[] = $r->ID;
					}
				}

				$details = array(
					'nof_jobs_completed'=> count( $completed ),
					'nof_jobs_cancelled'=> count( $cancelled ),
					'nof_jobs_remaining'=> count( $remaining ),
					'nof_jobs_total'	=> count( $completed ) + count( $cancelled ) + count( $remaining ),
					'completed_jobs'	=> $completed,
					'cancelled_jobs'	=> $cancelled,
					'remaining_jobs'	=> $remaining,
					'all_jobs'			=> $related,
				);
			}

			wp_cache_set( $identifier, $details );
		}

		return $details;
	}

	/**
     * Add cell content to Manage Bookings & List Shortcode
     */
	public function add_cell( $cell, $col, $app, $args ) {
		if ( substr( $col, 0, 9 ) != 'nof_jobs_' ) {
			return $cell;
		}

		if ( $cell ) {
			return $cell;
		}

		$details = $this->get_app_job_details( $app );

		$out = ! empty( $details[$col] ) ? $details[$col] : ' - ';
		return $cell . $out;
	}

	/**
     * Modify $app_id in admin emails and add children, so that all of them are searched
     */
	public function app_id_email( $app_id ) {
		$add = '';
		if ( $kids = $this->get_children( $app_id ) ) {
			foreach ( $kids as $child ) {
				$add .= '+'. $child->ID;
			}
		}

		return $app_id . $add;
	}

	/**
     * Modify $app_id in admin search to add parent, siblings and children
     */
	public function app_search( $s, $stype ) {
		if ( 'app_id' != $stype ) {
			return $s;
		}

		$app_id = $s;
		$app = wpb_get_app( $app_id );

		if ( empty( $app->ID ) ) {
			return $s;
		}

		$add_arr = array();
		if ( $kids = $this->get_children( $app_id ) ) {
			foreach ( $kids as $child ) {
				$add_arr[] = $child->ID;
			}
		}
		if ( $app->parent_id ) {
			$add_arr[] = $app->parent_id;
			// Find siblings
			if ( $kids = $this->get_children( $app->parent_id ) ) {
				foreach ( $kids as $child ) {
					$add_arr[] = $child->ID;
				}
			}
		}
		$add_arr = array_unique( array_filter( $add_arr ) );
		rsort( $add_arr );
		$add = implode( ' ', $add_arr );

		return $s . ' '. $add;
	}

}

	BASE('Multiple')->add_hooks();
}