<?php
/**
 * WPB Payment Gateway Addons
 *
 * Parent class & register function for payment gateways
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists('WpB_Gateway_API') ) {

  class WpB_Gateway_API {

    # Private gateway slug. Lowercase alpha (a-z) and dashes (-) only please!
    public $plugin_name = '';

    # Name of the gateway, for the admin side.
    public $admin_name = '';

    # Public name of the gateway, for lists and such.
    public $public_name = '';

    # Url for an image for checkout method. Displayed on method form
    public $method_img_url = '';

    # Contains the url to send payment notifications to if needed by your gateway. Populated by the parent class
    public $ipn_url;

	public $checkout_error = false;

	public $publishable_key, $private_key, $instructions, $email_note, $currency, $mode;

	/**
     * Constructor
     */
	public function __construct() {

		$this->a = BASE();

		$this->checkout_error = false;

		$this->_generate_ipn_url();

		$this->on_creation();

		if ( ! empty( $_GET['app_paymentgateway'] ) ) {
			do_action( 'app_handle_payment_return_' . $_GET['app_paymentgateway'] );
		}

		add_action( 'app_gateway_settings', array($this, 'gateway_settings_box') );
		add_filter( 'app_gateway_settings_filter', array($this, 'process_gateway_settings') );
		add_action( 'wp_ajax_nopriv_app_checkout_return_' . $this->plugin_name, array( $this, 'process_checkout_return' ) );
		add_action( 'wp_ajax_app_checkout_return_' . $this->plugin_name, array( $this, 'process_checkout_return' ) );
		add_action( 'wp_ajax_nopriv_app_ipn_' . $this->plugin_name, array($this, 'process_ipn_return')); // Send Paypal to IPN function
		add_action( 'wp_ajax_app_ipn_' . $this->plugin_name, array($this, 'process_ipn_return')); 		// Send Paypal to IPN function
		add_action( 'app_shortcode_found', array( $this, 'is_conf_page' ) );							// Set a flag if this is a conf page
		add_action( 'template_redirect', array( $this, 'template_redirect' ) );							// Print error/success message in js alert
		add_action( 'app_addon_settings_link', array( $this, 'settings_link' ) );						// Add settings link to Addons page
		add_filter( 'app_confirmation_total_price', array( $this, 'add_fee' ), 500, 2 );
	}


    /****** Below are the public methods you may overwrite via a plugin ******/

    /**
     * Runs when gateway class is instantiated.
     */
    function on_creation() {
    }

	/**
	 * Check if curl exists
	 */
	protected function curl_check() {
		if ( !function_exists( 'curl_init' ) ) {
			wpb_notice( sprintf( __( '%s requires curl function. Contact your hosting company to install curl extension.', 'wp-base' ), $this->admin_name ) );
			return false;
		}

		return true;
	}

	/**
	 * Check if running required WPB version
	 * @param $required		string		Minimum required version
	 * @since 3.5.7
	 */
	protected function check_wpb_version( $required ) {
		if ( version_compare( WPB_VERSION, $required, '<' ) ) {
			wpb_notice( sprintf( __( '%s Payment Gateway requires WP BASE version %s or above.', 'wp-base' ), $this->admin_name, $required ) );
			return false;
		}

		return true;
	}

	/**
	* Retreive latest settings
	*/
	public function apply_settings(){
		$this->method_img_url 	= WPB_PLUGIN_URL . '/images/credit_card_64.png';
		$this->public_name		= wpb_gateway_setting( $this->plugin_name, 'name', __('Credit Card', 'wp-base' ) );
		$this->currency			= wpb_gateway_setting( $this->plugin_name, 'currency', 'USD' );
		$this->publishable_key 	= wpb_gateway_setting( $this->plugin_name, 'publishable_key' );
		$this->private_key 		= wpb_gateway_setting( $this->plugin_name, 'private_key' );
		$this->mode 			= wpb_gateway_setting( $this->plugin_name, 'mode', 'sandbox' );
		$this->locale 			= wpb_gateway_setting( $this->plugin_name, 'locale', substr( $this->a->get_locale(), -2 ) );
		$this->instructions		= wpb_gateway_setting( $this->plugin_name, 'instructions' );
		$this->email_note 		= wpb_gateway_setting( $this->plugin_name, 'email-note' );
		$this->merchant_email	= wpb_gateway_setting( $this->plugin_name, 'merchant_email' );
	}

	/**
     * Places a wrapper around payment form
     */
	public function _payment_form_wrapper($app_id, $post_id, $amount) {

		$content = $this->payment_form($app_id, $post_id, $amount);

		return $content;
    }

    /**
     * Return fields you need to add to the payment screen, like your credit card info fields.
     *
     * @param integer $app_id. ID of the Appointment for which payment would be done. Cannot be zero at this point.
     * @param integer $post_id. ID of the Make an Appointment post/page
	 * @param float $amount. Total full or down payment amount
     */
    public function payment_form($app_id, $post_id, $amount) {
    }

    /**
     * Use this to process any fields you added. Use the $_POST global,
     *
     * @param integer $app_id. ID of the Appointment for which payment would be done. Cannot be zero at this point.
     * @param integer $post_id. ID of the Make an Appointment post/page
     */
	public function process_payment_form($app_id, $post_id) {
    }

    /**
     * Return the chosen payment details here for final confirmation. You probably don't need
     *  to post anything in the form as it should be in your $_SESSION var already.
     *
     * @param integer $app_id. ID of the Appointment for which payment would be done. Cannot be zero at this point.
     * @param integer $post_id. ID of the Make an Appointment post/page
     */
	public function confirm_payment_form($app_id, $post_id) {
    }

    /**
     * Use this to do the final payment. Create the order then process the payment. If
     *  you know the payment is successful right away go ahead and change the order status
     *  as well.
     *
     * @param integer $app_id. ID of the Appointment for which payment would be done. Cannot be zero at this point.
     * @param integer $post_id. ID of the Make an Appointment post/page
     */
	public function process_payment($app_id, $post_id) {
    }

    /**
     * Runs before page load incase you need to run any scripts before loading the success message page
     */
	public function order_confirmation($order) {
    }

	/**
     * Echo a settings meta box with whatever settings you need for you gateway.
     *  Form field names should be prefixed with mp[gateways][plugin_name], like "mp[gateways][plugin_name][mysetting]".
     *  You can access saved settings via $settings array.
     */
	public function gateway_settings_box($settings) {

    }

    /**
     * Filters posted data from your settings form. Do anything you need to the $settings['gateways']['plugin_name']
     *  array. Don't forget to return!
     */
	public function process_gateway_settings( $settings ) {

		return $settings;
    }

	/**
     * Use to handle any payment returns to the ipn_url. Do not display anything here. If you encounter errors
     *  return the proper headers. Exits after.
     */
	public function process_ipn_return() {
    }

/* Not to be overwritten */

	protected function common_settings( ){
	?>
	<tr>
		<th scope="row"><?php _e('Public Name', 'wp-base' ) ?></th>
		<td>
		<input value="<?php echo esc_attr( $this->public_name ); ?>" class="app-1" name="mp[gateways][<?php echo $this->plugin_name ?>][name]" type="text" />
		<?php do_action( 'app_admin_gateway_settings_after_public_name', $this ) ?>
		<span class="description app-btm"><?php _e('Enter a public name for this payment method that is displayed to users - No HTML', 'wp-base' ) ?></span>
		</td>
	</tr>
	<tr>
		<th scope="row"><?php WpBConstant::echo_setting_name('fee') ?></th>
		<td>
		<input value="<?php echo esc_attr( $this->get_value( 'fee' ) ); ?>" class="app-50" name="mp[gateways][<?php echo $this->plugin_name ?>][fee]" type="text" />
		<span class="description app-btm"><?php WpBConstant::echo_setting_desc('fee') ?></span>
		</td>
	</tr>
	<tr>
		<th scope="row"><?php WpBConstant::echo_setting_name('fee_fixed') ?></th>
		<td>
		<input value="<?php echo esc_attr( $this->get_value( 'fee_fixed' ) ); ?>" class="app-50" name="mp[gateways][<?php echo $this->plugin_name ?>][fee_fixed]" type="text" />
		<span class="description app-btm"><?php WpBConstant::echo_setting_desc('fee_fixed') ?></span>
		</td>
	</tr>

	<tr>
		<th scope="row"><label for="<?php echo $this->plugin_name ?>_instructions"><?php _e('Client Instructions', 'wp-base' ) ?></label></th>
		<td>
		<?php wp_editor( $this->instructions, $this->plugin_name.'_instructions', array_merge( wpb_editor_settings( __CLASS__ ), array( 'textarea_name'=>'mp[gateways]['.$this->plugin_name.'][instructions]' ) ) ); ?>
		<?php do_action( 'app_admin_gateway_settings_after_instructions', $this ) ?>
		<span class="description"><?php _e('These are the instructions to display in the tooltip of gateway image on confirmation form - HTML allowed', 'wp-base' ) ?></span>
		</td>
	</tr>
	<tr>
		<th scope="row"><?php WpBConstant::echo_setting_name('item_name') ?></th>
		<td>
		<input value="<?php echo esc_attr( $this->get_value( 'item_name' ) ); ?>" class="app-1" name="mp[gateways][<?php echo $this->plugin_name ?>][item_name]" type="text" />
		<?php do_action( 'app_admin_gateway_settings_after_item_name', $this ) ?>
		<span class="description app-btm"><?php WpBConstant::echo_setting_desc('item_name') ?></span>
		</td>
	</tr>
	<?php
	}

	/**
     * Adds a settings link
     */
	public static function settings_link( $filename ) {

		if ( $filename ==  pathinfo(__FILE__, PATHINFO_FILENAME) ) {
			echo ' | <a href="'.admin_url( 'admin.php?page=app_monetary&amp;tab=gateways#'.$filename ).'">'. __('Settings', 'wp-base' ) . '</a>';
		}
	}

	/**
	 * Get value for a setting. Fallout to default if not set
	 * @param $setting	string	Name of the setting, e.g. item_name, return
	 * @since 3.0
	 */
	public function get_value( $setting ){
		if ( $val = apply_filters( 'app_gateway_get_value', wpb_gateway_setting( $this->plugin_name, $setting ), $setting, $this ) ) {
			return $val;
		} else if ( 'paypal-standard' == $this->plugin_name && 'public_name' == $setting ) {
			return 'PayPal';
		} else {
			return WpBConstant::defaults( false, $setting );
		}
	}

	/**
     * Adds gateway fee to the total price
     */
	public function add_fee( $total_price, $items ) {
		$prev_price	= $total_price;
		$method 	= isset( $_POST["app_payment_method"] ) ? wpb_clean( $_POST["app_payment_method"] ) : '';

		$fee = str_replace( array( '%', ' ' ), '', wpb_gateway_setting( $method, 'fee' ) ) ?: 0;

		if ( $fee && $method == $this->plugin_name ) {
			$total_price = wpb_round( $total_price * ( 1 + $fee/100 ) );
			WpBDebug::set( 'fee', $total_price - $prev_price );
		}

		$fee_fixed = wpb_sanitize_price( wpb_gateway_setting( $method, 'fee_fixed' ) ) ?: 0;

		if ( $fee_fixed && $method == $this->plugin_name ) {
			$total_price = wpb_round( $total_price + $fee_fixed );
			WpBDebug::set( 'fee', $total_price - $prev_price );
		}

		wpb_set_session_val( 'fee', ( ( $fee + $fee_fixed ) ?: null ) );

		return $total_price;
	}

	/**
	 * Try to find $app_id from $_POST or session
	 */
	public function find_app_id(){
		$app_id = wpb_get_session_val('app_order_id');

		if ( ! $app_id ) {
			$app_id = ! empty( $_POST['app_id'] ) ? absint( $_POST['app_id'] ) : 0;
		}

		return $app_id;
	}

	/**
     * Find return url
     */
	public function return_url( $app_id = null ) {

		$ret = wpb_refresh_url( get_permalink( wpb_find_post_id() ) ) ?: home_url();
		$ret = apply_filters( 'app_paypal_return', $ret, $this ); # For backwards compatibility
		$ret = apply_filters( 'app_payment_gateway_return', $ret, $this );

		if ( $app_id ) {
			$ret = wpb_add_query_arg( 'app_hash', $this->a->create_hash( wpb_get_app( $app_id ), 'open_dialog' ), $ret );
		}

		return $ret;
	}

	/**
     * Automatically called if this is a conf page, i.e. it contains [app_book or [app_confirmation
     */
	public function is_conf_page( $context ){
		if ( 'confirmation' != $context ) {
			return;
		}

		$this->is_conf_page = true;
	}

	/**
     * Populates IPN url var
     */
	 function _generate_ipn_url() {
		$this->ipn_url = admin_url( 'admin-ajax.php?app_paymentgateway='. $this->plugin_name );
    }

	/**
     * Error handling
     */
	public function get_checkout_error(){
		return $this->a->checkout_error;
	}

	public function set_checkout_error( $msg ) {
		$this->a->checkout_error( $msg );
	}

	/**
     * Returns the HTML of a standard CC form
     */
	public function standard_cc_form( $app_id ) {
		$content = '';

		$non_mobile_cl = wpb_is_mobile() ? '' : ' app-non-mobile';

		$email = $name = $address1 = $address2 = $city = $state = $zip = $country = $phone = '';

		extract( BASE('User')->get_app_userdata( $app_id, get_current_user_id() ) );
		$address1 = $address;

		$content .= '<input type="hidden" class="app-cc-submit" value="'.$this->plugin_name.'" />
					<input type="hidden" class="app_id" value="'.$app_id.'" />

			<div class="app_billing_line">
				<label><span>'.$this->a->get_text( 'cc_name' ).'<sup> *</sup></span>
				'.apply_filters( 'app_checkout_error_name', '' ).'
				<input name="name" type="text" class="app-required app_cc_name'.$non_mobile_cl.'" value="'.esc_attr($name).'" /> </label>
			</div>

			<div class="app_billing_line">
				<label><span>'.$this->a->get_text( 'cc_number' ).'<sup> *</sup></span>
				'.apply_filters( 'app_checkout_error_card_num', '' ).'
				<input name="card_num" id="card_num" placeholder="1234 1234 1234 1234" inputmode="numeric" pattern="[0-9]*" class="app-required card-number app_cc_number credit_card_number input_field noautocomplete'.$non_mobile_cl.'" type="text" maxlength="22" />
				</label>
				<div class="hide_after_success nocard cardimage"  id="cardimage" style="background: url('.WPB_PLUGIN_URL.'/images/card_array.png) no-repeat;"></div></label>
			</div>

			<div class="app_billing_line app_expiry_security app-clearfix">
				<div class="app_billing_line_inner"><span class="'.$non_mobile_cl.'">'.$this->a->get_text( 'cc_expiry' ).'<sup> *</sup></span>
				'.apply_filters( 'app_checkout_error_exp', '' ).'
					<label class="app-cc-label" for="exp_month">
						<select name="exp_month" class="app-required app_cc_exp_month'.$non_mobile_cl.'" id="exp_month">
						'.$this->_print_month_dropdown().'
						</select>
					</label>
					<label class="app-cc-label" for="exp_year">
						<select name="exp_year" class="app-required app_cc_exp_year'.$non_mobile_cl.'" id="exp_year">
						'.$this->_print_year_dropdown('', true).'
						</select>
					</label>
				</div>

				<div class="app_billing_line_inner">
					<label><span>'.__('Security Code', 'wp-base').'<sup> *</sup></span>
					'.apply_filters( 'app_checkout_error_card_code', '' ).'
					<input id="card_code" placeholder="CVC/CVV" inputmode="numeric" pattern="[0-9]*" class="app-required app_cc_cvv input_field noautocomplete'.$non_mobile_cl.'" type="text" maxlength="4" /></label>
				</div>
			</div>
		';

		return $content;
	}

	/**
     * Returns the HTML of a standard CC form + Billing form
     */
	public function standard_billing_form( $app_id ) {
		$content = '';

		$email = $name = $address1 = $address2 = $city = $state = $zip = $country = $phone = '';

		extract( BASE('User')->get_app_userdata( $app_id, get_current_user_id() ) );
		$address1 = $address;

		$non_mobile_cl = wpb_is_mobile() ? ' app-mobile' : ' app-non-mobile';

		$content .= '<input type="hidden" class="app-cc-submit" value="'.$this->plugin_name.'" />
					<input type="hidden" class="app_id" value="'.$app_id.'" />

			<div class="app_billing_line">
				<label><span>'.$this->a->get_text('cc_email').'<sup> *</sup></span>
				'.apply_filters( 'app_checkout_error_email', '' ).'
				<input name="email" type="text" class="app-required app_cc_email'.$non_mobile_cl.'" value="'.esc_attr($email).'" /></label>
			</div>

			<div class="app_billing_line">
				<label><span>'.$this->a->get_text('cc_name').'<sup> *</sup></span>
				'.apply_filters( 'app_checkout_error_name', '' ).'
				<input name="name" type="text" class="app-required app_cc_name'.$non_mobile_cl.'" value="'.esc_attr($name).'" /> </label>
			</div>

			<div class="app_billing_line">
				<label><span>'.$this->a->get_text('cc_address1').'<sup> *</sup></span>
				'.apply_filters( 'app_checkout_error_address1', '' ).'
				<input name="address1" type="text" class="app-required app_cc_address1'.$non_mobile_cl.'" value="'.esc_attr($address1).'" placeholder="'.__('Street address, P.O. box, company name, c/o', 'wp-base').'"/>
				</label>
			</div>

			<div class="app_billing_line">
				<label><span>'.$this->a->get_text('cc_address2').'</span>
				<input name="address2" type="text" class="app_cc_address2'.$non_mobile_cl.'" value="'.esc_attr($address2).'" placeholder="'.__('Apartment, suite, unit, building, floor, etc.', 'wp-base').'"/>
				</label>
			</div>

			<div class="app_billing_line">
				<label><span>'.$this->a->get_text('cc_city').'<sup> *</sup></span>
				'.apply_filters( 'app_checkout_error_city', '' ).'
				<input name="city" type="text" class="app-required app_cc_city'.$non_mobile_cl.'" value="'.esc_attr($city).'" /></label>
			</div>

			<div class="app_billing_line">
				<label><span>'.$this->a->get_text('cc_state').'<sup> *</sup></span>
				'.apply_filters( 'app_checkout_error_state', '' ).'
				<input name="state" type="text" class="app-required app_cc_state'.$non_mobile_cl.'" value="'.esc_attr($state).'" /></label>
			</div>

			<div class="app_billing_line">
				<label><span>'.$this->a->get_text('cc_zip').'<sup> *</sup></span>
				'.apply_filters( 'app_checkout_error_zip', '' ).'
				<input id="mp_zip" name="zip" type="text" class="app-required app_cc_zip'.$non_mobile_cl.'" value="'.esc_attr($zip).'" /></label>
			</div>

			<div class="ui-field-contain app_billing_line'.$non_mobile_cl.'">
				<label for="app_cc_country"><span>'.$this->a->get_text('cc_country').'</span>
				'.apply_filters( 'app_checkout_error_country', '' ).'
				<select id="app_cc_country" class="filterable-select" name="country">';

		$content .= '<option >'.$this->a->get_text('select').'</option>';
		foreach( WpBConstant::countries() as $code=>$name) {
			$content .= '<option value="'.$code.'"'.selected($country, $code, false).'>'.esc_attr($name).'</option>';
		}

		$content .= '
				</select>
				</label>
			</div>';

		$content .= '
			<div class="app_billing_line">
				<label><span>'.$this->a->get_text('cc_phone').'<sup> *</sup></span>
				<input name="phone" type="text" class="app-required app_cc_phone'.$non_mobile_cl.'" value="'.esc_attr($phone).'" /></label>
			</div>

			<div class="app_billing_line">
				<label><span>'.$this->a->get_text('cc_number').'<sup> *</sup></span>
				'.apply_filters( 'app_checkout_error_card_num', '' ).'
				<input id="card_num" placeholder="1234 1234 1234 1234" inputmode="numeric" pattern="[0-9]*" class="app-required app_cc_number credit_card_number input_field noautocomplete'.$non_mobile_cl.'"
				type="text" maxlength="22" />
				</label>
				<div class="hide_after_success nocard cardimage"  id="cardimage" style="background: url('.WPB_PLUGIN_URL.'/images/card_array.png) no-repeat;"></div></label>
			</div>

			<div class="app_billing_line app_expiry_security app-clearfix">
				<div class="app_billing_line_inner"><span class="'.$non_mobile_cl.'">'.$this->a->get_text( 'cc_expiry' ).'<sup> *</sup></span>
				'.apply_filters( 'app_checkout_error_exp', '' ).'
					<label class="app-cc-label" for="exp_month">
						<select class="app-required app_cc_exp_month'.$non_mobile_cl.'" id="exp_month">
						'.$this->_print_month_dropdown().'
						</select>
					</label>
					<label class="app-cc-label" for="exp_year">
						<select class="app-required app_cc_exp_year'.$non_mobile_cl.'" id="exp_year">
						'.$this->_print_year_dropdown('', true).'
						</select>
					</label>
				</div>

				<div class="app_billing_line_inner">
					<label><span>'.__('Security Code', 'wp-base').'<sup> *</sup></span>
					'.apply_filters( 'app_checkout_error_card_code', '' ).'
					<input id="card_code" placeholder="CVC/CVV" inputmode="numeric" pattern="[0-9]*" class="app-required app_cc_cvv input_field noautocomplete'.$non_mobile_cl.'"
					type="text" maxlength="4" /></label>
				</div>
			</div>
		';

		return $content;
	}

	/**
     * Finds CC Type from CC number
	 * http://developer.ean.com/general-info/valid-card-types/
     */
	public function _get_card_type($number) {
		$number = str_replace( ' ', '', $number );
		$num_length = strlen($number);

		if ($num_length > 10 && preg_match('/[0-9]+/', $number) >= 1) {
			if((substr($number, 0, 1) == '4') && (($num_length == 13)||($num_length == 16))) {
				return "Visa";
			} else if((substr($number, 0, 1) == '5' && ((substr($number, 1, 1) >= '1') && (substr($number, 1, 1) <= '5'))) && ($num_length == 16)) {
				return "Mastercard";
			} else if(substr($number, 0, 4) == "6011" && ($num_length == 16)) {
				return "Discover Card";
			} else if((substr($number, 0, 1) == '3' && ((substr($number, 1, 1) == '4') || (substr($number, 1, 1) == '7'))) && ($num_length == 15)) {
				return "American Express";
			}
		}

		return '';
	}

	/**
	* Get a list of zero decimal currency
	*/
	public function zero_decimal_currency( ){
		return array( 'BIF', 'CLP', 'DJF', 'GNF', 'JPY', 'KMF', 'KRW', 'MGA', 'PYG', 'RWF', 'UGX', 'VND', 'VUV', 'XAF', 'XOF', 'XPF' );
	}

	/**
	* Print a dropdown of years
	* @param pfp: 4 digits
	*/
	public function _print_year_dropdown($sel='', $pfp = false) {
		$localDate=getdate();
		$minYear = $localDate["year"];
		$maxYear = $minYear + 20;

		$output = "<option value=''>".$this->a->get_text('year')."</option>";
		for($i=$minYear; $i<$maxYear; $i++) {
			if ($pfp) {
				$output .= "<option value='". substr($i, 0, 4) ."'".($sel==(substr($i, 0, 4))?' selected':'').
				">". $i ."</option>";
			} else {
				$output .= "<option value='". substr($i, 2, 2) ."'".($sel==(substr($i, 2, 2))?' selected':'').
				">". $i ."</option>";
			}
		}
		return($output);
	}

	/**
	* Print a dropdown of months
	*
	*/
	public function _print_month_dropdown($sel='') {
		$output =  "<option value=''>". ucwords( $this->a->get_text('month') )."</option>";
		$output .=  "<option " . ($sel==1?' selected':'') . " value='01'>01 - ". date_i18n("M",strtotime("1970-01-01")) ."</option>";
		$output .=  "<option " . ($sel==2?' selected':'') . "  value='02'>02 - ". date_i18n("M",strtotime("1970-02-01")) ."</option>";
		$output .=  "<option " . ($sel==3?' selected':'') . "  value='03'>03 - ". date_i18n("M",strtotime("1970-03-01")) ."</option>";
		$output .=  "<option " . ($sel==4?' selected':'') . "  value='04'>04 - ". date_i18n("M",strtotime("1970-04-01")) ."</option>";
		$output .=  "<option " . ($sel==5?' selected':'') . "  value='05'>05 - ". date_i18n("M",strtotime("1970-05-01")) ."</option>";
		$output .=  "<option " . ($sel==6?' selected':'') . "  value='06'>06 - ". date_i18n("M",strtotime("1970-06-01")) ."</option>";
		$output .=  "<option " . ($sel==7?' selected':'') . "  value='07'>07 - ". date_i18n("M",strtotime("1970-07-01")) ."</option>";
		$output .=  "<option " . ($sel==8?' selected':'') . "  value='08'>08 - ". date_i18n("M",strtotime("1970-08-01")) ."</option>";
		$output .=  "<option " . ($sel==9?' selected':'') . "  value='09'>09 - ". date_i18n("M",strtotime("1970-09-01")) ."</option>";
		$output .=  "<option " . ($sel==10?' selected':'') . "  value='10'>10 - ". date_i18n("M",strtotime("1970-10-01")) ."</option>";
		$output .=  "<option " . ($sel==11?' selected':'') . "  value='11'>11 - ". date_i18n("M",strtotime("1970-11-01")) ."</option>";
		$output .=  "<option " . ($sel==12?' selected':'') . "  value='12'>12 - ". date_i18n("M",strtotime("1970-12-01")) ."</option>";

		return($output);
	}

 	/**
	 * Handle error and open dialog requests
	 */
	public function template_redirect() {
		if ( $error = wpb_get_session_val( 'app_gateway_error' ) ) {
			wpb_notice( $error, 'error', 2000 );
			wpb_set_session_val( 'app_gateway_error', null );
		} else if ( ! empty( $_GET['app_gateway_error'] ) ) {
			wpb_notice( wpb_clean( urldecode( $_GET['app_gateway_error'] ) ) );
		} else if ( $error = $this->get_checkout_error() ) {
			wpb_notice( $error );
		} else if ( isset( $_GET['app_gw_success'] ) ) {

			wpb_set_session_val( 'app_total_amount', null );

			$app 		= wpb_get_app( wpb_clean( $_GET['app_gw_success'] ) );
			$ppal_paid	= ! empty( $app->status ) && ! empty( $app->payment_method ) && 'paid' == $app->status && 'paypal-standard' == $app->payment_method;
			$hash 		= ! empty( $_GET['app_hash'] ) ? wpb_clean( $_GET['app_hash'] ) : '';

			# PayPal return does not need hash
			if ( ! $ppal_paid && ! ( $hash && $this->a->create_hash( $app, 'open_dialog' ) == $hash ) ) {
				return;
			}

			BASE('Notices')->open_confirmation_dialog( $app );
		}
	}

 	/**
	 * Redirects back to the page with error message added
	 */
	public function do_error( $app_id, $post_id, $error ) {
		$ids = wpb_get_session_val( 'app_cart', array() );
		$ids[] = $app_id;

		if ( $this->a->change_status( 'cart', $app_id ) ) {
			wpb_set_session_val( 'app_cart', array_unique( $ids ) );
		}

		if ( wpb_doing_ajax() ) {
			WpBDebug::log( $error ? $error : sprintf(
				__('Unknown error with gateway %1$s for booking ID %2$s on page %3$s', 'wp-base'),
				$this->plugin_name,
				$app_id,
				$post_id )
			);

			die( json_encode( array( 'error' => $error )));
		}

		wpb_set_session_val( 'app_gateway_error', $error );

		wp_redirect( get_permalink( $post_id ) );
		exit;
	}

	/**
	 * Change appointment status to paid
	 * @param $confirmed	bool	In rare cases can be false, e.g. result comes as pending from gateway
	 */
	public function do_success( $app_id, $confirmed = true, $user_id = 0, $credits = 0 ) {
		$app		= wpb_get_app( $app_id );
		$prev_stat	= isset( $app->status ) ? $app->status : '';
		$problem	= false;

		if ( $app_id > 0 && $app && ! in_array( $prev_stat, array( 'completed' ) ) && $this->a->change_status( 'paid', $app_id ) ) {
			wpb_flush_cache();

			$slot = new WpB_Slot( new WpB_Booking( $app_id ) );
			$slot->exclude( $app_id );

			if ( $slot->is_busy() ) {
				$problem = true;
				$message = sprintf(
					__('Time slot %1$s looks overbooked. Please check booking with ID %2$s. \r\r
					Possible reasons: \r
					a) Booking has been deleted or its status manually modified before \r
					b) If Disable Pending Appointments Time setting is less than 180 minutes, client may have waited too long on Paypal website and as the time slot has been freed another client has booked the same time slot. \r
					c) If Strict Check is not enabled, a manual booking has been made for the same time slot \r
					d) Temporary server failure.', 'wp-base'),
					date_i18n( $this->a->dt_format, strtotime( $app->start ) ),
					$this->a->get_app_link( $app_id, true )
				);
			} else if ( 'confirmed' != $prev_stat && $confirmed ) {
				$this->a->maybe_send_message( $app_id, 'confirmation' );
			}

		} else if ( $app_id > 0 && $app && 'paid' != $prev_stat ) {
			$problem = true;
			$message = sprintf(
				__('Payment gateway confirmation arrived, but status could not be changed. Please check booking with ID %s. \r\rPossible reasons: \ra) Booking has been deleted or its status manually modified before \rb) If Disable Pending Appointments Time setting is less than 180 minutes, client may have waited too long on Paypal website and as the time slot has been freed another client has booked the same time slot. \rc) If Strict Check is not enabled, a manual booking has been made for the same time slot \rd) Temporary server failure.', 'wp-base'),
				$this->a->get_app_link( $app_id, true )
			);
		}

		# Something wrong. Warn admin
		if ( $problem ) {
			if ( 'yes' == wpb_setting('use_html') ) {
				$message = str_replace( '\r', '<br/>', $message );
			}

			wp_mail(
				BASE('User')->get_admin_email( ),
				sprintf( __( 'Problem with booking #%d', 'wp-base' ), $app_id ),
				$message,
				$this->a->message_headers()
			);
		}

		do_action( 'app_payment_received', $app_id, $user_id, $credits );

		$pname = $this->plugin_name;

		if ( ( 'paypal-express' == $pname || 'authorizenet-aim' == $pname || 'credits' == $pname ) && wpb_doing_ajax() ) {

			$this->delete_session();

			die ( json_encode( array(
					'confirm_text'	=> wpb_dialog_text( $app ),
					'confirm_title'	=> wpb_dialog_title( $app ),
					'refresh_url'	=> wpb_refresh_url( 1 ), # If no url defined, simply refresh
					'refresh'		=> 1,
				) )
			);
		}

		if ( $confirmed ) {
			$url = wpb_add_query_arg( array( 'app_gw_success' => $app_id, 'app_hash' => $this->a->create_hash( $app, 'open_dialog' ), 'app_test_ts' => $this->a->_time ), $this->return_url() );
		} else {
			$url = $this->return_url();
		}

		wp_redirect( $url );
		exit;
	}

	/**
	 * Delete session
	 */
	protected function delete_session(){
	}

  }

}

