<?php
/**
 * Financial Performance functions
 *
 * Functions about sales, number of bookings, payments etc
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Get and cache all transactions (Booking ID, User ID, Sum of transactions)
 * Note: Transactions are multipled by 100 (1234 means $12.34)
 * @since 2.0
 * @return array of objects
 */
function wpb_get_all_transactions( ) {
	$identifier = wpb_cache_prefix() . 'all_transactions';
	$trs = wp_cache_get( $identifier );

	if ( false === $trs ) {
		$tr_table	= BASE()->transaction_table;
		$app_table	= BASE()->app_table;

		$q = "SELECT app.ID, app.user, SUM(tr.transaction_total_amount) AS total_paid".
			 " FROM {$tr_table} AS tr, {$app_table} AS app".
			 " WHERE tr.transaction_app_ID=app.ID".
			 " AND app.ID IN (SELECT ID FROM {$app_table})".
			 " GROUP BY app.ID";

		$trs = BASE()->db->get_results( $q, OBJECT_K );

		wp_cache_set( $identifier, $trs );
	}

	return $trs;
}

/**
 * Get and cache all transactions of a user
 * Note: Transactions are multipled by 100 (1234 means $12.34)
 * @param user_id: User ID
 * @since 2.0
 * @return array of objects
 */
function wpb_get_transactions_by_user( $user_id ) {
	if ( ! $user_id ) {
		return false;
	}

	$identifier = wpb_cache_prefix() . 'transactions_by_user_'. $user_id;
	$trs = wp_cache_get( $identifier );

	if ( false === $trs ) {
		$tr_table	= BASE()->transaction_table;
		$app_table	= BASE()->app_table;

		$q = BASE()->db->prepare(
			"SELECT app.ID, SUM(tr.transaction_total_amount) AS total_paid".
			" FROM {$tr_table} AS tr, {$app_table} AS app".
			" WHERE tr.transaction_app_ID=app.ID".
			" AND app.ID IN (SELECT ID FROM {$app_table} WHERE user=%d)".
			" GROUP BY app.ID", $user_id
		);

		$trs = BASE()->db->get_results( $q, OBJECT_K );

		wp_cache_set( $identifier, $trs );
	}

	return $trs;
}

/**
 * Get total paid (sum of all transaction amounts) for an app
 * Note: Paid numbers are multipled by 100 (1234 means $12.34)
 * @param app_id: app ID or app stdClass Object
 * @param all: Also save all transactions by client into cache
 * @since 2.0
 * @return integer
 */
function wpb_get_paid_by_app( $app_id, $all = true ) {

	if ( is_object( $app_id ) ) {
		$app	= $app_id;
		$app_id = $app->ID;
	} else {
		$app = wpb_get_app( $app_id, $all );
	}

	# First check if a cache is available
	$trs = wp_cache_get( wpb_cache_prefix() . 'all_transactions' );

	if ( isset( $trs[$app_id] ) && isset( $trs[$app_id]->total_paid ) ) {
		return $trs[$app_id]->total_paid;
	}

	// If we are interested in just one payment. e.g. to use in email, it is resource saver if we call it from DB
	if ( !$all ) {
		$identifier = wpb_cache_prefix() . 'total_paid_by_app_'. $app_id;
		$paid = wp_cache_get( $identifier );

		if ( false === $paid ) {
			$tr_table = BASE()->transaction_table;

			$paid = BASE()->db->get_var( BASE()->db->prepare(
						"SELECT SUM(transaction_total_amount) FROM {$tr_table}
						WHERE transaction_app_ID=%d", $app_id )
					);

			wp_cache_set( $identifier, $paid );
		}

		return $paid;
	}

	// Logged in user
	// We are prefering the first method, because it caches user's all transactions
	if ( ! empty( $app->user ) ) {
		$trs = wpb_get_transactions_by_user( $app->user );
		if ( isset( $trs[$app_id] ) && isset( $trs[$app_id]->total_paid ) ) {
			return $trs[$app_id]->total_paid;
		} else {
			return 0;
		}
	} else {
		$trs = wpb_get_all_transactions( );
		if ( isset( $trs[$app_id]->total_paid ) ) {
			return $trs[$app_id]->total_paid;
		} else {
			return 0;
		}
	}
}

/**
 * Add a payment record to transactions table
 * @param 	$args		array	(Either app_id or transaction_ID is required):
 * 		app_id			integer			App ID payment belongs to
 *		amount			integer|string	Amount of the payment. Can be with two decimal degits or not. Internally multiplied by 100
 *		currency		string			3 digit currency code. Defaults to website currency. Do NOT use, because multiple currencies are not supported
 *		timestamp		integer			Timestamp of the transaction
 *		paypal_ID		string			Reference number from the gateway. Historically named after PayPal, but any gateway reference is ok
 *		status			string			Status of the transaction. Currently only 'paid' is supported
 *		note			string			Note to be saved with transaction, e.g. credit card last 4 digits
 *		gateway			string			Name of the gateway, e.g. manual-payments, paypal-standard, paypal-express, stripe, paymill, simplify, 2checkout, authorizenet, credits, woocommerce
 *		transaction_ID	integer			Transaction ID. Required to edit a payment record
 * @since 3.5.0
 * @return integer|false				Transaction ID on success, false on failure
 */
function wpb_add_payment( $args = array() ) {
	$defaults = array(
		'app_id'			=> 0,
		'amount'			=> 0,
		'currency'			=> wpb_setting( 'currency' , 'USD' ),
		'timestamp'			=> BASE()->_time,
		'paypal_ID'			=> '',
		'status'			=> 'paid',
		'note'				=> '',
		'gateway'			=> '',
		'transaction_ID'	=> 0,
	);

	$args = wp_parse_args( $args, $defaults );

	// Create a uniqe ID in manual payments, if not given by user
	if ( 'manual-payments' == $args['gateway'] && !$args['paypal_ID'] ) {
		$args['paypal_ID'] = uniqid('auto_');
	}

	$table 	= BASE()->transaction_table;
	$result	= $new_record = false;
	$data 	= array();
	$data['transaction_app_ID']			= $args['app_id'];
	$data['transaction_total_amount']	= $args['amount'] * 100;
	$data['transaction_currency']		= $args['currency'];
	$data['transaction_stamp']			= $args['timestamp'];
	$data['transaction_paypal_ID']		= $args['paypal_ID'];
	$data['transaction_status']			= $args['status'];
	$data['transaction_note']			= wp_unslash( $args['note'] );
	$data['transaction_gateway']		= $args['gateway'];

	# If we are editing a manual payment
	if ( $args['transaction_ID'] ) {
		# In the query we add manual payment check too, since we do not want an auto transaction to be edited
		if ( BASE()->db->update(
			$table,
			$data,
			array( 'transaction_ID' => $args['transaction_ID'], 'transaction_gateway' => 'manual-payments' )
		) ) {
			$result = $args['transaction_ID'];
		}
	} else {
		$existing_id = BASE()->db->get_var( BASE()->db->prepare(
			"SELECT transaction_ID FROM {$table} WHERE transaction_app_ID = %d AND transaction_paypal_ID = %s LIMIT 1",
			$args['app_id'], $args['paypal_ID']
		) );

		if ( $existing_id ) {
			if ( BASE()->db->update( $table, $data, array('transaction_ID' => $existing_id) ) ) {
				$result = $existing_id;
			}
		} else {
			if ( BASE()->db->insert( $table, $data ) ) {
				$result = BASE()->db->insert_id;
				$new_record = true;
			}
		}
	}

	do_action( 'app_payment_maybe_new', $result, $data );

	if ( $result ) {
		if ( $new_record ) {
			do_action( 'app_payment_new_record', $result, $data );
		} else {
			do_action( 'app_payment_updated', $result, $data );
		}
	}

	return $result;
}

/**
 * Add a payment record to transactions table
 * @param 	$args		array	(transaction_ID is required):
 *		transaction_ID	integer			Transaction ID. Required to edit a payment record
 *		For the rest see wpb_add_payment
 * @since 3.5.0
 * @return integer|false				Transaction ID on success, false on failure
 */
function wpb_edit_payment( $args = array() ) {
	if ( empty( $args['transaction_ID'] ) ) {
		return false;
	}

	return wpb_add_payment( $args );
}

/**
 * Helper function to normalize args
 * @since 3.5.0
 * @return array
 */
function _wpb_setup_report_args( $args ) {
	$time	= BASE()->_time;
	$today	= strtotime( 'today', $time );

	switch ( $args['range'] ) {
		case 'custom':			$args['start']	= wpb_strtotime( $args['start'] );
								$args['end']	= wpb_strtotime( $args['end'] );
								break;

		case 'yesterday':		$args['start'] = $today - DAY_IN_SECONDS;
								$args['end'] = $today;
								break;
		case 'today':			$args['start'] = $today;
								$args['end'] = $today + DAY_IN_SECONDS;
								break;
		case 'tomorrow':		$args['start'] = $today + DAY_IN_SECONDS;
								$args['end'] = $today + 2*DAY_IN_SECONDS;
								break;
		case 'last_week':		if ( '0' === (string)BASE()->start_of_week ) {
									$args['start'] = wpb_sunday( $time ) - WEEK_IN_SECONDS;
								} else if ( 1 == BASE()->start_of_week ) {
									$args['start'] = wpb_monday( $time ) - WEEK_IN_SECONDS;
								} else {
									$args['start'] = wpb_saturday( $time ) - WEEK_IN_SECONDS;
								}
								$args['end'] = $args['start'] + WEEK_IN_SECONDS;
								break;

		case 'this_week':		if ( '0' === (string)BASE()->start_of_week ) {
									$args['start'] = wpb_sunday( $time );
								} else if ( 1 == BASE()->start_of_week ) {
									$args['start'] = wpb_monday( $time );
								} else {
									$args['start'] = wpb_saturday( $time );
								}
								$args['end'] = $args['start'] + WEEK_IN_SECONDS;
								break;

		case 'next_week':		if ( '0' === (string)BASE()->start_of_week ) {
									$args['start'] = wpb_sunday( $time ) + WEEK_IN_SECONDS;
								} else if ( 1 == BASE()->start_of_week ) {
									$args['start'] = wpb_monday( $time ) + WEEK_IN_SECONDS;
								} else {
									$args['start'] = wpb_saturday( $time ) + WEEK_IN_SECONDS;
								}
								$args['end'] = $args['start'] + WEEK_IN_SECONDS;
								break;
		case 'last_30_days':	$args['start'] 	= $today - MONTH_IN_SECONDS;
								$args['end'] 	= $today;
								break;
		case 'last_month':		$args['start'] 	= wpb_first_of_month( $time, -1 );
								$args['end'] 	= wpb_last_of_month( $time, -1 );
								break;
		case 'this_month':		$args['start'] 	= wpb_first_of_month( $time );
								$args['end'] 	= wpb_last_of_month( $time );
								break;
		case 'next_month':		$args['start'] 	= wpb_first_of_month( $time, 1 );
								$args['end'] 	= wpb_last_of_month( $time, 1 );
								break;
		case 'last_year':		$year = date( "Y", $time );
								$args['start']	= mktime( 0, 0, 0, 1, 1, $year -1 );
								$args['end']	= mktime( 0, 0, 0, 1, 1, $year );
								break;
		case 'this_year':		$year = date( "Y", $time );
								$args['start']	= mktime( 0, 0, 0, 1, 1, $year );
								$args['end']	= mktime( 0, 0, 0, 1, 1, $year + 1 );
								break;
	}

	return $args;
}

/**
 * Retrieve sales (number of bookings) for a given time period, for a given set of lsw or all
 * @param $args	array	(All parameters optional):
 *		range		string			today, yesterday, tomorrow, this_week, last_week, next_week, last_30_days, this_month, last_month, next_month, last_year, this_year
 * 		start 		integer/string	Start of the report as timestamp or date time in any standard format, preferably Y-m-d H:i:s
 * 		end			integer/string	End of the report as timestamp or date time. If left empty, end time is calculated from duration of service
 * 		location	integer			ID of location. If left empty, all locations
 * 		service		integer			ID of service. If left empty, all services
 * 		worker		integer			ID of service provider. If left empty, all providers
 *		user		integer			ID of the client. If left empty, all clients
 *		day_by_day	bool			Whether x-axis will be daily or longer (15 days or monthly)
 *		by_created	bool			If true give the results by creation time. If false, give the results for start time
 *		exclude		string/array	Exclude booking ID(s)
 * @since 3.5.0
 * @return array
 */
function wpb_get_sales( $args = array() ) {
	return _wpb_get_sales_revenue( 'sales', $args );
}

/**
 * Retrieve earnings (revenue) for a given time period, for a given set of lsw or all
 * @param $args	array	(All parameters optional):
 * 		See wpb_get_sales
 * @since 3.5.0
 * @return array
 */
function wpb_get_revenue( $args ) {
	return _wpb_get_sales_revenue( 'revenue', $args );
}

/**
 * Helper to retrieve revenue and sales data
 * @param	$context	string	Either sales or revenue or seats
 * @param	$stats		array	Optionally select different stats
 * @param 	$args		array	(All optional):
 * 		See wpb_get_sales
 * @since 3.5.0
 * @return array
 */
function _wpb_get_sales_revenue( $context, $args, $stats = array() ) {
	global $wpdb;

	$defaults = array(
		'range'			=> 'this_week',
		'start'			=> false,
		'end'			=> false,
		'location'		=> false,
		'service'		=> false,
		'worker'		=> false,
		'user'			=> false,
		'day_by_day'	=> false,
		'by_created'	=> false,
		'stats'			=> $stats,
		'exclude'		=> false,
	);

	$args 			= _wpb_setup_report_args( wp_parse_args( $args, $defaults ) );
	$non_cachable	= ( ! empty( $args['range'] ) && 'custom' == $args['range'] ) || ! empty( $stats );

	if ( $non_cachable ) {
		$data	= false;
	} else {
		$cached	= get_transient( 'wpbase_' .$context );
		$key    = md5( json_encode( $args ) );
		$data 	= isset( $cached[ $key ] ) ? $cached[ $key ] : false;
	}

	if ( false === $data ) {

		$by = ! empty( $args['by_created'] ) ? 'created' : 'start';

		if ( ! $args['day_by_day'] ) {
			$select = "DATE_FORMAT(apps.{$by}, '%%m') AS m, YEAR(apps.{$by}) AS y, COUNT(DISTINCT apps.ID) as count";
			$grouping = "YEAR(apps.{$by}), MONTH(apps.{$by})";
		} else {
			if ( 'today' == $args['range'] || 'yesterday' == $args['range'] || 'tomorrow' == $args['range'] ) {
				$select = "DATE_FORMAT(apps.{$by}, '%%d') AS d, DATE_FORMAT(apps.{$by}, '%%m') AS m, YEAR(apps.{$by}) AS y, HOUR(apps.{$by}) AS h, COUNT(DISTINCT apps.ID) as count";
				$grouping = "YEAR(apps.{$by}), MONTH(apps.{$by}), DAY(apps.{$by}), HOUR(apps.{$by})";
			} else {
				$select = "DATE_FORMAT(apps.{$by}, '%%d') AS d, DATE_FORMAT(apps.{$by}, '%%m') AS m, YEAR(apps.{$by}) AS y, COUNT(DISTINCT apps.ID) as count";
				$grouping = "YEAR(apps.{$by}), MONTH(apps.{$by}), DAY(apps.{$by})";
			}
		}

		if ( 'today' == $args['range'] || 'yesterday' == $args['range'] || 'tomorrow' == $args['range'] ) {
			$grouping = "YEAR(apps.{$by}), MONTH(apps.{$by}), DAY(apps.{$by}), HOUR(apps.{$by})";
		}

		$statuses = apply_filters( 'app_report_status', ($stats ? $stats : array( 'confirmed', 'paid', 'completed', 'running', 'test' )) );
		$statuses = "'" . implode( "', '", $statuses ) . "'";
		
		$com_stasuses = apply_filters( 'app_report_commission_status', ($stats ? $stats : array( 'paid', 'unpaid', 'on_hold' )) );
		$com_stasuses = "'" . implode( "', '", $com_stasuses ) . "'";

		$location	= $args['location'] ? $wpdb->prepare( "location=%d", $args['location'] ) : "1=1";
		$service	= $args['service'] ? $wpdb->prepare( "service=%d", $args['service'] ) : "1=1";
		$worker		= $args['worker'] || '0' === (string)$args['worker'] ? $wpdb->prepare( "worker=%d", $args['worker'] ) : "1=1";
		$com_worker	= $args['worker'] || '0' === (string)$args['worker'] ? $wpdb->prepare( "coms.worker=%d", $args['worker'] ) : "1=1";
		$user		= $args['user'] ? $wpdb->prepare( "user=%d", $args['user'] ) : "1=1";

		$exclude	= $args['exclude']
					  ? (is_array( $args['exclude'] ) ? implode( ',', array_map( 'absint', $args['exclude'] ) ) : absint( $args['exclude'] ))
					  : 0;

		$limit		= apply_filters( 'app_report_limit', 10, $context, $args );

		if ( 'revenue' == $context ) {
			$data = $wpdb->get_results( $wpdb->prepare(
			"SELECT SUM(transaction_total_amount)/100 AS total, $select
			 FROM {$wpdb->prefix}base_bookings AS apps
			 INNER JOIN {$wpdb->prefix}base_transactions ON apps.ID = {$wpdb->prefix}base_transactions.transaction_app_ID
			 WHERE apps.{$by} >= %s
			 AND apps.{$by} < %s
			 AND apps.status IN ($statuses)
			 AND $location
			 AND $service
			 AND $worker
			 AND apps.ID NOT IN ($exclude)
			 GROUP BY $grouping
			 ORDER by apps.{$by} ASC", date( 'Y-m-d H:i:s', $args['start'] ), date( 'Y-m-d H:i:s', $args['end'] ) ), ARRAY_A );

		} else if ( 'commissions' == $context ) {
			$data = $wpdb->get_results( $wpdb->prepare(
			"SELECT SUM(amount) AS total_amount
			 FROM {$wpdb->prefix}base_bookings AS apps
			 INNER JOIN {$wpdb->prefix}base_commissions AS coms ON apps.ID = coms.booking_id
			 WHERE apps.{$by} >= %s
			 AND apps.{$by} < %s
			 AND apps.status IN ($com_stasuses)
			 AND $location
			 AND $service
			 AND $com_worker
			 ORDER by apps.{$by} ASC", date( 'Y-m-d H:i:s', $args['start'] ), date( 'Y-m-d H:i:s', $args['end'] ) ), ARRAY_A );

		} else if ( 'seats' == $context ) {
			$data = $wpdb->get_results( $wpdb->prepare(
			"SELECT SUM(seats) AS total_seats
			 FROM {$wpdb->prefix}base_bookings AS apps
			 WHERE apps.{$by} >= %s
			 AND apps.{$by} < %s
			 AND apps.status IN ($statuses)
			 AND $location
			 AND $service
			 AND $worker
			 AND apps.ID NOT IN ($exclude)
			 GROUP BY $grouping
			 ORDER by apps.{$by} ASC", date( 'Y-m-d H:i:s', $args['start'] ), date( 'Y-m-d H:i:s', $args['end'] ) ), ARRAY_A );

		} else if ( 'popular_services' == $context ) {
			$data = $wpdb->get_results( $wpdb->prepare(
			"SELECT service, SUM(seats) AS total_seats
			 FROM {$wpdb->prefix}base_bookings AS apps
			 WHERE apps.{$by} >= %s
			 AND apps.{$by} < %s
			 AND apps.status IN ($statuses)
			 AND $location
			 AND $worker
			 AND apps.ID NOT IN ($exclude)
			 GROUP BY service
			 ORDER by total_seats DESC
			 LIMIT $limit", date( 'Y-m-d H:i:s', $args['start'] ), date( 'Y-m-d H:i:s', $args['end'] ) ), ARRAY_A );

		} else if ( 'popular_services_by_revenue' == $context ) {
			$data = $wpdb->get_results( $wpdb->prepare(
			"SELECT service, SUM(transaction_total_amount)/100 AS total
			 FROM {$wpdb->prefix}base_bookings AS apps
			 INNER JOIN {$wpdb->prefix}base_transactions ON apps.ID = {$wpdb->prefix}base_transactions.transaction_app_ID
			 WHERE apps.{$by} >= %s
			 AND apps.{$by} < %s
			 AND apps.status IN ($statuses)
			 AND $location
			 AND $worker
			 AND apps.ID NOT IN ($exclude)
			 GROUP BY service
			 ORDER by total DESC
			 LIMIT $limit", date( 'Y-m-d H:i:s', $args['start'] ), date( 'Y-m-d H:i:s', $args['end'] ) ), ARRAY_A );

		} else if ( 'popular_workers' == $context ) {
			$data = $wpdb->get_results( $wpdb->prepare(
			"SELECT worker, SUM(seats) AS total_seats
			 FROM {$wpdb->prefix}base_bookings AS apps
			 WHERE apps.{$by} >= %s
			 AND apps.{$by} < %s
			 AND apps.status IN ($statuses)
			 AND $location
			 AND apps.ID NOT IN ($exclude)
			 GROUP BY worker
			 ORDER by total_seats DESC
			 LIMIT $limit", date( 'Y-m-d H:i:s', $args['start'] ), date( 'Y-m-d H:i:s', $args['end'] ) ), ARRAY_A );

		} else if ( 'popular_workers_by_revenue' == $context ) {
			$data = $wpdb->get_results( $wpdb->prepare(
			"SELECT worker, SUM(transaction_total_amount)/100 AS total
			 FROM {$wpdb->prefix}base_bookings AS apps
			 INNER JOIN {$wpdb->prefix}base_transactions ON apps.ID = {$wpdb->prefix}base_transactions.transaction_app_ID
			 WHERE apps.{$by} >= %s
			 AND apps.{$by} < %s
			 AND apps.status IN ($statuses)
			 AND $location
			 AND apps.ID NOT IN ($exclude)
			 GROUP BY worker
			 ORDER by total DESC
			 LIMIT $limit", date( 'Y-m-d H:i:s', $args['start'] ), date( 'Y-m-d H:i:s', $args['end'] ) ), ARRAY_A );

		} else if ( 'popular_clients' == $context ) {
			$data = $wpdb->get_results( $wpdb->prepare(
			"SELECT user, SUM(seats) AS total_seats
			 FROM {$wpdb->prefix}base_bookings AS apps
			 WHERE apps.{$by} >= %s
			 AND apps.{$by} < %s
			 AND apps.status IN ($statuses)
			 AND $location
			 AND $worker
			 AND apps.ID NOT IN ($exclude)
			 GROUP BY user
			 ORDER by total_seats DESC
			 LIMIT $limit", date( 'Y-m-d H:i:s', $args['start'] ), date( 'Y-m-d H:i:s', $args['end'] ) ), ARRAY_A );

		} else if ( 'popular_clients_by_revenue' == $context ) {
			$data = $wpdb->get_results( $wpdb->prepare(
			"SELECT user, SUM(transaction_total_amount)/100 AS total
			 FROM {$wpdb->prefix}base_bookings AS apps
			 INNER JOIN {$wpdb->prefix}base_transactions ON apps.ID = {$wpdb->prefix}base_transactions.transaction_app_ID
			 WHERE apps.{$by} >= %s
			 AND apps.{$by} < %s
			 AND apps.status IN ($statuses)
			 AND $location
			 AND $worker
			 AND apps.ID NOT IN ($exclude)
			 GROUP BY user
			 ORDER by total DESC
			 LIMIT $limit", date( 'Y-m-d H:i:s', $args['start'] ), date( 'Y-m-d H:i:s', $args['end'] ) ), ARRAY_A );

		} else {
			$data = $wpdb->get_results( $wpdb->prepare(
			"SELECT SUM(price) AS total_price, $select
			 FROM {$wpdb->prefix}base_bookings AS apps
			 WHERE apps.{$by} >= %s
			 AND apps.{$by} < %s
			 AND apps.status IN ($statuses)
			 AND $location
			 AND $service
			 AND $worker
			 AND $user
			 AND apps.ID NOT IN ($exclude)
			 GROUP BY $grouping
			 ORDER by apps.{$by} ASC", date( 'Y-m-d H:i:s', $args['start'] ), date( 'Y-m-d H:i:s', $args['end'] ) ), ARRAY_A );
		}

		if ( ! $non_cachable ) {
			$cached = $cached ?: array();
			$cached[ $key ] = $data;
			set_transient( 'wpbase_'.$context, $cached, HOUR_IN_SECONDS ); # Cleared with wpb_flush_cache
		}
	}

	return $data;
}

/**
 * Find total weekly service hours which can be provided to clients
 * @since 3.5.0
 * @return integer (hours)
 */
function wpb_usable_hours( $args = array() ){
	$hours	= array();

	if ( 'custom' == $args['range'] && ! empty( $args['time_diff'] ) ) {
		$days = intval( $args['time_diff'] / DAY_IN_SECONDS );
	} else if ( strpos( $args['range'], 'year' ) !== false ) {
		$days = 365;
	} else if ( strpos( $args['range'], 'month' ) !== false || strpos( $args['range'], 'last_30_days' ) !== false ) {
		$days = 30;
	} else if ( strpos( $args['range'], 'day' ) !== false || strpos( $args['range'], 'tomorrow' ) !== false ) {
		$days = 1;
	} else {
		$days = 7;
	}

	if ( ! BASE()->get_nof_workers() ) {
		foreach( BASE()->get_services() as $service ) {
			$ID 		= $service->ID;
			$cap		= 'yes' == wpb_setting( 'multitasking' ) || class_exists( 'WPBSeats' ) ? BASE()->get_capacity( $ID ) : 1;
			$wh 		= BASE('WH')->get( $ID, 'service' );
			$sum 		= array_sum( $wh )/$ID;
			$hours[$ID] = intval( $sum * $cap / 12 );
		}

		if ( 'yes' == wpb_setting( 'multitasking' ) ) {
			$total_hours = $hours ? array_sum( $hours ) : 0;
		} else {
			$total_hours = $hours ? max( $hours ) : 0;
		}
	} else {
		if ( ! empty( $args['worker'] ) ) {
			$ID 		= $args['worker'];
			$wh 		= BASE('WH')->get( $ID, 'worker' );
			$sum 		= array_sum( $wh )/$ID;
			$hours[$ID] = intval( $sum / 12 );
		} else {
			foreach( BASE()->get_workers() as $worker ) {
				$ID 		= $worker->ID;
				$wh 		= BASE('WH')->get( $ID, 'worker' );
				$sum 		= array_sum( $wh )/$ID;
				$hours[$ID] = intval( $sum / 12 );
			}
		}

		$total_hours = $hours ? array_sum( $hours ) : 0;
	}

	return intval( $total_hours * $days / 7 );
}

/**
 * Find total weekly hours which was worked or will be working for
 * @param $args	array	(All parameters optional):
 *		range		string			today, yesterday, tomorrow, this_week, last_week, next_week, last_30_days, this_month, last_month, next_month, last_year, this_year
 * 		start 		integer/string	Start of the report as timestamp or date time in any standard format, preferably Y-m-d H:i:s
 * 		end			integer/string	End of the report as timestamp or date time. If left empty, end time is calculated from duration of service
 * 		location	integer			ID of location. If left empty, all locations
 * 		service		integer			ID of service. If left empty, all services
 * 		worker		integer			ID of service provider. If left empty, all providers
 *		by_created	bool			If true give the results by creation time. If false, give the results for start time
 *		exclude		string/array	Exclude booking ID(s)
 * @since 3.5.0
 * @return integer (hours)
 */
function wpb_billable_hours( $args = array() ){
	global $wpdb;

	$defaults = array(
		'range'			=> 'last_week',
		'location'		=> false,
		'service'		=> false,
		'worker'		=> false,
	);

	$args 		= _wpb_setup_report_args( wp_parse_args( $args, $defaults ) );
	$statuses 	= apply_filters( 'app_worked_hours_status', array( 'confirmed', 'paid', 'pending', 'completed', 'test' ) );
	$statuses 	= "'" . implode( "', '", $statuses ) . "'";
	$location	= $args['location'] ? $wpdb->prepare( "location=%d", $args['location'] ) : "1=1";
	$service	= $args['service'] ? $wpdb->prepare( "service=%d", $args['service'] ) : "1=1";
	$worker		= $args['worker'] || '0' === (string)$args['worker'] ? $wpdb->prepare( "worker=%d", $args['worker'] ) : "1=1";
	$multi		= BASE()->get_nof_workers() ? 'seats' : '1';

	$seconds = $wpdb->get_var( $wpdb->prepare(
		"SELECT SUM( TIME_TO_SEC( TIMEDIFF( end, start )*$multi ) ) FROM ". BASE()->app_table .
		" WHERE start >= %s AND start < %s
			 AND status IN ($statuses)
			 AND $location
			 AND $service
			 AND $worker
		", date( 'Y-m-d H:i:s', $args['start'] ), date( 'Y-m-d H:i:s', $args['end'] )
	) );

	return intval( $seconds / HOUR_IN_SECONDS );
}

/**
 * Get total number of bookings based on status for a given time period, for a given set of lsw or all
 * @param $stats	array	Status/statii of bookings to be queried
 * @param $args		array	(All parameters optional):
 *		range		string			today, yesterday, tomorrow, this_week, last_week, next_week, last_30_days, this_month, last_month, next_month, last_year, this_year
 * 		location	integer			ID of location. If left empty, all locations
 * 		service		integer			ID of service. If left empty, all services
 * 		worker		integer			ID of service provider. If left empty, all providers
 *		user		integer			ID of the client. If left empty, all clients
 *		by_created	bool			If true give the results by creation time. If false, give the results for start time
 *		exclude		string/array	Exclude booking ID(s)
 * @since 3.5.0
 * @return integer
 */
function wpb_nof_bookings( $args, $stats ) {
	$data = _wpb_get_sales_revenue( 'sales', $args, $stats );

	return array_sum( wp_list_pluck( $data, 'count' ) );
}

/**
 * Get total number of booked seats based on status for a given time period, for a given set of lsw or all
 * @param $stats	array	Status/statii of bookings to be queried
 * @param $args		array	(All parameters optional):
 *		range		string			today, yesterday, tomorrow, this_week, last_week, next_week, last_30_days, this_month, last_month, next_month, last_year, this_year
 * 		location	integer			ID of location. If left empty, all locations
 * 		service		integer			ID of service. If left empty, all services
 * 		worker		integer			ID of service provider. If left empty, all providers
 *		user		integer			ID of the client. If left empty, all clients
 *		by_created	bool			If true give the results by creation time. If false, give the results for start time
 *		exclude		string/array	Exclude booking ID(s)
 * @since 3.7.8
 * @return integer
 */
function wpb_nof_booked_seats( $args, $stats ) {
	$data = _wpb_get_sales_revenue( 'seats', $args, $stats );

	return array_sum( wp_list_pluck( $data, 'total_seats' ) );
}

/**
 * Get popular services by number of bookings/seats
 * @param $stats	array	Status/statii of bookings to be queried
 * @param $args		array	(All parameters optional):
 *		range		string			today, yesterday, tomorrow, this_week, last_week, next_week, last_30_days, this_month, last_month, next_month, last_year, this_year
 * 		location	integer			ID of location. If left empty, all locations
 * 		worker		integer			ID of service provider. If left empty, all providers
 *		user		integer			ID of the client. If left empty, all clients
 *		by_created	bool			If true give the results by creation time. If false, give the results for start time
 *		exclude		string/array	Exclude booking ID(s)
 * @since 3.8.7
 * @return array of arrays
 */
function wpb_popular_services( $args, $stats = array() ) {
	$args['service'] = false;
	$data = _wpb_get_sales_revenue( 'popular_services', $args, $stats );

	return $data;
}

/**
 * Get popular services by revenue
 * @param $stats	array	Status/statii of bookings to be queried
 * @param $args		array	(All parameters optional):
 *		range		string			today, yesterday, tomorrow, this_week, last_week, next_week, last_30_days, this_month, last_month, next_month, last_year, this_year
 * 		location	integer			ID of location. If left empty, all locations
 * 		worker		integer			ID of service provider. If left empty, all providers
 *		user		integer			ID of the client. If left empty, all clients
 *		by_created	bool			If true give the results by creation time. If false, give the results for start time
 *		exclude		string/array	Exclude booking ID(s)
 * @since 3.8.7
 * @return array of arrays
 */
function wpb_popular_services_by_revenue( $args, $stats = array() ) {
	$args['service'] = false;
	$data = _wpb_get_sales_revenue( 'popular_services_by_revenue', $args, $stats );

	return $data;
}

/**
 * Get popular services by number of bookings/seats
 * @param $stats	array	Status/statii of bookings to be queried
 * @param $args		array	(All parameters optional):
 *		range		string			today, yesterday, tomorrow, this_week, last_week, next_week, last_30_days, this_month, last_month, next_month, last_year, this_year
 * 		location	integer			ID of location. If left empty, all locations
 * 		worker		integer			ID of service provider. If left empty, all providers
 *		user		integer			ID of the client. If left empty, all clients
 *		by_created	bool			If true give the results by creation time. If false, give the results for start time
 *		exclude		string/array	Exclude booking ID(s)
 * @since 3.8.7
 * @return array of arrays
 */
function wpb_popular_workers( $args, $stats = array() ) {
	$args['service'] = false;
	$args['worker'] = false;
	$data = _wpb_get_sales_revenue( 'popular_workers', $args, $stats );

	return $data;
}

/**
 * Get popular services by revenue
 * @param $stats	array	Status/statii of bookings to be queried
 * @param $args		array	(All parameters optional):
 *		range		string			today, yesterday, tomorrow, this_week, last_week, next_week, last_30_days, this_month, last_month, next_month, last_year, this_year
 * 		location	integer			ID of location. If left empty, all locations
 * 		worker		integer			ID of service provider. If left empty, all providers
 *		user		integer			ID of the client. If left empty, all clients
 *		by_created	bool			If true give the results by creation time. If false, give the results for start time
 *		exclude		string/array	Exclude booking ID(s)
 * @since 3.8.7
 * @return array of arrays
 */
function wpb_popular_workers_by_revenue( $args, $stats = array() ) {
	$args['service'] = false;
	$args['worker'] = false;
	$data = _wpb_get_sales_revenue( 'popular_workers_by_revenue', $args, $stats );

	return $data;
}

/**
 * Get popular clients by number of bookings/seats
 * @param $stats	array	Status/statii of bookings to be queried
 * @param $args		array	(All parameters optional):
 *		range		string			today, yesterday, tomorrow, this_week, last_week, next_week, last_30_days, this_month, last_month, next_month, last_year, this_year
 * 		location	integer			ID of location. If left empty, all locations
 * 		worker		integer			ID of service provider. If left empty, all providers
 *		by_created	bool			If true give the results by creation time. If false, give the results for start time
 *		exclude		string/array	Exclude booking ID(s)
 * @since 3.8.7
 * @return array of arrays
 */
function wpb_popular_clients( $args, $stats = array() ) {
	$args['service'] = false;
	$data = _wpb_get_sales_revenue( 'popular_clients', $args, $stats );

	return $data;
}

/**
 * Get popular clients by revenue
 * @param $stats	array	Status/statii of bookings to be queried
 * @param $args		array	(All parameters optional):
 *		range		string			today, yesterday, tomorrow, this_week, last_week, next_week, last_30_days, this_month, last_month, next_month, last_year, this_year
 * 		location	integer			ID of location. If left empty, all locations
 * 		worker		integer			ID of service provider. If left empty, all providers
 *		by_created	bool			If true give the results by creation time. If false, give the results for start time
 *		exclude		string/array	Exclude booking ID(s)
 * @since 3.8.7
 * @return array of arrays
 */
function wpb_popular_clients_by_revenue( $args, $stats = array() ) {
	$args['service'] = false;
	$data = _wpb_get_sales_revenue( 'popular_clients_by_revenue', $args, $stats );

	return $data;
}


/**
 * Get earned commissions s for a given time period, for a given set of lsw or all
 * @param $stats	array	Status/statii of commissions to be queried
 * @param $args		array	(All parameters optional):
 *		range		string			today, yesterday, tomorrow, this_week, last_week, next_week, last_30_days, this_month, last_month, next_month, last_year, this_year
 * 		location	integer			ID of location. If left empty, all locations
 * 		service		integer			ID of service. If left empty, all services
 * 		worker		integer			ID of service provider. If left empty, all providers
 *		user		integer			ID of the client. If left empty, all clients
 *		by_created	bool			If true give the results by creation time. If false, give the results for start time
 * @since 3.9.1.3
 * @return integer
 */
function wpb_commissions( $args, $stats = array() ) {
	$data = _wpb_get_sales_revenue( 'commissions', $args, $stats );
	
	return array_sum( wp_list_pluck( $data, 'total_amount' ) );
}