<?php
/**
 * Booking functions
 *
 * Functions about booking, services, workers and some other useful stuff
 *
 * Copyright © 2018-2021 Hakan Ozevin
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Generate a dropdown menu of countries with country code or country name as value
 * @param $value:	string	Optional selected country value (Code or full name that should match to WpBConstant::countries()
 * Since 3.0
 */
if ( ! function_exists('wpb_countries') ) {
	function wpb_countries( $value = '' ) {
		include_once( WPBASE_PLUGIN_DIR . '/includes/constant-data.php' );
		$countries = WpBConstant::countries();
		if ( !is_array( $countries ) || empty( $countries ) )
			return BASE()->debug_text( __('Check Countries array','wp-base') );

		$c  = '<select name="app_country">'. "\n";
		$c .= '<option value="">'. __('Select country','wp-base')."</option>\n";
		if ( '' === $value || ( $value && array_key_exists( $value, $countries ) ) ) {
			foreach ($countries as $code=>$country ) {
				$c .= '<option value="'.$code.'" '.selected($code,$value,false).'>'. esc_html( $country )."</option>\n";
			}
		} else {
			foreach ($countries as $country ) {
				$c .= '<option value="'.esc_attr( $country ).'" '.selected($country,$value,false).'>'. esc_html( $country )."</option>\n";
			}
		}
		$c .= "</select>\n";

		return $c;
	}
}

/**
 * Return appointments optionally filtered with several arguments
 * @param $args	array	Query parameters (All optional):
 * 		start 			integer|string			Start of the query as timestamp or date time in any standard format, preferably Y-m-d H:i:s. If left empty current time
 * 		end				integer|string			End of the query as time stamp or date time. If left empty, no limit
 * 		location		integer					ID of location. If left empty or related var does not exist, all
 * 		service			integer|string|array	ID of service. Multiple services separated with comma or as array is allowed. If left empty or service(s) does not exist, all
 * 		worker			integer					ID of service provider. If left empty or related var does not exist, all
 *		status			string|array			Array of statuses or single string or separated by comma
 *		wide_coverage	bool					Includes apps started before start and ending after end too. Only valid if start and end are both defined
 *		order_by		string					Order of the results. Allowed values: 'id', 'sort_order', 'name', 'start', 'end', 'duration', 'price',
												'id desc', 'sort_order desc', 'name desc', 'start desc', 'end desc', 'duration desc', 'price desc', 'rand()'
 * @since 3.5.0
 * @return array of StdClass objects
 */
function wpb_get_apps( $args = array() ) {
	$start			= ! empty( $args['start'] ) ? $args['start'] : BASE()->_time;
	$end 			= ! empty( $args['end'] ) ? $args['end'] : null;
	$service 		= ! empty( $args['service'] ) ? $args['service'] : 0;
	$statuses		= ! empty( $args['status'] ) ? $args['status'] : '';
	$wide_coverage	= ! empty( $args['wide_coverage'] ) ? $args['wide_coverage'] : true;

	$stat_sql		= BASE('Listing')->stat_sql( $statuses );
	$location_sql	= ! empty( $args['location'] ) ? BASE()->db->prepare( 'location=%d', $args['location'] ) : '1=1';
	$service_sql	= BASE('Listing')->service_sql( $service );
	$worker_sql		= ! empty( $args['worker']) ? BASE()->db->prepare( 'worker=%d', $args['worker'] ) : '1=1';
	$datetime_sql	= BASE('Listing')->dt_sql( $start, $end, $wide_coverage );
	$order_by_sql	= ! empty( $args['order_by'] ) ? BASE()->sanitize_order_by( $args['order_by'] ) : 'start';

	$query = " SELECT * FROM " . BASE()->app_table .
			 " WHERE (".$stat_sql.") AND ".$datetime_sql." AND (".$location_sql.") AND (".$service_sql.") AND (".$worker_sql.")
			   ORDER BY ".$order_by_sql;

	return 	BASE()->db->get_results( $query, OBJECT_K  );
}

/**
 * Check if a time slot free to be booked
 * @param $args_booking		array|object|integer
 * 		See wpb_why_not_free function
 *
 * @since 3.0
 * @return bool		true if slot is free, false if not free
 */
function wpb_is_free( $args_booking ) {
	return ! wpb_why_not_free( $args_booking );
}

/**
 * Return a code pointing why a time slot is not available to be booked
 * @param $args_booking		array|object|integer
 *		Time slot parameters (All except start optional):
 * 		start 		integer/string	Start of the slot as timestamp or date time in any standard format, preferably Y-m-d H:i:s
 * 		end			integer/string	End of the slot as time stamp or date time. If left empty, end time is calculated from duration of service
 * 		location	integer			ID of location. If left empty or related var does not exist, read from current values
 * 		service		integer			ID of service. If left empty or related var does not exist, read from current values
 * 		worker		integer			ID of service provider. If left empty or related var does not exist, read from current values
 *		OR
 *					object|int		Booking object or ID
 * @since 3.0
 * @return string (see wpb_code2reason function for return value) if slot is not available. false if slot is free
 */
function wpb_why_not_free( $args_booking ) {
	if ( $args_booking instanceof WpB_Booking ) {
		$booking = $args_booking;
	} else if ( is_array( $args_booking ) ) {
		$booking = new WpB_Booking();
		$booking->prepare( $args_booking );
	} else {
		$booking = new WpB_Booking( $args_booking );
	}

	$slot = new WpB_Slot( $booking );

	return wpb_code2reason( $slot->why_not_free() ); # e.g. 'busy'
}

/**
 * Return an appointment object
 * @param $app_id	integer		Unique appointment ID
 * @param $prime_cache	bool	Preload cache into memory
 * @since 3.0
 * @return StdClass  object
 */
function wpb_get_app( $app_id, $prime_cache = false ) {
	return apply_filters( 'app_get_app', BASE()->get_app( $app_id, $prime_cache ), $app_id, $prime_cache );
}

/**
 * Return current appointment object, if there is one
 * @since 3.5.0
 * @return StdClass  object or null
 */
function wpb_get_current_app(){
	if ( ! empty( $_REQUEST['app_id'] ) && $app = wpb_get_app( wpb_clean( $_REQUEST['app_id'] ) ) ) {
		return $app;
	}

	return $GLOBALS['appointment'];
}

/**
 * Return a booking object
 * @param $booking_id	integer		Unique booking ID
 * @since 3.0
 * @return WpB_Booking object		See class.booking.php
 */
function wpb_get_booking( $booking_id ) {
	return new WpB_Booking( $booking_id );
}

/**
 * Add a booking
 * Does not check availability of the slot. If required, use wpb_is_free before calling this
 * @param $arg	array	Booking parameters (All optional):
 *		parent_id		integer			Parent ID of the booking if this is a child booking (0 if left empty)
 *		created			integer/string	Creation date/time as timestamp or date time in any standard format, preferably Y-m-d H:i:s. If left empty, current date/time
 *		user			integer			Client ID (Wordpress user ID of the client)
 *		location		integer			Location ID. It must exist in the DB
 *		service			integer			Service ID. It must exist in the DB
 *		worker			integer			Worker ID (WordPress user ID of the service provider)
 *		status			integer			Status of the booking. Must be an existing status (pending, confirmed, paid, running, cart, removed, test, cart). If left empty, "confirmed"
 *		start			integer/string	Start of the slot as timestamp or date time in any standard format, preferably Y-m-d H:i:s. If not set, first free time will be used
 *		end				integer/string	End of the slot as time stamp or date time. If left empty, end time is calculated from duration of service
 *		price			string			Price of the booking. Comma and/or point allowed, e.g. $1.234,56
 *		deposit			string			Security deposit for the booking. Comma and/or point allowed, e.g. $1.234,56
 *		payment_method	string			Name of the payment method for the booking. Common values: manual-payments, paypal-standard, stripe, paymill
 * @since 3.0
 * @return integer (Booking ID) on success, false on fail. Reason of failure can be read from string: BASE()->db->last_error
 */
function wpb_add_booking( $args = array() ) {
	$booking = new WpB_Booking();
	return $booking->add( $args );
}

/**
 * Change a booking
 * @param $arg	array	Booking parameters (All optional except ID):
 *		ID	integer			Booking ID to be edited
 *		For the rest, see wpb_add_booking. 'created' only updated if explicitly set
 * @since 3.7.6.2
 * @return bool			True on success, false on fail. Reason of failure can be read from string: BASE()->db->last_error
 */
function wpb_update_booking( $args ) {
	$args['ID'] = isset( $args['ID'] ) ? $args['ID'] : 0;
	$booking = new WpB_Booking( $args['ID'] );
	return $booking->add( $args );
}

/**
 * Alias of wpb_update_booking
 * @since 3.0
 * @return bool
 */
function wpb_edit_booking( $args ) {
	return wpb_update_booking( $args );
}

/**
 * Delete one or more bookings
 * @param $ids 			array|string	Array or comma delimited string of booking IDs to be deleted
 * @param $status		string			Optional status of the bookings to delete
 * @since 3.0
 * @return bool
 */
function wpb_delete_booking( $ids, $status = '' ) {
	$ids = is_array( $ids ) ? $ids : wpb_explode( wpb_sanitize_commas( $ids ), ',' );

	if ( empty( $ids ) ) {
		return false;
	}

	$q_app = '';
	$result = false;

	foreach ( $ids as $app_id ) {
		$q_app .= " ID='". esc_sql( $app_id ). "' OR";
	}

	$q_app = rtrim( $q_app, " OR" );

	if ( $status ) {
		$q = "(".$q_app.") AND status='". esc_sql( $status ). "'";
	} else {
		$q = $q_app;
	}

	$result = BASE()->db->query( "DELETE FROM " .BASE()->app_table. " WHERE (".$q.") " );

	if ( $result ) {
		BASE()->db->query( "UPDATE ". BASE()->transaction_table . " SET transaction_app_ID=0 WHERE (".str_replace( 'ID','transaction_app_ID', $q_app ).") " );
		BASE()->db->query( "DELETE FROM " .BASE()->meta_table. " WHERE (".str_replace( 'ID','object_id', $q_app ).") AND meta_type='app'" );
		return true;
	}

	return $result;
}

/**
 * Change status of one or more bookings
 * Note: This doesn't automatically change status of the child bookings. Either include them in IDs or call 'app_change_status_children' action
 * @param $ids			array|string 	Array or comma delimited string of booking IDs
 * @param $new_status	string			New status. It should be a valid status (pending, paid, confirmed, completed, etc)
 * @param $old_status	string|array	Optional old statuses to change from. It should be a valid status. array or comma delimited string
 * @since 3.0
 * @return mix			false|integer	Number of records changed on success, false on failure
 */
function wpb_change_status( $ids, $new_status, $old_status = '' ) {
	$ids = is_array( $ids ) ? $ids : wpb_explode( wpb_sanitize_commas( $ids ), ',' );

	if ( empty( $ids ) ) {
		return false;
	}

	$q		= '';
	$result = false;
	$statii = array_keys( BASE()->get_statuses() );

	foreach ( $ids as $app_id ) {
		$q .= " ID=". esc_sql($app_id). " OR";
	}

	$q = rtrim( $q, " OR" );

	if ( $old_status ) {
		$valid		= array();
		$old_statii = array_filter( array_map( 'trim', (is_array( $old_status ) ? $old_status : explode( ',', $old_status ) ) ) );

		foreach ( $old_statii as $old ) {
			if ( in_array( $old, $statii ) ) {
				$valid[] = $old;
			}
		}

		if ( ! empty ( $valid ) ) {
			$q = '('.$q.') AND status IN ("' . implode('", "', $valid ) . '")';
		}
	}

	if ( in_array( $new_status, $statii ) ) {
		$result = BASE()->db->query( "UPDATE " . BASE()->app_table . " SET status='".$new_status."' WHERE " . $q );
	}

	return $result;
}

/**
 * Return a service StdClass object
 * @param	$ID		integer			ID of the service to be retrieved
 * @since 3.0
 * @return object	Properties:
 *		internal	integer			0=Not internal, 1=Internal
 *		sort_order	integer			Order of the service in the list
 *		name		string			Name of the service
 *		capacity	integer			Capacity of the service
 *		duration	integer			Duration of the service in minutes
 *		padding		integer			Padding before in minutes
 *		break_time	integer			Padding after in minutes
 *		locations	string			IDs of locations for which this service is available, : delimited, e.g. :1:3:4:
 *		categories	string			IDs of categories this service have, : delimited, e.g. :1:3:4:
 *		price		string			Price of service
 *		deposit		string			Deposit for the service
 *		page		integer			ID of the page/post that will be used as description page
 *
 */
function wpb_get_service( $ID ) {
	return BASE()->get_service( $ID );
}

/**
 * Add a service to the DB
 * @param $arg	array	Parameters (All optional):
 *		internal	bool			Whether service is internal
 *		sort_order	integer			Order of the service in the list. If left empty, service will be added to the end
 *		name		string			Name of the service. Defaults to custom text for 'service'
 *		capacity	integer			Capacity of the service
 *		duration	integer			Duration of the service in minutes. Defaults to time base setting
 *		padding		integer			Padding before in minutes
 *		break_time	integer			Padding after in minutes
 *		locations	array/string	An array or comma delimited string with IDs of locations for which this service is available. Location must be existing
 *		categories	array/string	An array or comma delimited string with IDs of categories for this service. Existence of category is not checked
 *		price		string			Price of service. Comma and/or point allowed, e.g. $1.234,56
 *		deposit		string			Security Deposit for the service. Comma and/or point allowed, e.g. $1.234,56
 *		page		integer			ID of the page/post that will be used as description page
 *		owner		integer			ID of owner, usually a service provider. If left empty, admin owns the service
 *		managed_by	string|array	Comma delimited string or array of provider IDs. IDs are not checked
 *		description	string			Description of service
 *		image_id	integer			Attachment ID of the service image. Existence of image is not checked
 *		created_by	integer			User ID of the creator. Must be an existing user, else or if left empty current user
 *
 * @since 3.0
 * @return service ID on success, false on fail. Reason of failure can be read from string: BASE()->db->last_error
 */
function wpb_add_service( $args = array() ) {
	$s = new WpB_Service();
	return $s->add( $args );
}

/**
 * Change a service
 * @param $arg	array	Service properties (All optional except ID):
 *		ID	integer			Service ID to be edited
 *		For the rest, see wpb_add_service. sort_order only updated if explicitly set
 *		Unset properties are kept as they are
 * @since 3.7.6.2
 * @return bool			True on success, false on fail. Reason of failure can be read from string: BASE()->db->last_error
 */
function wpb_update_service( $args ) {
	$args['ID'] = isset( $args['ID'] ) ? $args['ID'] : 0;
	return wpb_add_service( $args );
}

/**
 * Alias of wpb_update_service
 * @since 3.0
 * @return bool
 */
function wpb_edit_service( $args ) {
	return wpb_update_service( $args );
}

/**
 * Delete a service and related metas and Work Hours
 * @param $ID	integer		Service to be deleted
 * @since 3.0
 * @return		bool		True on success, false on fail. Reason of failure can be read from string: BASE()->db->last_error
 */
function wpb_delete_service( $ID ) {
	$r = BASE()->db->query( BASE()->db->prepare("DELETE FROM " .BASE()->services_table. " WHERE ID=%d LIMIT 1", $ID) );
	if ( $r ) {
		BASE()->db->query( BASE()->db->prepare("DELETE FROM " .BASE()->meta_table. " WHERE object_id=%d AND meta_type='service'", $ID) );
		BASE('WH')->remove( $ID, 'service' );
	}

	return $r;
}

/**
 * Return total duration (minutes) of a time slot
 * @param $slot_booking		object			WpB_Slot or WpB_Booking object
 * @return integer		Duration Time in minutes
 * @since 3.0
 */
function wpb_get_duration( $slot_booking ) {
	$slot		= $slot_booking instanceof WpB_Slot ? $slot_booking : new WpB_Slot( $slot_booking );
	$s			= BASE()->get_service( $slot->get_service() );
	$duration	= ! empty( $s->duration ) ? $s->duration : BASE()->get_min_time();

	return apply_filters( 'app_get_duration', intval( $duration ), $slot );
}

/**
 * Check if user is a service provider (worker)
 * @param $ID		integer			WP user ID. If not provided, current user is checked
 * @return bool
 * @since 3.0
 */
function wpb_is_worker( $ID = 0 ) {
	$uid = $ID ? $ID : ( function_exists( 'get_current_user_id' ) ? get_current_user_id() : 0 );
	return BASE()->is_worker( $uid );
}

/**
 * Returns service provider (worker) object
 * @param	$ID				integer			ID of the worker to be retrieved = WP User ID
 * @since 3.0
 * @return object	Properties:
 *		sort_order			integer			Order of the SP in the list
 *		name				string			Display name of the SP
 *		services_provided	string			IDs of services this SP is providing, : delimited, e.g. :1:3:4:
 *		price				string			Additional price of the SP
 *		page				integer			ID of the page/post of bio page of SP
 *		dummy				integer			0=Not dummy, 1=Dummy
 */
function wpb_get_worker( $ID ) {
	return BASE()->get_worker( $ID );
}

/**
 * Adds a service provider (Assigns a WP user as Service Provider)
 * Working hours will be taken from that of Business Representative
 * Person to be assigned as a service provider must be existing as a WP user
 * Role assignment should be done manually:
 * 		$user = new WP_User( $ID );
 *		$user->add_role( 'wpb_worker' );
 *
 * @param $arg	array	Parameters (All optional):
 *		ID					integer			User WP ID. If left empty, current user will be assigned.
 *		sort_order			integer			Order of the SP in the list. If left empty, SP will be at added to the end
 *		name				string			Display name. If left empty, it will created from display name or user_login
 *		services_provided	array/string	An array or comma delimited string with IDs of services provided by SP. If left empty, first service in list will be used
 *		price				string			Additional price of worker. Comma and/or point allowed, e.g. $1.234,56
 *		page				integer			ID of the page/post that will be used as bio page.
 *
 * @since 3.0
 * @return true on success, false on fail. Reason of failure can be read from string: BASE()->db->last_error
 */
function wpb_add_worker( $args = array() ) {
	$user_id	= ! empty( $args['ID'] ) ? $args['ID'] : ( function_exists('get_current_user_id') ? get_current_user_id() : false );
	$user		= get_user_by( 'ID', $user_id );

	if ( empty( $user->user_login ) ) {
		BASE()->db->last_error = __('User does not exist','wp-base');
		return false;
	}

	if ( $args['ID'] && $_worker = wpb_get_worker( $args['ID'] ) ) {
		$vars = get_object_vars( $_worker );
		$vars['services_provided'] = array_filter( explode( ':', $vars['services_provided'] ) );
		$args = wp_parse_args( $args, $vars );
	}

	$r			= false;
	$s_provided = array();
	$services	= isset( $args['services_provided'] ) ? $args['services_provided'] : array();
	$s_to_check	= is_array( $services ) ? $services : wpb_explode( wpb_sanitize_commas( $services ), ',' );

	foreach ( (array)$s_to_check as $s ) {
		if ( BASE()->service_exists( $s ) ) {
			$s_provided[] = $s;
		}
	}

	if ( empty( $s_provided ) ) {
		$s_provided[] = BASE()->get_first_service_id();
	}

	$data = array(
		'ID'				=> $user_id,
		'name'				=> ! empty($user->display_name) ? $user->display_name : $user->user_login,
		'services_provided'	=> ! empty( $s_provided ) ? wpb_implode( $s_provided ) : '',
		'price'				=> ! empty( $args['price'] ) ? wpb_sanitize_price( $args['price'] ) : 0,
		'page'				=> ! empty( $args['page'] ) ? (int)$args['page'] : '',
	);

	if ( ! empty( $args['sort_order'] ) ) {
		$data['sort_order'] = intval( $args['sort_order'] );
	} else if ( !BASE()->worker_exists( $user_id ) ) {
		$max_sort_order = BASE()->db->get_var( "SELECT MAX(sort_order) FROM " . BASE()->workers_table );
		$data['sort_order'] = intval($max_sort_order) + 1;
	}

	if ( BASE()->worker_exists( $user_id ) ) {
		if ( BASE()->db->update( BASE()->workers_table, $data, array( 'ID'=>$user_id ) ) ) {
			$r = $user_id;
		}
	} else if ( BASE()->db->insert( BASE()->workers_table, $data ) ) {
		BASE('WH')->add_default( $user_id, 'worker' );
		$r = $user_id;
	}

	if ( $r ) {
		wpb_flush_cache();
	}

	return $r;
}

/**
 * Change a service provider
 * @param $arg	array	Parameters are properties of worker object:
 *		See wpb_add_worker for description. sort_order only updated if explicitly set
 *		Undeclared properties are kept as they are
 * @since 3.7.6.2
 * @return bool		true on success, false on fail. Reason of failure can be read from string: BASE()->db->last_error
 */
function wpb_update_worker( $args ) {
	return wpb_add_worker( $args );
}

/**
 * Alias of wpb_update_worker
 * @since 3.0
 * @return bool
 */
function wpb_edit_worker( $args ) {
	return wpb_add_worker( $args );
}

/**
 * Delete a service provider and related metas and Work Hours (Unassign a WP user from SP)
 * WP user records are not deleted.
 * Role removal should be done manually:
 * 		$user = new WP_User( $ID );
 *		$user->remove_role( 'wpb_worker' );
 *
 * @param $ID	integer		Service provider to be deleted
 * @since 3.0
 * @return		bool		True on success, false on fail. Reason of failure can be read from string: BASE()->db->last_error
 */
function wpb_delete_worker( $ID ) {
	$r = BASE()->db->query( BASE()->db->prepare("DELETE FROM " .BASE()->workers_table. " WHERE ID=%d LIMIT 1", $ID) );
	if ( $r ) {
		BASE()->db->query( BASE()->db->prepare("DELETE FROM " .BASE()->meta_table. " WHERE object_id=%d AND meta_type='worker'", $ID) );
		if ( $ID != BASE()->get_def_wid() ) {
			BASE('WH')->remove( $ID, 'worker' );
		}
	}

	return $r;
}

/**
 * Create a WP user from an array of fields ($data['email'], $data['name'], etc ).
 * If user already exists, returns his ID
 * Also save user meta
 * @param $data			array		User info fields. Keys of array are:
 *		name			string		Either name, first_name or last_name required
 *		first_name		string		Ditto
 *		last_name		string		Ditto
 *		email			string		Should be valid email
 *		phone			string
 *		password		string		If left empty, a password will be automatically created
 *		address			string
 *		city			string
 *		zip				string		Postcode
 *		state			string		For future
 *		country			string		For future
 * Note: $data can be retrieved from a booking with: BASE( 'User' )->get_app_userdata( $booking->get_ID() )
 * @param $notify_admin	bool		Whether notify admin. Optional.
 * @param $notify_user	bool|null	Whether notify user (If null, "auto_register_client_notify" setting is in effect. If false, user not notified (default). If true, user is notified)
 * @since 3.0
 * @return mix		false|integer	User ID on success, false on failure
 */
function wpb_create_user( $data, $notify_admin = false, $notify_user = false ) {
	return BASE('User')->create_user( $data, $notify_admin, $notify_user );
}

/**
 * Returns a subset of WP Users who are clients
 *
 * @param $args		array	See get_users WordPress function
 * @since 3.9.0
 * @return array
 */
function wpb_get_clients( $args = array() ) {
	return get_users( wp_parse_args( $args, array( 'role' => 'wpb_client' ) ) );
}

