<?php
 /**
 * Pay Later
 *
 * Handles payment on front end when Pay Later option is selected
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( !class_exists( 'WpBPayLater' ) ) {

class WpBPayLater {
	
	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;
	
	const PAYMENT_SHORTCODE = 'app_pay';
	
	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}
	
	/**
     * Add hooks
     */
	public function add_hooks(){
		
		add_shortcode( self::PAYMENT_SHORTCODE, array( $this, 'shortcode' ) );

		add_action( 'init', array( $this, 'payment_from_email' ), 3 );
		add_filter( 'app_active_gateways', array( $this, 'disable_for_payment' ) );
		add_filter( 'app_confirmation_active_gateways', array( $this, 'disable_for_checkout' ) );
		add_action( 'app_pre_confirmation_start', array( $this, 'ajax_control' ), 3 );
		add_action( 'app_post_confirmation_start', array( $this, 'ajax_control' ), 3 );
		add_action( 'app_payment_received', array( $this, 'payment_received' ), 3 );
		
		add_filter( 'app_list_allowed_columns', array( $this, 'add_column' ), 10, 2 );
		add_filter( 'app_list_add_cell', array( $this, 'add_button' ), 10, 4 );
	}
	
	/**
	 * Get balance of the booking
     * Minus means client needs to pay
	 * @return string
     */
	private function get_balance( $app ) {
		$booking	= new WpB_Booking( $app );
		return $booking->get_balance();
	}
	
	/**
	 * Find url of the payment page
	 * @return string
     */
	public static function payment_page_url() {
		$url = get_permalink( wpb_gateway_setting( 'pay-later', 'payment-page' ) );
		return apply_filters( 'app_payment_page_url', $url ?: home_url() ); 
	}
	
	/**
	 * Check if this is the payment page
 	 * @return bool
     */
	private function is_payment_page( $post = null ){
		
		$page = wpb_gateway_setting( 'pay-later', 'payment-page' );
		$post = get_post( $post );
		
		if ( $page && !empty( $post->ID ) && $page == $post->ID ) {
			return true;
		}
		
		if ( !empty( $post->post_content ) && has_shortcode( $post->post_content, self::PAYMENT_SHORTCODE ) ) {
			return true;
		}
		
		if ( !empty( $post->post_content ) && has_shortcode( $post->post_content, 'app_book' ) ) {
			return false;
		}

		if ( ! $page && ( is_home() || is_front_page() ) ) {
			return true;
		}
		
		return false;
	}
	
	/**
	 * Helper to create payment hash
	 * @param $app		object|integer		Appointment object or app ID
 	 * @return string
     */
	private function calc_hash( $app ) {
		return $this->a->create_hash( is_object( $app ) ? $app : wpb_get_app( $app ), 'payment' );
	}
	
	/**
	 * Helper to get session val
 	 * @return	array	In the form: array( $app_id => $hash )
     */
	private static function session_val(){
		return wpb_get_session_val( 'pay_later' );
	}

	/**
	 * Handle confirmation of an appointment by the client using email link
	 * @since 3.0
	 */
	public function payment_from_email() {
		if ( !empty( $_GET['app_re_payment_login'] ) && !is_user_logged_in() ) {
			wpb_notice( $this->a->get_text('login_for_confirm') );
			return;
		}

		if ( !empty( $_GET['app_pay_now'] ) && !empty( $_GET['payment_nonce'] ) ) {
			
			$app = wpb_get_app( wpb_clean( $_GET['app_pay_now'] ) );
			
			if ( $app && $_GET['payment_nonce'] == $this->calc_hash( $app ) ) {
				add_filter( 'app_js_data', array( $this, 'js_data' ), 6, 2 );
				add_filter( 'the_content', array( $this, 'content' ), 6 );
				do_action( 'app_shortcode_found', 'confirmation', get_post() );
			}

			return;
		}

		/* confirm by the link in email */
		if ( empty( $_GET['app_payment'] ) || empty( $_GET['app_id'] ) || empty( $_GET['payment_nonce'] ) ) {
			return;
		}

		/* For future - This setting is not available yet */
		if ( 'yes' != wpb_setting( 'allow_payment', 'yes' ) ) {
			
			if ( isset( $_REQUEST['app_id'] ) && isset( $_REQUEST['payment_nonce'] ) ) {
				
				if ( wpb_doing_ajax() ) {
					die( json_encode( array( 'error' => $this->a->get_text('confirm_disabled') ) ) );
				} else {
					wpb_notice( 'confirm_disabled', 'error' );
				}
			}

			return;
		}

		$app_id	= wpb_clean( $_GET['app_id'] );
		$app	= wpb_get_app( $app_id );
		$hash	= $this->calc_hash( $app );

		if (  $_GET['payment_nonce'] != $hash ) {
			wpb_notice( 'error', 'error' );
			return;
		}

		if ( empty( $app->created ) ) {
			wpb_notice( 'not_possible', 'error' );
			return; # Appt deleted completely
		}

		if ( !empty( $app->user ) && !is_user_logged_in() ) {
			# Come back to the same page after login
			$redirect = wpb_add_query_arg( 'app_re_payment_login', $app_id, wp_login_url( esc_url_raw( $_SERVER['REQUEST_URI'] ) ) );
			wp_redirect( $redirect );
			exit;
		}

		if ( ! wpb_is_owner( $app ) ) {
			wpb_notice( 'not_owner', 'error' );
			return;
		}
		
		if ( (int)$this->get_balance( $app ) >= 0 ) {
			wpb_notice( 'paid_up' );
			return;
		}
		
		if ( 'pay-later' != $app->payment_method ) {
			$this->a->db->update( $this->a->app_table, array( 'payment_method' => 'pay-later' ), array( 'ID' => $app_id ) );
		}
		
		wpb_set_session_val( 'app_cart', array( $app_id ) );
		wpb_set_session_val( 'pay_later', array( $app_id => $hash ) );
		
		wp_redirect( wpb_add_query_arg( array( 
			'app_pay_now'	=> $app_id,
			'payment_nonce'	=> $hash, 
		), self::payment_page_url() ) );
		
		exit;
	}

	/**
	* Do not allow self and manual payments to make payment
	* @param $gateways		array of objects	Active gateways for website
	* @uses app_active_gateways filter hook
	* @return array
	*/
	public function disable_for_payment( $gateways ) {
		if ( $this->is_payment_page() && self::session_val() ) {
			foreach ( $gateways as $key => $gateway ) {
				if ( 'pay-later' == $gateway->plugin_name || 'manual-payments' == $gateway->plugin_name ) {
					unset( $gateways[$key] );
				}
			}
		}

		return $gateways;
	}

	/**
	* Do not allow self at checkout if set so
	* @param $gateways		array of objects	Active gateways for confirmation form
	* @uses app_confirmation_active_gateways filter hook
	* @return array
	*/
	public function disable_for_checkout( $gateways ) {
		if ( 'yes' == wpb_gateway_setting( 'pay-later', 'disable-checkout' ) ) {
			foreach ( $gateways as $key => $gateway ) {
				if ( 'pay-later' == $gateway->plugin_name ) {
					unset( $gateways[$key] );
				}
			}
		}

		return $gateways;
	}
	
	/**
	 * Modify data which controls jsavascript 
     * @param $data		array
	 * @return array
     */
	public function js_data( $data, $context ) {
		if ( 'front' != $context ) {
			return $data;
		}
		
		$slot = new WpB_Slot( new WpB_Booking( wpb_clean( $_GET['app_pay_now'] ) ) );
		$data['cart_values'] = array( $slot->pack() ); 
		
		return $data;
	}

	/**
	 * Modify content and add the shortcode, if it is not already there
     * @param $content		string		Post content
	 * @return string
     */
	public function content( $content ) {
		if ( has_shortcode( $content, self::PAYMENT_SHORTCODE ) ) {
			return $content;
		} else {
			return $content . '<div class="app-pay-later">['.self::PAYMENT_SHORTCODE.']</div>';
		}
	}
	
	/**
	 * app_payment shortcode
     * @param $atts		array		Shortcode attributes
	 * @return string
     */
	public function shortcode( $atts ) {
		
		extract( shortcode_atts( array(
			'fields'	=> '',
		), $atts, self::PAYMENT_SHORTCODE ) );
		
		add_filter( 'app_confirmation_after_fieldset', array( $this, 'legend' ) );
		
		remove_all_filters( 'app_confirmation_after_booking_fields' );
		remove_all_filters( 'app_confirmation_before_user_fields' );
		remove_all_filters( 'app_confirmation_after_user_fields' );
		remove_all_filters( 'app_confirmation_add_field' );
		
		return do_shortcode( '[app_confirmation continue_btn=0 countdown=0 fields="'.$fields.'"]' );
	}
	
	/**
	 * Add message in Legend tag of confirmation form 
     * @param $ret		string
	 * @return string
     */
	public function legend( $ret ) {
		
		if ( ! $this->is_payment_page() || ! $val = self::session_val() ) {
			return $ret;
		}
		
		$app_id = key( $val );
		
		if ( $text = wpb_gateway_setting( 'pay-later', 'form-header' ) ) {
			$legend = wpb_replace( $text, $app_id, 'confirmation_subject' );
		} else {
			$legend = sprintf( __( 'Please confirm payment for Booking %d', 'wp-base' ), $app_id );
		}
		
		return $ret . '<legend>' . $legend . '</legend>';
	}
	
	/**
	 * Handle ajax actions
	 * @return none
     */
	public function ajax_control(){
		
		if ( ! $plater = self::session_val() ) {
			return;
		}
		
		if ( current( $plater ) != $this->calc_hash( key( $plater ) ) ) {
			return;
		}
		
		if ( ! $this->is_payment_page( wpb_find_post_id() ) ) {
			return;
		}
		
		add_filter( 'app_options', array( $this, 'options' ) );
		add_filter( 'app_post_confirmation_save_data', array( $this, 'post_confirmation_save_data' ), 10, 3 );
		add_filter( 'app_confirmation_total_price', array( $this, 'confirmation_total_amount' ), 5000 );
		add_filter( 'app_confirmation_total_amount', array( $this, 'confirmation_total_amount' ), 5000 );
		add_filter( 'app_is_wc_page', '__return_false' );
		remove_all_filters( 'app_post_confirmation_userdata' );
	}
	
	/**
	 * Force payment required and hide some displays
     * @param $options		array		WP BASE options (settings)
	 * @return array
     */
	public function options( $options ) {
		$options['payment_required'] = 'yes';
		$options['conf_form_hidden_fields'] =  implode( ',', array_merge( array('details', 'price', 'deposit',), explode( ',', $options['conf_form_hidden_fields'] ) ) );
		return $options;
	}

	/**
	 * Set amount to be paid as balance and add fee
	 * @uses app_confirmation_total_amount filter hook
     * @param $amount	string		Previously calculated amount
	 * @return string
     */
	public function confirmation_total_amount( $amount ) {
		if ( ! $plater = self::session_val() ) {
			return $amount;
		}
		
		return -1 * $this->get_balance( key( $plater ) ) + ( wpb_get_session_val( 'fee' ) ?: 0 );
	}
	
	/**
	 * Refresh created date, otherwise booking may be removed
     * @param $data		array	Booking data to be saved/updated
	 * @return array
     */
	public function post_confirmation_save_data( $data, $old_booking, $val ) {
		
		if ( $plater = self::session_val() ) {
			$app_id	= key( $plater );
			$slot	= new WpB_Slot( $val );
			
			if ( $slot->get_app_id() == $app_id ) {
				
				$data['price'] = $old_booking->get_price() + ( wpb_get_session_val( 'fee' ) ?: 0 );
				$data['deposit'] = $old_booking->get_deposit();				
				$data['created'] = $this->a->_time;
				# Hash has to be refreshed
				add_action( 'app_new_appointment', array( $this, 'update_hash' ) );
			}
		}
		
		return $data;
	}
	
	/**
	 * Refresh hash
	 * @uses app_new_appointment
     * @param $app_id	integer		ID of the updated booking
	 * @return none
     */
	public function update_hash( $app_id ) {
		if ( ! $plater = self::session_val() ) {
			return;
		}
		
		if ( key( $plater ) != $app_id ) {
			return;
		}

		wpb_set_session_val( 'pay_later', array( $app_id => $this->calc_hash( $app_id ) ) );
	}
	
	/**
	 * Clear session cookie after payment received
	 * @uses app_payment_received action hook
     * @param $booking ID	integer		ID of the updated booking
	 * @return none
     */
	public function payment_received( $booking_id ) {
		if ( ! $plater = self::session_val()  ) {
			return;
		}
		
		if ( key( $plater ) == $booking_id ) {
			wpb_set_session_val( 'pay_later', null );
		}
	}
	
	/**
     * Add Pay column to List of Bookings
	 * @return array
     */
	public function add_column( $allowed, $args ) {
		
		if ( ! wpb_can_use_cc() ) {
			return $allowed;
		}
		
		if ( ! wpb_is_gateway_active( 'pay-later' ) ) {
			return $allowed;
		}
		
		# Supports only logged in user
		if ( is_user_logged_in() && 'yes' == wpb_setting( 'payment_required' ) && ((!empty( $args['what'] ) && 'client' == $args['what']) || current_user_can( 'wpb_client' )) ) {
			$allowed[] = 'pay';
		}

		return $allowed;
	}

	/**
     * Whether add a button to the cell in Bookings List
     */
	public function add_button( $ret, $col, $r, $args ) {
		if ( 'pay' != $col || $r->user != get_current_user_id() ) {
			return $ret;
		}
		
		$balance = $this->get_balance( $r );

		if ( $balance < 0 ) {
			$is_disabled = '';
			$title		 = '';
			$button_text = str_replace( 'AMOUNT', wpb_format_currency( abs( $balance ) ), 
					   (!empty( $args['pay_button'] ) 
					   ? $args['pay_button'] 
					   : $this->a->get_text('pay_now')) );
	   } else {
			$is_disabled	= ' app-disabled-button';
			$title			= 'title="'. $this->a->get_text('paid_up').'"';
			$button_text	= $this->a->get_text('paid');
		}
		
		$href = wpb_add_query_arg( array(
						'app_payment'	=> 1, 
						'app_id'		=> $r->ID, 
						'payment_nonce'	=> $this->calc_hash( $r ) 
					), 
					self::payment_page_url()
				);
		
		$ret .= '<button '.$title.' class="app-list-pay ui-button ui-state-default '.$is_disabled.
				'" data-app_id="'.$r->ID.'" name="app_pay['.$r->ID.']" data-href="'.esc_attr( $href ).'">'.$button_text.'</button>';

		return $ret;
	}
	
}
	BASE('PayLater')->add_hooks();
}