<?php
/**
 * WPB Front Ajax
 *
 * Handles ajax requests of front end
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBAjax' ) ) {

class WpBAjax {

	/**
     * Sum of cart values
     */
	public $total_list_price = 0;
	public $total_price = 0;
	public $total_amount = 0;
	public $total_deposit = 0;
	public $payment_method = '';

	/**
     * Array of submitted time slots after worker assigned
     */
	public $items = array();

	/**
     * Array of prices for each slot
     */
	public $slot_prices = array();

	 /**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}

	/**
     * Add hooks
     */
	public function add_hooks(){
		add_action( 'wp_ajax_update_cals', array( $this, 'update_cals' ) ); 								// Update calendars acc to pulldown menu selections
		add_action( 'wp_ajax_nopriv_update_cals', array( $this, 'update_cals' ) ); 							// Update calendars acc to pulldown menu selections
		add_action( 'wp_ajax_bring_timetable', array( $this, 'bring_timetable' ) ); 						// Get timetable html codes
		add_action( 'wp_ajax_nopriv_bring_timetable', array( $this, 'bring_timetable' ) ); 					// Get timetable html codes
		add_action( 'wp_ajax_bring_book_table', array( $this, 'book_table_ajax' ) ); 						// Get book table html codes
		add_action( 'wp_ajax_nopriv_bring_book_table', array( $this, 'book_table_ajax' ) ); 				// Get book table html codes
		add_action( 'wp_ajax_pre_confirmation', array( $this, 'pre_confirmation' ) ); 						// Get pre_confirmation results
		add_action( 'wp_ajax_nopriv_pre_confirmation', array( $this, 'pre_confirmation' ) ); 				// Get pre_confirmation results
		add_action( 'wp_ajax_pre_confirmation_update', array( $this, 'pre_confirmation_update' ) ); 		// Get pre_confirmation results in case update price
		add_action( 'wp_ajax_nopriv_pre_confirmation_update', array( $this, 'pre_confirmation_update' ) ); 	// Get pre_confirmation results in case update price
		add_action( 'wp_ajax_post_confirmation', array( $this, 'post_confirmation' ) ); 					// Do after final confirmation
		add_action( 'wp_ajax_nopriv_post_confirmation', array( $this, 'post_confirmation' ) );				// Do after final confirmation
		add_action( 'wp_ajax_bob_update', array( $this, 'update_user_fields' ) ); 							// Update conf form
		add_action( 'wp_ajax_app_show_children_in_tooltip', array( $this, 'show_children_in_tooltip' ) );
		add_action( 'wp_ajax_nopriv_app_show_children_in_tooltip', array( $this, 'show_children_in_tooltip' ) );
		add_action( 'wp_ajax_app_lsw_tooltip', array( $this, 'lsw_tooltip' ) );
		add_action( 'wp_ajax_nopriv_app_lsw_tooltip', array( $this, 'lsw_tooltip' ) );
	}

	/**
	 * Update calendars using ajax
	 * @since 2.0
	 */
	public function update_cals() {

		if ( ! defined( 'WPB_AJAX' ) ) {
			define( 'WPB_AJAX', true );
		}

		do_action( 'app_update_calendars' );

		$post_id 	= ! empty( $_POST["post_id"] ) ? absint( wpb_clean( $_POST["post_id"] ) ) : 0;
		$page		= get_post( $post_id );

		if ( $post_id ) {
			wpb_set_session_val( 'app_post_id', $post_id );
		}

		if ( wpb_is_bp_book_me( ) ) {
			$bp_book_me_tab = true;
			$content = BASE('BuddyPress')->screen_content_wpb_book_me( true, absint( wpb_clean( $_POST['bp_displayed_user_id'] ) ) ); // Get without do_shortcode
		} else {
			$bp_book_me_tab = false;
			$content = $page ? $this->a->post_content( $page->post_content, $page, true ) : '';
		}

		$aid = 0;
		$scode_arr = self::content_from_shortcodes();

		if ( ! empty( $_POST['active_id'] ) && array_key_exists( $_POST['active_id'], $scode_arr ) && $scode_arr['app_book_count'] > 1 ) {

			$aid		= $_POST['active_id'];
			$content	= '['.$scode_arr[ $aid ]['tag'].' '. $scode_arr[ $aid ]['attr'].']';
		}

		if ( ('yes' == wpb_setting( 'theme_builder_compat' ) &&  ! empty( $scode_arr['content'] )) || apply_filters( 'app_front_ajax_create_content_from_shortcode', ! trim( $content ) && ! empty( $scode_arr['content'] ), $content ) ) {

			$content = $scode_arr['content']; 
		}

		if ( has_shortcode( $content, 'app_account' ) ) {
			add_filter( 'app_is_admin_page', '__return_true' );
			add_filter( 'app_is_account_page', '__return_true' );
		}

		# Default step/unit values
		$step 		= 4;
		$unit 		= 'week';
		$keep_pag 	= false;

		if ( ! empty( $_POST['unit'] ) && ! empty( $_POST['step'] ) ) {
			$step = absint( wpb_clean( $_POST['step'] ) );
			$unit = wpb_clean( $_POST['unit'] );
		} else if ( $bp_book_me_tab || $page ) {
			if ( preg_match( '/' . get_shortcode_regex(array('app_next')) . '/', $content, $m ) ) {
				$atts = shortcode_parse_atts( $m[3] );
				if ( isset( $atts['step'] ) ) {
					$step = (int)$atts['step'];
				}
				
				if ( isset( $atts['unit'] ) ) {
					$unit = $atts['unit'];
				}
			} else {
				$keep_pag = true;
			}
		}

		$time = ! empty( $_REQUEST["app_timestamp"] ) ? absint( wpb_clean( $_REQUEST["app_timestamp"] ) ) : $this->a->_time;

		list( $prev, $prev_min, $ignore, $next, $next_max, $ignore ) = wpb_prev_next( $time, $step, $unit );

		if ( $prev <= $prev_min ) {
			$prev = "hide";
		}

		if ( $next >= $next_max ) {
			$next = "hide";
		}

		$result = $result_compact = $result_menus = $widgets = array();

		if ( $page || $bp_book_me_tab ) {
			if ( !wpb_is_mobile() ) {
				$content = preg_replace( '/' . get_shortcode_regex(array('app_is_mobile')) . '/','', $content );
			} else if ( wpb_is_mobile() ) {
				$content = preg_replace( '/' . get_shortcode_regex(array('app_is_not_mobile')) . '/','', $content );
			}

			# LSW menus
			$menus = apply_filters( 'app_menu_shortcodes', array( 'app_services' ) );

			while ( preg_match( '/' . get_shortcode_regex( $menus ) . '/', $content, $matches ) ) {
				$html = do_shortcode( $matches[0] );
				$result_menus[] = $html;
				$content = str_replace( $matches[0], $html, $content );
			}

			# Modular View Shortcodes
			$view_scodes = apply_filters( 'app_view_shortcodes', array('app_schedule','app_monthly_schedule','app_book_table','app_book_flex') );

			if ( preg_match_all( '/' . get_shortcode_regex( $view_scodes ) . '/', $content, $shortcode_arr ) ) {
				foreach( $shortcode_arr[0] as $shortcode ) {
					$result[] = do_shortcode( $shortcode );
				}
			}

			# app_book shortcode
			$compact_scodes = apply_filters( 'app_compact_shortcodes', array('app_book') );

			if ( preg_match_all( '/' . get_shortcode_regex( $compact_scodes ) . '/', $content, $shortcode_arr, PREG_SET_ORDER ) ) {
				foreach( $shortcode_arr as $key => $shortcode ) {
					$atts = shortcode_parse_atts( $shortcode[3] );
					$html = $this->a->book( array_merge( (array)$atts, array( '_just_calendar' => 1, 'id' => ($aid ?: '') ) ) );
					$content = str_replace( $shortcode[0], $html, $content );
					$result_compact[ $aid ?: $key ] = $html;
				}
			}

			// Widgets
			if ( isset( $_POST['used_widgets'] ) && is_array( $_POST['used_widgets'] ) ) {
				$ids	= wpb_clean( $_POST['used_widgets'] );
				$ops	= get_option( 'widget_appointments_shortcode' );
				include_once( WPBASE_PLUGIN_DIR . '/includes/constant-data.php' );
				$scodes	= array_keys( WpBConstant::shortcode_desc() );

				foreach ( $ids as $id ) {
					if ( isset( $ops[$id]['content'] ) && preg_match( '/' . get_shortcode_regex($scodes) . '/', $ops[$id]['content'], $irrelevant ) ) {
						$w_content = $ops[$id]['content'];

						if ( preg_match_all( '/' . get_shortcode_regex( wpb_shortcodes() ) . '/', $w_content, $shortcode_arr, PREG_SET_ORDER ) ) {
							foreach( $shortcode_arr as $shortcode ) {
								$w_content = str_replace( $shortcode[0], '['.$shortcode[2].' '. $shortcode[3].' _called_by_widget="'.$id.'"]', $w_content );
							}
						}

						$widgets[$id] = do_shortcode( $w_content );
					}
				}
			}
		}

		die( json_encode( apply_filters( 'app_update_cals_reply',
			array(
				'html'			=> $result,
				'htmlCompact'	=> $result_compact,
				'htmlMenus'		=> $result_menus,
				'prev'			=> $keep_pag ? '' : $prev,
				'next'			=> $keep_pag ? '' : $next,
				'widgets'		=> $widgets,
			))
		));
	}

	public static function content_from_shortcodes(){
		$scodes = ! empty( $_POST['scode'] ) ? json_decode( wpb_clean( wp_unslash( $_POST['scode'] ) ), true ) : false;

		$out = array( 'app_book_count' => 0, 'content' => '' );

		if ( is_array( $scodes ) && ! empty( $scodes ) ) {
			foreach ( $scodes as $key => $arr ) {
				$id		= ! empty( $arr['id'] ) ? $arr['id'] : $key;
				$scode	= ! empty( $arr['scode'] ) ? $arr['scode'] : array();

				foreach ( $scode as $tag => $attrs ) {
					$out[ $id ]['tag'] = $tag;
					$out['content'] = $out['content'] .'['. $tag.' ';

					if ( 'app_book' == $tag ) {
						$out['app_book_count'] = $out['app_book_count'] + 1;
					}

					foreach( (array)$attrs as $key => $val ) {
						$att = $key . '="'. $val. '" ';
						$out[ $id ]['attr'] = (! empty( $out[ $id ]['attr'] ) ? $out[ $id ]['attr'] : '') . $att;
						$out['content'] = $out['content'] . $att;
					}

					$out['content'] = $out['content'] .']';
				}
			}
		}

		return $out;
	}

	/**
	 * Handles ajax request for revealing available time tables for a day for monthly calendar
	 * @since 2.0
	 * @return json
	 */
	public function bring_timetable(){
		if ( isset( $_POST['app_value'] ) ) {
			$slot = new WpB_Slot( wpb_clean( $_POST['app_value'] ) );
		} else {
			wp_send_json_error( );
		}

		$html = '';
		$calendar = new WpB_Calendar( $slot );
		$calendar->setup( array(
			'display' 			=> ! empty( $_POST['display_mode'] ) ? wpb_clean( $_POST['display_mode'] ) : '',
			'force_min_time' 	=> ! empty( $_POST['force_min_time'] ) ? wpb_clean( $_POST['force_min_time'] ) : 0,
			'assign_worker'		=> ! $slot->get_worker(),
		) );

		foreach ( $calendar->find_slots_in_day( $slot->get_start() ) as $slot_start => $slot_obj ) {
			$html .= $slot_obj->monthly_calendar_cell_html( );
		}

		wp_send_json( apply_filters( 'app_bring_timetable_reply', array( 'data' => $calendar->wrap_cells( $slot->get_start(), $html ) ), $calendar ) );
	}

	/**
	 * Bring a book table or flex by ajax request and place it in html buffer
	 * Called by swipe js function
	 */
	public function book_table_ajax() {
		if ( ! isset( $_POST['start_ts'] ) ) {
			wp_send_json_error();
		}

		$start		= date('Y-m-d', wpb_clean( $_POST['start_ts'] ) );
		$title 		= isset( $_POST['title'] ) ? html_entity_decode( wpb_clean( $_POST['title'] ) ) : '';
		$logged 	= isset( $_POST['logged'] ) ? html_entity_decode( wpb_clean( $_POST['logged'] ) ) : '';
		$notlogged 	= isset( $_POST['notlogged'] ) ? html_entity_decode( wpb_clean( $_POST['notlogged'] ) ) : '';
		$pars		= array(
						'range'		=> '1day',
						'start'		=> $start,
						'title'		=> $title,
						'logged'	=> $logged,
						'notlogged'	=> $notlogged,
						'swipe'		=> 0,
						'mode'		=> 6,
		);

		if ( isset( $_POST['type'] ) && 'book_flex' == wp_unslash( $_POST['type'] ) && BASE('Pro') ) {
			$html = BASE('Pro')->book_flex( $pars );
		} else {
			$html = $this->a->book_table( $pars );
		}

		die( json_encode( array( 'html' => $html ) ) );
	}

	/**
	 * Check and return necessary fields to the front end when there is a change (usually to update price)
	 * @return json object
	 */
	public function pre_confirmation_update( ) {
		$this->pre_confirmation( true );
	}

	/**
	 * Check and return necessary fields to the front end
	 * @param update_price: Function is used for updating (price). Do not touch cart
	 * @return json object
	 */
	public function pre_confirmation( $update = false ) {

		self::check_ajax_referer();

		if ( ! defined( 'WPB_AJAX' ) ) {
			define( 'WPB_AJAX', true );
		}

		/* For Credits and Pay later */
		do_action( 'app_pre_confirmation_start' );

		$has_cart = ! empty( $_POST['has_cart'] ) ? true : false;

		// Clear previous cart items.
		if ( ! $update && ! $has_cart ) {
			BASE('Multiple')->empty_cart();
		}

		if ( ! $update ) {
			BASE('Multiple')->check_cart();
		}

		if ( ! empty( $_POST['deleted_value'] ) ) {
			$reply_array = BASE('Multiple')->remove_item( wpb_clean( $_POST['deleted_value'] ) );
		} else {
			$reply_array = array();
		}

		if ( ! isset( $_POST['value'] ) ) {
			die( json_encode( array_merge( $reply_array, array(
				'start'			=> $this->a->conf_line_html( 'date_time' ),
				'end'			=> $this->a->conf_line_html( 'end_date_time' ),
				'lasts'			=> $this->a->conf_line_html( 'lasts' ),
				'cart_contents'	=> $has_cart ? $this->a->conf_line_html( 'details' ) : '',
				'price'			=> $this->a->conf_line_html( 'price' ),
				'amount'		=> $this->a->conf_line_html( 'down_payment' ),
				'deposit'		=> $this->a->conf_line_html( 'deposit' ),
			))));
		} else {
			if ( is_array( $_POST['value'] ) ) {
				$value_arr = array_unique( wpb_clean( $_POST['value'] ) );
				sort( $value_arr );
				$count = count( $value_arr );
			} else {
				$value_arr = (array)wpb_clean( $_POST['value'] );
				$count = 1;
			}
		}

		# Check if this request is coming from BP Book Me tab. Then we will not touch provider.
		$bp_book_me = ! empty( $_POST['bp_tab'] ) && strpos( $_POST['bp_tab'], 'wpb-book-me' ) !== false;

		# Allow addons to make additional check here
		do_action( 'app_pre_confirmation_check', $value_arr );

		$there_is_daily = false;
		$lasts = $end_max = 0;
		$svs_sel_pre = $workers_sel_pre = $datetime_arr = array();

		$reply_array	= apply_filters( 'app_pre_confirmation_reply_initial', $reply_array, $value_arr );
		$value_arr		= apply_filters( 'app_pre_confirmation_modify_val', $value_arr );
		$is_multiple	= count( $value_arr ) > 1 || $has_cart ? true : false;
		$reply_array	= apply_filters( 'app_pre_confirmation_reply_pre', $reply_array, $value_arr );

		foreach ( $value_arr as $key => $val ) {

			$slot	= new WpB_Slot( $val );
			$service = $slot->get_service();

			# Assign a worker at this stage so that we can show final price to the client
			# It is possible to have worker=0 at this stage, if client_selects_worker is selected
			if ( ! ($bp_book_me && $slot->get_worker() ) ) {
				$worker = $slot->assign_worker( $is_multiple );
			}

			# Create a new "value pack" with new worker
			$new_val = $slot->pack( );

			# Check availability
			if ( ! $slot->get_app_id() && $reason = $slot->why_not_free( ) ) {

				switch ( $reason ) {
					case 1:
					case 7:		$text = $this->a->get_text('already_booked'); break;
					case 11:	$text = $this->a->get_text('too_late'); break;
					case 12:
					case 13:	$text = $this->a->get_text('past_date'); break;
					default:	$text = $this->a->get_text('not_working'); break;
				}

				$debug_text = WpBDebug::is_debug() ? ' {Debug: '. $slot->client_dt(). ' '. wpb_code2reason( $reason ) .' '.$reason.'/'. __LINE__ .'/'. $this->a->version .'}': '';
				die( json_encode( array( 'error' => $text . $debug_text )));
			}

			if ( $has_cart && ! $slot->get_app_id() ) {
				if ( ! $app_id = BASE('Multiple')->add( $new_val ) ) {
					$this->a->log( $this->a->db->last_error );
					die( json_encode( array(
						'error' => current_user_can(WPB_ADMIN_CAP)
							? sprintf(
								__('Booking cannot be saved. Last DB error: %s', 'wp-base' ),
								$this->a->db->last_error
							)
							: $this->a->get_text('error')
					) ) );
				}

				$slot->set_app_id( $app_id );
				# Create a new "value pack" with new app_id
				$new_val = $slot->pack( );
			}

			$this->items[] 			= $new_val;
			$pax					= $slot->get_pax();
			$price					= $slot->get_price();
			$sub_total				= apply_filters( 'app_confirmation_sub_total', $price * $pax, $slot );
			$this->total_price		+= $sub_total;
			$this->total_list_price	+= apply_filters( 'app_confirmation_sub_total_list', $slot->get_list_price() * $pax, $slot );
			$svs_sel_pre[] 			= apply_filters( 'app_pre_confirmation_service_name', $this->a->get_service_name( $service ), $val );
			$deposit 				= apply_filters( 'app_confirmation_deposit', $slot->get_deposit(), $new_val );

			# Client can select this: "In multi appointments, deposit is the maximum deposit and taken only once"
			if ( 'yes' === wpb_setting( 'deposit_cumulative' ) ) {
				$this->total_deposit += $deposit * $pax;
			} else {
				$this->total_deposit = max( $deposit * $pax, $this->total_deposit );
			}

			$lasts += apply_filters( 'app_pre_confirmation_duration', wpb_get_duration( $slot ), $new_val );

			// Conditions where we will not show workers selection menu
			if ( $slot->get_worker() && ( $bp_book_me || $is_multiple || 'no' != wpb_setting('client_selects_worker') ) ) {
				$workers_sel_pre[] = $this->a->get_worker_name( $slot->get_worker() );
			}

			// Check if there any daily services
			if ( $slot->is_daily() ) {
				$there_is_daily = true;
			}

			$datetime_arr[ $val ] = $slot->client_dt();

			$end_max = max( $end_max, $this->a->client_time( $slot->get_end() ) );
		}

		/* Prices */
		WpBDebug::set( 'pre_conf_filter', $this->total_price );

		$this->check_prices( );

		WpBDebug::set( 'final_price', $this->total_price );
		WpBDebug::set( 'list_price', $this->total_list_price );

		if ( wpb_is_hidden('price') ) {
			$price_html = '';
		} else {
			# If a discount is applied, highlight it
			if ( $this->total_list_price > $this->total_price ) {
				$price_html = '<label><span class="app-conf-title">' . $this->a->get_text('price') . '</span>'.
					'<span class="app-conf-text app_old_price">'. wpb_format_currency( $this->total_list_price ) . '</span>'.
					'<span title="'.WpBDebug::price_tt().'" class="app_new_price">'. wpb_format_currency( $this->total_price ) . '</span>'.
					'<div style="clear:both"></div></label>';
			} else if ( $this->total_price > 0 ) {
				$price_html = '<label><span class="app-conf-title">' . $this->a->get_text('price') . '</span>'.
					'<span title="'.WpBDebug::price_tt().'" class="app-conf-text app_current_price">'.wpb_format_currency( $this->total_price ).'</span>'.
					'<div style="clear:both"></div></label>';
			} else {
				$price_html = '';
			}
		}

		$end_max_str 	= $there_is_daily ? date_i18n( $this->a->date_format, $end_max ) : date_i18n( $this->a->dt_format, $end_max );
		$booking_fields = array(
			'new_value'	=> $this->items,
			'service'	=> wpb_is_hidden('service') ? '' : $this->a->conf_line_html( 'service_name', implode( ", ", array_unique($svs_sel_pre) ) ),
			'worker'	=> wpb_is_hidden('provider') ? '' : $this->pre_conf_worker_html( $workers_sel_pre, $slot, $is_multiple ),
			'start'		=> $this->a->conf_line_html( 'date_time', current( $datetime_arr ) ),
			'end'		=> $this->a->conf_line_html( 'end_date_time', $end_max_str ),
			'lasts'		=> $lasts ? $this->a->conf_line_html( 'lasts', wpb_format_duration( $lasts ) ) : '',
			'price'		=> $price_html,
			'disp_price'=> wpb_format_currency( $this->total_price, false, true ),
			'deposit'	=> $this->total_deposit > 0 ? $this->a->conf_line_html( 'deposit', wpb_format_currency( $this->total_deposit ) ) : '',
			'amount'	=> $this->is_pay_expected() ? $this->a->conf_line_html( 'down_payment', wpb_format_currency( $this->total_amount ) ) : '',
			'payment'	=> $this->is_pay_expected() ? 'ask':'',
		);

		die( wp_json_encode( apply_filters( 'app_pre_confirmation_reply', array_merge( $reply_array, $booking_fields ), $this->items, $this ) ) );
	}

	/**
	 * Prepare worker selection or display field for conf form
	 * @return string
	 */
	private function pre_conf_worker_html( $workers_sel_pre, $slot, $is_multiple ) {
		$worker_html = '';
		$_workers_by_service = ! $is_multiple ? $this->a->get_workers_by_service( $slot->get_service() ) : array();

		if ( ! empty( $_workers_by_service ) && 'forced_late' == wpb_setting('client_selects_worker') && $this->a->get_nof_workers() ) {
			$worker_html  = '<label for="app_select_workers_conf"><span class="app-conf-title">'. $this->a->get_text('provider_name').  '</span>';

			if ( ! empty( $_workers_by_service ) ) {
				$workers_pre = array();
				foreach ( $_workers_by_service as $_worker ) {
					$slot->set_worker( $_worker->ID );
					if ( $slot->why_not_free( 'single' ) ) {
						continue;
					} else {
						$workers_pre[ $_worker->ID ] = $this->a->get_worker_name( $_worker->ID );
					}
				}

				if ( count( $workers_pre ) > 1 ) {
					$with_page = array();

					$worker_html .= '<select %WORKERS_WITH_PAGE% data-desc="auto" data-ex_len="55" tabindex="1" id="app_select_workers_conf" class="app_select_workers_conf app_ms">';
					foreach ( $workers_pre as $wid => $wname ) {
						$worker_html .= '<option value="'.$wid.'">'. $wname . '</option>';

						$_worker = $this->a->get_worker( $wid );
						$page	 = apply_filters( 'app_worker_page_ajax', $_worker->page, $_worker, '' );
						if ( $page && ! empty( $_worker->ID ) ) {
							$with_page[] = $_worker->ID;
						}
					}
					$worker_html .= '</select>';

					# Replace "with page" placeholder
					$replace		= ! empty( $with_page ) ? 'data-with_page="'. implode( ',', $with_page ) .'"' : '';
					$worker_html	= str_replace( '%WORKERS_WITH_PAGE%', $replace, $worker_html );
				} else {
					$key = key( $workers_pre );
					$worker_html .= $key ? $workers_pre[$key] : '';
				}
			} else {
				$worker_html .= $this->a->get_text('not_working');
			}

			$worker_html .= '</label>';
		} else if ( ! empty( $workers_sel_pre ) ) {
			$worker_html  = $this->a->conf_line_html( 'provider_name', implode( ", ", array_unique( $workers_sel_pre ) ) );
		}

		return $worker_html;
	}

	/**
	 * Make checks on submitted fields and save appointment
	 * @return json object
	 */
	public function post_confirmation() {

		self::check_ajax_referer();

		if ( ! defined( 'WPB_AJAX' ) ) {
			define( 'WPB_AJAX', true );
		}

		/* For Credits and Pay later */
		do_action( 'app_post_confirmation_start' );

		ob_start();

		/* Login required check */
		if ( apply_filters( 'app_post_confirmation_login_check', ('yes' == wpb_setting('login_required') && ! is_user_logged_in() ) ) ) {
			die( json_encode( array( 'error' => $this->a->get_text('unauthorised') ) ) );
		}

		/* Update GCal or whoever modifies booking availability */
		do_action( 'app_cron' );

		/* If editing, this $_POST will arrive, but Paypal Express uses value=2: */
		$is_editing = isset( $_POST['editing'] ) && 1 == $_POST['editing'] ? 1 : 0;

		/* If we are editing we will have an app_id > 0 */
		$app_id_edit = isset( $_POST["app_id"] ) && $is_editing ? absint( wpb_clean( $_POST["app_id"] ) ) : 0;

		/* User check - See if user is a spammer, blacklisted or needs login. If everything OK create a sanitized userdata array  */
		$user_data = BASE('User')->post_confirmation_userdata( $app_id_edit );

		# Addons make additional handling here
		# Fee, Payment Gateways, Credits
		do_action( 'app_post_confirmation_handle', $app_id_edit, $user_data );

		/* If we are not editing, Double check whether value variable is emptied */
		if ( ! $app_id_edit && empty( $_POST['value'] ) ) {
			$error = WpBDebug::is_debug()
					 ? sprintf( __( 'POST["value"] is empty in %1$s line %2$d in version %3$s', 'wp-base' ), basename(__FILE__), __LINE__, $this->a->version )
					 : $this->a->get_text('error');
	
			die( json_encode( array('error' => $error ) ) );
		}

		if ( ! $this->check_spam() ) {
			die( json_encode( array( 'error' => $this->a->get_text('spam') ) ) );
		}

		/* Submitted booking data */
		if ( is_array( $_POST['value'] ) ) {
			$value_arr = array_unique( wpb_clean( $_POST['value'] ) );
			if ( apply_filters( 'app_post_confirmation_reverse_order', true, $value_arr ) ) {
				rsort( $value_arr );
			}
		} else {
			$value_arr = (array)wpb_clean( $_POST['value'] );
		}

		do_action( 'app_post_confirmation_check', $value_arr );

		$this->check_count( $value_arr );

		/* Let addons modify the value */
		$value_arr	= apply_filters( 'app_post_confirmation_modify_val', $value_arr );
		$is_multiple = count( $value_arr ) > 1;

		foreach ( $value_arr as $val ) {

			$slot = new WpB_Slot( $val );

			# Take booking to hold status: It will not count as reserved now
			BASE('Multiple')->hold( $slot->get_app_id() );

			$slot->assign_worker( $is_multiple, true );

			# Check assignment of workers
			# worker=0 is only allowed if service working hours cover it
			if( ! $slot->get_worker() && ! $slot->is_service_working( ) ) {
				# Unhold: Take back to In Cart - Nobody can book it yet until client fixes the selection, or session expires
				BASE('Multiple')->unhold( $slot->get_app_id() );
				die( json_encode( array( 'error' => $this->a->get_text('not_possible'). ' '. date_i18n( $this->a->dt_format, $slot->get_start() ) )));
			}

			/* Create a new "value pack" with new worker */
			$new_val 			= $slot->pack( );
			$this->items[] 		= $new_val;
			$pax 				= $slot->get_pax();
			$price 				= $slot->get_price();
			$sub_total			= apply_filters( 'app_confirmation_sub_total', $price * $pax, $slot );
			$this->total_price += $sub_total;
			$deposit 			= apply_filters( 'app_confirmation_deposit', $slot->get_deposit(), $new_val );

			/* Save each slot price */
			$this->slot_prices[ $new_val ] = $sub_total;

			/* In multi appointments, deposit calculus can be selected */
			if ( 'yes' == wpb_setting( 'deposit_cumulative' ) ) {
				$this->total_deposit += $deposit *$pax;
			} else {
				$this->total_deposit = max( $deposit, $this->total_deposit );
			}
		}

		/* Check pricing */
		$this->check_prices( );
		$sel_pay_method = $this->check_pay_method( );

		/* Do total price check for addons modifying prices */
		do_action( 'app_post_confirmation_final_price_check', $this->items, $this );

		/* Check availability */
		$this->check_avail( );

		# If price is zero or payment is not required, follow auto_confirm
		$pay_expected 	= $this->is_pay_expected();
		$status 		= ! $pay_expected && 'yes' === wpb_setting("auto_confirm") ? 'confirmed' : 'pending';
		$status 		= apply_filters( 'app_post_confirmation_status', $status, $this->items, $user_data, $this );

		/* Save */
		$parent_id	= $this->do_save( $user_data, $sel_pay_method, $status );
		$post_id	= ! empty( $_POST['post_id'] ) ? absint( wpb_clean( $_POST['post_id'] ) ) : 0;

		wpb_set_session_val( 'app_order_id', $parent_id );

		if ( $fee = wpb_get_session_val( 'fee' ) ) {
			wpb_update_app_meta( $parent_id, 'fee', $fee );
		}

		if ( $post_id ) {
			wpb_set_session_val( 'app_post_id', $post_id );
		}

		# Trigger send message for pending, payment not required cases */
		# If EDD or WC is the selected method, skip this
		if ( 'woocommerce' == $sel_pay_method || 'edd' == $sel_pay_method ) {
			// Leave to EDD or WC
		} else if ( ! $pay_expected && 'pending' == $status ) {
			$this->a->send_notification( $parent_id );
			$this->a->maybe_send_message( $parent_id, 'pending' );
		} else if ( $pay_expected && 'manual-payments' == $sel_pay_method && 'pending' == $status ) {
			$this->a->maybe_send_message( $parent_id, 'manual-payments' );
		} else if ( $pay_expected && 'pay-later' == $sel_pay_method && 'pending' == $status ) {
			$this->a->maybe_send_message( $parent_id, 'pay-later' );
		} else if ( 'confirmed' == $status ) {
			$this->a->maybe_send_message( $parent_id, 'confirmation' );
		}

		do_action( 'app_post_confirmation_after_email', $this );

		/* Confirm/Pending text && title */
		$app = wpb_get_app( $parent_id );
		if ( 'confirmed' == $status || 'paid' == $status ) {
			$popup_text = wpb_dialog_text( $app ) ?: $this->a->get_text( 'appointment_received' );
			$popup_title = wpb_dialog_title( $app );
		} else {
			$popup_text = wpb_dialog_text( $app, 'pending' ) ?: $this->a->get_text( 'appointment_received' );
			$popup_title = wpb_dialog_title( $app, 'pending' );
		}

		# There is no form for manual payments, so it is skipped
		# Also MP and WC may allow zero priced services
		if ( ( $pay_expected && 'manual-payments' != $sel_pay_method && 'pay-later' != $sel_pay_method ) ||
			in_array( $sel_pay_method, array( 'woocommerce', 'edd' ) ) ) {

			wpb_set_session_val( 'app_total_amount', $this->total_amount );

			wpb_delete_app_meta( $parent_id, 'no_commission' );

			$form = '';
			foreach ( wpb_active_gateways() as $plugin ) {
				// Insert only form of the selected gateway
				if ( $sel_pay_method == $plugin->plugin_name ) {
					$form = $plugin->_payment_form_wrapper( $parent_id, wpb_get_session_val('app_post_id'), $this->total_amount );
					break;
				}
			}

			# Connect to gateway
			do_action( 'app_payment_submit_' . $sel_pay_method, $parent_id, $this->total_amount );

			if ( $maybe_error = wpb_get_cart_error() ) {
				die( json_encode( array( 'error' => strip_tags( $maybe_error ) ) ) );
			}

			die( json_encode(
				apply_filters( 'app_post_confirmation_reply', array(
						'cell'				=> $this->items,
						'app_id'			=> $parent_id,
						'is_editing'		=> $is_editing,
						'refresh'			=> 0,
						'price'				=> $this->total_amount,
						'f_amount'			=> str_replace( 'AMOUNT', wpb_format_currency( $this->total_amount ), $this->a->get_text('pay_now') ),
						'method'			=> $sel_pay_method,
						'form'				=> $form,
						'confirm_text'		=> $popup_text,
						'confirm_title'		=> $popup_title,
						'status'			=> $status,
						), 'payment_expected', $this->items, $this
					)
				)
			);
		} else {
			BASE('Multiple')->empty_cart();

			die( json_encode(
				# refresh_url key can be used for custom redirect urls (e.g. depending on service)
				apply_filters( 'app_post_confirmation_reply', array(
						'cell'				=> $this->items,
						'app_id'			=> $parent_id,
						'is_editing'		=> $is_editing,
						'refresh'			=> 1,
						'method'			=> '',
						'confirm_text'		=> $popup_text,
						'confirm_title'		=> $popup_title,
						'status'			=> $status,
						), 'payment_not_expected', $this->items, $this
					)
				)
			);
		}
	}

	/**
	 * Check number of submitted time slots in post conf
	 * @since 3.0
	 * @return none
	 */
	private function check_count( $value_arr ) {
		$count = count( $value_arr );

		/* Check appt lower limit. In this case if it is zero */
		if ( ! $count ) {
			die( json_encode( array( 'error' => sprintf(
					$this->a->get_text('too_less'),
					BASE('Multiple')->get_apt_count_min( $value_arr ) )
			) ) );
		}

		/* Make max count check */
		if ( $count > 1 ) {
			# If multiple appts is not active picking more than 1 slot is not allowed
			if ( ! BASE('Multiple')->is_active() ) {
				$error = WpBDebug::is_debug()
						 ? sprintf( __('More than one time slot is submitted in %1$s line %2$d in version %3$s','wp-base'),
							basename(__FILE__),
							__LINE__,
							$this->a->version )
						 : $this->a->get_text('error');

				die( json_encode( array( 'error' => $error ) ) );
			}

			do_action( 'app_post_check_multiple', $value_arr );
		}
	}

	/**
	 * Check prices and disallow negative numbers
	 * @since 3.0
	 * @return none
	 */
	private function check_prices( ) {
		$this->total_price 		= apply_filters( 'app_confirmation_total_price', $this->total_price, $this->items, $this );
		$this->total_amount 	= apply_filters( 'app_confirmation_total_amount', $this->a->calc_downpayment($this->total_price), $this->items, $this );
		$this->total_amount 	= $this->total_amount > $this->total_price ?  $this->total_price : $this->total_amount;

		if ( 'yes' == wpb_setting( 'add_deposit' ) ) {
			$this->total_amount = $this->total_amount + $this->total_deposit;
		}

		# Do not allow negative amounts
		$this->total_price		= max( 0, $this->total_price );
		$this->total_amount		= max( 0, $this->total_amount );
		$this->total_deposit	= max( 0, $this->total_deposit );
	}

	/**
	 * Check if a payment is expected
	 * @since 3.0
	 * @return bool
	 */
	private function is_pay_expected() {
		return 'yes' === wpb_setting( 'payment_required' ) && $this->total_amount;
	}

	/**
	 * Check if payment method selected
	 * @since 3.0
	 * @return string
	 */
	private function check_pay_method( ) {

		$sel_pay_method = '';
		if ( 'yes' == wpb_setting( 'payment_required' ) ) {
			if ( ! empty( $_POST["app_payment_method"] ) ) {
				$sel_pay_method = wpb_clean( $_POST["app_payment_method"] );
			} else {
				$active = wpb_active_gateways();
				if ( 1 === count( $active ) ) {
					$sel_pay_method = $active[0]->plugin_name; 	# Auto select when there is a single gateway active
				} else if ( wpb_is_gateway_active( 'credits' ) && 'yes' == wpb_gateway_setting( 'credits', 'is_unique' ) ) {
					$sel_pay_method = 'credits';
				}
			}
		}

		$sel_pay_method = apply_filters( 'app_selected_payment_method', $sel_pay_method, $this->items, $this );

		if ( $this->is_pay_expected() && !$sel_pay_method ) {
			foreach ( $this->items as $item ) {
				$slot = new WpB_Slot( $item );
				BASE('Multiple')->unhold( $slot->get_app_id() );
			}

			die( json_encode( array( 'error' => $this->a->get_text('payment_method_error') ) ) );
		}

		return $this->payment_method = $sel_pay_method;
	}

	/**
	 * Check final availability after worker assigned
	 * @since 3.0
	 * @return none
	 */
	private function check_avail( ) {
		foreach ( $this->items as $val ) {

			$slot = new WpB_Slot( $val );

			if ( $reason = $slot->why_not_free( ) ) {
				switch ( $reason ) {
					case 1:
					case 7:		$text = $this->a->get_text('already_booked'); break;
					case 11:	$text = $this->a->get_text('too_late'); break;
					case 12:
					case 13:	$text = $this->a->get_text('past_date'); break;
					default:	$text = $this->a->get_text('not_working'); break;
				}
				BASE('Multiple')->unhold( $slot->get_app_id() );

				$debug_text = WpBDebug::is_debug() ? ' '. date_i18n( $this->a->dt_format, $slot->get_start() ). ' Code:'. $reason .' '. $val : '';
				die( json_encode( array( 'error' => $text . $debug_text )));
			}
		}
	}

	/**
	 * Do the actual saving to DB
	 * @since 3.0
	 * @return integer	Parent ID
	 */
	private function do_save( $user_data, $sel_pay_method, $status ) {
		$user_id 		= key( $user_data );
		$sub_user_data 	= current( $user_data );
		$parent_id 		= 0; // If possible, the first appointment (which is starting last) will be parent
		$suggested_id 	= (int)$this->a->db->get_var( "SELECT MAX(ID) FROM " . $this->a->app_table ) + count( $this->items );
		$child_prices	= array();

		/* Checks are ok. Do the last arrangements and save to database here */
		foreach ( $this->items as $val ) {

			$slot			= new WpB_Slot( $val );
			$category		= $slot->get_category();
			$service		= $slot->get_service();
			$worker			= $slot->get_worker();
			$app_id			= $slot->get_app_id();
			$start			= $slot->get_start();
			$end			= $slot->get_end();
			$old_booking	= new WpB_Booking( $app_id );

			$data = apply_filters( 'app_post_confirmation_save_data', array(
					'ID'			=> 	$app_id ? $app_id : $suggested_id,
					'parent_id'		=>	$parent_id,
					'created'		=>	date("Y-m-d H:i:s", $this->a->_time),
					'user'			=>	$user_id,
					'location'		=>	$slot->get_location(),
					'service'		=>	$service,
					'worker'		=> 	$worker,
					'price'			=>	!$parent_id ? $this->total_price : 0,	// In multi, price and deposit are written to parent
					'deposit'		=>	!$parent_id ? $this->total_deposit : 0,
					'status'		=>	apply_filters( 'app_post_confirmation_item_status', $status, $slot, $this ),
					'start'			=>	$slot->is_daily() ? date("Y-m-d " . "00:00:00", $start) : date("Y-m-d H:i:s", $start) ,
					'end'			=>	$slot->is_daily() ? date("Y-m-d " . "00:00:00", $end) : date("Y-m-d H:i:s", $end ),
					'seats'			=>	$slot->get_pax(),
					'payment_method'=>	$sel_pay_method,
				), $old_booking, $val );

			if ( $app_id ) {
				$result = $this->a->db->update( $this->a->app_table, $data, array( 'ID' => $app_id ) );
			} else {

				if ( !$result = $this->a->db->insert( $this->a->app_table, $data ) ) {
					// We might have a race condition. Forget parent having highest ID.
					$data['ID'] = 'null';
					$result = $this->a->db->insert( $this->a->app_table, $data );
				}
			}

			if ( ! $result ) {
				$this->a->log( $this->a->db->last_error );
				BASE('Multiple')->unhold( $app_id );
				die( json_encode( array(
					'error' => 	WpBDebug::is_debug()
								? sprintf( __('Booking cannot be saved. Last DB error: %s', 'wp-base' ), $this->a->db->last_error )
								: $this->a->get_text('save_error')
				)));
			}

			# Save Booking ID
			$ID = $app_id ?: $this->a->db->insert_id;
			
			if ( ! $ID ) {
				BASE('Multiple')->unhold( $app_id );
				die( json_encode( array(
					'error' => 	WpBDebug::is_debug()
								? sprintf( __('Booking ID was not received from database. Check database table: %s', 'wp-base' ), $this->a->app_table )
								: $this->a->get_text('save_error')
				)));
			}
			
			$suggested_id = $suggested_id - 1;

			if ( $category ) {
				wpb_update_app_meta( $ID, 'category', $category );
			}

			if ( ! empty( $_SERVER['REMOTE_ADDR'] ) ) {
				wpb_update_app_meta( $ID, 'client_ip', $_SERVER['REMOTE_ADDR'] );
			}

			# A new appointment is accepted, so clear cache
			wpb_flush_cache();

			if ( ! $parent_id ) {
				$parent_id = $ID;
			}

			$child_prices[ $ID ] = ! empty( $this->slot_prices[ $val ] ) ? $this->slot_prices[ $val ] : 0;

			/**
			 * Fires after a new booking has been saved to the database
			 * Use your hook with minimum priority 100 so that addons can complete their jobs
			 * @param $ID			integer		ID of the new booking
			 * @param $sub_udata	array		Submitted, sanitized user data, e.g. array( 'name' => 'Hakan Ozevin', 'email' => 'hakan@wp-base.com' )
			 * @param $parent_id	integer		ID of booking's parent. If zero or equal to $ID, booking is parent itself
			 */
			do_action( 'app_new_appointment', $ID, $sub_user_data, $parent_id );

		} /* End foreach */

		# Save last booked worker
		if ( $user_id && $worker && 'yes' === wpb_setting('preselect_latest_worker') ) {
			update_user_meta( $user_id, 'app_last_booked_worker', $worker );
		}

		if ( ! empty( $child_prices ) ) {
			wpb_add_app_meta( $parent_id, 'child_prices', $child_prices );
		}

		return $parent_id;
	}

	/**
	 * Update conf form user fields with data from selected user
	 * since 2.0
	 */
	public function update_user_fields(  ) {
		self::check_ajax_referer();

		$user_id = isset( $_POST['app_user_id'] ) ? absint( wpb_clean( $_POST['app_user_id'] ) ) : 0;

		$r = $user_id ? BASE('User')->get_app_userdata( 0, $user_id ) : array();
		$r = apply_filters( 'app_inline_edit_populate_user', $r, $user_id );

		if ( isset( $r['error'] ) || !is_array( $r ) ) {
			wp_send_json( array( 'error' => __( 'Unknown or empty user ID', 'wp-base' ) ) );
		} else {
			wp_send_json( $r );
		}
	}

	/**
	 * Dynamically show payment in qtip content
	 * @since 2.0
	 */
	public function show_children_in_tooltip() {
		if ( empty( $_POST['app_id'] ) ) {
			wp_send_json( array( 'result' => __('Unexpected error','wp-base') ) );
		}

		$args = isset( $_POST['args'] ) ? json_decode( wpb_clean( $_POST['args'] ) ) : array();
		$args['_children_of'] = absint( wpb_clean( $_POST['app_id'] ) );
		$args['title'] = __('Connected bookings','wp-base');

		wp_send_json( array( 'result' => preg_replace( '%<thead(.*?)</thead>%', '', BASE('Listing')->display( $args ) ) ) );
	}

	/**
	 * Generate an excerpt from the selected service/worker page
	 * @since 1.0
	 * @return string
	 */
	public function get_excerpt( $page_id, $id = 0 ) {
		$context	= ! empty( $_POST['lsw'] ) ? wpb_clean( $_POST['lsw'] ) : 'service';
		$page		= get_post( $page_id );
		$text		= ! empty( $page->post_content ) ? $page->post_content : '';
		$length		= apply_filters('app_excerpt_length', (! empty( $_POST['ex_len'] ) ? absint( wpb_clean( $_POST['ex_len'] ) ) : 55) );
		$text		= wp_trim_words( wpb_strip_shortcodes( $text ), $length );
		$text		= apply_filters( 'app_description_text', $text, $page_id, $id, $context, 'excerpt' );
		$text		= str_replace( ']]>', ']]&gt;', html_entity_decode( $text ) );
		$thumb		= $this->get_thumbnail( $page_id, $id );

		return apply_filters( 'app_excerpt', $thumb. $text, $page_id, $id );
	}

	/**
	 * Get the post excerpt for the selected service/worker page
	 * @since 2.0
	 * @return string
	 */
	public function get_post_excerpt( $page_id, $id = 0 ) {
		$page		= get_post( $page_id );
		$context 	= ! empty( $_POST['lsw'] ) ? wpb_clean( $_POST['lsw'] ) : 'service';
		$text		= empty( $page->post_excerpt ) ? '' : $page->post_excerpt;
		$text		= wpb_strip_shortcodes( $text );
		$text		= apply_filters( 'app_description_text', $text, $page_id, $id, $context, 'post_excerpt' );
		$thumb		= $this->get_thumbnail( $page_id, $id );

		return apply_filters( 'app_post_excerpt', $thumb. html_entity_decode( $text ), $page_id, $id );
	}

	/**
	 * Fetch content from the selected service/worker page
	 * @since 1.0
	 * @return string
	 */
	public function get_content( $page_id, $id = 0 ) {
		$page		= get_post( $page_id );
		$context 	= ! empty( $_POST['lsw'] ) ? wpb_clean( $_POST['lsw'] ) : 'service';
		$text		= empty( $page->post_content ) ? '' : $page->post_content;
		$text		= wpb_strip_shortcodes( $text );
		$text		= apply_filters( 'app_description_text', $text, $page_id, $id, $context, 'content' );
		$text		= apply_filters( 'app_pre_content', wpautop( wptexturize ( $text ) ), $page_id, $id  );
		$thumb		= $this->get_thumbnail( $page_id, $id );

		return apply_filters( 'app_content', $thumb. html_entity_decode( $text ), $page_id, $id );
	}

	/**
	 * Get html code for thumbnail or avatar
	 * @since 1.0
	 * @return string
	 */
	public function get_thumbnail( $page_id, $id ) {

		 $context		= ! empty( $_POST['lsw'] ) ? wpb_clean( $_POST['lsw'] ) : 'service';
		 $thumb_size	= apply_filters( 'app_thumbnail_size', '96,96', $context, $id );
		 $thumb_class	= apply_filters( 'app_thumbnail_class', (is_rtl() ? 'alignright' : 'alignleft'), $context, $id );

		if ( $thumb_size && 'none' !== $thumb_size ) {
			if ( strpos( $thumb_size, 'avatar' ) !== false ) {
				if ( strpos( $thumb_size, ',' ) !== false ) {
					$size_arr = explode( ",", $thumb_size );
					$size = $size_arr[1];
				} else {
					$size = 96;
				}

				$thumb = get_avatar( $id, $size );
				if ( $thumb_class ) {
					$thumb = str_replace( "class='", "class='".$thumb_class." ", $thumb );
					$thumb = str_replace( 'class="', 'class="'.$thumb_class.' ', $thumb );
				}
				$thumb = str_replace( "'",'"', $thumb );
			} else {
				if ( strpos( $thumb_size, ',' ) !== false )
					$size = explode( ",", $thumb_size );
				else
					$size = $thumb_size;

				$thumb = get_the_post_thumbnail( $page_id, $size, apply_filters( 'app_thumbnail_attr', array('class'=>$thumb_class) ) );
			}
		} else {
			$thumb = '';
		}

		return apply_filters( 'app_thumbnail', $thumb, $page_id, $id );
	}

	/**
	 * Combination of post_excerpt and excerpt
	 *
	 * @return string
	 */
	public function get_auto_excerpt( $page_id, $id = 0 ) {
		 $page = get_post( $page_id );
		if ( ! empty( $page->post_excerpt ) ) {
			return $this->get_post_excerpt( $page_id, $id );
		} else {
			return $this->get_excerpt( $page_id, $id );
		}
	}

	/**
	 * Prepare HTML for location/service/worker descriptions to be displayed in tooltip
	 *
	 * @return string
	 */
	 function lsw_tooltip(){
		 $id			= ! empty( $_POST['id'] ) ? wpb_clean( $_POST['id'] ) : 0;
		 $sub_context	= ! empty( $_POST['desc'] ) ? wpb_clean( $_POST['desc'] ) : 'auto';
		 $context		= ! empty( $_POST['lsw'] ) ? wpb_clean( $_POST['lsw'] ) : 'service';
		 $page_id		= 0;

		 switch ( $context ) {
			 case 'location':	$var = $this->a->get_location( $id );
								$page_id = ! empty( $var->page ) ? $var->page : 0;
								break;
			 case 'service':	$var = $this->a->get_service( $id );
								$page_id = ! empty( $var->page ) ? $var->page : 0;
								break;
			 case 'worker':		$var = $this->a->get_worker( $id );
								$page_id = ! empty( $var->page ) ? $var->page : 0;
								break;
			 case 'extra':		$page_id = BASE('Extras') ? BASE('Extras')->get_desc_page_id( $id ) : 0;
								break;
		 }

		 if ( ! $page_id && 'service' == $context ) {

			$thumb = $text = '';

			if ( $img_id = wpb_get_service_meta( $id, 'image_id' ) ) {
				$thumb_class = apply_filters( 'app_thumbnail_class', 'alignleft', $context, $id );
				$thumb_size	= apply_filters( 'app_thumbnail_size', 'thumbnail', $context, $id );
				$thumb = wp_get_attachment_image( $img_id, $thumb_size, false, array( 'class' => $thumb_class, 'alt' => BASE()->get_service_name( $id ) ) );
			}

			if ( $maybe_desc = wpb_get_service_meta( $id, 'description' ) ) {
				$text = $maybe_desc;
				$sub_context = 'description';
			}

			$html =	$thumb . html_entity_decode( $text );

		 } else {

			$sub_context = apply_filters( 'app_thumbnail_desc', $sub_context, $context, $id );

			switch ( $sub_context ) {
				case 'none':			break;
				case 'excerpt':			$html = $this->get_excerpt( $page_id, $id ); break;
				case 'post_excerpt':	$html = $this->get_post_excerpt( $page_id, $id ); break;
				case 'auto':			$html = $this->get_auto_excerpt( $page_id, $id ); break;
				case 'content':			$html = $this->get_content( $page_id, $id ); break;
				default:				$html = $this->get_excerpt( $page_id, $id ); break;
			}
		 }

		wp_send_json( array( 'result' => apply_filters( 'app_lsw_tooltip_html', $html, $context, $sub_context ) ) );
	}

	/**
	 * Check for too frequent back to back apps
	 * return true means no spam
	 * @return bool
	 */
	public function check_spam() {

		if ( ! wpb_setting( 'spam_time' ) || !$ids = $this->get_apps_from_cookie() ) {
			return true;
		}

		if ( ! is_array( $ids ) || empty( $ids ) ) {
			return true;
		}

		$ids_str 	= implode("','", array_map( 'esc_sql', $ids ) );
		$checkdate 	= date( 'Y-m-d H:i:s', $this->_time - wpb_setting("spam_time") );
		$count 		= $this->a->db->get_var( $this->a->db->prepare( "SELECT COUNT(ID) FROM " . $this->app_table .
						" WHERE created>%s AND status='pending' AND ID IN ('". $ids_str ."') ", $checkdate ) );

		if ( $count ) {
			return false;
		}

		return true;
	}

	/**
	 * Check ajax referer
	 * @since 3.7.8
	 */
	public static function check_ajax_referer(){
		if ( wpb_extra_security() && ! check_ajax_referer( 'front', 'ajax_nonce', false ) ) {
			die( json_encode( array('error' => BASE()->get_text('unauthorised') ) ) );
		}
	}

}
	BASE('Ajax')->add_hooks();
}