<?php
/**
 * WPB Debug
 *
 * Tools for debugging for admin
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( !class_exists( 'WpBDebug' ) ) {

class WpBDebug {

    public static $app_debug_global = array();

    public static function set($key,$value){
        self::$app_debug_global[$key] = $value;
    }

    public static function get($key){
        return isset( self::$app_debug_global[$key] ) ? self::$app_debug_global[$key] : null;
    }

	public static function is_debug() {
		if ( 'yes' == wpb_setting('debug_mode') && BASE('User')->_current_user_can( WPB_ADMIN_CAP ) )
			return true;

		return false;
	}

	/**
	 * Create tooltip text for time slots
	 * ● is black circle character in https://en.wikipedia.org/wiki/Interpunct
	 * @return string
	 */
	public static function time_slot_tt($slot) {
		
		if ( ! self::is_debug() ) {
			return;
		}
		
		$tt	= array();
		$sep = WPB_TT_SEPARATOR;

		if ( BASE('Timezones') ) {
			$tt[] = $sep. 'TZ &rarr; ' . BASE('Timezones')->get_user_pref() . $sep;
			if ( 'yes' == wpb_setting( 'enable_timezones' ) )
				$tt[] = __('Server time','wp-base'). ' &rarr; '. date_i18n(BASE()->dt_format, $slot->get_start() + BASE()->get_padding($slot->get_service())*60 ) ;
			else
				$tt[] = __('Time zone handling disabled','wp-base') ;
		}

		if ( null !== self::get('time_var_rule') ) {
			$tt[] = $sep. 'Applied Time Variant Durations rule: ' . self::get('time_var_rule');
			self::set('time_var_rule', null);
		}

		if ( $tt ) {
			return 'WP BASE debug:' . $sep. implode( $sep, $tt ) ;
		}
	}

	/**
	 * Create tooltip text for new price slot (Discounted price)
	 * @return string
	 */
	public static function price_tt() {

		if ( ! self::is_debug() ) {
			return;
		}
		
		$tt = array();
		$sep = WPB_TT_SEPARATOR;

		if ( null !== self::get('package_price') ) {
			$tt[] = 'P. Base price: ' . wpb_format_currency( self::get('package_price') );
		} else if ( null !== self::get('base_price') ) {
			$tt[] = 'Last item price: ' . wpb_format_currency( self::get('base_price') );
		}

		if ( null !== self::get('var_dur_pre') ) {
			$tt[] = 'Item price before Time Variant Durations: ' . wpb_format_currency( self::get('var_dur_pre') );
			self::set( 'var_dur_pre', null );
		}

		if ( null !== self::get('var_dur') ) {
			$tt[] = 'Item price after Time Variant Durations: ' . wpb_format_currency( self::get('var_dur') );
			self::set( 'var_dur', null );
		}

		if ( null !== self::get('sel_dur_pre') ) {
			$tt[] = 'Item price before Selectable Durations: ' . wpb_format_currency( self::get('sel_dur_pre') );
			self::set( 'sel_dur_pre', null );
		}

		if ( null !== self::get('sel_dur') ) {
			$tt[] = 'Item price after Selectable Durations: ' . wpb_format_currency( self::get('sel_dur') );
			self::set( 'sel_dur', null );
		}

		if ( null !== self::get('ep_pre') ) {
			$tt[] = 'Item price before Custom Pricing: ' . wpb_format_currency( self::get('ep_pre') );
		}

		if ( null !== self::get('ep') ) {
			$tt[] = 'Item price after Custom Pricing: ' . wpb_format_currency( self::get('ep') );
		}

		if ( 'yes' != BASE()->get_options('apply_coupon_to_extras') ) {
			if ( null !== self::get('coupon') ) {
				$coupon = self::get('coupon');
				$val = current( $coupon );
				$tt[] = 'Discount by coupon #' . key( $coupon ) . ' ['. $val['code'].']: -'. wpb_format_currency( $val['discount'] );
				self::set( 'coupon', null );
			}

			if ( null !==  self::get('after_coupons') ) {
				$tt[] = 'After Coupons: '. wpb_format_currency( self::get('after_coupons' ) );
				self::set( 'after_coupons', null );
			}
		}

		if ( null !== self::get('extras') ) {
			if ( null !== self::get('pre_conf_filter') ) {
				$tt[] = 'Total price before Extras: ' . wpb_format_currency( self::get('pre_conf_filter') );
			}

			$extras = self::get('extras');
			foreach ( $extras as $id => $e ) {
				$tt[] = $sep .esc_attr( wp_unslash( $e['name'] ) ). ': '. wpb_format_currency( $e['price'] );
			}
		}

		if ( null !== self::get('after_extras') ) {
			$tt[] = 'Price with extras: ' . wpb_format_currency( self::get('after_extras') );
		}

		if ( 'yes' == BASE()->get_options('apply_coupon_to_extras') ) {
			if ( null !== self::get('coupon') ) {
				$coupon = self::get('coupon');
				$val = current( $coupon );
				$tt[] = 'Discount by coupon #' . key( $coupon ) . ' ['. $val['code'].']: -'. wpb_format_currency( $val['discount'] );
				self::set( 'coupon', null );
			}
			if ( null !== self::get('after_coupons') ) {
				$tt[] = 'After Coupons: '. wpb_format_currency( self::get('after_coupons' ) );
				self::set( 'after_coupons', null );
			}
		}

		if ( null !== self::get('after_seats') ) {
			$nof_seats = self::get( 'nof_seats' ) ? self::get( 'nof_seats' ) : 1;
			$tt[] = 'After Group Bookings ('.$nof_seats.'): ' . wpb_format_currency( self::get('after_seats') );
			self::set( 'after_seats', null );
			self::set( 'nof_seats', null );
		}

		if ( null !== self::get('after_recurring') ) {
			$repeat = self::get( 'repeat' ) ? self::get( 'repeat' ) : 1;
			$tt[] = 'After Recurring ('.$repeat.'): ' . wpb_format_currency( self::get('after_recurring') );
			self::set( 'after_recurring', null );
			self::set( 'repeat', null );
		}

		if ( null !== self::get('fee') ) {
			$tt[] = 'Fee: ' . wpb_format_currency( self::get('fee') );
			self::set( 'fee', null );
		}

		if ( null !== self::get('final_price') ) {
			$tt[] = '-------------------';
			$tt[] = 'Final price: ' . wpb_format_currency( self::get('final_price') );
			self::set( 'final_price', null );
		}

		if ( null !== self::get('extra_not_available') ) {
			$tt[] = $sep . 'Service as extra is not available because: ' . self::get('extra_not_available');
			self::set( 'extra_not_available', null );
		}

		if ( $tt ) {
			return '== WP BASE debug ==' . $sep . $sep . implode( $sep, $tt );
		}
	}

	/**
	 * Return a debug text for admin
	 * @param $title:	string	Set a custom title
	 */
	public static function debug_text( $text, $title = false ) {
		if ( self::is_debug() ) {
			$title = $title ?: esc_html(__( 'Debug texts are only visible to admins. You can turn them off on Global Settings > Advanced', 'wp-base' ) );
			return '<abbr class="app-debug" data-titlebar="true" data-title="WpB Debug" title="' . $title . '" >'
					. esc_html( __( 'WpB Debug:', 'wp-base' ) ) . '</abbr> '. $text;
		}

		return '';
	}

	/**
	 * Return a debug text for admin to be used in tooltip, explaining why a slot is not available
	 * @param $code		array|integer	Reason code coming from class WpB_Slot
	 * @return string
	 */
	public static function display_reason( $code ) {
		if ( ! self::is_debug() ){
			return '';
		}
		
		$code = is_array( $code ) ? max( $code ) : $code;

		switch ( $code ) {
			case 1:		$reason = __( 'Providers on this time slot are busy', 'wp-base' ); break;
			case 2:		$reason = __( 'There are no providers for this time slot', 'wp-base' ); break;
			case 3:		$reason = __( 'Not enough time to accept an appointment here: Either there is another appointment or break in the proceeding slots', 'wp-base' ); break;
			case 4:
			case 5:		$reason = __( 'There is a break on this time slot', 'wp-base' ); break;
			case 6:		$reason = __( 'Proceeding days are not available', 'wp-base' ); break;
			case 7:		$reason = __( 'Location capacity full', 'wp-base' ); break;
			case 8:		$reason = __( 'Service is marked as not working for this time slot', 'wp-base' ); break;
			case 9:		$reason = __( 'Selected provider or service not working at this time', 'wp-base' ); break;
			case 10:	$reason = __( 'Time slot is holiday for the selected provider', 'wp-base' ); break;
			case 11:	$reason = __( 'Time slot is blocked due to Lead Time setting', 'wp-base' ); break;
			case 12:	$reason = __( 'Time slot is of a past time for today', 'wp-base' ); break;
			case 13:	$reason = __( 'Time slot is of a past date', 'wp-base' ); break;
			case 14:	$reason = __( 'Time slot exceeds Booking Submission Upper Limit setting', 'wp-base' ); break;
			case 15:	$reason = __( 'Service not published', 'wp-base' ); break;
			case 16:	$reason = __( 'Not available', 'wp-base' ); break;
			case 17:	$reason = __( 'Worker is not available possibly due to holiday or break', 'wp-base' ); break;
			case 18:	$reason = __( 'Slot is unavailable due to custom function', 'wp-base' ); break;
			case 19:	$reason = __( 'Service is not available possibly due to break', 'wp-base' ); break;
			default:	$reason = __( 'Not possible to book', 'wp-base' ); break;
		}

		return __('WpB Debug:','wp-base' ). ' '. $reason;
	}

	/**
	 * Log a message
	 * @param $message	mixed	Message or variable value to be logged
	 * @return none
	 */
	public static function log( $message ) {
		if ( ! self::is_debug() ){
			return;
		}

		BASE()->log( $message );
	}

}
}