<?php
/**
 * WPB Custom Texts
 *
 * Allows localization of front end texts without a translation tool
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBCustomTexts' ) ) {

class WpBCustomTexts {

	/**
     * Default texts
     */
	public $default_texts = null;

	/**
     * Default additional texts
     */
	public $add_default_texts = array();

	/**
     * Replace rules cache
     */
	private $replace_rules = null;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}

	/**
     * Add actions and filters
     */
	public function add_hooks() {
		add_filter( 'appointments_display_tabs', array( $this, 'add_tab' ), 16 ); 				// Add tab
		add_action( 'app_display_custom_texts_tab', array( $this, 'settings' ), 16 );			// Display HTML settings
		add_action( 'app_save_settings', array( $this, 'save_settings' ), 16 );					// Save settings

		add_filter( 'gettext', array( $this, 'global_text_replace' ), 10, 3 );
	}

	/**
     * Global quick text replace
	 * @param $tr_text: translated text
	 * @param $text: original text
	 * @param $domain: Text domain (wp-base is searched for)
     */
	public function global_text_replace( $tr_text, $text, $domain ) {
		if ( 'wp-base' != $domain || ! wpb_is_admin() || ! $rep_raw = apply_filters( 'app_replace_texts', get_option( 'wp_base_replace_texts' ), $domain ) ) {
			return $tr_text;
		}

		if ( null === $this->replace_rules ) {
			$rules = array();
			$reps = array_filter( array_map( 'trim', explode( ',', str_replace( "\n", ",", $rep_raw ) ) ) );

			foreach ( $reps as $rep ) {
				if ( strpos( $rep, '|' ) === false ) {
					continue;
				}

				list( $find, $replace ) = explode( '|', $rep );

				if ( $find ) {
					$rules[ $find ] = $replace;
					$rules[ ucwords( $find ) ] = ucwords( $replace );
				}

				if ( strpos( $find, ' ' ) ) {
					$rules[ ucwords( $find ) ] = ucwords( $replace );
				}
			}

			$this->replace_rules = array_filter( $rules );
		}

		if ( ! empty( $this->replace_rules ) ) {
			$tr_text = str_replace( array_keys( $this->replace_rules ), array_values( $this->replace_rules ), $tr_text );
		}

		return $tr_text;
	}

	/**
     * Get a custom text if $key is set or all custom texts if $key is not set
	 * To be used in Custom Texts page
	 * @return string or array
     */
	public function get_custom_texts( $key=null ) {
		# Simple cache
		if ( isset( $this->custom_texts ) ) {
			$custom_texts = $this->custom_texts;
		} else {
			$custom_texts = $this->custom_texts = get_option( 'wp_base_texts' );
		}

		if ( $key ) {
			if ( isset( $custom_texts[ $key ] ) ) {
				return $custom_texts[ $key ];
			} else {
				return '';
			}
		} else {
			return $custom_texts;
		}
	}

	/**
     * Get a help text if $key is set or all help texts if $key is not set
	 * @return string or array
     */
	public function get_help_text( $key=null ) {
		if ( ! isset( $this->help_texts ) ) :
			$this->help_texts = array(
					'action'								=> __('Column header of Action for Bookings shortcode', 'wp-base' ),
					'address'								=> __('Title for Address field in the confirmation form and user page','wp-base'),
					'address_placeholder'					=> __('Placeholder for Address field in the confirmation form','wp-base'),
					'add_to_cart'							=> __('Button text to add a product to shopping cart of WooCommerce or EDD', 'wp-base'),
					'added_to_cart'							=> __('Message displayed when an item is added to WooCommerce or EDD shopping cart. %s will be replaced with item name.', 'wp-base' ),
					'admin'									=> __('Label for admin user', 'wp-base'),
					'all'									=> __('Selection in provider dropdown when no particular provider is selected', 'wp-base'),
					'already_booked'						=> __('Javascript message displayed when selected time slot is no more available, e.g. because of another simultaneous booking', 'wp-base'),
					'amount'								=> __('Title for amount', 'wp-base'),
					'aop_placeholder'						=> __('Placeholder for participant address field. %d will be replaced by the order in the list.', 'wp-base' ),
					'appointment_edited'					=> __('Javascript message displayed after a successful editing', 'wp-base' ),
					'appointment_received'					=> __('Javascript message displayed after a successful booking and no confirmation/pending dialog text exists', 'wp-base' ),
					'app_id'								=> __('Column header for appointment ID for List of Bookings', 'wp-base' ),
					'auto_assign_intro'						=> __('Javascript message displayed after user is auto assigned as a Service Provider','wp-base'),
					'auto_assign_login'						=> __('Message displayed when login is required to be assigned as a service provider','wp-base'),
					'a_specialist'							=> __('Text displayed to client for service provider when no particular provider is selected', 'wp-base'),
					'available'								=> __('Legend label for free slot', 'wp-base'),
					'apps_to_change'						=> __('Label to confirm if chnages will be applied to child bookings', 'wp-base' ),
					'avatar'								=> __('Synonym for profile photo', 'wp-base'),
					'balance'								=> __('Column header of Balance for List of Bookings', 'wp-base'),
					'bio_page'								=> __('Column header for Bio Page','wp-base'),
					'biweekly'								=> __('Text for bookings recurring every other week','wp-base'),
					'blacklisted'							=> __('Javascript message displayed when client applies with a blacklisted email address', 'wp-base'),
					'booking'								=> __('Spinner panel message while booking is being saved','wp-base'),
					'booking_id'							=> __('Booking label used in carts','wp-base'),
					'booking_closed'						=> __('Book Now shortcode button text displayed when booking is not possible, i.e. fully booked or allowed booking time has passed', 'wp-base'),
					'book_now_long'							=> __('Book Now shortcode button text displayed when booking is possible. LOCATION, SERVICE, WORKER, START will be replaced by their real values', 'wp-base'),
					'book_now_short'						=> __('Book in Table View button text. LOCATION, SERVICE, WORKER, START will be replaced by their real values', 'wp-base'),
					'bp_3months'							=> __('Account page 3 Months Schedule tab title', 'wp-base' ),
					'bp_4weeks'								=> __('Account page 4 Weeks Schedule tab title', 'wp-base' ),
					'bp_annual'								=> __('Account or BuddyPress user profile page Seasonal Schedules tab title', 'wp-base' ),
					'bp_bookings'							=> __('Account or BuddyPress user profile page bookings tab title for client', 'wp-base' ),
					'bp_bookings_as_client'					=> __('BuddyPress user profile page bookings tab title for provider as client', 'wp-base' ),
					'bp_bookings_as_provider'				=> __('BuddyPress user profile page bookings tab title for provider as provider', 'wp-base' ),
					'bp_book_me'							=> __('BuddyPress user profile page Book Me tab title', 'wp-base'),
					'bp_commissions'						=> __('Account or BuddyPress user profile page Commissions tab title', 'wp-base'),
					'bp_credits'							=> __('Account or BuddyPress user profile page Credits tab title', 'wp-base'),
					'bp_dashboard'							=> __('Account or BuddyPress user profile page Dashboard tab title', 'wp-base'),
					'bp_gcal'								=> __('Account or BuddyPress user profile page Google Calendar tab title', 'wp-base' ),
					'bp_holidays'							=> __('Account or BuddyPress user profile page Holidays tab title', 'wp-base' ),
					'bp_manage_bookings'					=> __('Account or BuddyPress user profile page Manage Bookings tab title', 'wp-base'),
					'bp_monthly'							=> __('Account or BuddyPress user page Monthly Schedule tab title', 'wp-base' ),
					'bp_ntf_booking_cancelled'				=> __('BuddyPress notification when the client cancels a booking. You can use the booking placeholders which are listed on Emails page.', 'wp-base' ),
					'bp_ntf_booking_cancelled_by_admin'		=> __('BuddyPress notification when admin cancels a booking. You can use the booking placeholders which are listed on Emails page.', 'wp-base' ),
					'bp_ntf_booking_cancelled_by_worker'	=> __('BuddyPress notification when provider cancels a booking. You can use the booking placeholders which are listed on Emails page.', 'wp-base' ),
					'bp_ntf_booking_rescheduled'			=> __('BuddyPress notification when the client changes start time of a booking. You can use the booking placeholders which are listed on Emails page.', 'wp-base' ),
					'bp_ntf_booking_rescheduled_by_admin'	=> __('BuddyPress notification when admin changes start time of a booking. You can use the booking placeholders which are listed on Emails page.', 'wp-base' ),
					'bp_ntf_booking_rescheduled_by_worker'	=> __('BuddyPress notification when provider changes start time of a booking. You can use the booking placeholders which are listed on Emails page.', 'wp-base' ),
					'bp_ntf_booking_completed'				=> __('BuddyPress notification when status of booking has been changed to "completed". You can use the booking placeholders which are listed on Emails page.', 'wp-base' ),
					'bp_ntf_booking_confirmed'				=> __('BuddyPress notification when status of booking has been changed to "confirmed". You can use the booking placeholders which are listed on Emails page.', 'wp-base' ),
					'bp_ntf_booking_paid'					=> __('BuddyPress notification when status of booking has been changed to "paid". You can use the booking placeholders which are listed on Emails page.', 'wp-base' ),
					'bp_ntf_booking_running'				=> __('BuddyPress notification when a booking has started. You can use the booking placeholders which are listed on Emails page.', 'wp-base' ),
					'bp_ntf_new_booking'					=> __('BuddyPress notification when a new booking has been made by client. You can use the booking placeholders which are listed on Emails page.', 'wp-base' ),
					'bp_ntf_new_booking_by_admin'			=> __('BuddyPress notification when a new booking has been made by admin. You can use the booking placeholders which are listed on Emails page.', 'wp-base' ),
					'bp_ntf_new_booking_by_worker'			=> __('BuddyPress notification when a new booking has been made by provider. You can use the booking placeholders which are listed on Emails page.', 'wp-base' ),
					'bp_ntf_worker_changed'					=> __('BuddyPress notification when service provider of a booking has been changed. You can use the booking placeholders which are listed on Emails page.', 'wp-base' ),
					'bp_payments'							=> __('Account or BuddyPress user profile page Payment History tab title', 'wp-base'),
					'bp_schedules'							=> __('BuddyPress user profile page Schedules tab title', 'wp-base' ),
					'bp_services'							=> __('Account or BuddyPress user profile page Services tab title', 'wp-base' ),
					'bp_settings'							=> __('Account or BuddyPress user profile page Settings tab title', 'wp-base' ),
					'bp_title'								=> __('BuddyPress user profile page main tab title', 'wp-base' ),
					'bp_use_book_me'						=> __('BuddyPress WP BASE checkbox title to select whether to add a Book Me tab', 'wp-base'),
					'bp_weekly'								=> __('Account or BuddyPress user page Weekly Schedule tab title', 'wp-base' ),
					'bp_wh'									=> __('Account or BuddyPress user page Working Hours tab title', 'wp-base' ),
					'busy'						=> __('Legend label for busy slot', 'wp-base'),
					'calculating'				=> __('Spinner panel message while price is being recalculated after a form field change','wp-base'),
					'cancel'					=> __('Column header of Cancel for List of Bookings', 'wp-base' ),
					'cancelled'					=> __('Javascript text that is displayed after an appointment has been cancelled','wp-base'),
					'cancel_app_confirm'		=> __('Javascript text that is displayed before cancelling an appointment','wp-base'),
					'cancel_button'				=> __('Cancel button text in List of Bookings','wp-base'),
					'cancel_cart'				=> __('Button text on confirmation form to clear cart contents and refresh page', 'wp-base' ),
					'cancel_confirm_no'			=> __('Text to quit cancellation of current process', 'wp-base' ),
					'cancel_confirm_text'		=> __('Text displayed to confirm cancellation of current process (checkout, edit, etc)', 'wp-base' ),
					'cancel_confirm_yes'		=> __('Text to confirm cancellation of current process', 'wp-base' ),
					'cancel_disabled'			=> __('Javascript message displayed when client attempts to cancel an appointment, but cancellation is turned off','wp-base'),
					'capacity'					=> __('Column header of Capacity for Services table', 'wp-base'),
					'cart'						=> __('Text for status in Cart', 'wp-base'),
					'category'					=> __('Column header of Category for List of Bookings', 'wp-base' ),
					'cc_address1'				=> __('Address Line 1 title on credit card form', 'wp-base'),
					'cc_address2'				=> __('Address Line 2 title on credit card form', 'wp-base'),
					'cc_city'					=> __('City title on credit card form', 'wp-base'),
					'cc_country'				=> __('Country title on credit card form', 'wp-base'),
					'cc_cvv'					=> __('Security Code title on credit card form', 'wp-base'),
					'cc_declined'				=> __('Error message when credit card is declined by payment gateway.', 'wp-base'),
					'cc_email'					=> __('Email title on credit card form', 'wp-base'),
					'cc_error'					=> __('Error message coming from payment gateway other than declined. If you want to show details, keep ERROR_DETAIL placeholder which will be replaced by details of error.', 'wp-base'),
					'cc_expiry'					=> __('Expiration Date title on credit card form', 'wp-base'),
					'cc_form_legend'			=> __('Title above credit card form', 'wp-base' ),
					'cc_name'					=> __('Full Name title on credit card form', 'wp-base'),
					'cc_number'					=> __('Credit Card Number title on credit card form', 'wp-base'),
					'cc_phone'					=> __('Phone title on credit card form', 'wp-base'),
					'cc_state'					=> __('State/Province/Region title on credit card form', 'wp-base'),
					'cc_zip'					=> __('Postcode title on credit card form', 'wp-base'),
					'checkout'					=> __('Spinner panel message while processing checkout','wp-base'),
					'checkout_button'			=> __('Button text for checkout (finalise booking)', 'wp-base' ),
					'checkout_button_tip'		=> __('Tooltip text for checkout button', 'wp-base' ),
					'choose_image'				=> __('Label to select an image for service or avatar', 'wp-base'),
					'city'						=> __('Title for City field in the confirmation form and user page','wp-base'),
					'city_placeholder'			=> __('Placeholder for City field in the confirmation form','wp-base'),
					'clear'						=> __('Clear signature button text', 'wp-base' ),
					'click_to_book'				=> __('Hint text to add a booking in calendar','wp-base'),
					'click_to_remove'			=> __('Hint text to remove an appointment from confirmation form','wp-base'),
					'click_to_select_date'		=> __('Hint text to add select a date in monthly calendar','wp-base'),
					'client'					=> __('Column header of Client for List of Bookings', 'wp-base' ),
					'close'						=> __('Close button text', 'wp-base' ),
					'completed'					=> __('Text for status completed', 'wp-base'),
					'confirm'					=> __('Column header of Confirm for List of Bookings', 'wp-base' ),
					'confirm_button'			=> __('Confirm button text in List of Bookings','wp-base'),
					'confirm_cancel'			=> __( 'Javascript message displayed before cancelling a booking', 'wp-base' ),
					'confirm_confirm'			=> __( 'Javascript message displayed before confirming a booking', 'wp-base' ),
					'confirm_delete'			=> __( 'Javascript message displayed before deleting a booking', 'wp-base' ),
					'confirm_disabled'			=> __('Javascript message displayed when client attempts to confirm an appointment, but confirmation is turned off','wp-base'),
					'confirmation_title'		=> __('Title for confirmation form', 'wp-base' ),
					'confirmed'					=> __('Text for status confirmed', 'wp-base'),
					'confirmed_message'			=> __('Javascript message displayed after a booking has been confirmed', 'wp-base'),
					'conf_countdown_title'		=> __('Title of countdown on confirmation form', 'wp-base'),
					'connection_error'			=> __('Javascript text that is displayed when ajax request fails','wp-base'),
					'continue_button'			=> __('Button text for continue (finalise selections and display confirmation form)', 'wp-base' ),
					'countdown_title'			=> __('Title of Next Appointment Countdown', 'wp-base'),
					'country'					=> __('Title for Country field in the confirmation form and user page','wp-base'),
					'country_placeholder'		=> __('Placeholder for Country field in the confirmation form','wp-base'),
					'coupon'					=> __('Title for Coupon in the confirmation form', 'wp-base' ),
					'coupon_placeholder'		=> __('Description of coupon field in the confirmation form', 'wp-base' ),
					'created'					=> __('Column header of Creation Date for List of Bookings', 'wp-base' ),
					'created_by'				=> __('Column header of Creation by field for List of Bookings', 'wp-base' ),
					'credit'					=> __('Singular text for Credit', 'wp-base' ),
					'credit_card'				=> __('Title for Credit Card field on the payment form', 'wp-base' ),
					'credits'					=> __('Plural text for Credit', 'wp-base' ),
					'daily'						=> __('Text for bookings recurring every day','wp-base'),
					'dashboard'					=> __('Title for vendor account dashboard','wp-base'),
					'date'						=> __('Column header of Date in editing form', 'wp-base' ),
					'date_time'					=> __('Column header of starting time of booking for List of Bookings', 'wp-base' ),
					'day'						=> __('Singular text for day','wp-base'),
					'days'						=> __('Plural text for day','wp-base'),
					'day_of_week'				=> __('Column header of Week Day in editing form', 'wp-base' ),
					'default'					=> __('Indicates default value will be used', 'wp-base'),
					'delete'					=> __('Label for Delete Permanently', 'wp-base'),
					'deleted'					=> __('Javascript message after one or more records deleted.', 'wp-base'),
					'deposit'					=> __('Column header of Deposit for List of Bookings', 'wp-base'),
					'deposit_paid'				=> __('Label for Deposit Paid - This replaces Total in WooCommerce if payment is partial', 'wp-base'),
					'description_page'			=> __('Column header of Description Page for Services table', 'wp-base'),
					'details'					=> __('Title for items in the cart: Title for list of selected slots in the confirmation form when more than one time slot is selected', 'wp-base' ),
					'display_name'				=> __('Title for Display Name field in the confirmation form and user page','wp-base'),
					'done'						=> __('Spinner panel message when ajax jobs finished','wp-base'),
					'down_payment'				=> __('Column header of Prepayment for List of Bookings', 'wp-base'),
					'due_payment'				=> __('Label for Due Payment which is the amount to be paid after deposit is reduced', 'wp-base'),
					'duration'					=> __('Column header of Duration for Services table', 'wp-base'),
					'edd_payment'				=> __('Column header for EDD Payment ID', 'wp-base' ),
					'edit'						=> __('Column header of Edit for List of Bookings', 'wp-base' ),
					'edit_app_confirm'			=> __('Javascript text that is displayed before editing an appointment','wp-base'),
					'edit_button'				=> __('Edit button text in List of Bookings','wp-base'),
					'edit_disabled'				=> __('Javascript message displayed when client attempts to edit an appointment, but editing is turned off','wp-base'),
					'edit_saved'				=> __('Javascript message displayed after changes on booking have been saved', 'wp-base' ),
					'edit_too_early'			=> __('Javascript message displayed when client attempts to edit an appointment with a start time which falls in lead time (lower limit) margin. Keep %s which will be replaced by the latest time client can pick.','wp-base'),
					'email'						=> __('Title for email field in the confirmation form and user page','wp-base'),
					'email_placeholder'			=> __('Placeholder for email field in the confirmation form','wp-base'),
					'email_unmatch'				=> __('Javascript message displayed when logged in client attempts to make a booking with email of another registered user','wp-base'),
					'end_date_time'				=> __('Column header of end time of booking for List of Bookings', 'wp-base' ),
					'eod'						=> __('Text for bookings recurring every other day','wp-base'),
					'eod_except_sunday'			=> __('Text for bookings recurring every other day except Sunday','wp-base'),
					'error'						=> __('Javascript message displayed after a General/unknown error', 'wp-base'),
					'error_short'				=> __('Title for general/unknown error message display', 'wp-base'),
					'event_capacity'			=> __('Label for Event Capacity','wp-base'),
					'event_capacity_desc'		=> __('Description for Event Capacity field','wp-base'),
					'event_closed_from_pax'		=> __('Label for Closed from Pax','wp-base'),
					'event_closed_from_pax_desc'=> __('Description for Closed from Pax field','wp-base'),
					'event_enabled'				=> __('Label for Group Bookings Enable','wp-base'),
					'event_enabled_desc'		=> __('Description for Group Bookings Enable field','wp-base'),
					'event_max'					=> __('Label for Maximum Pax','wp-base'),
					'event_max_desc'			=> __('Description for Maximum Pax field','wp-base'),
					'event_min'					=> __('Label for Minimum Pax','wp-base'),
					'event_min_desc'			=> __('Description for Minimum Pax field','wp-base'),
					'event_price'				=> __('Label for Event Price','wp-base'),
					'event_price_desc'			=> __('Description for Event Price field','wp-base'),
					'event_title'				=> __('Title for Event Bookings list', 'wp-base'),
					'excerpt_more'				=> __('Text for more title/link for excerpts', 'wp-base' ),
					'expired_card'		 		=> __('Error message when credit card has expired', 'wp-base' ),
					'export_csv'				=> __('Button text for export bookings in CSV file format','wp-base'),
					'extra'						=> __('Title for Extra in confirmation form, bookings page table and list of bookings','wp-base'),
					'full_name'					=> __('Title for Full Name field in the forms','wp-base'),
					'first_name'				=> __('Title for First Name field in the confirmation form and user page','wp-base'),
					'first_name_placeholder'	=> __('Placeholder for First Name field in the confirmation form','wp-base'),
					'friday'					=> __('Friday',	'wp-base'),
					'friday_initial'			=> __('Initial letter of Friday', 'wp-base'),
					'friday_short'				=> __('Short form of Friday', 'wp-base'),
					'gcal'						=> __('Title for GCal column in List of Bookings','wp-base'),
					'gcal_button'				=> __('Tooltip title for Google Calendar button','wp-base'),
					'gdpr_udf_title'			=> __('Title for GDPR UDF Group','wp-base'),
					'gdpr_userdata_title'		=> __('Title for GDPR userdata Group','wp-base'),
					'hangout'					=> __('Label for Google Meet column header','wp-base'),
					'hangout_join'				=> __('Text for Google Meet Join button','wp-base'),
					'hold'						=> __('Text for status Temporary', 'wp-base'),
					'host'						=> __('Text for Host', 'wp-base'),
					'hour'						=> __('Singular text for hour','wp-base'),
					'hours'						=> __('Plural text for hour','wp-base'),
					'hour_short'				=> __('Short form of hour','wp-base'),
					'image'						=> __('Column header of Featured Image for Services table', 'wp-base'),
					'info'						=> __('Localization of pagination under tables. Keep _PAGE_ and _PAGES_ which is the current page no and total number of pages, respectively.', 'wp-base'),
					'internal'					=> __('Column header of Internal for Services table', 'wp-base'),
					'invalid_booking' 			=> __('Javascript message displayed when a mandatory field of a booking record is missing.', 'wp-base' ),
					'invalid_cardholder' 		=> __('Error message when credit cardholder is invalid', 'wp-base' ),
					'invalid_cc_number'	 		=> __('Error message when Credit Card Number field is empty or card number is invalid', 'wp-base' ),
					'invalid_cvc'				=> __('Error message when security code is empty or invalid', 'wp-base' ),
					'invalid_email'				=> __('Error message displayed when submitted email is not valid', 'wp-base' ),
					'invalid_expiration'		=> __('Error message when expiration month and/or year field is empty or invalid', 'wp-base' ),
					'jitsi'						=> __('Column header of Jitsi for List of Bookings', 'wp-base' ),
					'jitsi_already_started'		=> __('Javascript message displayed to the host when he/she is trying to start a meeting which has been already started', 'wp-base'),
					'jitsi_called'				=> __('Javascript message displayed to the participant when he/she is called to the meeting by the host.','wp-base'),
					'jitsi_meeting_note_host'	=> __('Default text displayed to the meeting host at the Meeting Room.','wp-base'),
					'jitsi_meeting_note'		=> __('Default text about general meeting rules.','wp-base'),
					'jitsi_meeting_started'		=> __('Text displayed to participant when meeting has started. Keep MEETING_LINK placeholder which will be replaced by the link to the Meeting Room page.','wp-base'),
					'jitsi_no_meeting'			=> __('Text displayed when user has no future or current meetings.','wp-base'),
					'jitsi_start_meeting'		=> __('Label for Start Meeting link','wp-base'),
					'jitsi_waiting_note'		=> __('Default text for client about how meeting will start.','wp-base'),
					'jitsi_waiting_note_host'	=> __('Default text for host about how meeting will start.','wp-base'),
					'lasts'						=> __('Title for Duration of the selected service(s) in the confirmation form', 'wp-base' ),
					'last_name'					=> __('Title for Last Name field in the confirmation form and user page','wp-base'),
					'last_name_placeholder'		=> __('Placeholder for Last Name field in the confirmation form','wp-base'),
					'length_menu'				=> __('Localization for pulldown menu that selects the number of records to be displayed in the tables. Keep _MENU_ which stands for the pulldown menu itself.', 'wp-base'),
					'limit_exceeded'			=> __('Javascript message displayed when selected number of appointments exceeds permitted number. Keep %d which will be replaced by actual limit', 'wp-base'),
					'loading'					=> __('Text message displayed while a data is being read','wp-base'),
					'location'					=> __('Column header of Location for List of Bookings', 'wp-base' ),
					'location_address'			=> __('Column header of Location Address for List of Bookings', 'wp-base' ),
					'location_name'				=> __('Title for Location in the confirmation form', 'wp-base' ),
					'location_names'			=> __('Plural of the Location title in the confirmation form', 'wp-base' ),
					'location_note'				=> __('Column header of Location Note for List of Bookings', 'wp-base' ),
					'logged_in'					=> __('Message displayed to client after a successful login', 'wp-base'),
					'logged_message'			=> __('Text displayed below calendar title when client is logged in', 'wp-base'),
					'logging_in'				=> __('Spinner panel message while login is being prepared','wp-base'),
					'login'						=> __('Text for the login link', 'wp-base'),
					'login_for_cancel'			=> __('Message displayed when login is required to cancel a booking','wp-base'),
					'login_for_confirm'			=> __('Message displayed when login is required to confirm a booking','wp-base'),
					'login_for_edit'			=> __('Message displayed when login is required to edit a booking','wp-base'),
					'login_message'				=> __('Text beside the login link', 'wp-base' ),
					'login_required'			=> __('Message displayed to client when they submitted an existing email and not logged in yet', 'wp-base' ),
					'login_with_facebook'		=> __('Button text to login with Facebook account', 'wp-base'),
					'login_with_google'			=> __('Button text to login with Google+ account', 'wp-base'),
					'login_with_twitter'		=> __('Button text to login with Twitter account', 'wp-base'),
					'login_with_wp'				=> __('Button text to login with WordPress account', 'wp-base'),
					'maybe_edited'				=> __('Message displayed when a record may have edited by another user and current value may not be valid any more.','wp-base'),
					'method'					=> __('Title for payment method','wp-base'),
					'minute'					=> __('Singular text for minute','wp-base'),
					'minutes'					=> __('Plural text for minute','wp-base'),
					'min_short'					=> __('Short form of minute','wp-base'),
					'missing_extra'				=> __('Javascript message displayed when selection of an extra is required, but client did not pick one','wp-base'),
					'missing_field'				=> __('Javascript message displayed when a required field is left empty','wp-base'),
					'missing_terms_check'		=> __('Javascript message displayed when a terms and conditions checkbox are not checked by the client','wp-base'),
					'monday'					=> __('Monday',	'wp-base'),
					'monday_initial'			=> __('Initial letter of Monday', 'wp-base'),
					'monday_short'				=> __('Short form of Monday', 'wp-base'),
					'month'						=> __('Singular text for month','wp-base'),
					'monthly'					=> __('Text for bookings recurring every month','wp-base'),
					'monthly_title'				=> __('Title of the monthly calendar. LOCATION, SERVICE, WORKER, START will be replaced by their real values', 'wp-base'),
					'months'					=> __('Plural text for month','wp-base'),
					'mop_placeholder'			=> __('Placeholder for participant email field. %d will be replaced by the order in the list.', 'wp-base' ),
					'mv_activate'				=> __('Registration successful message', 'wp-base' ),
					'mv_afternoon'				=> __('Label for Afternoon in availability filter','wp-base'),
					'mv_already_active'			=> __('Message displayed when an already active user attempts to activate themselves again', 'wp-base'),
					'mv_already_applied'		=> __('Message displayed when a user applies to become a vendor, but he/she already has past records. Keep %s which will be replaced with date of application.', 'wp-base' ),
					'mv_already_vendor'			=> __('Message displayed when a vendor applies to become a vendor again','wp-base'),
					'mv_apply_now'				=> __('Text of the button on Profile Settings to apply to become a member','wp-base'),
					'mv_become_vendor'			=> __('Text of the button to apply to become a vendor','wp-base'),
					'mv_become_vendor_desc'		=> __('Description text explaining Become a Vendor button','wp-base'),
					'mv_become_vendor_confirm'	=> __('Javascript message asking confirmation after clicking Become a Vendor button','wp-base'),
					'mv_bio'					=> __('Label for Bio field','wp-base'),
					'mv_bio_excerpt'			=> __('Label for Bio summary field','wp-base'),
					'mv_bio_page'				=> __('Label for Bio Page field','wp-base'),
					'mv_cannot_create_user'		=> __('Error message displayed when user could not be created', 'wp-base'),
					'mv_days_of_the_week'		=> __('Title for Days of the week in availability filter','wp-base'),
					'mv_declined'				=> __('Message displayed to the user whose vendor application has been declined','wp-base'),
					'mv_email_failed'			=> __('Error message displayed when activation email sending failed', 'wp-base'),
					'mv_email_banned'			=> __('Error message displayed when submitted email is banned', 'wp-base'),
					'mv_email_in_use'			=> __('Error message displayed when submitted email is already in use', 'wp-base'),
					'mv_email_invalid'			=> __('Error message displayed when submitted email is empty or not in valid format', 'wp-base'),
					'mv_evening'				=> __('Label for Evening in availability filter','wp-base'),
					'mv_invalid_key'			=> __('Error message displayed when an invalid or already used activation key is used.', 'wp-base'),
					'mv_is_vendor'				=> __('Text displayed instead of Become a Vendor button when user is already a vendor','wp-base'),
					'mv_late_afternoon'			=> __('Label for Late afternoon in availability filter','wp-base'),
					'mv_late_evening'			=> __('Label for Late Evening in availability filter','wp-base'),
					'mv_late_morning'			=> __('Label for Late morning in availability filter','wp-base'),
					'mv_late_night'				=> __('Label for Late Night in availability filter','wp-base'),
					'mv_login'					=> __('Account activation complete message. Displayed if vendor is approved', 'wp-base' ),
					'mv_morning'				=> __('Label for Morning in availability filter','wp-base'),
					'mv_night'					=> __('Label for Night in availability filter','wp-base'),
					'mv_no_privacy_policy'		=> __('Error message displayed when Privacy Policy check box is not checked', 'wp-base'),
					'mv_override_prices'		=> __('Label for Override Prices setting in vendor profile','wp-base'),
					'mv_override_prices_desc'	=> __('Description for Override Prices setting in vendor profile.','wp-base'),
					'mv_page_not_found'			=> __('Message displayed to the user when his page cannot be accessed, possibly due to suspension of the user.', 'wp-base'),
					'mv_password_empty'			=> __('Error message displayed when password field is empty', 'wp-base'),
					'mv_password_no_match'		=> __('Error message displayed when submitted password values do not match', 'wp-base'),
					'mv_required_field'			=> __('Error message displayed when a required form field was not filled. Keep %s which will be replaced by the name of the field', 'wp-base'),
					'mv_status_pending'			=> __('Description displayed to user when vendor status is pending', 'wp-base'),
					'mv_tagline_text'			=> __('Label for Tagline Text in user profile','wp-base'),
					'mv_tags'					=> __('Label for Tags in user profile','wp-base'),
					'mv_time_of_the_day'		=> __('Title for Time of the day in availability filter','wp-base'),
					'mv_title'					=> __('Label for Page Title in user profile','wp-base'),
					'mv_upload_photo'			=> __('Button text for Upload Photo','wp-base'),
					'mv_username_empty'			=> __('Error message displayed when an empty username is submitted', 'wp-base'),
					'mv_username_exists'		=> __('Error message displayed when an already existing username is submitted', 'wp-base'),
					'mv_username_illegal'		=> __('Error message displayed when an illegal username is submitted', 'wp-base'),
					'mv_username_invalid'		=> __('Error message displayed when a username having invalid characters is submitted', 'wp-base'),
					'mv_username_numeric'		=> __('Error message displayed when an all numeric username is submitted', 'wp-base'),
					'mv_username_short'			=> __('Error message displayed when a username shorter than 4 characters is submitted', 'wp-base'),
					'mv_username_underscore'	=> __('Error message displayed when a username having underscore character is submitted', 'wp-base'),
					'mv_signup'					=> __('Sign Up button text', 'wp-base'),
					'mv_video'					=> __('Label for Video in user profile','wp-base'),
					'mv_wait'					=> __('Account activation complete message. Displayed when vendor application awaits admin approval', 'wp-base' ),
					'name'						=> __('Title for Name field in the confirmation form','wp-base'),
					'name_placeholder'			=> __('Placeholder for Name field in the confirmation form','wp-base'),
					'next'						=> __('Pagination button text for Next','wp-base'),
					'next_day'					=> __('Note added to details field on confirmation form to notify a booking ending next day', 'wp-base' ),
					'next_month'				=> __('Pagination button text for Next Month','wp-base'),
					'next_months'				=> __('Pagination button text for Next Months','wp-base'),
					'next_week'					=> __('Pagination button text for Next Week','wp-base'),
					'next_weeks'				=> __('Pagination button text for Next Weeks','wp-base'),
					'no'						=> __('Translation for No, e.g. to show a check box is not checked','wp-base'),
					'nof_jobs_cancelled'		=> __('Title for number of cancelled jobs of a package in confirmation form, bookings page table and list of bookings','wp-base'),
					'nof_jobs_completed'		=> __('Title for number of completed jobs of a package in confirmation form, bookings page table and list of bookings','wp-base'),
					'nof_jobs_remaining'		=> __('Title for number of remaining jobs of a package in confirmation form, bookings page table and list of bookings','wp-base'),
					'nof_jobs_total'			=> __('Title for number of total jobs of a package in confirmation form, bookings page table and list of bookings','wp-base'),
					'nop_placeholder'			=> __('Placeholder for participant name field. %d will be replaced by the order in the list.', 'wp-base' ),
					'not_enough_credits'		=> __('Message displayed after a purchase attempt with insufficient credits', 'wp-base'),
					'noshow'					=> __('Text for status No Show', 'wp-base'),
					'note'						=> __('Title for Note field in the confirmation form','wp-base'),
					'nothing_changed'			=> __('Javascript message displayed when client did not make any changes', 'wp-base'),
					'notice'					=> __('Dialog title for notice type messages', 'wp-base'),
					'not_available'				=> __('Javascript text displayed when client attempts to edit date/time of a booking and time slot has been booked','wp-base'),
					'not_defined'				=> __('Text displayed when a service, provider, etc variable does not have a displayable name. Possibly because of a record being deleted', 'wp-base' ),
					'not_enough_capacity'		=> __('Javascript message displayed when client attempts to take seats more than the capacity. This may happen when different services with different capacities are to be booked', 'wp-base'),
					'not_logged_message'		=> __('Text displayed below calendar title when login is required and client is not logged in. LOGIN_PAGE will be replaced by the login url', 'wp-base'),
					'not_owner'					=> __('Javascript text displayed when user does not match to the owner of the booking.', 'wp-base'),
					'not_possible'				=> __('Javascript text displayed when requested action or selection is not possible','wp-base'),
					'not_published'				=> __('Javascript text displayed when client attempts to book a service which is not published (expired or not open yet)','wp-base'),
					'not_working'				=> __('Javascript text displayed when client attempts to edit date/time of a booking and provider is not working','wp-base'),
					'no_appointments'			=> __('Text for List of Bookings when there are no appointments to be displayed','wp-base'),
					'no_free_time_slots'		=> __('HTML text displayed when there are no free time slots in the table or block','wp-base'),
					'no_preference'				=> __('Selection in dropdowns when there is no preference, (e.g. when client does not particularly select a provider)', 'wp-base'),
					'no_repeat'					=> __('Text for Recurring Appointments pulldown menu to select no repeat','wp-base'),
					'no_service'				=> __('Message displayed when there are no matching services', 'wp-base'),
					'no_result'					=> __('Message displayed when there are no matching results', 'wp-base'),
					'notpossible'				=> __('Legend label for not available, e.g. break, holiday, slot', 'wp-base'),
					'order'						=> __('Title for Order in list of bookings','wp-base'),
					'our_staff'					=> __('Text displayed to admin for service provider when no particular provider is selected', 'wp-base'),
					'owner'						=> __('Column header of Owner for Services table', 'wp-base'),
					'padding_after'				=> __('Column header of Padding After for Services table', 'wp-base'),
					'padding_before'			=> __('Column header of Padding Before for Services table', 'wp-base'),
					'paid'						=> __('Text for status paid', 'wp-base'),
					'paid_up'					=> __('Javascript message informing that balance is not negative', 'wp-base'),
					'participants'				=> __('Header for List of Participants of Group Bookings', 'wp-base' ),
					'participant_title'			=> __('Title for each participant of Group Bookings. %d will be replaced by the order in the list.','wp-base'),
					'partly_busy'				=> __('Legend label for partly busy slot', 'wp-base'),
					'password'					=> __('Label for Password field', 'wp-base'),
					'password_confirm'			=> __('Label for Repeat Password field', 'wp-base'),
					'past_date'					=> __('Javascript text displayed when client attempts to edit a booking with past date/time','wp-base'),
					'pax'						=> __('Used for each guest/seat booked in Group Bookings', 'wp-base' ),
					'payment_method_error'		=> __('Message displayed when payment is required, but no payment method is selected or not available', 'wp-base'),
					'payment_id'				=> __('Title for Payment ID in Payment History','wp-base'),
					'pay'						=> __('Column header of Pay for List of Bookings', 'wp-base' ),
					'paypal'					=> __('Column header of PayPal for List of Bookings', 'wp-base' ),
					'paypal_express_button'		=> __('Button text to confirm PayPal Express payment final stage.', 'wp-base'),
					'paypal_express_note'		=> __('Text that will be displayed in final confirmation stage with Paypal Express Checkout. AMOUNT placeholder will be reapleced with the amount to be paid, and EMAIL is PayPal account email of the client.', 'wp-base'),
					'pay_now'					=> __('Button text for credit card data submit. AMOUNT placeholder will be replaced with formatted payable amount, including currency sign.', 'wp-base' ),
					'pay_with'					=> __('Text beside Payment field in the confirmation form when more than one payment option/gateway is active','wp-base'),
					'pdf'						=> __('Column header of Pdf download for List of Bookings', 'wp-base' ),
					'pdf'						=> __('Title text of confirmation attachment for my appointments table', 'wp-base' ),
					'pdf_download'				=> __('Button text to download a confirmation attachment for my appointments table', 'wp-base' ),
					'pdf_download'				=> __('Pdf Download button text in List of Bookings','wp-base'),
					'pdf_tooltip'				=> __('Tooltip text that explains function of pdf download button.','wp-base'),
					'pdf_tooltip'				=> __('Tooltip title for Pdf Download button','wp-base'),
					'pending'					=> __('Text for status pending', 'wp-base'),
					'pending_approval'			=> __('Text for status pending and an automatic payment is not expected, e.g. payment is not required, price is zero, or manual payment is selected', 'wp-base'),
					'pending_payment'			=> __('Text for status pending and an automatic payment via a gateway is expected', 'wp-base'),
					'phone'						=> __('Title for Phone field in the confirmation form and user page','wp-base'),
					'phone_placeholder'			=> __('Placeholder for Phone field in the confirmation form','wp-base'),
					'please_wait'				=> __('Message displayed while submitting a form', 'wp-base' ),
					'pop_placeholder'			=> __('Placeholder for participant phone field. %d will be replaced by the order in the list.', 'wp-base' ),
					'preferred_language'		=> __('Title text for language selection dropdown', 'wp-base'),
					'preparing_form'			=> __('Spinner panel message while booking views are being prepared','wp-base'),
					'preparing_timetable'		=> __('Spinner panel message while time table is being prepared','wp-base'),
					'previous'					=> __('Pagination button text for Previous','wp-base'),
					'previous_month'			=> __('Pagination button text for Previous Month','wp-base'),
					'previous_months'			=> __('Pagination button text for Previous Months','wp-base'),
					'previous_week'				=> __('Pagination button text for Previous Week','wp-base'),
					'previous_weeks'			=> __('Pagination button text for Previous Weeks','wp-base'),
					'price'						=> __('Title for Price in the confirmation form', 'wp-base' ),
					'price_mismatch'			=> __('Javascript message in case there is a mismatch of calculated price and price previously displayed to the client and new price is higher.','wp-base'),
					'price_mismatch_lower'		=> __('Javascript message in case there is a mismatch of calculated price and price previously displayed to the client and new price is lower.','wp-base'),
					'proceed'					=> __('Javascript message displayed when client is asked to confirm to proceed', 'wp-base'),
					'processing' 				=> __('Processing text when connecting to payment gateway', 'wp-base' ),
					'provider'					=> __('Column header of Provider for List of Bookings', 'wp-base' ),
					'provider_name'				=> __('Title for Service Provider in the confirmation form', 'wp-base' ),
					'provider_names'			=> __('Plural of the Service Provider title in the confirmation form', 'wp-base' ),
					'quota'						=> __('Javascript text displayed when client attempts to book a service which is out of quota','wp-base'),
					'reading'					=> __('Spinner panel message while ajax data read is in progress','wp-base'),
					'redirect'					=> __('Javascript message displayed before client is redirected to the login page', 'wp-base' ),
					'reference'					=> __('Title for Payment Reference','wp-base'),
					'refreshing'				=> __('Spinner panel message while page is being refreshed','wp-base'),
					'register'					=> __('Placeholder value in login Register field', 'wp-base'),
					'remember'					=> __('Text beside Remember Me field in the confirmation form','wp-base'),
					'removed'					=> __('Text for status removed', 'wp-base'),
					'removed'					=> __('Text in tooltip when an appointment has been removed from confirmation form','wp-base'),
					'required'					=> __('Note added under confirmation form when there is at least one required field', 'wp-base' ),
					'reserved'					=> __('Text for status reserved by Google Calendar', 'wp-base'),
					'room'						=> __('Jitsi meeting room ID', 'wp-base'),
					'running'					=> __('Text for status happening now', 'wp-base'),
					'saturday'					=> __('Saturday',	'wp-base'),
					'saturday_initial'			=> __('Initial letter of Saturday', 'wp-base'),
					'saturday_short'			=> __('Short form of Saturday', 'wp-base'),
					'saved'						=> __('Javascript message after settings saved.', 'wp-base'),
					'save_button'				=> __('Button text for save (finalise edit)', 'wp-base' ),
					'save_error'				=> __('Javascript message displayed when appointment could not be saved possibly because of a server error', 'wp-base'),
					'saving'					=> __('Spinner panel message while data is being saved','wp-base'),
					'search'					=> __('Placeholder value in table Search field', 'wp-base'),
					'seats'						=> __('Title for number of seats in the confirmation form', 'wp-base'),
					'seats_left'				=> __('Column header of available capacity', 'wp-base' ),
					'seats_total'				=> __('Column header of total capacity', 'wp-base' ),
					'seats_total_left'			=> __('Column header of total and available capacity separated with "/"', 'wp-base' ),
					'second'					=> __('Singular text for second','wp-base'),
					'seconds'					=> __('Plural text for seconds','wp-base'),
					'select'					=> __('General select text', 'wp-base'),
					'selected'					=> __('Text displayed to show a slot has been selected', 'wp-base'),
					'select_button'				=> __('Text for select button of locations/services/service providers dropdowns', 'wp-base'),
					'select_credits'			=> __('Text displayed above credit selection dropdown in Sell Credits', 'wp-base'),
					'select_date'				=> __('Text displayed above date selection field', 'wp-base'),
					'select_duration'			=> __('Text displayed above duration selection dropdown', 'wp-base'),
					'select_language'			=> __('Text displayed above language selection dropdown', 'wp-base'),
					'select_limit_exceeded'		=> __('Javascript message displayed when selected limit exceeded', 'wp-base'),
					'select_location'			=> __('Text displayed above locations dropdown', 'wp-base'),
					'select_provider'			=> __('Text displayed above service providers dropdown', 'wp-base'),
					'select_recurring'			=> __('Text displayed above number of repeats and repeat unit selection dropdowns', 'wp-base'),
					'select_seats'				=> __('Text displayed above pax/seats selection dropdown', 'wp-base'),
					'select_service'			=> __('Text displayed above services dropdown', 'wp-base'),
					'select_theme'				=> __('Title of Theme selection pulldown menu', 'wp-base'),
					'select_user'				=> __('Text displayed above users dropdown', 'wp-base'),
					'server_date_time'			=> __('Column header of Server Date/time for List of Bookings', 'wp-base' ),
					'server_day'				=> __('Column header of Server Day for List of Bookings', 'wp-base' ),
					'server_timezone'			=> __('Text for Server Timezone', 'wp-base' ),
					'service'					=> __('Column header of Service for List of Bookings', 'wp-base' ),
					'services'					=> __('Title for Services Given on account page', 'wp-base' ),
					'service_id'				=> __('Title for Service ID', 'wp-base' ),
					'service_name'				=> __('Title for Service in the confirmation form', 'wp-base' ),
					'service_names'				=> __('Plural of the Service title in the confirmation form', 'wp-base' ),
					'services_provided'			=> __('Title for Services Provided', 'wp-base' ),
					'service_required'			=> __('Message displayed when no service is selected, but selection of at least one is required', 'wp-base' ),
					'sort_order'				=> __('Column Header for Sort Order in Services List', 'wp-base' ),
					'spam'						=> __('Javascript message displayed after a too frequent booking attempt', 'wp-base'),
					'standby'					=> __('Text for status stand by', 'wp-base'),
					'state'						=> __('Title for State field in the confirmation form and user page','wp-base'),
					'state_placeholder'			=> __('Placeholder for State field in the confirmation form','wp-base'),
					'status'					=> __('Column header of Status for List of Bookings', 'wp-base' ),
					'submit_confirm'			=> __('Button text for confirming an appointment', 'wp-base'),
					'sunday'					=> __('Sunday',	'wp-base'),
					'sunday_initial'			=> __('Initial letter of Sunday', 'wp-base'),
					'sunday_short'				=> __('Short form of Sunday', 'wp-base'),
					'tba'						=> __('To be announced: Exact time or link of the booking is not determined yet.', 'wp-base'),
					'test'						=> __('Text for status test', 'wp-base'),
					'thursday'					=> __('Thursday',	'wp-base'),
					'thursday_initial'			=> __('Initial letter of Thursday', 'wp-base'),
					'thursday_short'			=> __('Short form of Thursday', 'wp-base'),
					'time'						=> __('Column header of Time in editing form', 'wp-base' ),
					'timezone_title'			=> __('Title of Time Zone selection pulldown menu', 'wp-base'),
					'too_late'					=> __('Javascript message displayed when client attempts to cancel/edit/add an appointment, but it is too late','wp-base'),
					'too_less'					=> __('Javascript message displayed when selected number of appointments is less than the permitted number. Keep %d which will be replaced by actual limit', 'wp-base'),
					'total_paid'				=> __('Column header of Total Paid amount for List of Bookings', 'wp-base'),
					'trial_service_limit'		=> __('Javascript error message when client attempts to select a time slot which would exceed trial limit. Keep LIMIT placeholder which will be replaced by set limit value.', 'wp-base'),
					'trial_service_worker_limit'=> __('Javascript error message when client attempts to select a time slot which would result in selecting same provider second time.', 'wp-base'),
					'tt_coupon'					=> __('Tooltip text displayed for price discounted by coupon','wp-base'),
					'tt_discounted_price'		=> __('Tooltip text displayed for price discounted by coupon or custom pricing','wp-base'),
					'tt_regular_price'			=> __('Tooltip text displayed for regular price when there is a discounted price','wp-base'),
					'tuesday'					=> __('Tuesday',	'wp-base'),
					'tuesday_initial'			=> __('Initial letter of Tuesday', 'wp-base'),
					'tuesday_short'				=> __('Short form of Tuesday', 'wp-base'),
					'unauthorised'				=> __('Message displayed after an unauthorised access', 'wp-base'),
					'unknown'					=> __('Text displayed when details of a setting is unknown. Possibly because the addon which made the setting has been disabled.', 'wp-base' ),
					'updated'					=> __('Javascript message after one or more records updated.', 'wp-base'),
					'updating'					=> __('Spinner panel message while an update is in progress','wp-base'),
					'username'					=> __('Placeholder value in login Username field', 'wp-base'),
					'use_server_timezone'		=> __('Text for selection of using server timezone instead of dedected client local timezone', 'wp-base'),
					'vendor'					=> __('Singular text for vendor', 'wp-base'),
					'vendors'					=> __('Plural text for vendor', 'wp-base'),
					'waiting'					=> __('Text for status in Waiting List', 'wp-base'),
					'waiting_button'			=> __('Confirmation form submit button text when booking will be added to Waiting List', 'wp-base'),
					'waiting_claim_failed'		=> __('Javascript message displayed when client claims an opening but slot is no more available.', 'wp-base'),
					'wednesday'					=> __('Wednesday',	'wp-base'),
					'wednesday_initial'			=> __('Initial letter of Wednesday', 'wp-base'),
					'wednesday_short'			=> __('Short form of Wednesday', 'wp-base'),
					'week'						=> __('Singular text for week','wp-base'),
					'weekday'					=> __('Text for bookings recurring every weekday','wp-base'),
					'weekend'					=> __('Text for bookings recurring every weekend','wp-base'),
					'weekly'					=> __('Text for bookings recurring every week','wp-base'),
					'weekly_title'				=> __('Title above the weekly calendar. LOCATION, SERVICE, WORKER, START, END will be replaced by their real values', 'wp-base'),
					'weeks'						=> __('Plural text for week','wp-base'),
					'worker'					=> __('Column header of Provider', 'wp-base' ),
					'wrong_value'				=> __('Javascript message displayed when submitted field is not acceptable. Keep %s which will be replaced by field name', 'wp-base'),
					'year'						=> __('Text for year','wp-base'),
					'yes'						=> __('Translation for Yes, e.g. to show a check box is checked','wp-base'),
					'your_price'				=> __('Label for Your Rate field in user profile. The same for hourly and fixed rates.','wp-base'),
					'zip'						=> __('Title for Postcode field in the confirmation form and user page','wp-base'),
					'zip_placeholder'			=> __('Placeholder for Postcode field in the confirmation form','wp-base'),
					'zoom'						=> __('Column header of Zoom for List of Bookings', 'wp-base' ),
					'zoom_id'					=> __('Meeting ID label in List of Bookings','wp-base'),
					'zoom_join'					=> __('Button label for Join to Meeting','wp-base'),
					'zoom_password'				=> __('Meeting Password label in List of Bookings','wp-base'),
					'zoom_start'				=> __('Button label for Start Meeting','wp-base'),
			);
		endif;

		if ( $key ) {
			if ( isset( $this->help_texts[ $key ] ) ) {
				return $this->help_texts[ $key ];
			} else {
				return '';
			}
		} else {
			return $this->help_texts;
		}
	}

	/**
     * Get a default text if $key is set or all default texts if $key is not set
	 * @return string or array
     */
	public function get_default_texts( $key = null ) {

		if ( null !== $this->default_texts ) {
			if ( null === $key ) {
				return $this->default_texts;
			}

			if ( isset( $this->default_texts[ $key ] ) ) {
				return $this->default_texts[ $key ];
			} else {
				return '';
			}
		}

		global $wp_locale;
		$method_exists = method_exists( $wp_locale, 'get_weekday' );

		$this->default_texts = array(
			'action'								=> __('Action', 'wp-base' ),
			'address'								=> __('Address','wp-base'),
			'address_placeholder'					=> '&nbsp;',
			'add_to_cart'							=> __('Add to Cart', 'wp-base'),
			'added_to_cart'							=> __( '%s has been added to your shopping cart.', 'wp-base' ),
			'admin'									=> __('Admin', 'wp-base'),
			'all'									=> __('All', 'wp-base'),
			'already_booked'						=> __('We are sorry, but this time slot is no longer available. Please refresh the page and try another time slot. Thank you.', 'wp-base'),
			'amount'								=> __('Amount', 'wp-base'),
			'aop_placeholder'						=> __('Address of the %d. guest', 'wp-base' ),
			'appointment_edited'					=> __('Your booking has been successfully changed.', 'wp-base' ),
			'appointment_received'					=> __('We have received your submission. Thanks!', 'wp-base' ),
			'app_id'								=> __('Booking ID', 'wp-base' ),
			'apps_to_change'						=> __('Change children', 'wp-base' ),
			'auto_assign_intro'						=> sprintf( __('Congratulations! You are a service provider of SITE_NAME now. Using this page you can set your profile, arrange working hours and define services you want to give.', 'wp-base'), wp_specialchars_decode(get_option('blogname'), ENT_QUOTES) ),
			'auto_assign_login'						=> __('Please login to be assigned as a service provider','wp-base'),
			'a_specialist'							=> __('A specialist', 'wp-base'),
			'available'								=> __('Available', 'wp-base'),
			'avatar'								=> __('Profile Picture', 'wp-base'),
			'balance'								=> __('Balance', 'wp-base'),
			'bio_page'								=> __('Bio Page','wp-base'),
			'biweekly'								=> __('Every other week','wp-base'),
			'blacklisted'							=> __('We are sorry, but the provided email cannot be accepted. Please contact website admin for details.', 'wp-base'),
			'booking'								=> __('Processing booking...','wp-base'),
			'booking_id'							=> __('Booking','wp-base'),
			'booking_closed'						=> __('Booking closed',	'wp-base'),
			'book_now_long'							=> __('Book Now for START', 'wp-base'),
			'book_now_short'						=> __('Book Now', 'wp-base'),
			'bp_3months'							=> __('3 Months Schedule', 'wp-base' ),
			'bp_4weeks'								=> __('4 Weeks Schedule', 'wp-base' ),
			'bp_annual'								=> __('Seasonal Schedules', 'wp-base'),
			'bp_bookings'							=> __('My Bookings', 'wp-base'),
			'bp_bookings_as_client'					=> __('My Bookings as Client', 'wp-base'),
			'bp_bookings_as_provider'				=> __('As Provider', 'wp-base'),
			'bp_book_me'							=> __('Book Me', 'wp-base'),
			'bp_commissions'						=> __('Commissions', 'wp-base'),
			'bp_credits'							=> __('Credit History', 'wp-base'),
			'bp_dashboard'							=> __('Dashboard', 'wp-base'),
			'bp_gcal'								=> __('Google Calendar', 'wp-base'),
			'bp_holidays'							=> __('Holidays', 'wp-base'),
			'bp_manage_bookings'					=> __('Manage Bookings', 'wp-base'),
			'bp_monthly'							=> __('Monthly Schedule', 'wp-base' ),
			'bp_ntf_booking_cancelled'				=> __('SERVICE booking #APP_ID supposed to start at DATE_TIME has been cancelled by the client CLIENT', 'wp-base' ),
			'bp_ntf_booking_cancelled_by_admin'		=> __('SERVICE booking #APP_ID supposed to start at DATE_TIME has been cancelled by admin', 'wp-base' ),
			'bp_ntf_booking_cancelled_by_worker'	=> __('SERVICE booking #APP_ID supposed to start at DATE_TIME has been cancelled by provider WORKER', 'wp-base' ),
			'bp_ntf_booking_completed'				=> __('SERVICE booking #APP_ID started at DATE_TIME completed', 'wp-base' ),
			'bp_ntf_booking_confirmed'				=> __('SERVICE booking #APP_ID for DATE_TIME has been confirmed', 'wp-base' ),
			'bp_ntf_booking_paid'					=> __('SERVICE booking #APP_ID for DATE_TIME has been paid', 'wp-base' ),
			'bp_ntf_booking_rescheduled'			=> __('Client CLIENT has rescheduled SERVICE booking #APP_ID to DATE_TIME', 'wp-base' ),
			'bp_ntf_booking_rescheduled_by_admin'	=> __('Admin has rescheduled SERVICE booking #APP_ID to DATE_TIME', 'wp-base' ),
			'bp_ntf_booking_rescheduled_by_worker'	=> __('Provider WORKER has rescheduled SERVICE booking #APP_ID to DATE_TIME', 'wp-base' ),
			'bp_ntf_booking_running'				=> __('SERVICE booking #APP_ID started. Estimated finish time: END_DATE_TIME', 'wp-base' ),
			'bp_ntf_new_booking'					=> __('CLIENT made a SERVICE booking #APP_ID for DATE_TIME. It will be served by WORKER', 'wp-base' ),
			'bp_ntf_new_booking_by_admin'			=> __('Admin created a new SERVICE booking #APP_ID scheduled for DATE_TIME', 'wp-base' ),
			'bp_ntf_new_booking_by_worker'			=> __('WORKER created a new SERVICE booking #APP_ID for client CLIENT for DATE_TIME', 'wp-base' ),
			'bp_ntf_worker_changed'					=> __('WORKER has been assigned as service provider for SERVICE booking #APP_ID scheduled for DATE_TIME', 'wp-base' ),
			'bp_payments'							=> __('Payment History', 'wp-base'),
			'bp_schedules'							=> __('Other Schedules', 'wp-base'),
			'bp_services'							=> __('Services', 'wp-base' ),
			'bp_settings'							=> __('Settings', 'wp-base'),
			'bp_title'								=> __('Bookings', 'wp-base'),
			'bp_use_book_me'						=> __('Add a Book Me tab', 'wp-base'),
			'bp_weekly'								=> __('Weekly Schedule', 'wp-base' ),
			'bp_wh'									=> __('Working Hours', 'wp-base'),
			'busy'						=> __('Busy', 'wp-base'),
			'calculating'				=> __('Calculating','wp-base'),
			'cancel'					=> __('Cancel', 'wp-base' ),
			'cancelled'					=> __('Selected booking cancelled.','wp-base'),
			'cancel_app_confirm'		=> __('Are you sure to cancel the selected booking?','wp-base'),
			'cancel_button'				=> __('Cancel','wp-base'),
			'cancel_cart'				=> __('Clear', 'wp-base'),
			'cancel_confirm_no'			=> __('No, I want to continue', 'wp-base' ),
			'cancel_confirm_text'		=> __('Are you sure to cancel current process?', 'wp-base' ),
			'cancel_confirm_yes'		=> __('Yes, I want to cancel', 'wp-base' ),
			'cancel_disabled'			=> __('Cancellation of appointments is disabled. Please contact website admin.','wp-base'),
			'capacity'					=> __('Capacity', 'wp-base'),
			'cart'						=> __('In Cart', 'wp-base'),
			'category'					=> __('Category', 'wp-base' ),
			'cc_address1'				=> __('Address Line 1', 'wp-base'),
			'cc_address2'				=> __('Address Line 2', 'wp-base'),
			'cc_city'					=> __('City', 'wp-base'),
			'cc_country'				=> __('Country', 'wp-base'),
			'cc_cvv'					=> __('Security Code', 'wp-base'),
			'cc_declined'				=> __('We are sorry, but your bank did not approve your credit card. Please try with another card.', 'wp-base'),
			'cc_email'					=> __('Email', 'wp-base'),
			'cc_error'					=> __('There was a problem with your submission. ERROR_DETAIL', 'wp-base'),
			'cc_expiry'					=> __('Expiration Date', 'wp-base'),
			'cc_form_legend'			=> __('Please enter your credit card details below and confirm', 'wp-base' ),
			'cc_name'					=> __('Name on the Card', 'wp-base'),
			'cc_number'					=> __('Credit Card Number', 'wp-base'),
			'cc_phone'					=> __('Phone', 'wp-base'),
			'cc_state'					=> __('State/Province/Region', 'wp-base'),
			'cc_zip'					=> __('Postcode', 'wp-base'),
			'checkout'					=> __('Processing checkout...','wp-base'),
			'checkout_button'			=> __('Checkout', 'wp-base'),
			'choose_image'				=> __('Choose Image', 'wp-base'),
			'checkout_button_tip'		=> __('Click to apply for the appointment', 'wp-base' ),
			'city'						=> __('City','wp-base'),
			'city_placeholder'			=> '&nbsp;',
			'clear'						=> __('Clear', 'wp-base' ),
			'click_to_book'				=> __('Click to add a booking','wp-base'),
			'click_to_remove'			=> __('Click to remove','wp-base'),
			'click_to_select_date'		=> __('Click to pick date','wp-base'),
			'client'					=> __('Client', 'wp-base' ),
			'close'						=> __('Close', 'wp-base' ),
			'completed'					=> __('Completed', 'wp-base'),
			'confirm'					=> __('Confirm', 'wp-base' ),
			'confirm_button'			=> __('Confirm','wp-base'),
			'confirm_cancel'			=> __( 'Are you sure to cancel selected appointment?', 'wp-base' ),
			'confirm_confirm'			=> __( 'Are you sure to confirm selected appointment?', 'wp-base' ),
			'confirm_delete'			=> __( 'Are you sure to delete selected record(s)?', 'wp-base' ),
			'confirm_disabled'			=> __('This action is disabled. Please contact website admin.','wp-base'),
			'confirmation_title'		=> __('Please fill in the form and confirm:', 'wp-base' ),
			'confirmed'					=> __('Confirmed', 'wp-base'),
			'confirmed_message'			=> __('Selected booking confirmed', 'wp-base'),
			'conf_countdown_title'		=> __('We are holding your selection for:', 'wp-base'),
			'connection_error'			=> __('A connection error occurred.','wp-base'),
			'continue_button'			=> __('Add another time slot', 'wp-base' ),
			'countdown_title'			=> __('Your next appointment', 'wp-base'),
			'country'					=> __('Country','wp-base'),
			'country_placeholder'		=> '&nbsp;',
			'coupon'					=> __('Discount code', 'wp-base' ),
			'coupon_placeholder'		=> __('Use discount coupon here', 'wp-base' ),
			'created'					=> __('Created', 'wp-base' ),
			'created_by'				=> __('Created by', 'wp-base' ),
			'credit'					=> __('Credit', 'wp-base' ),
			'credit_card'				=> __('Credit Card', 'wp-base' ),
			'credits'					=> __('Credits', 'wp-base' ),
			'daily'						=> __('Daily','wp-base'),
			'dashboard'					=> __('Booking System Dashboard','wp-base'),
			'date'						=> __('Date', 'wp-base' ),
			'date_time'					=> __('Starts', 'wp-base' ),
			'day'						=> __('day','wp-base'),
			'days'						=> __('days','wp-base'),
			'day_of_week'				=> __('Day', 'wp-base' ),
			'default'					=> __('Default', 'wp-base'),
			'delete'					=> __('Delete Permanently', 'wp-base'),
			'deleted'					=> __('Selected record(s) deleted.', 'wp-base'),
			'deposit'					=> __('Deposit', 'wp-base'),
			'deposit_paid'				=> __('Deposit Paid:', 'wp-base'),
			'description_page'			=> __('Description Page', 'wp-base' ),
			'details'					=> __('Details', 'wp-base' ),
			'display_name'				=> __('Display Name','wp-base'),
			'done'						=> __('Done...','wp-base'),
			'down_payment'				=> __('Amount', 'wp-base'),
			'due_payment'				=> __('Due Payment:', 'wp-base'),
			'duration'					=> __('Duration', 'wp-base'),
			'edd_payment'				=> __('EDD', 'wp-base' ),
			'edit'						=> __('Edit', 'wp-base' ),
			'edit_app_confirm'			=> __('You are about to edit an existing appointment. Click OK to continue.','wp-base'),
			'edit_button'				=> __('Edit','wp-base'),
			'edit_disabled'				=> __('Editing of appointments is disabled. Please contact website admin.','wp-base'),
			'edit_saved'				=> __('Booking successfully changed.', 'wp-base' ),
			'edit_too_early'			=> __('New time of the booking is too close. Please select a time later than %s.','wp-base'),
			'email'						=> __('Email','wp-base'),
			'email_placeholder'			=> '&nbsp;',
			'email_unmatch'				=> __('Submitted email does not belong to you','wp-base'),
			'end_date_time'				=> __('Ends', 'wp-base' ),
			'eod'						=> __('Every other day','wp-base'),
			'eod_except_sunday'			=> __('EOD except Sunday','wp-base'),
			'error'						=> __('Something went wrong. Please try again, if applicable, after refreshing the page. If problem persists, contact website admin.', 'wp-base'),
			'error_short'				=> __('Error', 'wp-base'),
			'event_capacity'			=> __('Capacity','wp-base'),
			'event_capacity_desc'		=> __('Total event capacity. Leave empty for no limit','wp-base'),
			'event_closed_from_pax'		=> __('Closed from Pax','wp-base'),
			'event_closed_from_pax_desc'=> __('When group bookings is enabled, event will be regarded full when this number is reached','wp-base'),
			'event_enabled'				=> __('Group Bookings Enable','wp-base'),
			'event_enabled_desc'		=> __('Tick if more than one pax can be selected/sold per booking','wp-base'),
			'event_max'					=> __('Maximum Pax','wp-base'),
			'event_max_desc'			=> __('When group bookings is enabled, maximum pax that can be selected per booking','wp-base'),
			'event_min'					=> __('Minimum Pax','wp-base'),
			'event_min_desc'			=> __('When group bookings is enabled, minimum pax that can be selected per booking','wp-base'),
			'event_price'				=> __('Price','wp-base'),
			'event_price_desc'			=> __('Unit price. Leave empty if event is free','wp-base'),
			'event_title'				=> __('Book now for EVENT - START_END', 'wp-base'),
			'excerpt_more'				=> __('More information <span class="meta-nav">&rarr;</span>', 'wp-base' ),
			'expired_card'		 		=> __('Card is no longer valid or has expired', 'wp-base' ),
			'export_csv'				=> __('Export bookings as CSV file','wp-base'),
			'extra'						=> __('Extra','wp-base'),
			'full_name'					=> __('Full Name','wp-base'),
			'first_name'				=> __('First Name','wp-base'),
			'first_name_placeholder'	=> '&nbsp;',
			'friday'					=> $method_exists ? $wp_locale->get_weekday( 5 ) : 'Friday',
			'friday_initial'			=> $method_exists ? $wp_locale->get_weekday_initial( $wp_locale->get_weekday( 5 ) ) : 'F',
			'friday_short'				=> $method_exists ? $wp_locale->get_weekday_abbrev( $wp_locale->get_weekday( 5 ) ) : 'Fr',
			'gcal'						=> __('GCal','wp-base'),
			'gcal_button'				=> __('Click to submit this appointment to your Google Calendar account','wp-base'),
			'gdpr_udf_title'			=> __('Additional User Info for Booking','wp-base'),
			'gdpr_userdata_title'		=> __('User Info for Booking','wp-base'),
			'hangout'					=> __('Google Meet','wp-base'),
			'hangout_join'				=> __('Join','wp-base'),
			'hold'						=> __('Temporary', 'wp-base'),
			'host'						=> __('Host', 'wp-base'),
			'hour'						=> __('hour','wp-base'),
			'hours'						=> __('hours','wp-base'),
			'hour_short'				=> __('h','wp-base'),
			'image'						=> __('Image', 'wp-base'),
			'info'						=> __('Showing page _PAGE_ of _PAGES_', 'wp-base'),
			'internal'					=> __('Internal', 'wp-base'),
			'invalid_booking' 			=> __('There is something wrong about this booking. Please contact website admin.', 'wp-base' ),
			'invalid_cardholder' 		=> __('Invalid cardholder', 'wp-base' ),
			'invalid_cc_number'	 		=> __('Please enter a valid Credit Card Number.', 'wp-base' ),
			'invalid_cvc'				=> __('Please enter a valid Card Security Code', 'wp-base' ),
			'invalid_email'				=> __('Please enter a valid email', 'wp-base' ),
			'invalid_expiration'		=> __('Please choose a valid Expiration Date.', 'wp-base' ),
			'jitsi'						=> __('Jitsi', 'wp-base' ),
			'jitsi_already_started'		=> __('This meeting has already started.', 'wp-base'),
			'jitsi_called'				=> __('Meeting has been started by the host and you are being called. Please confirm to join the meeting.','wp-base'),
			'jitsi_meeting_note'		=> __('&nbsp;','wp-base'),
			'jitsi_meeting_note_host'	=> __('You can start by clicking START_MEETING link immediately. If you do so, participant will be able to attend to the meeting even before meeting time arrives.','wp-base'),
			'jitsi_meeting_started'		=> __('Meeting already started. Please click MEETING_ROOM to join.','wp-base'),
			'jitsi_no_meeting'			=> __('You don\'t have any upcoming or running meetings.','wp-base'),
			'jitsi_start_meeting'		=> __('Start Meeting','wp-base'),
			'jitsi_waiting_note'		=> __('When countdown expires, you will be redirected to the Meeting Room. Please keep this page open until then.','wp-base'),
			'jitsi_waiting_note_host'	=> __('You are the host of this meeting. You can click MEETING_ROOM to go to the Meeting Room, or wait here. If you prefer to stay, when countdown expires, you will be redirected to the Meeting Room. Please keep this page open until then.','wp-base'),
			'lasts'						=> __('Lasts', 'wp-base' ),
			'last_name'					=> __('Last Name','wp-base'),
			'last_name_placeholder'		=> '&nbsp;',
			'length_menu'				=> __('Display _MENU_ records', 'wp-base'),
			'limit_exceeded'			=> __('We are sorry, but number of appointments limit (%d) has been reached.', 'wp-base'),
			'loading'					=> __('Loading...','wp-base'),
			'location'					=> __('Location', 'wp-base' ),
			'location_address'			=> __('Address', 'wp-base' ),
			'location_name'				=> __('Location name', 'wp-base' ),
			'location_names'			=> __('Location names', 'wp-base' ),
			'location_note'				=> __('Location Note', 'wp-base' ),
			'logged_in'					=> __('You are now logged in', 'wp-base'),
			'logged_message'			=> __('Click a free day to apply for an appointment.', 'wp-base'),
			'logging_in'				=> __('Preparing login...','wp-base'),
			'login'						=> __('Login', 'wp-base'),
			'login_for_cancel'			=> __('Please login in order to cancel the appointment','wp-base'),
			'login_for_confirm'			=> __('Please login in order to confirm the booking','wp-base'),
			'login_for_edit'			=> __('Please login in order to edit the booking','wp-base'),
			'login_message'				=> __('Click here to login:', 'wp-base' ),
			'login_required'			=> __('It looks like you have previously registered to our website. Please login to proceed.', 'wp-base' ),
			'login_with_facebook'		=> __('Login with Facebook', 'wp-base'),
			'login_with_google'			=> __('Login with Google+', 'wp-base'),
			'login_with_twitter'		=> __('Login with Twitter', 'wp-base'),
			'login_with_wp'				=> __('Login with WordPress', 'wp-base'),
			'maybe_edited'				=> __('This record may have been edited by another user and current value may not be valid any more.','wp-base'),
			'method'					=> __('Method','wp-base'),
			'minute'					=> __('minute','wp-base'),
			'minutes'					=> __('minutes','wp-base'),
			'min_short'					=> __('min','wp-base'),
			'missing_extra'				=> __('Please select at least one option from the list','wp-base'),
			'missing_field'				=> __('Please fill in the required field','wp-base'),
			'missing_terms_check'		=> __('Please accept Terms and Conditions','wp-base'),
			'monday'					=> $method_exists ? $wp_locale->get_weekday( 1 ) : 'Monday',
			'monday_initial'			=> $method_exists ? $wp_locale->get_weekday_initial( $wp_locale->get_weekday( 1 ) ) : 'M',
			'monday_short'				=> $method_exists ? $wp_locale->get_weekday_abbrev( $wp_locale->get_weekday( 1 ) ) : 'Mo',
			'month'						=> __('month','wp-base'),
			'monthly'					=> __('Monthly','wp-base'),
			'monthly_title'				=> __('SERVICE - START', 'wp-base'),
			'months'					=> __('months','wp-base'),
			'mop_placeholder'			=> __('email of the %d. guest', 'wp-base' ),
			'mv_activate'				=> __('You have successfully created your account! To begin using this site you will need to activate your account via the email we have just sent to your address.', 'wp-base' ),
			'mv_afternoon'				=> __('Afternoon','wp-base'),
			'mv_already_active'			=> __('The user is already active.', 'wp-base'),
			'mv_already_applied'		=> __('We have already received your application at %s. Please give us some time to evaluate it. If you have questions, please use our contact form to send us a message.', 'wp-base' ),
			'mv_already_vendor'			=> __('You are already a registered vendor. You don\'t need to apply again. If you need assistance please contact admin.','wp-base'),
			'mv_apply_now'				=> __('Apply Now','wp-base'),
			'mv_become_vendor'			=> __('Become a Vendor','wp-base'),
			'mv_become_vendor_desc'		=> __('By clicking this button, you can apply to be a vendor on our website.','wp-base'),
			'mv_become_vendor_confirm'	=> __('Do you want to apply to become a vendor?','wp-base'),
			'mv_bio'					=> __('About You','wp-base'),
			'mv_bio_page'				=> __('Bio Page','wp-base'),
			'mv_bio_excerpt'			=> __('About You (Summary)','wp-base'),
			'mv_cannot_create_user'		=> __('Could not create user', 'wp-base'),
			'mv_commission_amount'		=> __('Commission', 'wp-base'),
			'mv_commission_id'			=> __('Commission ID', 'wp-base'),
			'mv_commission_rate'		=> __('Rate', 'wp-base'),
			'mv_commission_status'		=> __('Status', 'wp-base'),
			'mv_days_of_the_week'		=> __('Days of the week','wp-base'),
			'mv_declined'				=> __('Please contact admin for assistance','wp-base'),
			'mv_email_failed'			=> __('Activation email could not be sent.', 'wp-base'),
			'mv_email_banned'			=> __('Sorry, that email address is not allowed!', 'wp-base'),
			'mv_email_in_use'			=> __('Sorry, that email address is already used!', 'wp-base'),
			'mv_email_invalid'			=> __('Please check your email address.', 'wp-base'),
			'mv_evening'				=> __('Evening','wp-base'),
			'mv_invalid_key'			=> __('Invalid activation key. If you used activation key before, please wait for our response email.', 'wp-base'),
			'mv_is_vendor'				=> __('You are already a vendor','wp-base'),
			'mv_late_afternoon'			=> __('Late afternoon','wp-base'),
			'mv_late_evening'			=> __('Late Evening','wp-base'),
			'mv_late_morning'			=> __('Late morning','wp-base'),
			'mv_late_night'				=> __('Late Night','wp-base'),
			'mv_login'					=> __('You have successfully created your account! Please log in using the username and password you have just created. Click here to LOGIN_PAGE', 'wp-base' ),
			'mv_morning'				=> __('Morning','wp-base'),
			'mv_night'					=> __('Night','wp-base'),
			'mv_no_privacy_policy'		=> __('You must indicate that you have read and agreed to the Privacy Policy.', 'wp-base'),
			'mv_override_prices'		=> __('Override Prices','wp-base'),
			'mv_override_prices_desc'	=> __('You can override prices of each service here. If you leave a price field empty, its regular price will be used. If you set a price as zero, it will be free.','wp-base'),
			'mv_page_not_found'			=> __('Your personal page could not be found. Please contact admin.', 'wp-base'),
			'mv_password_empty'			=> __('Please make sure you enter your password twice', 'wp-base'),
			'mv_password_no_match'		=> __('The passwords you entered do not match', 'wp-base'),
			'mv_required_field'			=> __('%s is a required field', 'wp-base'),
			'mv_tags'					=> __('Tags','wp-base'),
			'mv_signup'					=> __('Complete Sign Up', 'wp-base'),
			'mv_status_pending'			=> __('Pending', 'wp-base'),
			'mv_tagline_text'			=> __('Tagline','wp-base'),
			'mv_time_of_the_day'		=> __('Time of the day','wp-base'),
			'mv_title'					=> __('Title','wp-base'),
			'mv_upload_photo'			=> __('Upload Photo','wp-base'),
			'mv_username_empty'			=> __('Please enter a username', 'wp-base'),
			'mv_username_exists'		=> __('Sorry, that username already exists!', 'wp-base'),
			'mv_username_illegal'		=> __('That username is not allowed', 'wp-base'),
			'mv_username_invalid'		=> __('Usernames can contain only letters, numbers, ., -, and @', 'wp-base'),
			'mv_username_numeric'		=> __('Sorry, usernames must have letters too!', 'wp-base'),
			'mv_username_short'			=> __('Username must be at least 4 characters', 'wp-base'),
			'mv_username_underscore'	=> __('Sorry, usernames may not contain the character "_"!', 'wp-base'),
			'mv_video'					=> __('Video','wp-base'),
			'mv_wait'					=> __('Your application is successful! Please wait until you receive an email about the result of your submission.', 'wp-base' ),
			'name'						=> __('Name','wp-base'),
			'name_placeholder'			=> '&nbsp;',
			'next'						=> __('Next','wp-base'),
			'next_day'					=> __('Ends next day', 'wp-base' ),
			'next_month'				=> __('Next Month','wp-base'),
			'next_months'				=> __('Next Months','wp-base'),
			'next_week'					=> __('Next Week','wp-base'),
			'next_weeks'				=> __('Next Weeks','wp-base'),
			'no'						=> __('No','wp-base'),
			'nof_bookings'				=> __('Bookings','wp-base'),
			'nof_jobs_cancelled'		=> __('Cancelled Jobs','wp-base'),
			'nof_jobs_completed'		=> __('Completed Jobs','wp-base'),
			'nof_jobs_remaining'		=> __('Remaining Jobs','wp-base'),
			'nof_jobs_total'			=> __('Total Jobs','wp-base'),
			'nop_placeholder'			=> __('Name of the %d. guest', 'wp-base'),
			'noshow'					=> __('No Show', 'wp-base'),
			'not_enough_credits'		=> __('Not enough credits', 'wp-base'),
			'note'						=> __('Note','wp-base'),
			'nothing_changed'			=> __('You did not make any changes.', 'wp-base'),
			'notice'					=> __('Notice', 'wp-base'),
			'not_available'				=> __('This time slot has been already booked. Please pick another time.', 'wp-base'),
			'not_defined'				=> __('Not defined', 'wp-base'),
			'not_enough_capacity'		=> __('We are sorry, but we do not have enough capacity to fulfil the request at the moment. Please refresh the page and try other time slots. Thank you.', 'wp-base'),
			'not_logged_message'		=> __('You need to login to make an appointment. Click here to login: LOGIN_PAGE OR click here to register: REGISTRATION_PAGE', 'wp-base'),
			'not_owner'					=> __('Action failed, because our records could not confirm that this booking was made by you in the first place.', 'wp-base'),
			'not_possible'				=> __('Action or selection is not possible.', 'wp-base'),
			'not_published'				=> __('Booking is not possible at the moment.', 'wp-base'),
			'not_working'				=> __('Sorry, service or provider is not available for the selected date/time. Please pick another time.', 'wp-base'),
			'no_appointments'			=> __('No appointments','wp-base'),
			'no_free_time_slots'		=> __('Not available', 'wp-base'),
			'no_preference'				=> __('No preference', 'wp-base'),
			'no_repeat'					=> __('No repeat','wp-base'),
			'no_service'				=> __('No services', 'wp-base'),
			'no_result'					=> __('No result', 'wp-base'),
			'notpossible'				=> __('Not available', 'wp-base'),
			'order'						=> __('Order No', 'wp-base'),
			'our_staff'					=> __('Staff', 'wp-base'),
			'owner'						=> __('Owner', 'wp-base'),
			'padding_after'				=> __('Padding After', 'wp-base'),
			'padding_before'			=> __('Padding Before', 'wp-base'),
			'paid'						=> __('Paid', 'wp-base'),
			'paid_up'					=> __('All paid up.', 'wp-base'),
			'participants'				=> __('Participants', 'wp-base' ),
			'participant_title'			=> __('%d. Guest - %s','wp-base'),
			'partly_busy'				=> __('Partly Busy','wp-base'),
			'password'					=> __('Password', 'wp-base'),
			'password_confirm'			=> __('Repeat Password', 'wp-base'),
			'past_date'					=> __('You cannot select a past date/time.', 'wp-base'),
			'pax'						=> __('Pax', 'wp-base' ),
			'payment_method_error'		=> __('Payment method is not selected or not available', 'wp-base'),
			'pay'						=> __('Make Payment', 'wp-base' ),
			'paypal'					=> __('PayPal', 'wp-base' ),
			'paypal_express_button'		=> __('Confirm Payment', 'wp-base'),
			'paypal_express_note'		=> __('Please confirm payment of <strong>AMOUNT</strong> for this booking that will be collected from your <strong>EMAIL</strong> PayPal account.', 'wp-base'),
			'pay_now'					=> __('Pay AMOUNT', 'wp-base' ),
			'pay_with'					=> __('Pay with','wp-base'),
			'payment_id'				=> __('Payment ID','wp-base'),
			'pdf'						=> __('Pdf','wp-base'),
			'pdf_download'				=> __('Download','wp-base'),
			'pdf_tooltip'				=> __('Confirmation in pdf form will be downloaded to your PC.','wp-base'),
			'pending'					=> __('Pending', 'wp-base'),
			'pending_approval'			=> __('Pending Approval', 'wp-base'),
			'pending_payment'			=> __('Pending Payment', 'wp-base'),
			'phone'						=> __('Phone','wp-base'),
			'phone_placeholder'			=> '&nbsp;',
			'please_wait'				=> __('Please Wait...', 'wp-base' ),
			'pop_placeholder'			=> __('Phone of the %d. guest', 'wp-base' ),
			'preferred_language'		=> __('Preferred Language', 'wp-base'),
			'preparing_form'			=> __('Preparing form...','wp-base'),
			'preparing_timetable'		=> __('Checking...','wp-base'),
			'previous'					=> __('Previous','wp-base'),
			'previous_month'			=> __('Previous Month','wp-base'),
			'previous_months'			=> __('Previous Months','wp-base'),
			'previous_week'				=> __('Previous Week','wp-base'),
			'previous_weeks'			=> __('Previous Weeks','wp-base'),
			'price'						=> __('Price', 'wp-base' ),
			'price_mismatch'			=> __('We are sorry, but this price is no more valid.','wp-base'),
			'price_mismatch_lower'		=> __('We are glad to inform you that we can offer even a better price now. Please review the new price and confirm again.','wp-base'),
			'proceed'					=> __('Click OK to proceed.', 'wp-base'),
			'processing' 				=> __('Processing...', 'wp-base' ),
			'provider'					=> __('Provider', 'wp-base' ),
			'provider_name'				=> __('Specialist', 'wp-base' ),
			'provider_names'			=> __('Specialists', 'wp-base' ),
			'quota'						=> __('Sorry, but you have reached the booking quota. No additional bookings are allowed.', 'wp-base'),
			'reading'					=> __('Reading data...','wp-base'),
			'redirect'					=> __('Login required to make an appointment. Now you will be redirected to login page.', 'wp-base' ),
			'reference'					=> __('Reference','wp-base'),
			'refreshing'				=> __('Refreshing','wp-base'),
			'register'					=> __('Register', 'wp-base'),
			'remember'					=> __('Remember me','wp-base'),
			'removed'					=> __('Removed!','wp-base'),
			'removed'					=> __('Removed', 'wp-base'),
			'required'					=> __('Required', 'wp-base' ),
			'reserved'					=> __('Reserved by GCal', 'wp-base'),
			'room'						=> __('Room ID', 'wp-base'),
			'running'					=> __('In Progress', 'wp-base'),
			'saturday'					=> $method_exists ? $wp_locale->get_weekday( 6 ) : 'Saturday',
			'saturday_initial'			=> $method_exists ? $wp_locale->get_weekday_initial( $wp_locale->get_weekday( 6 ) ) : 'S',
			'saturday_short'			=> $method_exists ? $wp_locale->get_weekday_abbrev( $wp_locale->get_weekday( 6 ) ) : 'Sa',
			'saved'						=> __('Settings saved.', 'wp-base'),
			'save_button'				=> __('Save changes', 'wp-base' ),
			'save_error'				=> __('Appointment could not be saved. Please contact website admin.', 'wp-base'),
			'saving'					=> __('Saving...','wp-base'),
			'search'					=> __('Search', 'wp-base'),
			'seats'						=> __('Pax', 'wp-base'),
			'seats_left'				=> __('Seats Left', 'wp-base' ),
			'seats_total'				=> __('Seats Total', 'wp-base' ),
			'seats_total_left'			=> __('Seats Total/Left', 'wp-base' ),
			'second'					=> __('second','wp-base'),
			'seconds'					=> __('seconds','wp-base'),
			'select'					=> __('Please select', 'wp-base'),
			'selected'					=> __('Selected', 'wp-base'),
			'select_button'				=> __('Refresh', 'wp-base'),
			'select_credits'			=> __('Select credits to buy', 'wp-base'),
			'select_date'				=> __('Date', 'wp-base'),
			'select_duration'			=> __('Duration', 'wp-base'),
			'select_language'			=> __('Select Language', 'wp-base'),
			'select_limit_exceeded'		=> __('Selection limit exceeded', 'wp-base'),
			'select_location'			=> __('Location', 'wp-base'),
			'select_provider'			=> __('Specialist', 'wp-base'),
			'select_recurring'			=> __('Repeat', 'wp-base'),
			'select_repeat_count'		=> __('Repeat Number', 'wp-base'),
			'select_repeat_unit'		=> __('Occurence', 'wp-base'),
			'select_seats'				=> __('Number of Guests', 'wp-base'),
			'select_service'			=> __('Service', 'wp-base'),
			'select_theme'				=> __('Theme', 'wp-base'),
			'select_user'				=> __('User', 'wp-base'),
			'server_date_time'			=> __('Server Date/time', 'wp-base' ),
			'server_day'				=> __('Server Day', 'wp-base' ),
			'server_timezone'			=> __('Server Timezone', 'wp-base'),
			'service'					=> __('Service', 'wp-base' ),
			'services'					=> __('Services', 'wp-base' ),
			'service_id'				=> __('Service ID', 'wp-base' ),
			'service_name'				=> __('Service name', 'wp-base' ),
			'service_names'				=> __('Service names', 'wp-base' ),
			'service_required'			=> __('Please select at least one service', 'wp-base' ),
			'services_provided'			=> __('Services Provided', 'wp-base' ),
			'sort_order'				=> __('Sort Order', 'wp-base' ),
			'spam'						=> __('You have already applied for an appointment. Please wait until you hear from us.', 'wp-base'),
			'standby'					=> __('Stand by', 'wp-base'),
			'state'						=> __('State','wp-base'),
			'state_placeholder'			=> '&nbsp;',
			'status'					=> __('Status', 'wp-base' ),
			'submit_confirm'			=> __('Submit', 'wp-base'),
			'sunday'					=> $method_exists ? $wp_locale->get_weekday( 0 ) : 'Sunday',
			'sunday_initial'			=> $method_exists ? $wp_locale->get_weekday_initial( $wp_locale->get_weekday( 0 ) ) : 'S',
			'sunday_short'				=> $method_exists ? $wp_locale->get_weekday_abbrev( $wp_locale->get_weekday( 0 ) ) : 'Su',
			'tba'						=> __('To be announced', 'wp-base'),
			'test'						=> __('Test', 'wp-base'),
			'thursday'					=> $method_exists ? $wp_locale->get_weekday( 4 ) : 'Thursday',
			'thursday_initial'			=> $method_exists ? $wp_locale->get_weekday_initial( $wp_locale->get_weekday( 4 ) ) : 'T',
			'thursday_short'			=> $method_exists ? $wp_locale->get_weekday_abbrev( $wp_locale->get_weekday( 4 ) ) : 'Th',
			'time'						=> __('Time', 'wp-base' ),
			'timezone_title'			=> __('Select your timezone', 'wp-base'),
			'too_late'					=> __('We are sorry, but it is too late for this action.','wp-base'),
			'too_less'					=> __('You should select at least %d time slot to proceed.', 'wp-base'),
			'total_paid'				=> __('Paid', 'wp-base'),
			'trial_service_limit'		=> __('You have exceeded trial services limit LIMIT', 'wp-base'),
			'trial_service_worker_limit'=> __('You can take only one trial service from the same person', 'wp-base'),
			'tt_coupon'					=> __('Price after coupon applied','wp-base'),
			'tt_discounted_price'		=> __('Special price for you','wp-base'),
			'tt_regular_price'			=> __('Regular price','wp-base'),
			'tuesday'					=> $method_exists ? $wp_locale->get_weekday( 2 ) : 'Tuesday',
			'tuesday_initial'			=> $method_exists ? $wp_locale->get_weekday_initial( $wp_locale->get_weekday( 2 ) ) : 'T',
			'tuesday_short'				=> $method_exists ? $wp_locale->get_weekday_abbrev( $wp_locale->get_weekday( 2 ) ) : 'Tu',
			'unauthorised'				=> __('Authorization check failed. Try to refresh page. If you are logged in, try to log out and login again.', 'wp-base'),
			'unknown'					=> __('Unknown', 'wp-base'),
			'updated'					=> __('Selected record(s) updated.', 'wp-base'),
			'updating'					=> __('Updating...','wp-base'),
			'username'					=> __('Username', 'wp-base'),
			'use_server_timezone'		=> __('Use Server Timezone', 'wp-base'),
			'vendor'					=> __('Vendor', 'wp-base'),
			'vendors'					=> __('Vendors', 'wp-base'),
			'waiting'					=> __('Waiting List', 'wp-base'),
			'waiting_button'			=> __('Add to Waiting List', 'wp-base'),
			'waiting_claim_failed'		=> __('Unfortunately this slot has been reserved. Please wait for another notification message from us.', 'wp-base'),
			'wednesday'					=> $method_exists ? $wp_locale->get_weekday( 3 ) : 'Wednesday',
			'wednesday_initial'			=> $method_exists ? $wp_locale->get_weekday_initial( $wp_locale->get_weekday( 3 ) ) : 'W',
			'wednesday_short'			=> $method_exists ? $wp_locale->get_weekday_abbrev( $wp_locale->get_weekday( 3 ) ) : 'We',
			'week'						=> __('week','wp-base'),
			'weekday'					=> __('Weekday','wp-base'),
			'weekend'					=> __('Weekend','wp-base'),
			'weekly'					=> __('Weekly','wp-base'),
			'weekly_title'				=> __('SERVICE - START_END', 'wp-base'),
			'weeks'						=> __('weeks','wp-base'),
			'worker'					=> __('Provider', 'wp-base' ),
			'wrong_value'				=> __('Please check submitted %s entry!', 'wp-base'),
			'year'						=> __('Year','wp-base'),
			'yes'						=> __('Yes','wp-base'),
			'your_price'				=> __('Your Rate','wp-base'),
			'zip'						=> __('Postcode','wp-base'),
			'zip_placeholder'			=> '&nbsp;',
			'zoom'						=> __('Zoom', 'wp-base' ),
			'zoom_id'					=> __('ID','wp-base'),
			'zoom_join'					=> __('Join','wp-base'),
			'zoom_password'				=> __('Password','wp-base'),
			'zoom_start'				=> __('Start','wp-base'),

		);

		$this->default_texts = apply_filters( 'app_default_texts', $this->default_texts );

		if ( null === $key ) {
			return $this->default_texts;
		}

		if ( isset( $this->default_texts[ $key ] ) ) {
			return $this->default_texts[ $key ];
		} else {
			return '';
		}
	}

	/**
	 * Add "Custom Texts" tab
	 */
	public function add_tab( $tabs ) {
		$tabs['custom_texts'] = __('Custom Texts', 'wp-base');
		return $tabs;
	}

	/**
	 * Save settings
	 */
	public function save_settings() {

		if ( 'save_custom_texts' != $_POST["action_app"] ) {
			return;
		}

		$saved = false;

		if ( isset( $_POST['app_custom_texts'] ) && is_array( $_POST['app_custom_texts'] ) ) {
			foreach ( $_POST['app_custom_texts'] as $key => $custom_text ) {
				if ( isset( $_POST['app_custom_texts'][ $key ] ) )
					$this->texts[ $key ] = trim( sanitize_text_field( $custom_text ) ); // Immediately update cache
			}

			if ( update_option( 'wp_base_texts', $this->texts, false ) ) {
				$saved = true;
			}
		}

		if ( isset( $_POST['replace_texts'] ) ) {
			if ( update_option( 'wp_base_replace_texts', trim($_POST['replace_texts']) ) ) {
				$saved = true;
			}
		}

		do_action( 'app_custom_texts_maybe_updated', $saved );

		if ( $saved ) {
			wpb_notice( 'saved' );
		}
	}

	/**
	 * Admin settings HTML code
	 */
	public function settings() {

		wpb_admin_access_check( 'manage_custom_texts' );
?>
<div class="wrap app-page">
	<div id="poststuff" class="metabox-holder">
		<?php
		wpb_infobox(
			__( 'Here you can quickly customize <b>admin side</b> terms and localize (translate) some or all of the <b>front end</b> texts (javascript messages, column names, day names, field titles, etc).', 'wp-base'),
			__( '<b>Admin side text replace</b> is intended for quick terminology change, not for translation. To make a full localization for the admin side, use a translation tool, e.g. Loco Translate, instead.','wp-base' )
			.'<br />'.
			__( '<b>Front end text replace</b> usage: Search for the original text using the Search field. Matching results will be dynamically filtered. Enter your custom text in "Your Text" field for the desired text. Repeat this for every text you wish. Click Save at the end. If you are using a caching plugin, do not forget to clear the cache. Now, your text will be in effect. If no custom text is entered for a certain message or front end field, then the default text will be used.', 'wp-base' )
			.'<br />'.
			__( 'Note for <b>shortcode texts:</b> When a shortcode parameter (e.g. monthly_title: Title of monthly calendar) is not explicitly set, it will be replaced by your custom text here or the default text if no custom text is entered. For such parameters the text is selected as: 1) Text entered in the shortcode for the parameter, 2) If (1) is not set, custom text, 3) If (2) is not set, the default text.', 'wp-base' )
		);

		do_action( 'app_admin_custom_texts_after_info' );
		?>
		<form class="app-form" method="post" action="<?php echo wpb_add_query_arg(null,null) ?>">

			<?php remove_filter( 'gettext', array( $this, 'global_text_replace' ) ); ?>
			<div class="postbox">
				<h3 class="hndle"><span><?php _e('Admin Side Text Replace', 'wp-base' ) ?></span></h3>
			<table class="widefat">
			<tr>
				<td>
				<textarea placeholder="<?php _e('Enter find|replace pairs. Example: service provider|specialist, provider|specialist, services|facilities, service|facility','wp-base') ?>" cols="90" style="width:100%" name="replace_texts"><?php echo stripslashes( get_option( 'wp_base_replace_texts' ) ); ?></textarea>
				<br />
				<span class="description app-btm"><?php _e('Here you can enter old text|new text pairs to replace a term globally on admin side. Separate multiple entries by comma, e.g. "service provider|specialist, service|facility" will replace all instances of "service provider" with "specialist" and "service" with "facility". Replace is done from left to right, therefore if there are coinciding words, use more complex one earlier, e.g. service provider before service. If the replacement matches a word and its first letter is uppercase, it is replaced as well. For example, Service will be replaced with Facility.', 'wp-base' ) ?></span>
				</td>
			</tr>
			</table>
			</div>
			<?php add_filter( 'gettext', array( $this, 'global_text_replace' ), 10, 3 ); ?>

			<div class="postbox">
				<h3 class="hndle"><span><?php _e('Front End Text Replace', 'wp-base' ) ?></span></h3>
				<div class="inside">
					<table class="fixed striped display dt-responsive no-wrap app-custom-texts" style="width:99%">
						<thead>
						<tr>
							<th style="width:15%"><?php _e( 'Name', 'wp-base' ) ?></th>
							<th style="width:25%"><?php _e( 'Descr.', 'wp-base' ) ?></th>
							<th ><?php _e( 'Default Text (Not editable)', 'wp-base' ) ?></th>
							<th ><?php _e( 'Your Text (Editable)', 'wp-base' ) ?></th>
						</tr>
						</thead>
					<?php
					$default_texts = $this->get_default_texts( );
					$default_texts = array_merge( $default_texts, $this->add_default_texts );
					ksort( $default_texts );
					foreach ( $default_texts as $key => $default_text ) {
						?>
						<tr><td tabindex="-1" class="key">
							<?php echo $key ?>
						</td><td>
						<?php
							echo $this->get_help_text( $key );
						?>
						</td>
						<td>
							<textarea class="app-1" readonly="readonly"><?php echo stripslashes( $default_text ) ?></textarea>
						</td>
						<td>
							<textarea class="app-1" name="app_custom_texts[<?php echo $key ?>]"><?php
								echo esc_textarea(stripslashes( $this->get_custom_texts( $key ) ));
						?></textarea><?php
							do_action( 'app_admin_custom_texts_after_editable', $key, $default_text )
						?>
						</td>
						</tr>
						<?php
					}
					?>
					</table>
				</div>
			</div>
			<?php
			echo wp_nonce_field( 'update_app_settings', 'app_nonce', true, false );
			?>
			<p class="submit">
				<div class="app_custom_texts_hidden" style="display:none"></div>
				<input type="hidden" value="save_custom_texts" name="action_app" />
				<input type="submit" class="button-primary app_custom_texts_submit" value="<?php _e('Save Custom Texts','wp-base'); ?>" />
			</p>

		</form>
	</div>

</div>
	<?php
	}

}

	BASE('CustomTexts')->add_hooks();
}

