<?php
/**
 * WPB Compat
 *
 * Provides classes to maintain compatibility for 3rd party plugins and software
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 *
 * Handles:
 * Automatic cleaning of page caches for selected Caching plugins
 * Also Automatic Adaptation for selected page builders & templates having WP BASE shortcodes
 *
 * In addition to the classes below, tested and already compatible with following page builders:
 * "Beaver Builder": https://www.wpbeaverbuilder.com/
 * "Elementor": https://elementor.com/
 * "Fusion Builder": https://avada.theme-fusion.com/fusion-builder-2/
 * "Layers": https://www.layerswp.com/
 * "Make": https://thethemefoundry.com/wordpress-themes/make/
 * "WP Bakery" (as text block): https://vc.wpbakery.com/
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBCache' ) ) {

class WpBCache {

	public function __construct(){
		$this->a = BASE();
	}

	protected function load_hooks(){
		add_action( 'app_flush_cache', array( $this, 'flush' ) );
		add_action( 'app_ajax_init', array( $this, 'ajax_init' ) );
	}

	/**
	 * Return ID's of all pages having a WP BASE shortcode
	 * @return array
	 */
	protected function get_page_ids(){
		$pages = BASE()->get_app_pages();

		return is_array( $pages ) ? array_keys( $pages) : array();
	}

	public function flush(){}
	public function ajax_init(){}
}
}

/*
 * Clean WP BASE related page caches created by W3 Total Cache
 *
 */
class WpBW3T extends WpBCache {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBW3T();
		$me->load_hooks();
	}

	/**
	 * @uses app_flush_cache action
	 */
	public function flush(){
		if ( ! function_exists( 'w3tc_pgcache_flush_post' ) ) {
			return;
		}

		# https://gist.github.com/avioli/0cfd7987a66f418ab563
		foreach( $this->get_page_ids() as $page_id ) {
			w3tc_pgcache_flush_post( $page_id );
		}
	}

	/**
	 * Do not allow DB caching in ajax
	 * @uses app_ajax_init action
	 */
	public function ajax_init(){
		if ( ! defined( 'DONOTCACHEDB' ) ) {
			define( 'DONOTCACHEDB', true );
		}
	}

}

WpBW3T::serve();

/*
 * Clean WP BASE related page caches created by Super Cache
 *
 */
class WpBSuperCache extends WpBCache {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBSuperCache();
		$me->load_hooks();
	}

	/**
	 * @uses app_flush_cache action
	 */
	public function flush(){
		if ( ! function_exists( 'wp_cache_post_change' ) ) {
			return;
		}

		# http://z9.io/wp-super-cache-developers/
		$GLOBALS["super_cache_enabled"] = 1;

		foreach( $this->get_page_ids() as $page_id ) {
			wp_cache_post_change( $page_id );
		}
	}

}

WpBSuperCache::serve();

/*
 * Clean WP BASE related page caches created by WP-Rocket
 *
 */
class WpBRocket extends WpBCache {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBRocket();
		$me->load_hooks();
	}

	/**
	 * @uses app_flush_cache action
	 */
	public function flush(){
		if ( ! function_exists( 'rocket_clean_post' ) ) {
			return;
		}

		# Function is in wp-rocket/inc/common/pure.php
		foreach( $this->get_page_ids() as $page_id ) {
			rocket_clean_post( $page_id );
		}
	}

}

WpBRocket::serve();

/*
 * Clean WP BASE related page caches created by WP Fastest Cache
 *
 */
class WpBFastestCache extends WpBCache {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBFastestCache();
		$me->load_hooks();
	}

	/**
	 * @uses app_flush_cache action
	 */
	public function flush(){
		if ( ! class_exists( 'WpFastestCache' ) ) {
			return;
		}

		$instance = is_callable( array( $GLOBALS["wp_fastest_cache"], 'singleDeleteCache' ) )
					? $GLOBALS["wp_fastest_cache"]
					: new WpFastestCache();

		if ( ! is_callable( array( $instance, 'singleDeleteCache' ) ) ) {
			return;
		}

		foreach( $this->get_page_ids() as $page_id ) {
			$instance->singleDeleteCache( false, $page_id );
		}
	}

}

WpBFastestCache::serve();


/*
 * Clean WP BASE related page caches created by WP Cloudflare
 * @requires https://wordpress.org/plugins/cloudflare/
 */
class WpBCloudflare extends WpBCache {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBCloudflare();
		$me->load_hooks();
	}

	/**
	 * @uses app_flush_cache action
	 */
	public function flush(){
		if ( ! class_exists( '\CF\WordPress\Hooks' ) ) {
			return;
		}

		$instance = new \CF\WordPress\Hooks();

		if ( ! is_callable( array( $instance, 'purgeCacheByRelevantURLs' ) ) ) {
			return;
		}

		$pages = $this->get_page_ids();

		foreach( $pages as $page_id ) {
			$instance->purgeCacheByRelevantURLs( $page_id );
		}
	}

}

WpBCloudflare::serve();


/*
 * Clean WP BASE related page caches created by WP Cloudflare Super Page Cache
 * @requires https://wordpress.org/plugins/wp-cloudflare-page-cache/
 */
class WpBCloudflareCache extends WpBCache {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBCloudflareCache();
		$me->load_hooks();
	}

	/**
	 * @uses app_flush_cache action
	 */
	public function flush(){
		if ( ! class_exists( 'SW_CLOUDFLARE_PAGECACHE' ) || ! class_exists( 'SWCFPC_Cache_Controller' ) ) {
			return;
		}

		$instance = new SWCFPC_Cache_Controller( SWCFPC_CACHE_BUSTER, new SW_CLOUDFLARE_PAGECACHE() );

		if ( ! is_callable( array( $instance, 'purge_cache_on_post_edit' ) ) ) {
			return;
		}

		foreach( $this->get_page_ids() as $page_id ) {
			$instance->purge_cache_on_post_edit( $page_id );
		}
	}

}

WpBCloudflareCache::serve();


/*
 * Clean WP BASE related page caches created by Litespeed Cache
 *
 */
class WpBLiteSpeed extends WpBCache {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBLiteSpeed();
		$me->load_hooks();
	}

	/**
	 * @uses app_flush_cache action
	 */
	public function flush(){
		if ( ! function_exists( 'run_litespeed_cache' ) || ! class_exists( 'Purge' ) ) {
			return;
		}

		if ( ! is_callable( array( 'Purge', 'get_instance' ) ) ) {
			return;
		}

		foreach( $this->get_page_ids() as $page_id ) {
			Purge::get_instance()->purge_post( $page_id );
		}
	}

}

WpBLiteSpeed::serve();


/*
 * Clean WP BASE related page caches created by SG Optimizer
 *
 */
class WpBSGOptimizer extends WpBCache {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBSGOptimizer();
		$me->load_hooks();
	}

	/**
	 * @uses app_flush_cache action
	 */
	public function flush(){
		if ( ! function_exists( 'sg_cachepress_purge_cache' ) ) {
			return;
		}

		foreach( $this->get_page_ids() as $page_id ) {
			sg_cachepress_purge_cache( get_permalink( $page_id ) );
		}
	}

}

WpBSGOptimizer::serve();



# Page Builders
if ( ! class_exists( 'WpBCompat' ) ) {

class WpBCompat {

	public function __construct(){
		$this->a = BASE();
	}

	protected function load_hooks(){
		add_action( 'plugins_loaded', array( $this, 'plugins_loaded' ) );
		add_filter( 'app_js_data', array( $this, 'js_parameters' ) );
		add_filter( 'app_post_content', array( $this, 'post_content' ), 10, 3 );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
		add_action( 'app_scripts_enqueued', array( $this, 'app_scripts_enqueued' ) );
		add_action( 'wp_print_scripts', array( $this, 'wp_print_scripts' ), 1 );
		add_action( 'wp_enqueue_scripts', array( $this, 'wp_enqueue_scripts_pre' ), 1 );
		add_action( 'wp_enqueue_scripts', array( $this, 'wp_enqueue_scripts' ), 100 );
		add_filter( 'template_include', array( $this, 'template_check' ) );
		add_filter( 'embed_template', array( $this, 'template_check' ) );
		add_action( 'update_postmeta', array( $this, 'update_postmeta'), 10, 4 );
		add_action( 'app_admin_styles_enqueued', array( $this, 'admin_styles_enqueued') );
	}
	
	public function plugins_loaded() {}

	public function js_parameters( $param ) {
		return $param;
	}

	public function post_content( $content, $post, $ajax = false ) {
		return $content;
	}

	public function wp_print_scripts() {}
	public function wp_enqueue_scripts_pre() {}
	public function wp_enqueue_scripts() {}
	public function admin_enqueue_scripts(){}
	public function app_scripts_enqueued(){}

	public function the_posts($posts) {
		return $posts;
	}

	public function template_check( $template ) {
		return $template;
	}

	public function update_postmeta( $meta_id, $object_id, $meta_key, $meta_value ) {}

	public function admin_styles_enqueued(){}

}
}

/*
 * Support for Divi
 * There is also code in front-scripts.js
 */
class WpBDivi extends WpBCompat {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBDivi();
		$me->load_hooks();
	}

	public function wp_print_scripts() {
		if ( wpb_is_mobile() ) {
			wp_dequeue_script( 'et-jquery-touch-mobile' );
			wp_dequeue_script( 'et-builder-modules-script' );
		}
	}
}

WpBDivi::serve();

/*
 * Support for Live Composer
 *
 */
class WpBWPBakery extends WpBCompat {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBWPBakery();
		$me->load_hooks();
	}

	public function the_posts_content( $prev_content, $post ) {
		$content = (isset( $post->post_excerpt ) ? $post->post_excerpt : '') .' '. (isset( $post->post_content ) ? $post->post_content : '');
		return $prev_content . ' ' . $content;
	}

	public function post_content( $content, $post, $ajax = false ) {
		if ( ! function_exists( 'dslc_json_decode' ) || ! $ajax || empty( $post->ID ) ) {
			return $content;
		}

		$meta = maybe_unserialize( get_post_meta( $post->ID, 'dslc_code', true ) );
		$page_code_array = dslc_json_decode( $meta );

		foreach( (array)$page_code_array as $element ) {
			if ( ! isset( $element['content'] ) ) {
				continue;
			}

			$content .= print_r( $element['content'], true );
		}

		return $content;
	}

}
WpBWPBakery::serve();

/*
 * Support for OnePager
 *
 */
class WpBOnePager extends WpBCompat {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBOnePager();
		$me->load_hooks();
	}

	public function the_posts($posts) {
		if ( empty( $posts ) || is_admin() || ! function_exists( 'onepager' ) ) {
			return $posts;
		}

		$sc_found = false;

		foreach( $posts as $post ) {
			if ( ! isset( $post->ID ) ) {
				continue;
			}

			$meta = maybe_unserialize( get_post_meta( $post->ID, 'onepager_sections', true ) );

			foreach( (array)$meta as $section ) {
				$title = ! empty( $section['contents']['title'] ) ? $section['contents']['title'] : '';
				$desc = ! empty( $section['contents']['description'] ) ? $section['contents']['description'] : '';
				if ( has_shortcode( $title, 'app_confirmation') || has_shortcode( $title, 'app_book') ||
					has_shortcode( $desc, 'app_confirmation') || has_shortcode( $desc, 'app_book')) {
					do_action( 'app_shortcode_found', 'confirmation' );
					$sc_found = true;
					break;
				}
				if ( strpos( $title, '[app_' ) !== false || strpos( $desc, '[app_' ) !== false ) {
					$sc_found = true;
				}
			}
		}

		if ( $sc_found ) {
			do_action( 'app_load_assets' );
		}

		return $posts;
	}

	public function post_content( $prev_content, $post, $ajax = false ) {
		if ( ! function_exists( 'onepager' ) || !$ajax || empty($post->ID) ) {
			return $prev_content;
		}

		$content = '';
		$sections = maybe_unserialize(get_post_meta( $post->ID, 'onepager_sections', true ));

		if ( is_array( $sections ) ) {
			$all_valid = array_filter( $sections,
				function ( $section ) {
					return array_key_exists( 'slug', $section )
					 && array_key_exists( 'id', $section )
					 && array_key_exists( 'title', $section );
			} );

			if ( ! empty( $all_valid ) ) {
				foreach( $all_valid as $section ) {
					$title = ! empty( $section['contents']['title'] ) ? $section['contents']['title'] : '';
					$desc = ! empty( $section['contents']['description'] ) ? $section['contents']['description'] : '';
					$content .= $title.$desc;
				};
			}
		}

		return $prev_content. $content;
	}

}

WpBOnePager::serve();


/*
 * Support for Site Origin (Text Widget)
 *
 */
class WpBSiteOrigin extends WpBCompat {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBSiteOrigin();
		$me->load_hooks();
	}

	public function admin_enqueue_scripts() {
		global $pagenow;

		if ( ! is_admin() || 'post.php' != $pagenow || ! defined( 'SITEORIGIN_PANELS_VERSION' ) ) {
			return;
		}

		wp_enqueue_script( 'app-editor-widget-js', WPB_PLUGIN_URL . '/js/admin/admin-widgets.js', array( 'jquery' ), $this->a->version );
	}

}

WpBSiteOrigin::serve();

/*
 * Support for Unyson
 *
 */
class WpBUnyson extends WpBCompat {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBUnyson();
		$me->load_hooks();
	}

	public function post_content( $prev_content, $post, $ajax = false ) {
		if ( ! function_exists('fw_ext_page_builder_get_post_content') ) {
			return $prev_content;
		}

		global $shortcode_tags;
		$stack = $shortcode_tags;
		if ( $ajax ) {
			$app_shortcodes = wpb_shortcodes();
			foreach( $shortcode_tags as $scode=>$val ) {
				if ( in_array( $scode, $app_shortcodes ) )
					unset( $shortcode_tags[$scode] );
			}
		}

		$c = do_shortcode(fw_ext_page_builder_get_post_content($post));
		$shortcode_tags = $stack;

		if ( $c != $prev_content && empty( $this->lss_called ) &&  strpos( $c, 'app-sc' ) !== false ) {
			do_action( 'app_load_assets' );
			$this->lss_called = true;
		}

		return $prev_content. $c;
	}

}

WpBUnyson::serve();

/*
 * Support for X Theme
 *
 */
class WpBXTheme extends WpBCompat {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBXTheme();
		$me->load_hooks();
	}

	/*
	 * X Theme has a js conflict with qtip as of March 2017:
	 * https://community.theme.co/forums/topic/apparent-conflict-with-qtip-library/#post-872048
	 * Solution is to load qtip after x-body.min.js of X Theme
	 */
	public function app_scripts_enqueued( ) {
		if ( ! function_exists( 'x_enqueue_site_scripts' ) ) {
			return;
		}

		if ( WpBDebug::is_debug() ) {
			wp_deregister_script( 'jquery-qtip' );
			wp_enqueue_script( 'jquery-qtip', WPB_PLUGIN_URL . '/js/dev/jquery.qtip.min.js', array('jquery','x-site-body'), $this->a->version );
		} else {
			wp_deregister_script( 'wp-base-libs' );
			wp_enqueue_script( 'wp-base-libs', WPB_PLUGIN_URL . '/js/libs.js', array('jquery-ui-widget','jquery-ui-button','jquery-ui-dialog','jquery-ui-datepicker','x-site-body'), $this->a->version, true );
		}
	}

}

WpBXTheme::serve();

/*
 * Support for Popup Maker
 * Only one popup per page is supported
 * "Disable Overlay" checkbox should be checked for scrolling to work
 */
class WpBPopupMaker extends WpBCompat {

	private $popup_id = null;

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBPopupMaker();
		$me->load_hooks();
		add_action( 'pum_preload_popup', array( $me, 'preload' ) );
		add_action( 'app_update_calendars', array( $me, 'update_calendars' ) );
		add_filter( 'pum_popup_get_title', array( $me, 'popup_title' ), 10, 2 );
	}
	
	/*
	 * Loading popups late is causing problem
	 */
	public function wp_enqueue_scripts_pre() {
		if ( class_exists( 'PUM_Site_Popups' ) ) {
			if ( $pri = has_action( 'wp_enqueue_scripts', array( 'PUM_Site_Popups', 'load_popups' ) ) ) {
				remove_action( 'wp_enqueue_scripts', array( 'PUM_Site_Popups', 'load_popups' ), $pri );
				add_action( 'wp_enqueue_scripts', array( 'PUM_Site_Popups', 'load_popups' ), 9 );
			}
		}
	}
	
	/*
	 * Load assets when preload_popup action fires
	 */
	public function preload( $ID ) {
		
		if ( $this->popup_id ) {
			return;
		}
		
		# Check if we have shortcode inside the popup
		$post = get_post( $ID );

		if ( empty( $post->post_content ) ) {
			return;
		}

		if ( ! has_shortcode( $post->post_content, 'app_book' ) ) {
			return;
		}
		
		add_filter( 'body_class', array( $this, 'body_class' ) );

		$this->popup_id = $ID;
		do_action( 'app_load_assets' );
	}
	
	/**
	 * Add css class to body tag
	 * @return array
	 */
	public function body_class( $classes ) {
		$classes[] = 'app-has-popup';
		return $classes;
	}


	public function update_calendars() {
		if ( wpb_is_admin() ) {
			return;
		}
		
		if ( ! empty( $_POST['popup_id'] ) ) {
			add_filter( 'app_front_ajax_create_content_from_shortcode', '__return_true' );
		}
	}
	
	/*
	 * Add css class to popup container and overlay
	 */
	public function popup_classes( $classes ) {
		$classes['container'][] = 'app-pum-container';
		$classes['overlay'][] = 'app-pum-overlay';
		
		return $classes;
	}
	
	/*
	 * Change WORKER_NAME in title
	 */
	public function popup_title( $title, $popup_id ) {
		$post = get_post();
		$popup = get_post( $popup_id );
		
		if ( ! empty( $popup->post_type ) && 'popup' == $popup->post_type ) {
			if ( $worker_id = $this->a->find_worker_for_page( $post ) ) {
				$title = str_replace( array( 'WORKER', 'VENDOR_NAME' ), $this->a->get_worker_name( $worker_id ), $title );
			} else if ( $service_id = $this->a->find_service_for_page( $post ) ) {
				$title = str_replace( array( 'SERVICE' ), $this->a->get_service_name( $service_id ), $title );
			} 
		}
		
		return $title;
	}

	/*
	 * When popup is active, WP BASE checks popup content instead of page content
	 * Therefore WP BASE shortcodes on normal page content will not update
	 */
	public function js_parameters( $param ) {
		if ( ! $this->popup_id ) {
			return $param;
		}

		$param['popup_id'] = $this->popup_id;
		$param['is']['noMS'] = true;
		return $param;
	}

}

WpBPopupMaker::serve();

/*
 * Support for WooCommerce Bookings
 * This extention loads JQuery UI styles remotely and on every page! (Not only to WC product page)
 * It forces using smoothness theme and it does not provide a theme selection option
 * This class fixes this for WP BASE pages
 */
class WpBWCBookings extends WpBCompat {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBWCBookings();
		$me->load_hooks();
	}

	/*
	 * Remove style on WP BASE pages as it will otherwise override ours
	 */
	public function wp_enqueue_scripts( ) {
		if ( ! class_exists( 'WC_Bookings' ) ) {
			return;
		}

		# If not WP BASE page, do not touch
		if ( ! wp_style_is( 'wp-base' ) ) {
			return;
		}

		wp_dequeue_style( 'jquery-ui-style' );
	}

}

WpBWCBookings::serve();

/*
 * Support for LearnDash Gradebook
 * This extention loads JQuery UI styles on every admin page
 * This class fixes this for WP BASE admin pages
 */
class WpBLearnDash_Gradebook extends WpBCompat {

	public function __construct(){
		parent::__construct();
	}

	public static function serve(){
		$me = new WpBLearnDash_Gradebook();
		$me->load_hooks();
	}

	/*
	 * Remove style on WP BASE pages as it will otherwise override ours
	 */
	public function admin_styles_enqueued( ) {
		if ( ! class_exists( 'LearnDash_Gradebook' ) ) {
			return;
		}

		# If not WP BASE page, do not touch
		if ( ! wp_style_is( 'wp-base' ) ) {
			return;
		}

		if ( strpos( wpb_get_current_screen_id(), 'learndash' ) !== false ) {
			return;
		}

		wp_dequeue_style( 'ld-gb-admin' );
		wp_dequeue_style( 'rbm-fh-admin' );
	}

}

WpBLearnDash_Gradebook::serve();
