<?php
/**
 * WPB Service
 *
 * Handles service functions
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.7.6.2
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpB_Service' ) ) {

class WpB_Service {

	# Unique ID
	# integer
 	public $ID;

	# Whether service is internal
	# integer
 	private $internal;

	# Sort order in the DB
	# integer
 	private $sort_order;

	# Name of the service
	# sttring
	private $name;

	# Capacity
	# integer
 	private $capacity = 1;

	# Duration in minutes
	# integer
 	private $duration;

	# Padding before in minutes
	# integer
 	private $padding;

	# Padding after in minutes
	# integer
 	private $break_time;

	# Location IDs service available in. Coded with ":"
	# string
	private $locations;

	# category IDs service has. Coded with ":"
	# sttring
	private $categories;

	# Price of the booking including decimal sign (.), without currency symbol
	# string
	private $price;

	# Security deposit for the booking including decimal sign (.), without currency symbol
	# string
	private $deposit;

	# Page ID
	# integer
 	private $page;

	# WP BASE Core instance
	protected $a;

	/**
	 * Constructor
	 * @param	$_service	integer|object|null		Service ID or service StdClass object
	 */
	public function __construct( $_service = null ) {
		$this->a = BASE();

		if ( $_service ) {
			if ( ! is_object( $_service ) ) {
				$_service = $this->a->get_service( $_service );
			}

			if ( $_service ) {
				foreach ( get_object_vars( $_service ) as $key => $value ) {
					$this->$key = $value;
				}
			}

		}
	}

	/**
	 * Getters
	 */
	public function get_ID(){
		return $this->ID;
	}
	public function get_internal(){
		return $this->internal;
	}
	public function get_sort_order(){
		return $this->sort_order;
	}
	public function get_name(){
		return $this->name;
	}
	public function get_capacity(){
		return $this->capacity;
	}
	public function get_duration(){
		return $this->duration;
	}
	public function get_padding(){
		return $this->padding;
	}
	public function get_break_time(){
		return $this->break_time;
	}
	public function get_locations(){
		return wpb_explode( $this->locations );
	}
	public function get_categories(){
		return wpb_explode( $this->categories );
	}
	public function get_price(){
		return wpb_round( $this->price );
	}
	public function get_deposit(){
		return wpb_round( $this->deposit );
	}
	public function get_page(){
		return $this->page;
	}

	/**
	 * Setters
	 */
	public function set_ID( $ID ){
		$this->ID = $ID;
	}
	public function set_internal( $_internal ){
		$this->internal = $_internal;
	}
	public function set_sort_order( $_order ){
		$this->sort_order = $_order;
	}
	public function set_name( $_name ){
		$this->name = $_name;
	}
	public function set_capacity( $number ){
		$this->capacity = $number;
	}
	public function set_duration( $minutes ){
		$this->duration = $minutes;
	}
	public function set_padding( $minutes ){
		$this->padding = $minutes;
	}
	public function set_break_time( $minutes ){
		$this->break_time = $minutes;
	}
	public function set_locations( $locations ){
		$this->locations = wpb_implode( $locations );
	}
	public function set_categories( $cats ){
		$this->categories = wpb_implode( $cats );
	}
	public function set_price( $price ){
		$this->price = $price;
	}
	public function set_deposit( $deposit ){
		$this->deposit = $deposit;
	}
	public function set_page( $page_id ){
		$this->page = $page_id;
	}

	/**
	 * Aliases
	 */
	public function is_internal(){
		return (bool)$this->get_internal();
	}
	public function get_padding_before(){
		return $this->get_padding();
	}
	public function get_padding_after(){
		return $this->get_break_time();
	}
	public function set_padding_before( $pad ){
		return $this->set_padding( $pad );
	}
	public function set_padding_after( $pad ){
		return $this->set_break_time( $pad );
	}

	/**
	 * Get Description page permalink
	 * @since 3.8.0
	 * @return string
	 */
	public function get_page_permalink() {
		$page_id = $this->get_page();
		return apply_filters( 'app_service_description_page_permalink', ($page_id ? get_permalink( $page_id ) : ''), $page_id, $this );
	}

	/**
	 * Name of the Services DB table
	 * @since 3.7.7.2
	 * @return string
	 */
	public function get_table_name(){
		return $this->a->services_table;
	}

	/**
	 * Return a list of current property values
	 * to be readily used in $wpdb insert/update functions
	 * @since 3.7.7.2
	 * @return array
	 */
	public function get_save_data(){
		return array(
			'sort_order'	=> $this->sort_order,
			'internal'		=> $this->internal,
			'name'			=> $this->name,
			'capacity'		=> $this->capacity,
			'duration'		=> $this->duration,
			'padding'		=> $this->padding,
			'break_time'	=> $this->break_time,
			'locations'		=> $this->locations,
			'categories'	=> $this->categories,
			'price'			=> $this->price,
			'deposit'		=> $this->deposit,
			'page'			=> $this->page,
		);
	}

	/**
	 * Writes service to DB by updated values
	 * @since 3.7.7.2
	 * @return true on success, false on fail. Reason of failure can be read from string: BASE()->db->last_error
	 */
	public function save(){
		return $this->a->db->update( $this->get_table_name(), $this->get_save_data(), array( 'ID' => $this->ID ) );
	}

	/**
	 * Inserts a service record to DB
	 * @since 3.7.7.2
	 * @return Service ID on success, false on fail. Reason of failure can be read from string: BASE()->db->last_error
	 */
	public function insert(){
		if ( $this->a->db->insert( $this->get_table_name(), $this->get_save_data() ) ) {
			return $this->ID = $this->a->db->insert_id;
		} else {
			return false;
		}
	}

	/**
	 * Add a service to DB
	 * @param $args	array	Service parameters (All optional) see wpb_add_service
	 * @since 3.7.7.2
	 * @return Service ID on success, false on fail. Reason of failure can be read from string: BASE()->db->last_error
	 */
	 public function add( $args = array() ) {

		$this->prepare( $args );

		if ( $this->ID ) {
			$this->save();
		} else {
			if ( $this->a->db->insert( $this->get_table_name(), $this->get_save_data() ) ) {
				BASE('WH')->add_default( $this->ID, 'service' );
				$this->add_created_at( date( 'Y-m-d H:i:s', $this->a->_time ) );
				$this->add_created_by( ! empty( $args['created_by'] ) && get_user_by( 'ID', $args['created_by'] ) ? $args['created_by'] : get_current_user_id() );
			} else {
				return false;
			}
		}

		if ( isset( $args['owner'] ) ) {
			$this->update_owner( wpb_clean( $args['owner'] ) );
		}

		if ( isset( $args['managed_by'] ) ) {
			$this->update_managed_by( wpb_clean( $args['managed_by'] ) );
		}

		if ( ! empty( $args['description'] ) ) {
			$this->update_description( wp_kses_post( $args['description'] ) );
		}

		if ( ! empty( $args['image_id'] ) ) {
			$this->update_image_id( intval( $args['image_id'] ) );
		}

		return $this->ID;
	}

	/**
     * Normalize and set properties from arguments
	 * @param $args		array	Values to create a service
	 * @since 3.7.7.2
     */
	public function prepare( $args ) {

		if ( ! empty( $args['ID'] ) && $_service = wpb_get_service( $args['ID'] ) ) {
			$args = wp_parse_args( $args, get_object_vars( $_service ) );
		}

		$locs_incl	= array();
		$locations	= isset( $args['locations'] ) ? $args['locations'] : array();
		$l_to_check	= is_array( $locations ) ? $locations : wpb_explode( wpb_sanitize_commas( $locations ), ',' );

		foreach ( (array)$l_to_check as $loc ) {
			if ( $this->a->location_exists( $loc ) ) {
				$locs_incl[] = $loc;
			}
		}

		$data = array(
			'internal'			=> ! empty( $args['internal'] ) ? 1 : 0,
			'name'				=> ! empty( $args['name'] ) ? $args['name'] : $this->a->get_text('service'),
			'capacity'			=> ! empty( $args['capacity'] ) ? preg_replace('/[^0-9]/', '', $args['capacity']) : '',
			'duration'			=> ! empty( $args['duration'] ) ? preg_replace('/[^0-9]/', '', $args['duration']) : $this->a->get_min_time(),
			'padding'			=> ! empty( $args['padding'] ) ? preg_replace('/[^0-9]/', '', $args['padding']) : '',
			'break_time'		=> ! empty( $args['break_time'] ) ? preg_replace('/[^0-9]/', '', $args['break_time']) :'',
			'locations'			=> ! empty( $locs_incl ) ? wpb_implode( $locs_incl ) : '',
			'categories'		=> ! empty( $args['categories'] ) && is_array($args['categories']) ? wpb_implode( wpb_sanitize_commas($args['categories']) ) : '',
			'price'				=> ! empty( $args['price'] ) ? wpb_sanitize_price( $args['price'] ) : 0,
			'deposit'			=> ! empty( $args['deposit'] ) ? wpb_sanitize_price( $args['deposit'] ) : '',
			'page'				=> ! empty( $args['page'] ) ? (int)$args['page'] : '',
		);

		if ( ! empty( $args['sort_order'] ) ) {
			$data['sort_order'] = intval($args['sort_order']);
		} else if ( empty( $args['ID'] ) ) {
			$max_sort_order = $this->a->db->get_var( "SELECT MAX(sort_order) FROM " . $this->get_table_name() );
			$data['sort_order'] = intval($max_sort_order) + 1;
		}

		$this->set_props( $data );
	}

	/**
     * Set multiple properties at once
	 * @since 3.7.7.2
     */
	public function set_props( $data ) {
		foreach ( get_object_vars( $this ) as $key => $value ) {
			if ( isset( $data[ $key ] ) ) {
				$this->$key = $data[ $key ];
			}
		}
	}

	/**
     * Permanently delete a service
	 * @since 3.8.0
     */
	public function delete() {
		return wpb_delete_service( $this->ID );
	}

	/**
	 * Number of bookings using the service
	 * @since 3.8.0
	 * @return integer
	 */
	public function nof_bookings() {
		return (int)$this->a->db->get_var( $this->a->db->prepare(
			"SELECT COUNT(ID) FROM " .$this->a->app_table. 	" WHERE service=%d AND status<>'test'",
			$this->ID
		) );
	}

	/**
	 * Whether service has any booking
	 * @since 3.8.0
	 * @return bool
	 */
	public function has_booking() {
		return (bool)$this->nof_bookings();
	}

	/**
	 * Metas
	 */

	/**
	 * Return value of the desired meta
	 * @param $meta_name	string
	 * @return mixed
	 */
	public function get_meta( $meta_name ){
		return wpb_get_service_meta( $this->ID, $meta_name );
	}

	/**
	 * Add a value to the desired meta
	 * @param $meta_name	string
	 * @param $value		mixed		Value to write
	 * @return bool|integer
	 */
	public function add_meta( $meta_name, $value, $single = true ){
		return wpb_add_service_meta( $this->ID, $meta_name, $value, $single );
	}

	/**
	 * Update the desired meta with a value
	 * @param $meta_name	string
	 * @param $value		mixed		Value to write
	 * @return bool|integer
	 */
	public function update_meta( $meta_name, $value, $prev_value = '' ){
		return wpb_update_service_meta( $this->ID, $meta_name, $value, $prev_value );
	}

	/**
	 * Delete a meta
	 * @param $meta_name	string
	 * @return bool
	 */
	public function delete_meta( $meta_name ){
		return wpb_delete_service_meta( $this->ID, $meta_name );
	}

	/**
	 * Get ID of the user who created the service
	 * @return integer
	 */
	public function get_created_by(){
		return wpb_get_service_meta( $this->ID, 'created_by' );
	}

	/**
	 * Add ID of the user who created the service
	 * @param $user_id		integer
	 * @return bool
	 */
	public function add_created_by( $user_id = null ){
		return wpb_add_service_meta( $this->ID, 'created_by', ($user_id ?: get_current_user_id()) );
	}

	/**
	 * Get ID of the service owner
	 * @return integer
	 */
	public function get_owner(){
		return wpb_get_service_meta( $this->ID, 'owner' );
	}

	/**
	 * Add service owner
	 * @param $user_id		integer
	 * @return bool
	 */
	public function add_owner( $user_id = null ){
		return wpb_add_service_meta( $this->ID, 'owner', ($user_id ?: get_current_user_id()) );
	}

	/**
	 * Update service owner
	 * @param $user_id		integer
	 * @return bool
	 */
	public function update_owner( $user_id = null ){
		return wpb_update_service_meta( $this->ID, 'owner', ($user_id ?: get_current_user_id()) );
	}

	/**
	 * Delete service owner
	 * @return bool
	 */
	public function delete_owner(){
		return wpb_delete_service_meta( $this->ID, 'owner' );
	}

	/**
	 * Get translated creation time of the service
	 * @return string
	 */
	public function get_created_at(){
		if ( $dt = wpb_get_service_meta( $this->ID, 'created_at' ) ) {
			return is_numeric( $dt ) ? date_i18n( $this->a->dt_format, $dt ) : date_i18n($this->a->dt_format, strtotime( $dt ) );
		}

		return '';
	}

	/**
	 * Add creation time of the service
	 * @param $time		integer
	 * @return bool
	 */
	public function add_created_at( $time = null ){
		return wpb_add_service_meta( $this->ID, 'created_at', ($time ?: $this->a->_time) );
	}

	/**
	 * Get worker IDs who can use the service
	 * @return array
	 */
	public function get_managed_by() {
		return wpb_explode( wpb_get_service_meta( $this->ID, 'managed_by' ) );
	}

	/**
	 * Update worker IDs who can use the service
	 * @param $worker_ids	null|string|array	If null, current user. Else comma delimited string or array of worker IDs
	 * @param $update		bool				If true do update, if false do add
	 * @return bool
	 */
	public function update_managed_by( $worker_ids = null, $update = true ) {
		$ids = is_array( $worker_ids )
			   ? $worker_ids
			   : ( null === $worker_ids ? array( get_current_user_id() ) : explode( ',', $worker_ids ) );

		if ( ! empty( $ids ) ) {

			$val = wpb_implode( array_map( 'intval', $ids ) );

			if ( $update ) {
				return wpb_update_service_meta( $this->ID, 'managed_by', $val );
			} else {
				return wpb_add_service_meta( $this->ID, 'managed_by', $val );
			}
		} else {
			return wpb_delete_service_meta( $this->ID, 'managed_by' );
		}
	}

	/**
	 * Add worker IDs who can use the service
	 * @param $worker_ids	null|string|array	If null, current user. Else comma delimited string or array of worker IDs
	 * @return bool
	 */
	public function add_managed_by( $worker_ids = null ) {
		return $this->update_managed_by( $worker_ids, false );
	}

	/**
	 * Delete worker IDs who can use the service
	 * @return bool
	 */
	public function delete_managed_by() {
		return wpb_delete_service_meta( $this->ID, 'managed_by' );
	}

	/**
	 * Get description of the service
	 * @return string
	 */
	public function get_description() {
		return wpb_get_service_meta( $this->ID, 'description' );
	}

	/**
	 * Update description of the service
	 * @return bool
	 */
	public function update_description( $desc ) {
		return wpb_update_service_meta( $this->ID, 'description', $desc );
	}

	/**
	 * Get Featured Image Url
	 * @return string
	 */
	public function get_image_url() {
		return wpb_get_service_meta( $this->ID, 'image_url' );
	}

	/**
	 * Update Featured Image Url
	 * @return bool
	 */
	public function update_image_url( $url ) {
		return wpb_update_service_meta( $this->ID, 'image_url', $url );
	}

	/**
	 * Get Featured Image Attachment/Post ID
	 * @return string
	 */
	public function get_image_id() {
		return wpb_get_service_meta( $this->ID, 'image_id' );
	}

	/**
	 * Update Featured Image Attachment/Post ID
	 * @return bool
	 */
	public function update_image_id( $id ) {
		return wpb_update_service_meta( $this->ID, 'image_id', $id );
	}

	/**
	 * Find workers giving this service
	 * @return array of worker IDs
	 */
	public function get_workers( $order_by = '' ) {
		return (array)$this->a->get_worker_ids_by_service( $this->ID, $order_by );
	}

	/**
	 * Whether this is a trial service
	 * @return bool
	 */
	public function is_trial() {
		return in_array( $this->ID, explode( ',', wpb_setting( 'trial_services' ) ) );
	}
}
}