<?php
/**
 * WPB LSW Normalize
 *
 * Normalizes lsw (location/service/worker) values, i.e. filter out unallowed values 
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( !class_exists( 'WpB_Norm' ) ) {

class WpB_Norm {
	
	private $location, $service, $worker, $category;
	
	public $show_locations_menu	= true;
	public $show_services_menu	= true;
	public $show_workers_menu	= true;
	public $display_errors		= true;
	
	public $error = array();
	
	protected $a;
	
	/**
	 * Constructor
	 * @param	$location		integer|string		Set location ID or 'auto' or location name
	 * @param	$service		integer|string		Set service ID or 'all' or 'auto' or service name
	 * @param	$worker			integer|string		Set worker ID or 'all' or 'auto'
	 * @param	$category		integer				Limit services with the category
	 */
	function __construct( $location = false, $service = false, $worker = false, $category = 0 ) {
		$this->a = BASE();
		
		if ( 'all' === $service || 'all' === $worker ) {
			$this->service = 'all';
			$this->worker = $this->a->get_def_wid();
			return;
		}

		$this->normalize_location( $location );
		$this->normalize_service( $service );
		$this->normalize_worker( $worker );
		$this->category = $category;
	}
	
	/**
	 * Getters
	 * @return 	mixed		false|null|integer|string
	 */
	public function get_location(){
		return $this->location;
	}
	public function get_service(){
		return $this->service;
	}
	public function get_worker(){
		return $this->worker;
	}
	public function get_category(){
		return $this->category;
	}

	/**
	 * Setters
	 */
	public function set_location( $l ){
		$this->location = $l;
	}
	public function set_service( $s ){
		$this->service = $s;
	}
	public function set_worker( $w ){
		$this->worker = $w;
	}
	
	/**
	 * Normalize location ID, i.e if name is given find its ID
	 * @param $ID	integer|string		ID to be normalized
	 * @return 		integer				Location ID
	 */
	private function normalize_location( $location_id ) {
		if ( ! class_exists( 'WpBLocations' ) || ! $this->a->get_nof_locations() )
			return false;
		
		if ( ! empty( $_GET['app_location_id'] ) && $this->a->location_exists( $_GET['app_location_id'] ) ) {
			$this->location = wpb_clean( $_GET['app_location_id'] );
			return;
		}

		$ID = defined( 'WPB_AJAX' ) && ! $location_id 
			  ? (! empty( $_POST['app_location_id'] ) ? wpb_clean( $_POST['app_location_id'] ) : 0)
			  : $location_id;
		
		if ( '0' === (string)$ID || false === $ID ) {
			$this->location = $ID;
			return;
		}

		if ( is_numeric( $ID ) && $this->a->location_exists( $ID ) ) {
			$this->location = $ID;
			if ( $location_id ) {
				$this->show_locations_menu = false;
			}
			return;
		}
		
		$maybe_id = false;
		if ( 'auto' === $ID ) {
			if ( $maybe_id = $this->a->find_location_for_page( ) ) {
				$this->location = $maybe_id;
				$this->show_locations_menu = false;
			} else {
				$this->location = ! empty( $_POST['app_location_id'] ) ? wpb_clean( $_POST['app_location_id'] ) : 0;
			}
			
			return;
		} 
		
		if ( !is_numeric( $ID ) && $maybe_id = $this->a->find_location_id_from_name( urldecode( $ID ) ) ) {
			$this->location = $maybe_id;
			return;
		}

		$this->error[] = sprintf( __( 'Location %s is not correct', 'wp-base' ), print_r( $ID, true ) );
	}
	
	/**
	 * Normalize service ID, i.e if name is given find its ID
	 * @param $service_id	integer|string		ID to be normalized
	 * @return 	integer		Service ID
	 */
	private function normalize_service( $service_id ) {
		if ( $maybe_id = apply_filters( 'app_norm_pref_service', false, $service_id ) ) {
			$this->service = is_array( $maybe_id ) ? key( $maybe_id ) : $maybe_id;
			return;
		}
		
		if ( ! empty( $_GET['app_service_id'] ) && $this->a->service_exists( $_GET['app_service_id'] ) ) {
			$this->service = wpb_clean( $_GET['app_service_id'] );
			return;
		}
	
		$ID = defined( 'WPB_AJAX' ) && !$service_id 
			  ? (! empty( $_POST['app_service_id'] ) ? wpb_clean( $_POST['app_service_id'] )  : 0)
			  : $service_id;

		if ( '0' === (string)$ID || false === $ID ) {
			$this->service = $ID;
			return;
		}
		
		if ( is_numeric( $ID ) && $this->a->service_exists( $ID ) ) {
			$this->service = $ID;
			if ( $service_id ) {
				$this->show_services_menu = false;
			}
			return;
		}

		$maybe_id = false;
		if ( 'auto' === $ID ) {
			if ( $maybe_id = $this->a->find_service_for_page( ) ) {
				$this->service = $maybe_id;
				$this->show_services_menu = false;
			} else {
				$this->service = ! empty( $_POST['app_service_id'] ) ? wpb_clean( $_POST['app_service_id'] ) : 0;
			}
			
			return;
		}
		
		if ( ! is_numeric( $ID ) && $maybe_id = $this->a->find_service_id_from_name( urldecode( $ID ) ) ) {
			$this->service = $maybe_id;
			return;
		}

		$this->error[] = sprintf( __( 'Service %s is not correct', 'wp-base' ), print_r( $ID, true ) );
	}
	
	/**
	 * Normalize worker ID, i.e if auto is selected find ID
	 * @param $ID	integer|string		ID to be normalized
	 * @return 		integer				Worker ID
	 */
	private function normalize_worker( $worker_id ) {
		if ( ! class_exists( 'WpBSP' ) || ! $this->a->get_nof_workers() ) {
			return false;
		}
		
		if ( ! empty( $_GET['app_worker_id'] ) && $this->a->worker_exists( $_GET['app_worker_id'] ) ) {
			$this->worker = wpb_clean( $_GET['app_worker_id'] );
			if ( $this->worker == $this->a->find_worker_for_page() ) {
				$this->show_workers_menu = false;
			}
			
			return;
		}

		if ( defined( 'WPB_AJAX' ) && !$worker_id ) {
			if ( wpb_is_bp_book_me() ) {
				$ID = ! empty( $_POST['bp_displayed_user_id'] ) ? wpb_clean( $_POST['bp_displayed_user_id'] ) : 0;
			} else {
				$ID = ! empty( $_POST['app_worker_id'] ) ? wpb_clean( $_POST['app_worker_id'] ) : 0;
			}
		} else {
			$ID = $worker_id;
		}
		
		if ( '0' === (string)$ID || false === $ID  ) {
			$this->worker = $ID;
			return;
		}

		if ( is_numeric( $ID ) && $this->a->worker_exists( $ID ) ) {
			$this->worker = $ID;
			if ( $worker_id ) {
				$this->show_workers_menu = false;
			}
			
			return;
		}
		
		if ( 'auto' === $ID ) {
			if ( $maybe_id = $this->a->find_worker_for_page( ) ) {
				$this->worker = $maybe_id;
				$this->show_workers_menu = false;
			} else {
				if ( wpb_is_bp_book_me() ) {
					$this->worker = ! empty( $_POST['bp_displayed_user_id'] ) ? wpb_clean( $_POST['bp_displayed_user_id'] ) : 0;
				} else {
					$this->worker = ! empty( $_POST['app_worker_id'] ) ? wpb_clean( $_POST['app_worker_id'] ) : 0;
				}
			}
			
			return;
		}

		$this->error[] = sprintf( __( 'worker %s is not correct', 'wp-base' ), print_r( $ID, true ) );
	}
	
	/**
	 * Display errors
	 * @return string
	 */
	public function display_errors(){
		if ( empty( $this->error ) || !$this->display_errors || ! WpBDebug::is_debug() ) {
			return;
		}
		
		$out = '<ul>';
		foreach ( $this->error as $error ) {
			$out .= '<li>'. $error . '</li>';
		}

		$out .= '</ul>';
		$out .= '<div class="app-clearfix" style="clear:both"></div>';
		
		return $out;
	}
}
}