<?php
/**
 * WPB Client Class
  *
 * Copyright © 2018-2021 Hakan Ozevin
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.9.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

class WpB_Client {

	/**
     * User ID
     */
	private $ID;

	/**
     * User WP role
     */
	protected $role;

	/**
     * User email
     */
	private $email;

	/**
     * Copy of WP User
     */
	private $user;

	/**
     * Constructor
	 * @param $id	integer|object		WP user ID or WP_User object
     */
	public function __construct( $id ) {
		$this->user		= new WP_User( $id );
		$this->ID		= $this->user->ID;
		$this->email	= $this->user->user_email;
	}

	/**
     * ID of the client
	 * @return	integer
     */
	public function get_ID(){
		return $this->ID;
	}

	/**
     * Role of the client
	 * @return	array
     */
	public function get_role(){
		return $this->role;
	}

	/**
     * email of the client
	 * @return	string
     */
	public function get_email(){
		return $this->email;
	}

	/**
     * User WP object
	 * @return	object
     */
	public function get_user(){
		return $this->user;
	}
	
	/**
     * Check if user has been assigned as client
	 * @return	bool
     */
	public function is_client(){
		if ( is_multisite() && ! is_user_member_of_blog( $this->ID ) ) {
			return false;
		}

		return in_array( 'wpb_client', (array)$this->role );
	}

	/**
     * Assign a user as client
	 * @return	bool
     */
	public function add(){
		$this->user->add_role( 'wpb_client' );
		return $this->is_client();
	}

	/**
     * Unassign user being a client
	 * @return	none
     */
	public function remove(){
		$this->user->remove_role( 'wpb_client' );
	}

	/**
     * Add a line to the log
	 * @param $text		string		Text to be logged
	 * @return	integer|false
     */
	public function log( $text ) {
		return add_user_meta( $this->ID, 'app_client_log', $text );
	}

	/**
     * Get all log records of a client
	 * @return	array
     */
	public function get_log( ) {
		return (array)get_user_meta( $this->ID, 'app_client_log', false );
	}

	/**
     * Get a user meta
	 * @param $key	string		Name of meta, e.g. name, email, address
	 * @return	string
     */
	public function get_meta( $key, $single = true ) {
		if ( 'email' == $key ) {
			return $this->email;
		}

		return get_user_meta( $this->ID, 'app_'. $key, $single );
	}

	/**
     * Update a user meta
	 * @param $key			string		Name of meta, e.g. name, email, address
	 * @param $value		string		Value to write
	 * @param $prev_value	string		See update_user_meta WP function
	 * @return bool
     */
	public function update_meta( $key, $value, $prev_value = '' ) {
		if ( update_user_meta( $this->ID, 'app_'. $key, $value, $prev_value ) ) {

			if ( in_array( $key, array( 'display_name', 'first_name', 'last_name' ) ) ) {
				wp_update_user( array( 'ID' => $this->ID, $key => $value ) );
			}

			return true;
		}

		return false;
	}

	/**
     * Find display name of client
	 * @return	string
     */
	public function	get_display_name() {
		if ( ! empty( $this->user->display_name ) ) {
			return $this->user->display_name;
		}

		# aka Full Name
		if ( $name = $this->get_meta( 'name' ) ) {
			return $name;
		}

		$maybe_name = trim( $this->get_first_name() .' '. $this->get_last_name() );

		if ( ! empty( $maybe_name ) ) {
			return $maybe_name;
		}

		return $this->user->user_login;
	}

	/**
     * Find first name of client
	 * @return	string
     */
	public function	get_first_name() {
		if ( ! empty( $this->user->first_name ) ) {
			return $this->user->first_name;
		}

		return $this->get_meta( 'first_name' );
	}

	/**
     * Find last name of client
	 * @return	string
     */
	public function	get_last_name() {
		if ( ! empty( $this->user->last_name ) ) {
			return $this->user->last_name;
		}

		return $this->get_meta( 'last_name' );
	}

	/**
     * Avatar of client
	 * @return	string
     */
	public function get_avatar( $size = 96 ){
		return get_avatar( $this->email, $size );
	}

	/**
     * Uploaded image of the client, *just the url part*
	 * @return	string
     */
	public function get_profile_image(){
		return get_user_meta( $this->ID, 'app_profile_image', true );
	}
	
	/**
     * Get HTML image. Fallback to avatar
	 * @since
	 * @return	string
     */
	public function get_image_html( $size = 96 ) {
		$img = $this->get_profile_image();
		return $img ? '<img height="'.$size.'" width="'.$size.'" class="avatar avatar-'.$size.'" src="'.$img.'">' : $this->get_avatar( $size );
	}
	
	/**
     * Localized date client accepted as being client
	 * @return	string
     */
	public function start_date(){
		$start_date = '';
		
		if ( $start = $this->get_meta( 'client_since' ) ) {
			$start_date = date_i18n( BASE()->date_format, abs( $start ) );
		}
		
		return $start_date;
	}

	/**
     * Timestamp client accepted as client
	 * @return	integer
     */
	public function start_ts(){
		return $this->get_meta( 'client_since' ) ?: 0;
	}
	
	/**
     * Total revenue (payments) of the client
	 * @return integer (Amount is multiplied by 100, e.g. 1234 means $12.34)
     */	
	public function get_revenue() {
		$sum = '';
		if ( $trs = wpb_get_transactions_by_user( $this->ID ) ) {
			if ( $tr_arr = wp_list_pluck( array_values( $trs ), 'total_paid' ) ) {
				$sum = array_sum( $tr_arr );
			}
		}
		
		return intval( $sum );
	}
	
	/**
     * Total number of valid bookings (paid, confirmed, pending, completed) of the client
	 * @param	$start	integer/string		Timestamp or date in any standard format
	 * @param	$end	integer/string		Timestamp or date in any standard format
	 * @return integer
     */	
	public function get_nof_bookings( $start = 0, $end = null ) {
		$args = array(
			'user'	=> $this->ID,
			'range' => 'custom',
			'start'	=> $start,
			'end'	=> $end ?: strtotime( 'today', time() ) + DAY_IN_SECONDS,
		);
		
		$stats = array( 'paid', 'confirmed', 'pending', 'completed' );
		
		return wpb_nof_bookings( $args, $stats );
	}
	
	/**
     * Preferred timezone
	 * @since 3.9
	 * @return string
     */
	public function	get_timezone() {
		return $this->get_meta( 'timezone' );
	}
	
	/**
     * Update preferred timezone
	 * @since 3.9
	 * @return bool
     */
	public function	update_timezone( $tz ) {
		return $this->update_meta( 'timezone', $tz );
	}
	
	/**
     * Preferred language
	 * @since 3.9
	 * @return string
     */
	public function	get_lang() {
		return $this->get_meta( 'user_lang' );
	}
	
	/**
     * Update preferred language
	 * @since 3.9
	 * @return bool
     */
	public function	update_lang( $lang ) {
		return $this->update_meta( 'user_lang', $lang );
	}	
}
