<?php
/**
 * WPB Booking
 *
 * Handles booking functions
 *
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpB_Booking' ) ) {

class WpB_Booking {

	# Unique booking ID
	# integer
 	private $ID;

	# Parent booking ID, if there is one
	# integer
	private $parent_id;

	# Creation time of the booking in mysql format (Y-m-d H:i:s)
	# string
	private $created;

	# WordPress user ID of the client booking belongs to. For not logged in clients, zero
	# integer
	private $user;

	# Location ID of the booking
	# integer
	private $location;

	# Service ID of the booking
	# integer
	private $service;

	# Service Provider ID assigned to the booking. WP user ID of the SP
	# integer
	private $worker;

	# Status of the booking. Common values: pending, confirmed, paid, running, cart, removed, test, waiting
	# string
	private $status;

	# Start time of the booking in mysql format (Y-m-d H:i:s)
	# string
	private $start;

	# End time of the booking in mysql format (Y-m-d H:i:s)
	# string
	private $end;

	# Number of seats/pax booking occupies. 1 for single bookings. May be greater than 1 for Group Bookings
	# integer
	private $seats = 1;

	# Unique gcal ID created by Google Calendar
	# string
	private $gcal_ID;

	# Latest update time of the Google Calendar entry for the booking in mysql format (Y-m-d H:i:s)
	# string
	private $gcal_updated;

	# Price of the booking including decimal sign (.), without currency symbol
	# string
	private $price;

	# Security deposit for the booking including decimal sign (.), without currency symbol
	# string
	private $deposit;

	# Name of the payment method for the booking.
	# Common values: manual-payments, paypal-standard, paypal-express, stripe, paymill, simplify, 2checkout, authorizenet, credits, woocommerce
	# string
	private $payment_method;

	# A unique value to work instead of ID for new booking created at instantiation
	# string
	private $uniqid;

	# ID of the Source appointment record, if there is one
	# integer
	private $source;

	# Whether this is an event
	# bool
	private $is_event;

	# WP BASE Core instance
	protected $a;

	/**
	 * Constructor
	 * @param	$app	integer|object|null		Booking ID or Appointment StdClass object
	 */
	public function __construct( $app = null ) {
		$this->a = BASE();
		$this->uniqid = uniqid();

		if ( $app ) {
			if ( ! is_object( $app ) ) {
				$this->source = $app;
				$app = wpb_get_app( $app );
			} else if ( ! empty( $app->ID ) ) {
				$this->source = $app->ID;
			}

			if ( $app ) {
				foreach ( get_object_vars( $app ) as $key => $value ) {
					$this->$key = $value;
				}
			}

			$this->uniqid = null;
		}
	}

	/**
	 * Getters
	 * @since 3.0
	 */
	public function get_ID(){
		return $this->ID;
	}
	public function get_parent_id(){
		return $this->parent_id;
	}
	public function get_created(){
		return $this->created;
	}
	public function get_user(){
		return $this->user;
	}
	public function get_location(){
		return $this->location;
	}
	public function get_service(){
		return $this->service;
	}
	public function get_worker(){
		return $this->worker;
	}
	public function get_status(){
		return $this->status;
	}
	public function get_start(){
		return $this->start;
	}
	public function get_end(){
		return $this->end;
	}
	public function get_seats(){
		return $this->seats;
	}
	public function get_price(){
		return wpb_round( $this->price );
	}
	public function get_gcal_ID(){
		return $this->gcal_ID;
	}
	public function get_gcal_updated(){
		return $this->gcal_updated;
	}
	public function get_deposit(){
		return wpb_round( $this->deposit );
	}
	public function get_payment_method(){
		return (string)$this->payment_method;
	}
	public function get_uniqid(){
		return $this->uniqid;
	}
	public function get_source(){
		return $this->source;
	}

	/**
	 * Setters
	 * @since 3.0
	 */
	public function set_ID( $ID ){
		$this->ID = $ID;
	}
	public function set_parent_id( $parent_id ){
		$this->parent_id = $parent_id;
	}
	public function set_created( $created ){
		$this->created = wpb_date( $created );
	}
	public function set_user( $user ){
		$this->user = $user;
	}
	public function set_location( $location ){
		$this->location = $location;
	}
	public function set_service( $service ){
		$this->service = $service;
	}
	public function set_worker( $worker ){
		$this->worker = $worker;
	}
	public function set_status( $status ){
		$this->status = $status;
	}
	public function set_start( $start ){
		$this->start = wpb_date( $start );
	}
	public function set_end( $end ){
		$this->end = wpb_date( $end );
	}
	public function set_seats( $seats ){
		$this->seats = $seats;
	}
	public function set_price( $price ){
		$this->price = $price;
	}
	public function set_gcal_ID( $gcal_ID ){
		$this->gcal_ID = $gcal_ID;
	}
	public function set_gcal_updated( $gcal_updated ){
		$this->gcal_updated = $gcal_updated;
	}
	public function set_deposit( $deposit ){
		$this->deposit = $deposit;
	}
	public function set_payment_method( $method ){
		$this->payment_method = $method;
	}
	public function set_source( $ID ){
		$this->source = $ID;
	}

	/**
	 * Return a list of current property values
	 * to be readily used in $wpdb insert/update functions
	 * @since 3.0
	 * @return array
	 */
	public function get_save_data(){
		return array(
			'parent_id'		=> $this->parent_id,
			'created'		=> $this->created,
			'user'			=> $this->user,
			'location'		=> $this->location,
			'service'		=> $this->service,
			'worker'		=> $this->worker,
			'status'		=> $this->status,
			'start'			=> $this->start,
			'end'			=> $this->end,
			'seats'			=> $this->seats,
			'price'			=> $this->price,
			'deposit'		=> $this->deposit,
			'gcal_ID'		=> $this->gcal_ID,
			'gcal_updated'	=> $this->gcal_updated,
			'payment_method'=> $this->payment_method,
		);
	}
	
	/**
	 * Writes booking to DB by updated values
	 * @since 3.7.3.1
	 * @return true on success, false on fail. Reason of failure can be read from string: BASE()->db->last_error
	 */
	public function save(){
		return (bool)$this->a->db->update( $this->a->app_table, $this->get_save_data(), array( 'ID' => $this->ID ) );
	}
	
	/**
	 * Inserts a booking record to DB
	 * @since 3.7.7
	 * @return Booking ID on success, false on fail. Reason of failure can be read from string: BASE()->db->last_error
	 */
	public function insert(){
		if ( $this->a->db->insert( $this->a->app_table, $this->get_save_data() ) ) {
			return $this->ID = $this->a->db->insert_id;
		} else {
			return false;
		}
	}
	
	/**
	 * Add a booking to DB
	 * Does not check availability of the slot. If required, use wpb_is_free before calling this
	 * @param $args	array	Booking parameters (All optional) see wpb_add_booking
	 * @since 3.0
	 * @return Booking ID on success, false on fail. Reason of failure can be read from string: BASE()->db->last_error
	 */
	 public function add( $args = array() ) {

		$this->prepare( $args );

		if ( $this->ID && $this->save() ) {
			return $this->ID;
		} else {
			if ( $this->a->db->insert( $this->a->app_table, $this->get_save_data() ) ) {
				return $this->ID = $this->a->db->insert_id;
			} else {
				return false;
			}
		}
	}
	
	/**
     * Normalize and set properties from arguments
	 * @param $args		array	Values to create a booking
	 * @since 3.7.4.1
     */
	public function prepare( $args ) {
		 if ( ! empty( $args['ID'] ) && $_app = wpb_get_app( $args['ID'] ) ) {
			$args = wp_parse_args( $args, $this->get_save_data() );
		}

		$args['location']	= isset( $args['location'] ) && $this->a->location_exists( $args['location'] ) ? $args['location'] : $this->a->read_location_id();
		$args['service']	= isset( $args['service'] ) && $this->a->service_exists( $args['service'] ) ? $args['service'] : $this->a->read_service_id();
		$args['worker']		= isset( $args['worker'] ) && $this->a->worker_exists( $args['worker'] ) ? $args['worker'] : $this->a->read_worker_id();

		$start	= ! empty( $args['start'] )
				  ? wpb_strtotime( $args['start'] )
				  : ( $this->a->find_first_free_slot( $args['location'], $args['service'], $args['worker'] ) ?: $this->a->_time );

		$args['start']	= date( 'Y-m-d H:i:s', $start );
		$args['end']	= empty( $args['end'] ) 
						? date( 'Y-m-d H:i:s', $start + wpb_get_duration( $this )*60 ) 
						: date( 'Y-m-d H:i:s', wpb_strtotime( $args['end'] ) );

		$created = ! empty( $args['created'] ) 
				   ? wpb_strtotime( $args['created'] ) 
				   : ( isset( $args['ID'] ) ? '' : $this->a->_time );
				   
		if ( $created ) {
			$args['created'] = date( 'Y-m-d H:i:s', $created );
		}

		$args['status']		= !empty( $args['status'] ) && array_key_exists( $args['status'], $this->a->get_statuses() ) ? $args['status'] : 'confirmed';
		$args['parent_id']	= isset( $args['parent_id'] ) ? $args['parent_id'] : 0;

		$args['price']		= isset( $args['price'] ) ? wpb_sanitize_price( $args['price'] ) : 0;
		$args['deposit']	= isset( $args['deposit'] ) ? wpb_sanitize_price( $args['deposit'] ) : '';

		$this->set_props( $args );
	}

	/**
     * Set multiple properties at once
	 * @since 3.7.4.1
     */
	public function set_props( $data ) {
		foreach ( get_object_vars( $this ) as $key => $value ) {
			if ( isset( $data[ $key ] ) ) {
				$this->$key = $data[ $key ];
			}
		}
	}
	
	/**
	 * Update price based on current values
	 * @return string	Current price of the booking
	 */
	public function update_price() {
		$slot = new WpB_Slot( $this );
		return $this->price = $slot->get_price();
	}

	/**
	 * Check whether updated value is within limits
	 * @param $old_booking		object		WpB_Booking object for old booking (existing one before update)
	 * @param $die				bool		If true, dies with json response. If false, returns error message.
	 * @return none|string					if error, dies with json response or returns string. If no error, returns null
	 */
	public function check_update( $old_booking, $die = true ) {

		# Check if this service available in this location
		if ( $this->a->get_nof_locations() ) {
			if ( ! (bool)$this->a->get_services_by_location( $this->location ) ) {
				$error = __('Selected service is not available in this location!', 'wp-base' );
				if ( $die ) {
					die( json_encode( array( 'error' => $error ) ) );
				} else {
					return $error;
				}
			}
		}

		# Check if provider gives this service
		if ( $this->a->get_nof_workers() && $this->worker ) {
			if ( ! (bool)$this->a->get_workers_by_service( $this->service ) ) {
				$error = __('Selected provider is not giving selected service!', 'wp-base' );
				if ( $die ) {
					die( json_encode( array( 'error' => $error ) ) );
				} else {
					return $error;
				}
			}
		}

		$slot = new WpB_Slot( $this );

		$slot->exclude( $this->ID );

		add_filter( 'app_seats_no_pax_filter', '__return_true' );

		$error = '';

		if ( $this->location == $old_booking->get_location() && $this->service == $old_booking->get_service()
			&& $this->worker == $old_booking->get_worker() && $this->start == $old_booking->get_start() ) {

			$seats = apply_filters( 'app_bookings_seats', (!empty($_POST['app_seats']) ? wpb_clean( $_POST['app_seats'] ) : 1), $this, $old_booking );

			if ( $seats <= $old_booking->get_seats() ) {
				# Same lsw, same or less seats. No need to check.
			} else if ( $reason = $slot->is_busy() ) {
				$error = sprintf( __('This time slot is not available! Reason: %s', 'wp-base'), wpb_code2reason( $reason ) );
			}
		} else {
			if ( $this->worker && !$slot->is_working( ) )
				$error = __('Selected service provider is not working at selected time interval!', 'wp-base');

			# Unassigned provider case
			if ( !$this->worker && 'yes' == wpb_setting( 'service_wh_check' ) && !$slot->is_service_working( ) ) {
				$error = __('Selected service is not available at selected time interval!', 'wp-base');
			}

			if ( $reason = $slot->is_busy( ) )
				$error = sprintf( __('This time slot is not available! Reason: %s', 'wp-base'), wpb_code2reason( $reason ) );
		}

		if ( $error ) {
			if ( $die ) {
				die( json_encode( array( 'error' => $error ) ) );
			} else {
				return $error;
			}
		}
	}

	/**
	 * Function to normalize args
	 * DEPRECATED
	 * @param args		array
	 * @since 3.0
	 * @until 3.7.4.1
	 * @return array
	 */
	public function normalize_args( $args ) {

		return $args;
	}

	/**
     * Get start date of the booking in Y-m-d format
	 * @return string
     */
	public function date( ) {
		return date( 'Y-m-d', wpb_strtotime( $this->start ) );
	}

	/**
     * Get start time part of the booking in WP set format
	 * @return string
     */
	public function time( ) {
		return date( $this->a->time_format, wpb_strtotime( $this->start ) );
	}

	/**
     * Get start hour of the booking in military time - H:i format
	 * @return string
     */
	public function military_time( ) {
		return date( 'H:i', wpb_strtotime( $this->start ) );
	}

	/**
     * Get end time part of the booking in WP set format
	 * @return string
     */
	public function end_time( ) {
		return date( $this->a->time_format, wpb_strtotime( $this->end ) );
	}

	/**
     * Get total paid for the booking in all transactions
	 * @return string
     */
	public function get_total_paid( ) {
		if ( ! $this->ID ) {
			return 0;
		}

		$paid = $this->a->db->get_var( $this->a->db->prepare(
			"SELECT SUM(transaction_total_amount)
			FROM {$this->a->transaction_table}
			WHERE transaction_app_ID = %d ",
			$this->ID )
		);

		return wpb_round( $paid/100 );
	}

	/**
     * Get balance of the booking
	 * @since 3.6.0
	 * @return string
     */
	public function get_balance(){
		return ($this->get_total_paid() - $this->get_price() - $this->get_deposit());
	}

	/**
     * Get fee of a booking
	 * @since 3.6.3
	 * @return string
     */
	public function get_fee(){
		return wpb_get_app_meta( $this->ID, 'fee' );
	}

	/**
     * Human readable payment method name
	 * @return string
     */
	public function payment_method_name(){
		return wpb_payment_method_name( $this->payment_method );
	}

	/**
     * Find who created this booking except client himself
	 * @return ID of editor
     */
	public function get_created_by(){
		return (int)wpb_get_app_meta( $this->ID, 'booking_on_behalf' );
	}

	/**
     * Set who created this booking
	 * @return bool
     */
	public function set_created_by( $user_id ){
		if ( $user_id ) {
			return wpb_update_app_meta( $this->ID, 'booking_on_behalf', $user_id );
		} else {
			return wpb_delete_app_meta( $this->ID, 'booking_on_behalf' );
		}
	}

	/**
     * Get children of the booking
	 * @return array of objects
     */
	public function get_children( ) {
		return BASE('Multiple')->get_children( $this->ID );
	}

	/**
     * Find other children of booking's parent
	 * @return array of objects
     */
	public function get_siblings( ) {
		return BASE('Multiple')->get_siblings( $this->ID );
	}

	/**
     * Get WooCommerce Order ID
	 * @return integer
     */
	public function get_wc_order_id(){
		return (integer)wpb_get_app_meta( $this->ID, 'woocommerce_order_id' );
	}

	/**
     * Get WooCommerce product/post ID
	 * @return integer
     */
	public function get_wc_post_id(){
		return (integer)wpb_get_app_meta( $this->ID, 'woocommerce' );
	}

	/**
	 * Return extras selected by client during booking
	 * @since 3.0
	 * @return array of numerically indexed extra array each having key/value pairs:
	 *		ID			integer		ID of the corresponding extra
	 *		type		string		Empty value for product, 'service' for Service as extra
	 *		name		string		Name of the extra
	 *		service_id	integer		If extra is a service, ID of the service. Otherwise 0
	 */
	public function get_extras(){
		if ( ! BASE('Extras') ) {
			return array();
		}

		$extras = BASE('Extras')->get_extras();
		
		if ( empty( $extras ) ) {
			return array();
		}

		$out = array();

		foreach ( array_filter( explode(',', wpb_get_app_meta( $this->ID, 'extra' ) ) ) as $id ) {

			if ( empty( $extras[$id] ) ) {
				continue;
			}

			$is_service = !empty($extras[$id]["type"]) && 'service' == $extras[$id]["type"];

			$out[] = array(
				'ID'			=> $extras[ $id ],
				'type'			=> $extras[ $id ]["type"],
				'price'			=> ! empty( $extras[ $id ]['price'] ) ? $extras[ $id ]['price'] : 0,
				'name'			=> $is_service ? BASE()->get_service_name( $extras[$id]["name"] ) : $extras[$id]["name"],
				'service_id'	=> $is_service ? $extras[$id]["name"] : 0,
			);
		}

		return $out;
	}

	/**
	 * Return the coupon and discount applied to the booking
	 * @since 3.0
	 * @return array	key is coupon id, value is an array with 'discount' net discount applied, 'code' is coupon code
	 */
	public function get_coupon(){
		return (array)wpb_get_app_meta( $this->ID, 'coupon' );
	}

	/**
	 * Return the timezone selected by client during booking
	 * The timezone is in tz string format: https://en.wikipedia.org/wiki/List_of_tz_database_time_zones
	 * @since 3.0
	 * @return string
	 */
	public function get_timezone(){
		return (string)wpb_get_app_meta( $this->ID, 'timezone' );
	}

	/**
	 * Return the language selected by client during booking
	 * @since 3.0
	 * @return string
	 */
	public function get_lang(){
		return (string)wpb_get_app_meta( $this->ID, 'language' );
	}

	/**
	 * Return value of the desired meta
	 * @param $meta_name	string		Name of the meta, e.g. email, phone, note, udf_1, udf_3, etc
	 * @since 3.5.6
	 * @return mixed
	 */
	public function get_meta( $meta_name ){
		return wpb_get_app_meta( $this->ID, $meta_name );
	}

	/**
	 * Return the category ID of the service of the booking
	 * This will only return a value if category is explicitly selected
	 * @since 3.5.6
	 * @return integer
	 */
	public function get_category(){
		return wpb_get_app_meta( $this->ID, 'category' );
	}

	/**
	 * Set category ID
	 * This will only return a value if category is explicitly selected
	 * @since 3.7.7
	 * @return bool
	 */
	public function update_category( $category ){
		if ( $category ) {
			return wpb_update_app_meta( $this->ID, 'category', $category );
		} else {
			return wpb_delete_app_meta( $this->ID, 'category' );
		}
	}	

	/**
	 * Determine if service of booking lasts all day
	 * @since 3.0
	 * @return bool
	 */
	public function is_daily(){
		return $this->a->is_daily( $this->service );
	}
	
	/**
	 * Localized booking start time
	 * Only the time part in local format
	 * @param $mysql	bool	Return in mysql format, i.e. H:i
	 * @since 3.0
	 * @return string
	 */
	public function client_time( $mysql = false ){
		$format = $mysql ? 'H:i' : $this->a->time_format;
		return date_i18n( $format, $this->a->client_time( wpb_strtotime( $this->start ) ) );
	}

	/**
	 * Localized booking start date
	 * Only the date part in local format
	 * @param $mysql	bool	Return in mysql format, i.e. Y-m-d
	 * @since 3.0
	 * @return string
	 */
	public function client_date( $mysql = false ){
		$format = $mysql ? 'Y-m-d' : $this->a->date_format;
		return date_i18n( $format, $this->a->client_time( wpb_strtotime( $this->start ) ) );
	}

	/**
	 * Localized booking start day
	 * Only the day part in local format
	 * @since 3.0
	 * @return string
	 */
	public function client_day(){
		return date_i18n( 'l', $this->a->client_time( wpb_strtotime( $this->start ) ) );
	}

	/**
	 * Localized booking end time
	 * Only the time part in local format
	 * @param $mysql	bool	Return in mysql format, i.e. H:i
	 * @since 3.0
	 * @return string
	 */
	public function client_end_time( $mysql = false ){
		$format = $mysql ? 'H:i' : $this->a->time_format;
		return date_i18n( $format, $this->a->client_time( wpb_strtotime( $this->end ) ) );
	}

	/**
	 * Localized booking end date/time
	 * @since 3.0
	 * @return string
	 */
	public function client_end_dt(){
		$format = $this->is_daily() ? $this->a->date_format : $this->a->dt_format;
		return date_i18n( $format, $this->a->client_time( wpb_strtotime( $this->end ) ) );
	}

	/**
	 * Localized booking start date/time
	 * @since 3.0
	 * @return string
	 */
	public function client_dt(){
		$format = $this->is_daily( ) ? $this->a->date_format : $this->a->dt_format;
		return date_i18n( $format, $this->a->client_time( wpb_strtotime( $this->start ) ) );
	}

	/**
	 * Time difference between client local time and server time in seconds
	 * @since 3.0
	 * @return integer
	 */
	public function client_offset(){
		return $this->a->get_client_offset( $this->start, $this->user, $this->ID );
	}

	/**
	 * Externally set as WP BASE event, e.g. for a new booking
	 * @since 3.2
	 * @return none
	 */
	public function set_as_event(){
		$this->is_event = true;
	}

	/**
	 * Determine if this is a WP BASE event booking
	 * @since 3.2
	 * @return bool
	 */
	public function is_event(){
		if ( $this->service < 0 && $this->service != WPB_GCAL_SERVICE_ID ) {
			$this->is_event = true;
		}

		return $this->is_event;
	}

	/**
	 * Title of the WP BASE event that booking is made for
	 * @since 3.2
	 * @return string
	 */
	public function get_event_title(){
		if ( ! $this->is_event() ) {
			return '';
		}

		$post = get_post( abs( $this->service ) );

		return ( ! empty( $post->post_title ) ? $post->post_title : '' );
	}

	/**
	 * Link of the WP BASE event that booking is made for
	 * @since 3.2
	 * @return string
	 */
	public function get_event_link(){
		if ( ! $this->is_event() ) {
			return '';
		}

		$post = get_post( abs( $this->service ) );

		return ( ! empty( $post->ID ) ? get_permalink( $post->ID ) : '' );
	}

	/**
	 * Get security code
	 * If none available, create one
	 * @since 3.2
	 * @return string
	 */
	public function get_security_code() {
		if ( $code = wpb_get_app_meta( $this->ID, 'security_code' ) ) {
			return $code;
		}

		$code = strtoupper( substr( md5( rand() . '_' . $this->user ), 0, 10 ) );

		if ( wpb_add_app_meta( $this->ID, 'security_code', $code ) ) {
			return $code;
		} else {
			return false;
		}
	}

	/**
	 * Determine if this is a GCal event
	 * @since 3.5.5
	 * @return bool
	 */
	public function is_gcal_event(){
		return ($this->gcal_ID && WPB_GCAL_SERVICE_ID == $this->service && 'reserved' == $this->status);
	}

	/**
	 * Get ID of the user who created the booking
	 * DEPRECATED: Use get_created_by instead
	 * @since 3.5.6
	 * @return integer
	 */
	public function created_by(){
		return $this->get_created_by();
	}

	/**
	 * Get ID of the user who made the last update
	 * @since 3.5.6
	 * @return integer
	 */
	public function updated_by(){
		$uid = 0;
		if ( $meta = wpb_get_app_meta( $this->ID, '_edit_last' ) ) {
			$meta = explode( ':', $meta );
			$uid = isset( $meta[1] ) ? $meta[1] : 0;
		}

		return $uid;
	}

	/**
	 * Get last update time
	 * @param $ts		bool	If true return timestamp, if false return human readable date/time
	 * @since 3.5.6
	 * @return integer|string
	 */
	public function updated_at( $ts = false ){
		$time = 0;
		if ( $meta = wpb_get_app_meta( $this->ID, '_edit_last' ) ) {
			$meta = explode( ':', $meta );
			$time = isset( $meta[0] ) ? $meta[0] : 0;
		}
		
		if ( $ts ) {
			return $time;
		} else {
			return ($time ? date_i18n( $this->a->dt_format, $time ) : '');
		}
	}

	/**
	 * Get UDF value
	 * @param $ID		integer		ID of the UDF
	 * get_meta can also be used
	 * @since 3.5.7
	 * @return mixed
	 */
	public function get_udf_value( $ID ){
		return wpb_get_app_meta( $this->ID, 'udf_'.$ID );
	}

	/**
	 * Get Google Hangout URL link
	 * @since 3.6.0
	 * @return string
	 */
	public function hangout_link(){
		return (string)wpb_get_app_meta( $this->ID, 'hangout_link' );
	}

	/**
	 * Add a value to the desired meta
	 * @param $meta_name	string		Name of the meta, e.g. email, phone, note, udf_1, etc
	 * @param $value		mixed		Value to write
	 * @since 3.6.0
	 * @return bool|integer
	 */
	public function add_meta( $meta_name, $value, $single = true ){
		return wpb_add_app_meta( $this->ID, $meta_name, $value, $single );
	}

	/**
	 * Update the desired meta with a value
	 * @param $meta_name	string		Name of the meta, e.g. email, phone, note, udf_1, etc
	 * @param $value		mixed		Value to write
	 * @since 3.6.0
	 * @return bool|integer
	 */
	public function update_meta( $meta_name, $value, $prev_value = '' ){
		return wpb_update_app_meta( $this->ID, $meta_name, $value, $prev_value );
	}

	/**
	 * Delete a meta
	 * @param $meta_name	string		Name of the meta, e.g. email, phone, note, udf_1, etc
	 * @since 3.7.7
	 * @return bool
	 */
	public function delete_meta( $meta_name ){
		return wpb_delete_app_meta( $this->ID, $meta_name );
	}
	
	/**
	 * Calculate duration in minutes
	 * @since 3.6.5
	 * @return integer
	 */
	public function get_duration(){
		return intval( ( strtotime( $this->end ) - strtotime( $this->start ) ) / 60 );
	}

	/**
	 * Set Zoom meeting details
	 * @param $details		string		JSON encoded details
	 * @since 3.6.5
	 */
	public function set_zoom_details( $details ){
		return wpb_update_app_meta( $this->ID, 'zoom_details', $details );
	}

	/**
	 * Get Zoom meeting details
	 * @since 3.6.5
	 * @return string	JSON encoded
	 */
	public function get_zoom_details(){
		return wpb_get_app_meta( $this->ID, 'zoom_details' );
	}

	/**
	 * Get Zoom meeting ID
	 * @since 3.6.5
	 * @return integer
	 */
	public function get_zoom_id(){
		$zoom = json_decode( $this->get_zoom_details() );
		return ! empty( $zoom->id ) ? $zoom->id : 0;
	}

	/**
	 * Get Zoom Host ID
	 * @since 3.6.5
	 * @return string
	 */
	public function get_zoom_host_id(){
		$zoom = json_decode( $this->get_zoom_details() );
		return ! empty( $zoom->host_id ) ? $zoom->host_id : '';
	}

	/**
	 * Get Zoom meeting start url
	 * Per Zoom API, this is valid only for 2 hours after zoom details retrieved
	 * @since 3.6.5
	 * @return string
	 */
	public function get_zoom_start_url(){
		$zoom = json_decode( $this->get_zoom_details() );
		return ! empty( $zoom->start_url ) ? $zoom->start_url : '';
	}

	/**
	 * Get Zoom meeting join url
	 * @since 3.6.5
	 * @return string
	 */
	public function get_zoom_join_url(){
		$zoom = json_decode( $this->get_zoom_details() );
		return ! empty( $zoom->join_url ) ? $zoom->join_url : '';
	}

	/**
	 * Get Zoom meeting password
	 * @since 3.6.5
	 * @return string
	 */
	public function get_zoom_password(){
		$zoom = json_decode( $this->get_zoom_details() );
		return ! empty( $zoom->password ) ? $zoom->password : '';
	}

	/**
	 * Get Zoom meeting topic
	 * @since 3.6.5
	 * @return string
	 */
	public function get_zoom_topic(){
		$zoom = json_decode( $this->get_zoom_details() );
		return ! empty( $zoom->topic ) ? $zoom->topic : '';
	}

	/**
	 * Get Zoom meeting agenda
	 * @since 3.6.5
	 * @return string
	 */
	public function get_zoom_agenda(){
		$zoom = json_decode( $this->get_zoom_details() );
		return ! empty( $zoom->agenda ) ? $zoom->agenda : '';
	}

	/**
	 * Get Client IP that he/she used during booking
	 * @since 3.6.6.2
	 * @return string
	 */
	public function get_client_ip(){
		return wpb_get_app_meta( $this->ID, 'client_ip' );
	}

	/**
	 * Get price of each booking before discounts
	 * @since 3.6.7
	 * @return array	key is booking ID of child booking, value is the price
	 */
	public function get_child_prices(){
		return wpb_get_app_meta( $this->ID, 'child_prices' );
	}

	/**
	 * Update price of each booking before discounts
	 * @param $prices		array		key is booking ID of child booking, value is the price
	 * @since 3.7.7
	 * @return array	
	 */
	public function update_child_prices( $prices ){
		if ( ! empty( $prices ) ) {
			return wpb_update_app_meta( $this->ID, 'child_prices', $prices );
		} else {
			return wpb_delete_app_meta( $this->ID, 'child_prices' );
		}
	}
	
	/**
	 * Get explanation when a commission is not paid to a vendor
	 * @since 3.7.1
	 * @return string
	 */
	public function get_no_commission_reason(){
		return wpb_get_app_meta( $this->ID, 'no_commission' );
	}

	/**
	 * Get event summary (title) of an imported GCal event
	 * @since 3.7.2.1
	 * @return string
	 */
	public function get_gcal_summary(){
		return wpb_get_app_meta( $this->ID, 'gcal_summary' );
	}

	/**
	 * Return reason code why this booking has been removed/cancelled
	 * @since 3.7.3.1
	 * @return string
	 */
	public function reason_removed(){
		return wpb_get_app_meta( $this->ID, 'abandoned' );
	}

	/**
	 * Return human readable reason why this booking has been removed/cancelled
	 * @since 3.7.3.1
	 * @return string
	 */
	public function why_removed(){
		$reason = '';
		if ( $stat = $this->reason_removed() ) {
			switch( $stat ) {
				case 'hold':
				case 'cart':				$reason = __( 'Abandoned in cart', 'wp-base' ); break;
				case 'pending':				$reason = __( 'Abandoned after checkout', 'wp-base' ); break;
				case 'cancelled':			$reason = __( 'Client cancelled', 'wp-base' ); break;
				case 'worker_cancelled':	$reason = __( 'Provider cancelled', 'wp-base' ); break;
				case 'editor_cancelled':	$reason = __( 'Editor cancelled', 'wp-base' ); break;
				case 'admin_cancelled':		$reason = __( 'Admin cancelled', 'wp-base' ); break;
				case 'removed':
				case 'expired':				$reason = __( 'Expired', 'wp-base' ); break;
				default:					$reason = __( 'Abandoned', 'wp-base' ); break;
			}
		}

		return $reason;
	}

	/**
     * Get EDD Payment ID
	 * @since 3.7.7
	 * @return integer
     */
	public function get_edd_payment_id(){
		return (integer)wpb_get_app_meta( $this->ID, 'edd_payment_id' );
	}

	/**
     * Get EDD product/post ID
	 * @since 3.7.7
	 * @return integer
     */
	public function get_edd_post_id(){
		return (integer)wpb_get_app_meta( $this->ID, 'edd_product_id' );
	}

	/**
	 * Determine if this is a package booking
	 * @since 3.0
	 * @return integer|false	ID of the service if this is package, false if not package
	 */
	public function is_package(){
		if ( ! BASE('Packages') ) {
			return false;
		}

		$service_id = wpb_get_app_meta( $this->ID, 'package' );

		if ( $service_id && BASE('Packages')->is_package( $service_id ) ) {
			return $service_id;
		}

		return false;
	}

	/**
	 * Mark as package booking
	 * @since 3.7.7
	 * @return bool
	 */
	public function set_as_package( $main = 0, $step = null ){
		if ( ! BASE('Packages') || ! BASE('Packages')->is_package( $main ) ) {
			return false;
		}
		
		if ( $step ) {
			wpb_update_app_meta( $this->ID, 'package_step', $step );
		}

		return wpb_update_app_meta( $this->ID, 'package', $main );
	}
	
	/**
	 * Determine if this is a recurring booking
	 * @since 3.7.7
	 * @return integer|false	ID of the service if this is recurring, false if not
	 */
	public function is_recurring(){
		if ( ! BASE('Recurring') ) {
			return false;
		}

		$service_id = wpb_get_app_meta( $this->ID, 'recurring' );

		if ( $service_id && BASE('Recurring')->is_recurring( $service_id ) ) {
			return $service_id;
		}

		return false;
	}

	/**
	 * Mark as recurring booking
	 * @since 3.7.7
	 * @return bool
	 */
	public function set_as_recurring( $main = 0, $step = null ){
		if ( ! BASE('Recurring') || ! BASE('Recurring')->is_recurring( $main ) ) {
			return false;
		}
		
		if ( $step ) {
			wpb_update_app_meta( $this->ID, 'recurring_step', $step );
		}

		return wpb_update_app_meta( $this->ID, 'recurring', $main );
	}
	
	/**
	 * Get step of the booking in the series
	 * @since 3.7.7
	 * @return integer
	 */
	public function	get_step(){
		return wpb_get_app_meta( $this->ID, 'recurring_step' );
	}
	
	/**
	 * Get repeat unit of a booking
	 * @since 3.7.7
	 * @return string (daily, weekly, etc)
	 */
	public function	get_repeat_unit(){
		return wpb_get_app_meta( $this->ID, 'repeat_unit' );
	}

	/**
	 * Get repeat unit of a booking
	 * @param $unit		string	(daily, weekly, etc)	
	 * @since 3.7.7
	 * @return bool
	 */
	public function	update_repeat_unit( $unit ){
		return wpb_update_app_meta( $this->ID, 'repeat_unit', $unit );
	}

	/**
	 * Get repeat number of a booking
	 * @since 3.7.7
	 * @return integer
	 */
	public function	get_repeat_count(){
		return (int)wpb_get_app_meta( $this->ID, 'repeat_count' );
	}

	/**
	 * Get repeat count of a booking
	 * @param $count		integer	
	 * @since 3.7.7
	 * @return bool
	 */
	public function	update_repeat_count( $count ){
		return wpb_update_app_meta( $this->ID, 'repeat_count', $count );
	}
	
	/**
	 * Whether this is a parent booking
	 * @since 3.7.7
	 * @return bool
	 */
	 public function is_parent(){
		return ! (bool)$this->get_parent_id();
	}
	
	/**
	 * Whether this is a child booking
	 * @since 3.7.7
	 * @return integer|bool
	 */
	 public function is_child(){
		return $this->get_parent_id();
	}

	/**
	 * Get session ID
	 * @since 3.7.7
	 * @return mixed
	 */
	public function	get_session_id(){
		return wpb_get_app_meta( $this->ID, 'session_id' );
	}

	/**
	 * Update session ID
	 * @since 3.7.7
	 * @return bool
	 */
	public function	update_session_id( $session_id ){
		return wpb_update_app_meta( $this->ID, 'session_id', $session_id );
	}	
}
}