<?php
/**
 * WPB Admin Transactions
 *
 * Display and recording of transactions
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBTransactions' ) ) {

class WpBTransactions {

	/**
     * Number of visible columns
     */
	private $colspan = 0;

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Name of user option for Hidden Columns
     */
	const HIDDEN_COLUMNS = 'app_manage_payments_columnshidden';

	/**
     * Name of user option for Hidden Navbar
     */
	const HIDDEN_NAVBAR = 'app_manage_payments_navbarhidden';

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}

	/**
     * Add hooks
     */
	public function add_hooks(){
		add_action( 'app_menu_before_all', array( $this, 'add_menu' ), 30 );
		add_action( 'wp_ajax_app_save_hidden_columns_payments', array( $this, 'save_hidden_columns' ) );
		add_action( 'wp_ajax_app_save_navbar_payments', array( $this, 'save_navbar' ) );
	}

	/**
     * Add submenu page
     */
	public function add_menu(){
		$this->transactions_page = wpb_add_submenu_page(
			'appointments',
			__( 'Payment History', 'wp-base' ),
			__( 'Payment History', 'wp-base' ),
			array( WPB_ADMIN_CAP, 'manage_transactions' ),
			'app_transactions',
			array( $this, 'transaction_list' )
		);

	}

	/**
	 * Define an array of allowed columns in bookings page
	 * Called after addons loaded
	 */
	public static function get_allowed_columns( $user_id = 0 ) {
		$allowed = array( 'payment_id','client','app_id','date','location','service','worker','date_time','end_date_time','amount','reference','note','status','method' );
		if ( $user_id && !user_can( $user_id, WPB_ADMIN_CAP ) ) {
			if ( wpb_is_worker( $user_id ) ) {
				$allowed = array_diff( $allowed, array( 'payment_id', 'worker', 'end_date_time', 'note' ) );
			} else {
				$allowed = array_diff( $allowed, array( 'payment_id', 'client', 'end_date_time', 'note' ) );
			}
		}

		return apply_filters( 'app_transactions_allowed_columns', $allowed, $user_id  );
	}

	/**
     * Add some columns as hidden to Bookings page if there is no selection before
     */
	private static function default_hidden_columns( ) {
		$hidden_cols = array('location','worker');
		return apply_filters( 'app_transactions_default_hidden_columns', $hidden_cols );
	}

	/**
	 * Get hidden columns for the table
	 * @since 3.0
	 * @return array
	 */
	private function get_hidden_columns(){

		$hidden_columns = get_user_option( self::HIDDEN_COLUMNS );

		if ( empty( $hidden_columns['hidden_check'] ) ) {
			$hidden_columns = self::default_hidden_columns();
		}

		return $hidden_columns;
	}

	/**
	 * Select element to display columns on the table
	 * @since 3.8.2
	 * @return string
	 */
	private function displayed_columns_selection(){
		$cols	= self::get_allowed_columns();
		$hidden = self::get_hidden_columns();

		$html = '<select size="15" multiple data-scope="payments" data-table="app-transactions" class="app_displayed_columns app-ms-small" data-selectedtext="'.esc_attr( __( 'Columns', 'wp-base' ) ).'">';
		foreach ( $cols as $col ) {

			if ( 'delete' == $col ) {
				continue;
			}

			if ( 'id' == $col ) {
				$text = $this->a->get_text( 'app_id' );
			} else {
				$text = $this->a->get_text( $col );
			}

			$html .= '<option value="'.$col.'" '. selected( in_array( $col, $hidden ), false, false ).'>'.$text.'</option>';
		}
		$html .= '</select>';

		return $html;
	}

	/**
	 * Ajax save hidden columns
	 * @since 3.8.2
	 * @return json
	 */
	public function save_hidden_columns(){
		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array('error' => $this->a->get_text('unauthorised') ) ) );
		}

		if ( wpb_is_demo() ) {
			die( json_encode( array( 'error' => __('Changes cannot be saved in DEMO mode!', 'wp-base' ) ) ) );
		}

		if ( ! empty( $_POST['hidden'] ) ) {
			update_user_option( get_current_user_id(), self::HIDDEN_COLUMNS, array_merge( $_POST['hidden'], array( 'hidden_check' => 1 ) ) );
		}

		wp_send_json_success();
	}

	/**
	 * Ajax save navbar
	 * @since 3.8.2
	 * @return json
	 */
	public function save_navbar(){
		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array('error' => $this->a->get_text('unauthorised') ) ) );
		}

		if ( wpb_is_demo() ) {
			die( json_encode( array( 'error' => __('Changes cannot be saved in DEMO mode!', 'wp-base' ) ) ) );
		}

		update_user_option( get_current_user_id(), self::HIDDEN_NAVBAR, ! empty( $_POST['hidden'] ) );

		wp_send_json_success();
	}

	/**
	 * Get transaction records
	 */
	public function get_admin_transactions( $startat, $num, $force_app = false, $only_own = 0, $sel_location = 0 ) {

		// Force app_id for payment tooltip
		if ( $force_app ) {
			$_GET['app_s'] = $force_app;
			$_GET['stype'] = 'app_id';
		}

		if ( $only_own ) {
			$user_id = get_current_user_id();

			if ( wpb_is_worker( $user_id ) ) {
				$_GET['app_worker_id'] = $user_id;
			} else {
				$_GET['app_s'] = $user_id;
				$_GET['stype'] = 'user_id';
			}
		}

		if ( $sel_location ) {
			$_GET['app_location_id'] = $sel_location;
		}

		// Search. Also selecting of one or more bookings
		if( ! empty( $_GET['app_s'] ) && ! empty( $_GET['stype'] ) ) {
			$stype 	= esc_sql( $_GET['stype'] );
			$s 		= apply_filters( 'app_search', esc_sql( $_GET['app_s'] ), $stype );

			switch ( $stype ) {

				case 'app_id':	$sarr = array_filter( array_map( 'esc_sql', array_map( 'trim', explode( ' ', str_replace( ',', ' ', $s ) ) ) ) );
								if ( ! empty( $sarr ) ) {
									$add = " AND (transaction_app_ID='". implode( "' OR transaction_app_ID='", $sarr ) . "')";
								} else {
									$add = "";
								}
								break;

				case 'user_id': $add = " AND transaction_app_ID IN ( SELECT ID FROM {$this->a->app_table} WHERE user='{$s}' ) ";
								break;

				case 'name':	$add = " AND transaction_app_ID IN ( SELECT ID FROM {$this->a->app_table}
										 WHERE ( ( user IN ( SELECT ID FROM {$this->a->db->users} AS users
										 WHERE user_login LIKE '%{$s}%' OR user_nicename LIKE '%{$s}%'
										 OR ID IN ( SELECT user_id FROM {$this->a->db->usermeta} AS usermeta
										 WHERE users.ID=usermeta.user_id AND meta_value LIKE '%{$s}%'
										 AND (meta_key='app_first_name' OR meta_key='app_last_name')  ) ) )
										 OR ( user=0 AND ID IN ( SELECT object_id FROM {$this->a->meta_table}
										 WHERE meta_type='app' AND (meta_key='name' OR meta_key='first_name'
										 OR meta_key='last_name') AND meta_value LIKE '%{$s}%' )	)
										) ) ";
								break;

				case 'email':	$add = " AND transaction_app_ID IN ( SELECT ID FROM {$this->a->app_table}
										 WHERE ( ( user IN ( SELECT ID FROM {$this->a->db->users} WHERE user_email LIKE '%{$s}%' ) )
										 OR ( user=0 AND ID IN ( SELECT object_id FROM {$this->a->meta_table}
										 WHERE meta_type='app' AND meta_key='email' AND meta_value LIKE '%{$s}%' ) )
										) ) ";
								break;

				case 'payment_id':
				case 'transaction_id':
								$add = " AND transaction_ID='{$s}' "; break;

				case 'amount':	$add = " AND ( IFNULL(transaction_total_amount,0) /100 ) = '{$s}' "; break;

				case 'amount_gt':
								$add = " AND ( IFNULL(transaction_total_amount,0) /100 ) > '{$s}' "; break;

				case 'amount_lt':
								$add = " AND ( IFNULL(transaction_total_amount,0) /100 ) < '{$s}' "; break;

				default:		$add = apply_filters( 'app_transaction_search_switch', $add, $stype, $s ); break;
			}
		} else {
			$add = '';
		}

		# Get a sanitized filter params
		$filt = wpb_sanitize_search();

		// Filters
		if ( ! isset( $_GET['app_or_fltr'] ) ) {
			if ( $filt['location_id'] )
				$add .= $this->a->db->prepare(" AND transaction_app_ID IN ( SELECT ID FROM {$this->a->app_table} WHERE location=%d ) ", $filt['location_id'] );

			if ( $filt['service_id'] )
				$add .= $this->a->db->prepare(" AND transaction_app_ID IN ( SELECT ID FROM {$this->a->app_table} WHERE service=%d )  ", $filt['service_id'] );

			if ( $filt['worker_id'] )
				$add .= $this->a->db->prepare( " AND transaction_app_ID IN ( SELECT ID FROM {$this->a->app_table} WHERE worker=%d ) ", $filt['worker_id'] );

			if ( $filt['monweek'] ) {
				// Year + Week
				if ( 'w' == substr( $filt['monweek'], 0, 1 ) ) {
					$mode = $this->a->start_of_week ? 7 : 4;
					$add .= $this->a->db->prepare(
								" AND YEARWEEK(FROM_UNIXTIME(transaction_stamp),%d)=%s ",
								 $mode,
								 substr( $filt['monweek'], 1 )
							);
				} else {
					$year = substr( $filt['monweek'], 0, 4 );
					$month = substr( $filt['monweek'], 4, 2 );
					if ( $year && $month )
						$add .= $this->a->db->prepare(
									" AND YEAR(FROM_UNIXTIME(transaction_stamp))=%d AND MONTH(FROM_UNIXTIME(transaction_stamp))=%d ",
									$year,
									$month
								);
				}
			}
		}

		$test = str_replace( array('desc', 'asc', '|',' ' ), '', strtolower( $filt['order_by'] ) );

		if ( $test && in_array( $test, array('transaction_app_id','transaction_stamp') ) ) {
			$order_by = str_replace( '|', ' ', $filt['order_by'] );
		} else {
			$order_by = "transaction_stamp DESC";
		}

		$select = "SELECT SQL_CALC_FOUND_ROWS * FROM {$this->a->transaction_table} ";
		$sql = $select . $this->a->db->prepare( "WHERE (1=1) {$add} ORDER BY {$order_by} LIMIT %d,%d", $startat, $num );

		return $this->a->db->get_results( $sql, OBJECT_K );
	}

	/**
	 * Find if a Paypal transaction is duplicate or not
	 */
	public function duplicate_transaction($app_id, $amount, $currency, $timestamp, $paypal_ID, $status, $note,$content=0) {
		$sql = $this->a->db->prepare(
				"SELECT transaction_ID FROM {$this->a->transaction_table} WHERE transaction_app_ID = %d AND transaction_paypal_ID = %s AND transaction_stamp = %d LIMIT 1",
				$app_id, $paypal_ID, $timestamp );

		$trans = $this->a->db->get_var( $sql );
		if( ! empty($trans) ) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Save or edit a Transaction record in the database
	 * TO BE DEPRECATED and replaced with wpb_add_payment function
	 */
	public function record( $app_id,
							$amount,
							$currency = '',
							$timestamp = '',
							$paypal_ID = '',
							$status = '',
							$note = '',
							$gateway = '',
							$transaction_ID = 0) {

		// Create a uniqe ID in manual payments, if not given by user
		if ( 'manual-payments' == $gateway && !$paypal_ID ) {
			$paypal_ID = uniqid('auto_');
		}

		$result	= $new_record = false;
		$data = array();
		$data['transaction_app_ID']			= $app_id;
		$data['transaction_total_amount']	= $amount * 100;
		$data['transaction_currency']		= $currency ? $currency : $this->a->get_options('currency');
		$data['transaction_stamp']			= $timestamp ? $timestamp : $this->a->_time;
		$data['transaction_paypal_ID']		= $paypal_ID;
		$data['transaction_status']			= $status;
		$data['transaction_note']			= wp_unslash( $note );
		$data['transaction_gateway']		= $gateway;

		# If we are editing a manual payment
		if ( $transaction_ID ) {
			# In the query we add manual payment check too, since we do not want an auto transaction to be edited
			if ( $this->a->db->update(
				$this->a->transaction_table,
				$data,
				array( 'transaction_ID' => $transaction_ID, 'transaction_gateway' => 'manual-payments' )
			) ) {
				$result = $transaction_ID;
			}
		} else {
			$existing_id = $this->a->db->get_var( $this->a->db->prepare(
				"SELECT transaction_ID FROM {$this->a->transaction_table} WHERE transaction_app_ID = %d AND transaction_paypal_ID = %s LIMIT 1",
				$app_id, $paypal_ID
			) );

			if( $existing_id ) {
				if ( $this->a->db->update( $this->a->transaction_table, $data, array('transaction_ID' => $existing_id) ) ) {
					$result = $existing_id;
				}
			} else {
				if ( $this->a->db->insert( $this->a->transaction_table, $data ) ) {
					$result = $this->a->db->insert_id;
					$new_record = true;
				}
			}
		}

		do_action( 'app_payment_maybe_new', $result, $data );

		if ( $result ) {
			if ( $new_record ) {
				do_action( 'app_payment_new_record', $result, $data );
			} else {
				do_action( 'app_payment_updated', $result, $data );
			}
		}

		return $result;
	}

	/**
	 * Get total row number of previous query
	 */
	public function get_total() {
		return $this->a->db->get_var( "SELECT FOUND_ROWS();" );
	}

	/**
	 * Helper to pass page and tab variables
	 */
	public static function print_fields(){
		if ( ! empty( $_GET['tab'] ) ) { ?>
			<input type="hidden" name="tab" value="<?php echo wpb_clean( $_GET['tab'] ) ?>"/>
		<?php }

		if ( 'users_page_your_bookings' == wpb_get_current_screen_id( ) ) { ?>
			<input type="hidden" name="page" value="your_bookings"/> <?php
		} else if ( is_admin() ) { ?>
			<input type="hidden" name="page" value="app_transactions"/>
		<?php }
	}

	/**
	 * Generate html for Transactions page
	 * @param $user_id		integer		User ID in case list is displayed in profile page of user (only own)
	 */
	public function transaction_list( $user_id = 0 ) {

		if ( $user_id ) {
			if ( ! wpb_user_can_view_payments( $user_id ) ) {
				wpb_notice( __('You do not have sufficient permissions to access this page.','wp-base') );
				return;
			}
		} else {
			wpb_admin_access_check( 'manage_transactions' );
		}

		$stype 		= ! empty( $_GET['stype'] ) ? sanitize_text_field( $_GET['stype'] ) : '';
		$s 			= ! empty( $_GET['app_s'] ) ? sanitize_text_field( $_GET['app_s'] ) : '';
		$s 			= apply_filters( 'app_search', $s, $stype );
		$filt 		= wpb_sanitize_search();
		$is_hidden	= (bool)get_user_option( self::HIDDEN_NAVBAR );
	?>
<div class="wrap app-page app-payment-history <?php echo wpb_get_current_screen_id() ?>">
	<div id="app-open-navbar"<?php echo(! $is_hidden ? ' style="display:none"' : '')?>>
		<button title="<?php echo esc_attr( __( 'Open toolbar', 'wp-base' ) ) ?>" data-scope="payments"></button>
	</div>

	<div id="app-navbar" class="app-control-wrap metabox-holder"<?php echo ($is_hidden ? ' style="display:none"' : '')?>>

		<div id="app-close-navbar">
			<button title="<?php echo esc_attr( __( 'Close toolbar', 'wp-base' ) ) ?>" data-scope="payments"></button>
		</div>

		<div class="postbox app-controls">
			<div id="payments-filter" class="tablenav top">
				<div class="app-actions actions">

					<div class="app-dash-title"><span><?php echo ($user_id ? BASE()->get_text('bp_payments') : sprintf( __('%s Payments','wp-base'), WPB_NAME ) ); ?></span></div>

					<div class="app-manage-first-column">
						<form id="app-search-form" method="get" action="" class="search-form">
							<input type="hidden" value="app_transactions" name="page" />
							<input type="hidden" value="all" name="type" />
							<input type="hidden" value="1" name="app_or_fltr" />
							<input type="search" value="<?php echo esc_attr($s); ?>" name="app_s"/>

							<?php $add_class = $stype === 'app_id' ? 'class="app-option-selected"' : ''; ?>
							<select name="stype" <?php echo $add_class ?>>
								<option value="name" <?php selected( $stype, 'name' ); ?>><?php _e('Name','wp-base'); ?></option>
								<option value="email" <?php selected( $stype, 'email' ); ?>><?php _e('email','wp-base'); ?></option>
								<option value="app_id" <?php selected( $stype, 'app_id' ); ?> title="<?php _e('Multiple IDs separated with comma or space is possible','wp-base')?>"><?php _e('Booking ID','wp-base'); ?></option>
								<option value="payment_id" <?php selected( $stype, 'payment_id' ); ?>><?php _e('Payment ID','wp-base'); ?></option>
								<option value="amount" <?php selected( $stype, 'amount' ); ?>><?php _e('Amount =','wp-base'); ?></option>
								<option value="amount_gt" <?php selected( $stype, 'amount_gt' ); ?>><?php _e('Amount &gt;','wp-base'); ?></option>
								<option value="amount_lt" <?php selected( $stype, 'amount_lt' ); ?>><?php _e('Amount &lt;','wp-base'); ?></option>
								<?php do_action( 'app_transaction_search_options', $stype ) ?>
							</select>

							<?php self::print_fields() ?>
							<input type="submit" class="button app-search-button" value="<?php _e('Search','wp-base'); ?>" />
						</form>
					</div>
				</div>

				<div class="app-actions actions">

					<div class="app-manage-first-column">
						<?php $this->filter_form() ?>
					</div>

					<div class="app-manage-second-column">
						<form method="get" action="">
							<?php self::print_fields() ?>
							<input type="hidden" name="app_filter_reset" value="1"/>
							<input type="submit" class="button" value="<?php _e('Clear Filters','wp-base'); ?>" />
						</form>
					</div>
				</div>
			</div>
		</div>
	</div>

	<div class="app-manage-row third-row app-mt">
		<div class="app-manage-first-column">
			<form id="app-sort-form" method="get" action="">
				<select name="app_order_by" style='float:none;'>
					<option value=""><?php _e('Sort by','wp-base'); ?></option>
					<option value="transaction_stamp" <?php selected( $filt['order_by'], 'transaction_stamp' ); ?>><?php _e('Payment date (01/01 &rarr; 31/12)','wp-base'); ?></option>
					<option value="transaction_stamp|DESC" <?php selected( $filt['order_by'], 'transaction_stamp|DESC' ); ?>><?php _e('Payment date (31/12 &rarr; 01/01)','wp-base'); ?></option>
					<option value="transaction_app_ID" <?php selected( $filt['order_by'], 'transaction_app_ID' ); ?>><?php _e('Booking ID (1 &rarr; 99)','wp-base'); ?></option>
					<option value="transaction_app_ID|DESC" <?php selected( $filt['order_by'], 'transaction_app_ID|DESC' ); ?>><?php _e('Booking ID (99 &rarr; 1)','wp-base'); ?></option>
				</select>
				<?php self::print_fields() ?>
				<input type="hidden" value="<?php echo $filt['location_id'] ?>" name="app_location_id" />
				<input type="hidden" value="<?php echo $filt['service_id'] ?>" name="app_service_id" />
				<input type="submit" class="button" value="<?php _e('Sort','wp-base'); ?>"  />
			</form>
		</div>

		<div class="app-manage-second-column app-crowded">
			<?php echo $this->displayed_columns_selection(); ?>
		</div>


	<?php
		if ( is_admin() ) {
			$paged = empty( $_GET['paged'] ) ? 1 : absint( $_GET['paged'] );
		} else {
			$paged = get_query_var( 'paged' ) ? absint( get_query_var( 'paged' ) ) : 1;
		}

		$rpp 			= wpb_setting("records_per_page", 50);
		$startat 		= ($paged - 1) * $rpp;
		$transactions 	= $this->get_admin_transactions( $startat, $rpp, false, $user_id );
		$total 			= $this->get_total();

		wpb_paginate_links( $total, $paged, $rpp );
	?>
	</div>
	<form class="app-form" method="post">
	<?php
		$this->display_table( $transactions, false, $user_id );
	?>
	</form>
</div>
	<?php
	}

	/**
	 * Helper to produce a filter form
	 * @return string
	 */
	private function filter_form( ){
		# Get a sanitized filter params
		$filt = wpb_sanitize_search();
	?>
	<form method="get" >
		<?php
		$where = apply_filters( 'app_admin_transactions_where', " WHERE 1=1 ", $filt );

		$add_class = $filt['monweek'] ? 'class="app-option-selected"' : '';
		?>
		<select name="app_monweek" <?php echo $add_class ?>>
			<option value=""><?php _e('Filter by month/week','wp-base'); ?></option>
			<optgroup label="<?php echo ucwords( $this->a->get_text('month') ) ?>">
				<?php wpb_month_options( $where, 'transactions' ) ?>
			</optgroup>
			<optgroup label="<?php echo ucwords( $this->a->get_text('week') ) ?>">
				<?php wpb_week_options( $where, 'transactions' ) ?>
			</optgroup>
		</select>
		<?php

		$locations = $this->a->get_locations( 'name' );
		if ( $locations ) {
			$add_class = $filt['location_id'] ? 'class="app-option-selected"' : '';
		?>
		<select name="app_location_id" <?php echo $add_class ?>>
			<option value=""><?php _e('Filter by location','wp-base'); ?></option>
		<?php
			foreach ( $locations as $location ) {
				echo '<option '.selected( $filt['location_id'], $location->ID ).' value="'.$location->ID.'">'. $this->a->get_location_name( $location->ID ) .'</option>';
			}
		?>
		</select>
		<?php }
			$add_class = $filt['service_id'] ? 'class="app-option-selected"' : '';
		?>
		<select name="app_service_id" <?php echo $add_class ?>>
			<option value=""><?php _e('Filter by service','wp-base'); ?></option>
		<?php
		$services = $this->a->get_services( 'name' );
		foreach ( (array)$services as $service ) {
			echo '<option '.selected( $filt['service_id'], $service->ID ).' value="'.$service->ID.'">'. $this->a->get_service_name( $service->ID ) .'</option>';
		}
		?>
		</select>

		<?php self::print_fields() ?>
		<input type="submit" class="button" value="<?php _e('Filter','wp-base'); ?>" />
	</form>
	<?php
	}

	/**
	 * Render table header
	 * @param short: Render table for tooltip (hide unnecessary columns)
	 */
	private function table_header( $transactions, $columns, $short ){
		// Columns which will be hidden when short table is selected (e.g. to display transactions in tooltip)
		$hide_when_short = array('client','app_id','service','note','status','date_time', 'end_date_time');

		foreach ( $columns as $key => $col ) {
			if ( $transactions && $short && ( in_array( $col, $hide_when_short ) ) ) {
				continue;
			}

			if ( 'location' == $col && !$this->a->get_nof_locations() ) {
				continue;
			}

			$hidden			= in_array( $col, $this->get_hidden_columns() ) ? " hidden" : "";
			$col_primary	= in_array( $col, array( 'user' ) ) ? " column-primary" : "";
			$title			= '<span>'. apply_filters( 'app_transactions_column_title', str_replace( ':', '', $this->a->get_text($col)), $col) .'</span>';

			if ( ! $hidden ) {
				$this->colspan++;
			}

			?>
			<th class="manage-column column-<?php echo $col.$col_primary.$hidden; ?>" id="<?php echo $col; ?>" scope="col"><?php echo $title; ?></th>
			<?php
		}
	}

	/**
	 * Render transaction table
	 * @param short: Render table for tooltip (hide unnecessary columns)
	 */
	public function display_table( $transactions, $short = false, $user_id = 0 ) {

		$columns = self::get_allowed_columns( $user_id );

	?>
<table class="wp-list-table widefat dt-responsive display striped dataTable app-transactions<?php echo ($short ? ' app-tr-tooltip' : '') ?>">
	<thead>
		<tr>
			<?php
				$this->colspan = 0;
				$this->table_header( $transactions, $columns, $short );
			?>
		</tr>
	</thead>
	<tbody>
		<?php
		if ( $transactions ) {
			foreach ( $transactions as $tkey => $t ) {
				$hide 		= $short ? ' style="display:none"' : '';
				$service_id = $this->find_service_id( $t->transaction_app_ID );	?>

				<tr class="app-tr app-trans-tr app-trans-<?php echo $t->transaction_ID ?>">
					<th scope="row" class="column-payment_id">
						<?php $this->table_cell_ID( $t ) ?>
					</th>

		<?php foreach ( $columns as $col ) {

				if ( 'payment_id' == $col ) {
					continue;
				}

				if ( $short && in_array( $col, array( 'client', 'app_id', 'service', 'note', 'status', 'date_time', 'end_date_time' ) ) ) {
					continue;
				}

				if ( 'location' == $col && !$this->a->get_nof_locations() ) {
					continue;
				}

				$hidden	= in_array( $col, $this->get_hidden_columns() ) ? " hidden" : "";

				if ( 'client' == $col ) {
					?><td class="column-<?php echo $col.$hidden ?> column-primary has-row-actions"><?php
				} else {
					?><td class="column-<?php echo $col.$hidden ?>" data-colname="<?php echo esc_attr( strip_tags( $col ) ) ?>"><?php
				}

				switch ( $col ) {
					case 'client': ?>
						<div class="user-inner">
							<?php $this->table_cell_client( $t, $service_id ) ?>
						</div>
					<?php break;

					case 'app_id':
						echo $this->a->get_app_link( $t->transaction_app_ID );
						break;

					case 'date':
						echo '<span title="'. date_i18n("l", $t->transaction_stamp). '">'.date_i18n( $this->a->dt_format, $t->transaction_stamp ). '</span>';
						break;

					case 'location':
						echo apply_filters( 'app_transactions_location_name', wpb_cut( $this->a->get_location_name( $this->find_location_id( $t->transaction_app_ID ) ) ), $t );
						break;

					case 'service':
						echo apply_filters( 'app_transactions_service_name', wpb_cut( $this->a->get_service_name( $service_id ) ), $t );
						break;

					case 'worker':
					case 'provider':
						echo apply_filters( 'app_transactions_worker_name', wpb_cut( $this->a->get_worker_name( $this->find_worker_id( $t->transaction_app_ID ) ) ), $t );
						break;

					case 'date_time':
						echo apply_filters( 'app_transactions_booking_starts', $this->booking_start( $t->transaction_app_ID ), $t );
						break;

					case 'end_date_time':
						echo apply_filters( 'app_transactions_booking_ends', $this->booking_end( $t->transaction_app_ID ), $t );
						break;

					case 'amount': ?>
						<span>
						<?php echo wpb_format_currency( $t->transaction_total_amount / 100, $t->transaction_currency );	?>
						</span>
						<input type="hidden" class="app-trans-amount-net" value="<?php echo $t->transaction_total_amount / 100 ?>" />
						<input type="hidden" class="app-trans-currency" value="<?php echo $t->transaction_currency ?>" />
					<?php break;

					case 'reference':
						echo $t->transaction_paypal_ID;
						break;

					case 'note':
						if (! empty($t->transaction_note))
							echo '<span class="dashicons dashicons-admin-comments" title="'.$t->transaction_note.'"></span>';
						else
							echo '&nbsp;';
						break;

					case 'status':
						echo ( ! empty($t->transaction_status) ? ucwords( str_replace( '_', ' ', $t->transaction_status ) ) : __('None yet','wp-base') );
						break;

					case 'method':
						echo wpb_payment_method_name( $t->transaction_gateway ); ?>
						<input type="hidden" class="app-trans-gateway" value="<?php echo $t->transaction_gateway ?>" />
					<?php break;

					case $col: do_action( 'app_transactions_add_cell', '', $col, $t, $columns ); break;

					} ?>
					</td>
			<?php } ?>
				</tr>
			<?php
			}
		} else {
			?>
			<tr class="alternate" >
				<td colspan="<?php echo $this->colspan; ?>" scope="row"><?php _e('No payments have been found.','wp-base'); ?></td>
			</tr>
			<?php
		}
		?>
	</tbody>
	<tfoot>
		<tr>
			<?php $this->table_header( $transactions, $columns, $short ); ?>
		</tr>
	</tfoot>
</table>
	<?php
		do_action( 'app_mytransactions_after', $transactions );
	}

	/**
	 * Render ID cell
	 */
	private function table_cell_ID( $t ){
		global $app_gateway_plugins;

		if ( isset( $app_gateway_plugins[$t->transaction_gateway] ) ) {
			if ( 'manual-payments' == $t->transaction_gateway ) {
				if ( wpb_is_gateway_active( 'manual-payments' ) ) {
					echo '<a href="javascript:void(0)" data-app_id="'.$t->transaction_app_ID.'" data-db_id="'.$t->transaction_ID.'" class="app-trans-edit" title="'.__('Click to edit manual payment','wp-base').'">' .
					$t->transaction_ID .
					'</a>';
				} else {
					echo $t->transaction_ID;
				}
			} else {
				echo '<span title="'.__('Payments coming from payment processors cannot be edited','wp-base').'">' . $t->transaction_ID .'</span>';
			}
		} else {
			echo $t->transaction_ID;
		}
	}

	/**
	 * Render client cell
	 */
	private function table_cell_client( $t, $service_id ){
		global $app_gateway_plugins;

		echo BASE('User')->get_client_name( $t->transaction_app_ID, null, true );
		echo '<span class="booking-info">'.$this->a->get_app_link( $t->transaction_app_ID ) .'</span>';
		echo '<span class="booking-info">'.wpb_cut( $this->a->get_service_name( $service_id ) ) .'</span>';
		echo '<span class="booking-info">'. wpb_format_currency( $t->transaction_total_amount / 100, $t->transaction_currency ) .'</span>';
		echo '<span class="booking-info">'. date_i18n( $this->a->dt_format, $t->transaction_stamp ) .'</span>';
		echo '</div><div class="row-actions">';
		if ( isset( $app_gateway_plugins[$t->transaction_gateway] ) ) {
			if ( 'manual-payments' == $t->transaction_gateway ) {
				if ( wpb_is_gateway_active( 'manual-payments' ) ) {
					$tt = __('Click to edit manual payment','wp-base');
					$style = '';
				} else {
					$tt = __('Manual Payments gateway is not active','wp-base');
					$style = ' style="opacity:0.3"';
				}

				echo '<div'.$style.'><span class="dashicons dashicons-edit"></span><a href="javascript:void(0)" data-app_id="'.$t->transaction_app_ID.'" data-db_id="'.$t->transaction_ID.'" class="app-trans-edit" title="'.$tt.'">' .
				__('Edit', 'wp-base' ) .'</a></div>';

			} else {
				echo '<span title="'.__('Payments coming from payment processors cannot be edited','wp-base').'">' . __('Cannot be edited', 'wp-base').'</span>';
			}
		}
	}

	/**
	 * Get appointment with caching the result
	 * @since 3.5.10
	 * @return object
	 */
	private function get_app( $app_id ) {
		$id	= wpb_cache_prefix() . 'payment_app_'. $app_id;
		$app = wp_cache_get( $id );

		if ( false === $app ) {
			$app = wpb_get_app( $app_id, true ); # Cache latest results
			wp_cache_set( $id, $app );
		}

		return $app;
	}

	/**
	 * Get location ID given transaction app ID
	 * @since 3.5.10
	 * @return integer
	 */
	private function find_location_id( $app_id ) {
		$app = $this->get_app( $app_id );
		return ! empty( $app->location ) ? $app->location : 0;
	}

	/**
	 * Get service ID given transaction app ID
	 * @since 3.0
	 * @return integer
	 */
	private function find_service_id( $app_id ) {
		$app = $this->get_app( $app_id );
		return ! empty( $app->service ) ? $app->service : 0;
	}

	/**
	 * Get worker ID given transaction app ID
	 * @since 3.5.10
	 * @return integer
	 */
	private function find_worker_id( $app_id ) {
		$app = $this->get_app( $app_id );
		return ! empty( $app->worker ) ? $app->worker : 0;
	}

	/**
	 * Get worker ID given transaction app ID
	 * @since 3.5.10
	 * @return integer
	 */
	private function booking_start( $app_id ) {
		$app = $this->get_app( $app_id );
		return ! empty( $app->start ) ? date_i18n( $this->a->dt_format, wpb_strtotime( $app->start ) ) : '';
	}

	/**
	 * Get worker ID given transaction app ID
	 * @since 3.5.10
	 * @return integer
	 */
	private function booking_end( $app_id ) {
		$app = $this->get_app( $app_id );
		return ! empty( $app->end ) ? date_i18n( $this->a->dt_format, wpb_strtotime( $app->end ) ) : '';
	}
}
	BASE('Transactions')->add_hooks();
}