<?php
/**
 * WPB Admin Global Settings
 *
 * Displays and manages global settings on admin side
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBGlobalSettings' ) ) {

class WpBGlobalSettings{

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Constructor
     */
	public function __construct(){
		$this->a = BASE();
	}

	/**
     * Add admin actions
     */
	public function add_hooks() {
		add_action( 'app_submenu_before_tools', array( $this, 'add_submenu' ), 12 );
		add_action( 'app_save_settings', array( $this, 'save_settings' ), 12 );
	}

	/**
     * Add submenu page to main admin menu
     */
	public function add_submenu(){
		add_submenu_page('appointments', __('Global Settings','wp-base'), __('Global Settings','wp-base'), WPB_ADMIN_CAP, "app_settings", array($this,'settings'));
	}

	/**
     * Save admin settings
     */
	public function save_settings() {

		if ( isset( $_POST['app_nonce'] ) && ! wp_verify_nonce( $_POST['app_nonce'], 'update_app_settings' ) ) {
			wpb_notice( 'unauthorised', 'error' );
			return;
		}

		$saved = $settings_changed = $flush_cache = false;

		$old_options = $options = wpb_setting();

		// Save general settings
		if ( 'save_general' == $_POST['action_app'] ) {
			$settings_changed = true;
			$flush_cache = true;

			if ( ! empty( $_POST['create_make_page'] ) && isset( $_POST['app_page_type'] ) ) {
				$options['app_page_type'] = $_POST['app_page_type'];
			}

			$this->create_app_page();

			$options['refresh_url']					= trim( $_POST['refresh_url'] );

			# Calculate optimum time base
			if ( 'auto' == $_POST['min_time'] ) {
				$options['calc_min_time'] 			= $this->a->find_optimum_time_base();
			}

			$options['min_time']					= $_POST['min_time'];
			$options['additional_min_time']			= intval(round( preg_replace( '/[^0-9]/', '',  (int)$_POST['additional_min_time'] )/5 )*5);
			$options['app_lower_limit']				= preg_replace( '/[^0-9]/', '', $_POST['app_lower_limit'] );
			$options['app_limit']					= preg_replace('/[^0-9\-]/', '', $_POST['app_limit'] );
			$options['app_limit_unit']				= $_POST['app_limit_unit'];
			$options['clear_time']					= preg_replace('/[^0-9]/', '', $_POST['clear_time'] );
			$options['clear_time_pending_payment']	= preg_replace('/[^0-9]/', '', $_POST['clear_time_pending_payment'] );
			$options['countdown_time']				= preg_replace('/[^0-9]/', '', $_POST['countdown_time'] );

			$options['auto_confirm']				= $_POST['auto_confirm'];
			$options['time_slot_calculus_legacy']	= $_POST['time_slot_calculus_legacy'];
			$options['preselect_latest_service']	= $_POST['preselect_latest_service'];
			$options['lsw_priority']				= $_POST['lsw_priority'];
			$options['allow_confirm']				= $_POST['allow_confirm'];
			$options['allow_cancel'] 				= $_POST['allow_cancel'];
			$options['cancel_limit'] 				= preg_replace( '/[^0-9]/', '', $_POST['cancel_limit'] );
			$options['cancel_page'] 				= isset( $_POST['cancel_page'] ) ? $_POST['cancel_page'] : '';
			$options['allow_now']					= $_POST['allow_now'];
			$options['late_booking_time']			= preg_replace( '/[^0-9]/', '', $_POST['late_booking_time'] );
			$options['apt_count_max']				= preg_replace( '/[^0-9]/', '', $_POST['apt_count_max'] );
			$options['apt_count_min']				= preg_replace( '/[^0-9]/', '', $_POST['apt_count_min'] );

			$options['theme_builder_compat']		= $_POST['theme_builder_compat'];
			$options['lazy_load']					= $_POST['lazy_load'];
			$options['cache']						= $_POST['cache'];
			$options['debug_mode']					= $_POST['debug_mode'];

			$options['auto_delete']					= $_POST['auto_delete'];
			$options['auto_delete_time']			= preg_replace( '/[^0-9]/', '', $_POST['auto_delete_time'] );
			$options['log_settings']				= $_POST['log_settings'];
			$options['strict_check']				= $_POST['strict_check'];
			$options['admin_edit_collapse']			= $_POST['admin_edit_collapse'];
			$options['admin_toolbar']				= $_POST['admin_toolbar'];
			$options['records_per_page']			= preg_replace( '/[^0-9]/', '',  $_POST['records_per_page'] );
			$options['records_per_page_business']	= preg_replace( '/[^0-9]/', '',  $_POST['records_per_page_business'] );
			$options['schedule_content']			= stripslashes( $_POST['schedule_content'] );
			$options['description_post_type']		= $_POST['description_post_type'];

		} else if ( 'save_email_settings' == $_POST['action_app'] ) {
			$settings_changed = true;
			$options['admin_email']					= (strpos( $_POST['admin_email'], ',') || is_email( $_POST['admin_email'] )) ? $_POST['admin_email'] : get_option('admin_email');
			$options['from_name']					= stripslashes( $_POST['from_name'] );
			$options['from_email']					= is_email( trim($_POST['from_email']) ) ?  trim($_POST['from_email']) : '';
			$options['log_emails']					= $_POST['log_emails'];
			$options['use_html']					= $_POST['use_html'];

			$options['send_confirmation']			= $_POST['send_confirmation'];
			$options['send_confirmation_bulk']		= $_POST['send_confirmation_bulk'];
			$options['send_notification']			= $_POST['send_notification'];
			$options['confirmation_subject']		= stripslashes( $_POST['confirmation_subject'] );
			$options['confirmation_message']		= stripslashes( $_POST['confirmation_message'] );

			$options['send_pending']				= $_POST['send_pending'];
			$options['send_pending_bulk']			= $_POST['send_pending_bulk'];
			$options['pending_subject']				= stripslashes( $_POST['pending_subject'] );
			$options['pending_message']				= stripslashes( $_POST['pending_message'] );

			$options['send_completed']				= $_POST['send_completed'];
			$options['send_completed_bulk']			= $_POST['send_completed_bulk'];
			$options['completed_subject']			= stripslashes( $_POST['completed_subject'] );
			$options['completed_message']			= stripslashes( $_POST['completed_message'] );

			$options['send_cancellation']			= $_POST['send_cancellation'];
			$options['send_cancellation_bulk']		= $_POST['send_cancellation_bulk'];
			$options['cancellation_subject']		= stripslashes( $_POST['cancellation_subject'] );
			$options['cancellation_message']		= stripslashes( $_POST['cancellation_message'] );

			do_action( 'app_admin_email_settings_maybe_updated' );

			if ( isset( $_POST['send_test_email'] ) && 1 == $_POST['send_test_email'] ) {
				$this->send_test_email( $options['admin_email'] ); // This is the email entered in the setting, NOT the one from get_admin_email()
			}
		}


		if ( ! $settings_changed ) {
			return;
		}

		if ( $flush_cache ) {
			wpb_flush_cache();
			update_user_meta( get_current_user_id(), 'app_service_check_needed', true );
		}

		if ( $this->a->update_options( $options ) ) {
			wpb_notice( 'saved' );
			$this->settings_changed( $old_options, $options );
		}
	}

	/**
	 * Send a test email
	 * @since 2.0
	 */
	public function send_test_email( $admin_email ) {

		$r = $this->a->test_app_record();

		if ( empty( $r->service ) ) {
			return;
		}

		$body = $this->a->_replace( wpb_setting("confirmation_message"), $r, 'test_message' );

		// Find who will get this test email
		if ( $admin_email ) {
			$temp = explode( ',', $admin_email );
			foreach( $temp as $email ) {
				$e = trim( $email );
				if ( is_email( $e ) ) {
					$to[] = $e;
				}
			}
		} else {
			$to = BASE('User')->get_admin_email( );
		}

		$mail_result = wp_mail(
			$to,
			$this->a->_replace( __('This is a test email of WP BASE', 'wp-base'), $r, 'test_subject' ),
			$body,
			$this->a->message_headers( 'send_test_email', $r ),
			apply_filters( 'app_email_attachment', array(), $r, 'test' )
		);

		if ( $mail_result ) {
			wpb_notice( 'test_email_success' );
			do_action( 'app_email_sent', $body, $r, 0, 'test' );
			return true;
		} else {
			wpb_notice( 'test_email_fail' );
			do_action( 'app_email_failed', $body, $r, 0, 'test' );
			return false;
		}
	}

	/**
	 * When Time Base setting has been changed, reset/reschedule the scheduled event
	 * When caching setting changed accordingly, make a wp_remote_get test
	 * @since 3.0
	 */
	public function settings_changed( $old_options, $options ) {
		if ( ! empty( $old_options['min_time'] ) && $old_options['min_time'] != $options['min_time'] ) {
			wp_reschedule_event( strtotime( current_time( 'Y-m-d' ) ) - 24*3600, 'wpb_time_base_tick', 'app_time_base_tick' );
		}
	}

	/**
	 * Create a Make an Appointment/Book a Service Page
	 */
	public function create_app_page( $ecommerce = false ) {
		if ( ! empty( $_POST["create_make_page"] ) && ! empty( $_POST["app_page_type"] ) && isset($_POST["create_page_btn"]) ) {

			$monthly 		= '[app_book]';
			$weekly 		= '[app_book type="weekly"]';
			$table_10 		= '[app_book type="table" range="10"]';
			$table_2_days 	= '[app_book type="table" range="2 days"]';
			$flex_vertical	= '[app_book type="flex" range="2weeks" mode="1"]';
			$flex_hor		= '[app_book type="flex" range="10days" mode="5" from_week_start="0"]';

			$shortcode = trim( ${$_POST["app_page_type"]} );

			if ( $ecommerce ) {
				if ( 'woocommerce' == $ecommerce ) {
					$post_type		= 'product';
					$post_excerpt	= $shortcode;
					$post_content	= '&nbsp;';
				} else if ( 'edd' == $ecommerce ) {
					$post_type		= 'download';
					$post_excerpt	= '';
					$post_content	= $shortcode;
				}
			} else {
				$post_type		= 'page';
				$post_excerpt 	= '';
				$post_content 	= $shortcode;
			}

			$page_title = $ecommerce ? __( 'Book a Service', 'wp-base' ) : __( 'Make an Appointment', 'wp-base' );

			if ( $page_id = wp_insert_post(
					array(
						'post_title'	=> $page_title,
						'post_status'	=> 'publish',
						'post_type'		=> $post_type,
						'post_content'	=> $post_content,
						'post_excerpt'	=> $post_excerpt,
					)
				)
			) {
				BASE('Notices')->set_page_id( $page_id );
				wpb_notice( 'page_created' );

				if ( isset( $_POST['create_account_page'] ) ) {

					$page_found = $this->a->db->get_var(
								"SELECT ID FROM " . $this->a->db->posts .
								" WHERE post_name = 'app_account' LIMIT 1" );

					if ( ! $page_found && $list_page_id = wp_insert_post(
							array(
								'post_title'	=> 'My Account',
								'post_status'	=> 'publish',
								'post_type'		=> 'page',
								'pst_name'		=> 'app_account',
								'post_content'	=> '[app_account]'
							)
						)
					) {
						BASE('Notices')->set_account_page_id( $list_page_id );
						wpb_notice( 'page_created_list' );
						$permalink = get_permalink( $list_page_id );

						if ( $permalink && isset( $_POST['set_refresh_url'] ) && $this->a->update_options( $permalink, 'refresh_url' ) ) {
							if ( empty( $_POST['refresh_url'] ) ) {
								$_POST['refresh_url'] = $permalink;
							}
						}
					}
				}

				return $page_id;
			}
		}
	}

	/**
	 * HTML for create a Make an Appointment/Book a Service Page
	 * @param $ecommerce	string		Either 'woocommerce, 'edd' or false
	 */
	public function create_app_page_html( $ecommerce = false ) {

		$page_name	= 'app_booking';
		$page_title = $ecommerce ? __( 'Book a Service', 'wp-base' ) : __( 'Make an Appointment', 'wp-base' );
		$link 		=  $ecommerce ? '<a href="'.admin_url('admin.php?page=app_settings#refresh-url').'">'. __( 'here', 'wp-base' ).'</a>' : __( 'below', 'wp-base' );
		$ptype 		= wpb_setting("app_page_type");
		$disabled_cl = !$ecommerce || ( 'woocommerce' == $ecommerce && function_exists( 'WC' ) ) || ( 'edd' == $ecommerce && function_exists( 'EDD' ) ) ? '' : ' app-disabled-button';
		$title 		= $disabled_cl ? sprintf( __('%s is not activated','wp-base' ), ucwords( $ecommerce ) ) : '';

		?>
		<select name="app_page_type" class="app-no-save-alert" >
			<option value="monthly" <?php selected( 'monthly', $ptype ) ?>><?php _e('Monthly calendar', 'wp-base')?></option>
			<option value="weekly" <?php selected( 'weekly', $ptype ) ?>><?php _e('Weekly calendar - 2 weeks', 'wp-base')?></option>
			<option value="table_10" <?php selected( 'table_10', $ptype ) ?>><?php _e('Table view - 10 booking rows', 'wp-base')?></option>
			<option value="table_2_days" <?php selected( 'table_2_days', $ptype ) ?>><?php _e('Table view - 2 days', 'wp-base')?></option>
			<?php if ( class_exists( 'WpBPro' ) ) { ?>
				<option value="flex_vertical" <?php selected( 'flex_vertical', $ptype ) ?>><?php _e('Flex view - Vertical, 2 weeks', 'wp-base')?></option>
				<option value="flex_hor" <?php selected( 'flex_hor', $ptype ) ?>><?php _e('Flex view - Horizontal, 10 days', 'wp-base')?></option>
			<?php } ?>
		</select>

		<input title="<?php echo $title ?>" type="submit" class="app-no-save-alert button-secondary create_page_btn <?php echo $disabled_cl?> <?php echo ($ecommerce ? $ecommerce : 1) ?>" name="create_page_btn" value="<?php _e('Create Page Now','wp-base') ?>"/>

		<?php /* Value of the following field determines what page type we will create */ ?>
		<input type="hidden" class="app-no-save-alert disabled-when-hidden" name="create_make_page" value="<?php echo ($ecommerce ? $ecommerce : 1) ?>"/>

        <?php if ( ! $ecommerce ) { ?>
		<div class="app-mtmb">
			<input type="checkbox" name="create_account_page" class="app-no-save-alert" <?php checked( wpb_setting('list_page'), false ) ?>/>
			<?php _e('Also create a My Account Page', 'wp-base' ) ?>

			&nbsp;&nbsp;&nbsp;&nbsp;
			<input type="checkbox" name="set_refresh_url" class="app-no-save-alert" <?php checked( wpb_setting('refresh_url'), false ); if ( wpb_setting('refresh_url') ) echo " disabled='disabled'" ?> />
			<?php _e('Also redirect client to My Account Page', 'wp-base' ) ?>
		</div>
        <?php } ?>

		<span class="description app-btm">
		<?php printf( __('Immediately creates a front end booking page with title "%s" with the selected booking layout by inserting <code>[app_book]</code> shortcode with required parameters. You can edit, add parameters to the shortcode and customize this page later.', 'wp-base'), $page_title ); ?>
		<?php printf( __('For layout examples, please visit our %s.','wp-base'), '<a href="'.WPB_DEMO_WEBSITE.'" target="_blank">'. __('demo website', 'wp-base'). '</a>'); ?>

        <?php if ( ! $ecommerce ) { ?>
        <?php _e('Checking "Create a My Account Page" checkbox will additionally create a page with title "My Account" with <code>[app_account]</code> shortcode.', 'wp-base'); ?>
		<?php printf( __('Checking "Redirect client..." checkbox will automatically fill the Redirect Url field %s.', 'wp-base'), $link ); ?>
		<?php } ?>

        </span>
		<?php
		$existing_page_id = $this->a->first_page_id( $page_name, $page_title );
		if ( $existing_page_id ) { ?>
			<br />
			<span class="description app-btm"><?php printf( __('<b>Note:</b> You already have a "%s" page. If you click Create page Now button, another page with the same title will be created. To edit existing page:' , 'wp-base'), $page_name ); ?>
			<a class="app-btm" href="<?php echo admin_url('post.php?post='.$existing_page_id.'&amp;action=edit')?>" target="_blank"><?php _e('Click here', 'wp-base')?></a>
			&nbsp;
			<?php _e('To view the page:', 'wp-base' ) ?>
			<a class="app_existing_make_page app-btm" href="<?php echo get_permalink( $existing_page_id)?>" target="_blank"><?php _e('Click here', 'wp-base')?></a>
			</span>
		<?php }
	}

	/**
	 * Admin Global Settings HTML code
	 */
	public function settings() {

		wpb_admin_access_check( 'manage_global_settings' );
	?>
<div class="wrap app-page app-global-settings">
	<h2 class="app-dashicons-before dashicons-admin-settings"><?php echo __('Global Settings','wp-base'); ?></h2>
	<h3 class="nav-tab-wrapper">
	<?php
		$tab 	= ! empty( $_GET['tab'] ) ? wpb_clean( $_GET['tab'] ) : 'general';
		$tabhtml 	= array();

		foreach ( wpb_admin_tabs( 'global' ) as $stub => $title ) {
			$class = $stub == $tab ? ' nav-tab-active' : '';
			$tabhtml[] = '	<a href="' . admin_url( 'admin.php?page=app_settings&amp;tab=' . $stub ) . '" class="nav-tab'.$class.'" id="app_tab_'.$stub.'">'.$title.'</a>';
		}

		echo implode( "\n", $tabhtml );
	?>
	</h3>
	<div class="clear"></div>
	<?php switch( $tab ) {
	/*******************************
	* Main tab
	********************************
	*/
	case 'general':
	?>
	<div id="poststuff" class="metabox-holder meta-box-sortables">
		<?php wpb_infobox( sprintf( __('WP BASE plugin makes it possible for your clients to apply for appointments from the front end and for you to enter appointments from backend. On this page, you can set settings which will be valid throughout the website. Please note that some settings can be overridden per page basis by setting appropriate shortcode parameters. For example, user info fields can be set on the Confirmation shortcode overriding settings on this page. More information can be found in our %s.', 'wp-base'), '<a href="'. WPBASE_URL.'knowledge-base/" title="'. __('Click to access Knowledge Base', 'wp-base').'" target="_blank">'. __( 'Knowledge Base', 'wp-base') .'</a>' ) ); ?>
		<form id="app_main_settings_form" class="app-form" method="post" action="<?php echo wpb_add_query_arg( null, null )?>">

			<div id="wpb-quick-start" class="postbox">
				<div class="postbox-header">
					<h3 class="hndle">
						<span><?php _e('Quick Start', 'wp-base' ) ?></span>
					</h3>
				</div>
				<div class="inside">
					<table class="form-table">
						<tr>
							<th scope="row" ><?php _e('Create a Booking Page', 'wp-base')?></th>
							<td>
								<?php $this->create_app_page_html(); ?>
							</td>
						</tr>
						<?php wpb_setting_text( 'refresh_url', 'app-600' ) ?>
					</table>
				</div>
			</div>

			<div id="tabs" class="app-tabs">
				<ul></ul>

				<div class="postbox">
					<h3 class="hndle"><span class="dashicons dashicons-clock"></span><span><?php _e('Time Settings', 'wp-base' ) ?></span></h3>
					<div class="inside">
						<table class="form-table"><?php
							$this->print_time_base_setting();

							wpb_setting_text( 'additional_min_time' );
							wpb_setting_text( 'app_lower_limit', 'app-w-select app-50' );

							$limit 		= wpb_setting("app_limit");
							$limit_unit = wpb_setting('app_limit_unit');
							?>
							<tr id="app-upper-limit">
								<th scope="row" ><?php WpBConstant::echo_setting_name('app_limit') ?></th>
								<td>
									<input type="text" class="app-w-select app-50" name="app_limit" value="<?php echo $limit ?>" />
									<select name="app_limit_unit" class="app-btm">
										<option value="day" <?php selected( $limit_unit, 'day' ); ?>><?php echo _n('day', 'days', $limit, 'wp-base')?></option>
										<option value="month" <?php selected( $limit_unit, 'month' ); ?>><?php echo _n('month', 'months', $limit, 'wp-base')?></option>
									</select>
									<span class="description app-btm"><?php WpBConstant::echo_setting_desc('app_limit') ?></span>
								</td>
							</tr><?php
							wpb_setting_text( 'clear_time', 'app-w-select app-50' );
							wpb_setting_text( 'clear_time_pending_payment', 'app-w-select app-50' );
							wpb_setting_text( 'countdown_time' );
							?>
						</table>
					</div>
				</div>

				<div class="postbox">
					<h3 class="hndle"><span class="dashicons dashicons-tickets-alt"></span><span><?php _e('Booking & Cancelling', 'wp-base' ) ?></span></h3>
					<div class="inside">
						<table class="form-table"><?php
							$this->print_auto_confirm();
							wpb_setting_yn( 'time_slot_calculus_legacy' );
							$this->print_priority();
							wpb_setting_yn( 'preselect_latest_service' );
							wpb_setting_yn( 'allow_confirm' );
							wpb_setting_yn( 'allow_cancel' );
							wpb_setting_text( 'cancel_limit' ); ?>
							<tr>
								<th scope="row"><?php WpBConstant::echo_setting_name('cancel_page') ?></th>
								<td>
								<?php
									if ( $dropdown = wp_dropdown_pages( array( "echo"=>false,"show_option_none"=>__('Home page', 'wp-base'),"option_none_value "=>0,"name"=>"cancel_page", "selected"=>wpb_setting("cancel_page") ) ) )
										echo $dropdown;
									else
										echo '<span class="app-b app-btm">' . __( 'There are no pages!', 'wp-base' ) .'</span>';
									?>
									<span class="description app-btm"><?php WpBConstant::echo_setting_desc('cancel_page') ?></span>
								</td>
							</tr><?php
							wpb_setting_yn( 'allow_now' );
							wpb_setting_text( 'late_booking_time' );
							wpb_setting_text( 'apt_count_max' );
							wpb_setting_text( 'apt_count_min' ); ?>

						</table>
					</div>
				</div>

				<?php do_action( 'app_admin_settings_after_booking' ) ?>

				<div class="postbox">
					<h3 class="hndle"><span class="dashicons dashicons-performance"></span><span><?php _e('Performance & Setup', 'wp-base' ) ?></span></h3>
					<div class="inside">
						<table class="form-table">
						
							<?php 
								wpb_setting_yn( 'theme_builder_compat' );
								wpb_setting_yn( 'lazy_load' );
							?>
							<tr>
								<th scope="row" ><?php WpBConstant::echo_setting_name('cache') ?></th>
								<td>
								<select name="cache">
									<option value="" <?php if ( !wpb_setting('cache') ) echo "selected='selected'"?>><?php _e('Off', 'wp-base')?></option>
									<option value="on" <?php if ( wpb_setting('cache') == 'on' || wpb_setting('cache') == 'preload' ) echo "selected='selected'"?>><?php _e('On', 'wp-base')?></option>
								</select>
								<span class="description app-btm"><?php WpBConstant::echo_setting_desc('cache') ?></span>
								</td>
							</tr>
							
							<tr>
								<th scope="row"><?php WpBConstant::echo_setting_name('description_post_type') ?></th>
								<td>
								<select name="description_post_type">
								<?php
								$dpt = wpb_setting( 'description_post_type', 'page' );
								$post_types = get_post_types( array( 'public' => true ), 'objects' );

								foreach ( $post_types as $post_type ) {
									if ( 'attachment' != $post_type->name ) {
										echo '<option value="'.$post_type->name .'"'. selected( $dpt, $post_type->name, false) . '>'. $post_type->labels->singular_name . '</option>';
									}
								}
								?>
								</select>
								<span class="description app-btm"><?php WpBConstant::echo_setting_desc('description_post_type') ?></span>
								</td>
							</tr>
						</table>
					</div>
				</div>

				<div class="postbox">
					<h3 class="hndle"><span class="dashicons dashicons-forms"></span><span><?php _e('Admin Side Preferences', 'wp-base' ) ?></span></h3>
					<div class="inside">
						<table class="form-table"><?php
							wpb_setting_yn( 'debug_mode' );
							wpb_setting_yn( 'auto_delete' );
							wpb_setting_text( 'auto_delete_time' );
							wpb_setting_yn( 'log_settings' );
							wpb_setting_yn( 'strict_check' );
							wpb_setting_yn( 'admin_edit_collapse' );
							wpb_setting_yn( 'admin_toolbar' );
							wpb_setting_text( 'records_per_page' );
							wpb_setting_text( 'records_per_page_business' );
							wpb_setting_text( 'schedule_content', 'app-1' );
							?>

						</table>
					</div>
				</div>
			</div><!-- tabs -->

			<div class="submit app-manage-row">
				<input type="hidden" name="action_app" value="save_general" />
				<?php wp_nonce_field( 'update_app_settings', 'app_nonce', true ) ?>
				<div>
					<input type="submit" class="button-primary" value="<?php _e('Save General Settings', 'wp-base' ) ?>" />
				</div>
				<div>
				<?php if ( BASE('EXIM') ) : ?>
					<a href="<?php echo admin_url('admin.php?page=app_tools&tab=impex'); ?>" title="<?php _e('Link for export/import of settings', 'wp-base' ) ?>" ><?php _e('Export/Import Settings', 'wp-base' ) ?></a>
				<?php endif; ?>
				</div>
			</div>

		</form>
	</div>
	<?php break;
	/*******************************
	* Email tab
	********************************
	*/
		case 'email':
			$this->print_email_settings() ;
			break;
	/*******************************
	* Advanced tab
	********************************
	*/
		case 'advanced':
			$this->print_advanced_tab();
			break;

	/*******************************
	* Additional tabs
	********************************
	*/
		case $tab:
			do_action( 'app_'.$tab.'_tab' );
			break;

		} ?>
</div><!-- Wrap -->
	<?php
	}

	/**
	 * Render options for time base (min time)
	 */
	private function print_time_base_setting() {
	?>
	<tr id="time-base">
		<th scope="row" ><?php WpBConstant::echo_setting_name('min_time') ?></th>
		<td>
			<select name="min_time">
	<?php
		$class = '';
		if ( 'auto' == wpb_setting("min_time") ) {
			if ( wpb_setting("calc_min_time") ) {
				$ctext = wpb_readable_duration( $this->a->get_min_time() );
			} else {
				$ctext = __('Not possible to calculate. 60 mins is in effect.','wp-base');
				$class = " class='error'";
			}
		}

		echo '<option'.$class.' value="auto">'. __('Auto','wp-base') . '</option>';

		$min_time_set = wpb_setting("min_time");
		foreach ( $this->a->time_base() as $min_time ) {
			echo '<option value="'.$min_time .'" '. selected( $min_time_set, $min_time, false ) . '>'.
				  wpb_readable_duration( $min_time ) . '</option>';
		}
	?>		</select>
			<?php
			if ( 'auto' == wpb_setting("min_time") ) {
				echo '<span class="description app-btm app-b">'. __('Calculated value: ','wp-base') . $ctext .'</span>  ';
			}
			?>
			<span class="description app-btm"><?php WpBConstant::echo_setting_desc('min_time') ?></span>
		</td>
	</tr>
	<?php
	}

	/**
	 * Render auto confirm setting
	 */
	private function print_auto_confirm() {
		$val = wpb_setting('auto_confirm');
	?>
	<tr id="auto-confirm">
		<th scope="row" ><?php WpBConstant::echo_setting_name('auto_confirm') ?></th>
		<td>
			<select name="auto_confirm">
				<option value="no" <?php selected( $val, 'no' ) ?>><?php _e('No', 'wp-base')?></option>
				<option value="yes" <?php selected( $val, 'yes' ) ?>><?php _e('Yes', 'wp-base')?></option>
			</select>
			<span class="description app-btm"><?php WpBConstant::echo_setting_desc('auto_confirm') 	. '&nbsp;'. _e( 'To display the table summarizing effect of different settings please click the "i" icon:', 'wp-base'); ?></span>
			&nbsp;
			<a href="javascript:void(0)" class="info-button" title="<?php _e('Click to toggle details', 'wp-base')?>">
				<img src="<?php echo WPB_PLUGIN_URL . '/images/information.png'?>" alt="" />
			</a>
			<div class="app-instructions" style="display:none">
				<table class="widefat">
					<tr>
						<th><?php _e( 'Setting of Auto Confirm' ) ?></th>
						<th style="text-align:center" colspan="2"><?php _e( 'Resulting Status When...' ) ?></th>
					</tr>
					<tr>
						<th>&nbsp;</th>
						<th><?php _e( 'Payment Required = NO OR Price = 0' ) ?></th>
						<th><?php _e( 'Payment Required = YES AND Price > 0' ) ?></th>
					</tr>
					<tr>
						<td><?php _e( 'No' ) ?></td>
						<td><?php _e( 'Pending' ) ?></td>
						<td><?php _e( 'Pending&rarr;{Payment}&rarr;Paid' ) ?></td>
					</tr>
					<tr>
						<td><?php _e( 'Yes' ) ?></td>
						<td><?php _e( 'Confirmed' )?></td>
						<td><?php _e( 'Pending&rarr;{Payment}&rarr;Paid' ) ?></td>
					</tr>
				</table>
			</div>
		</td>
	</tr>
	<?php
	}

	/**
	 * Render priority setting
	 */
	private function print_priority() {
	?>
	<tr>
		<th scope="row" ><?php WpBConstant::echo_setting_name('lsw_priority') ?></th>
		<td>
		<?php $pri = wpb_setting( 'lsw_priority', WPB_DEFAULT_LSW_PRIORITY); ?>
		<select name="lsw_priority">
			<option value="LSW" <?php selected( $pri, 'LSW' ) ?>><?php _e('Location &gt; Service &gt; Provider', 'wp-base')?></option>
			<option value="LWS" <?php selected( $pri, 'LWS' ) ?>><?php _e('Location &gt; Provider &gt; Service', 'wp-base')?></option>
			<option value="SLW" <?php selected( $pri, 'SLW' ) ?>><?php _e('Service &gt; Location &gt; Provider', 'wp-base')?></option>
			<option value="SWL" <?php selected( $pri, 'SWL' ) ?>><?php _e('Service &gt; Provider &gt; Location', 'wp-base')?></option>
			<option value="WLS" <?php selected( $pri, 'WLS' ) ?>><?php _e('Provider &gt; Location &gt; Service', 'wp-base')?></option>
			<option value="WSL" <?php selected( $pri, 'WSL' ) ?>><?php _e('Provider &gt; Service &gt; Location', 'wp-base')?></option>
		</select>
		<span class="description app-btm"><?php WpBConstant::echo_setting_desc('lsw_priority') ?></span>
		</td>
	</tr>
	<?php
	}

	/**
	 * Render email tab
	 */
	private function print_email_settings() {
	?>
<div id="poststuff" class="metabox-holder meta-box-sortables">
	<?php
	wpb_desc_infobox( 'email' );

	do_action( 'app_admin_email_settings_after_info' );
	?>
	<form id="app_email_settings_form" class="app-form" method="post" action="<?php echo wpb_add_query_arg( null, null )?>">
		<div id="wpb-general-email-settings" class="postbox">
			<h3 class="hndle"><span class="notification_settings"><?php _e('General', 'wp-base' ) ?></span></h3>
            <div class="inside">
				<table class="form-table">
				<?php
					wpb_setting_text( 'admin_email', 'app-600' );
					wpb_setting_text( 'from_name', 'app-600' );
					wpb_setting_text( 'from_email', 'app-600' );
					wpb_setting_yn( 'log_emails' );
					wpb_setting_yn( 'use_html' );
				?>
					<tr id="send-test-email">
						<th scope="row" ><?php _e('Send a Test email', 'wp-base')?></th>
						<td>
						<input type="button" class="app-no-save-alert button-secondary" name="send_test_email_btn" value="<?php _e('Send Now', 'wp-base' ) ?>"/>
						<input type="hidden" class="app-no-save-alert" name="send_test_email" />
						<span class="description app-btm"><?php _e('Clicking this button will IMMEDIATELY try to send a test email to the admin email(s) above using confirmation message template below. Most email problems are related to incorrect hosting email installations. In other words, email application may not be correctly configured for your website. This may help to figure out if you have such a problem in the first place.', 'wp-base' ) ?></span>
						</td>
					</tr>
				</table>
			</div>
		</div>

		<div id="tabs" class="app-tabs">
			<ul></ul>

			<div class="postbox">
				<h3 class="hndle">
					<span class="notification_settings">
					<abbr title="<?php _e( 'Emails sent when status is confirmed.', 'wp-base' ) ?>">
						<?php _e('Confirmation', 'wp-base' ) ?>
					</abbr>
					</span>
				</h3>
				<div class="inside">
					<table class="form-table">
						<?php wpb_setting_yn( 'send_confirmation' ) ?>
						<?php wpb_setting_yn( 'send_confirmation_bulk' ) ?>
						<tr>
							<th scope="row"><?php WpBConstant::echo_setting_name('confirmation_subject') ?></th>
							<td>
							<input value="<?php echo esc_attr( wpb_setting('confirmation_subject')); ?>" class="app-1" name="confirmation_subject" type="text" />
							<?php do_action( 'app_admin_email_settings_after_subject', 'confirmation' ) ?>
							</td>
						</tr>
						<?php
						$this->print_email_msg_field( 'confirmation' );
						do_action( 'app_admin_after_email', 'confirmation' ) ?>
					</table>
				</div>
			</div>
			
			<?php do_action( 'app_admin_after_email_confirmation' ) ?>

			<div class="postbox">
				<h3 class="hndle">
					<span class="notification_settings">
					<abbr title="<?php _e( 'Settings for email messages when appointment status is Pending.', 'wp-base' ) ?>">
						<?php _e('Pending', 'wp-base' ) ?>
					</abbr>
					</span>
				</h3>
				<div class="inside">
					<table class="form-table">
						<?php
						wpb_setting_yn( 'send_pending' );
						wpb_setting_yn( 'send_pending_bulk' );
						wpb_setting_yn( 'send_notification' );
						?>
						<tr>
							<th scope="row"><?php WpBConstant::echo_setting_name('pending_subject') ?></th>
							<td>
							<input value="<?php echo esc_attr( wpb_setting('pending_subject')); ?>" class="app-1" name="pending_subject" type="text" />
							<?php do_action( 'app_admin_email_settings_after_subject', 'pending' ) ?>
							</td>
						</tr>
						<?php
						$this->print_email_msg_field( 'pending' );
						do_action( 'app_admin_after_email', 'pending' ) ?>
					</table>
				</div>
			</div>

			<?php do_action( 'app_admin_after_email_pending' ) ?>

			<div class="postbox">
				<h3 class="hndle">
					<span class="notification_settings">
					<abbr title="<?php _e( 'Settings for email messages when appointment has just completed or its status is Completed.', 'wp-base' ) ?>">
						<?php _e('Completed', 'wp-base' ) ?>
					</abbr>
					</span>
				</h3>
				<div class="inside">
					<table class="form-table">
						<?php wpb_setting_yn( 'send_completed' ) ?>
						<?php wpb_setting_yn( 'send_completed_bulk' ) ?>
						<tr>
							<th scope="row"><?php WpBConstant::echo_setting_name('completed_subject') ?></th>
							<td>
							<input value="<?php echo esc_attr( wpb_setting('completed_subject')); ?>" class="app-1" name="completed_subject" type="text" />
							<?php do_action( 'app_admin_email_settings_after_subject', 'completed' ) ?>
							</td>
						</tr>
						<?php
						$this->print_email_msg_field( 'completed' );
						do_action( 'app_admin_after_email', 'completed' ) ?>
					</table>
				</div>
			</div>

			<?php do_action( 'app_admin_after_email_completed' ) ?>

			<div class="postbox">
				<h3 class="hndle">
					<span class="notification_settings">
					<abbr title="<?php _e( 'Settings for email messages when appointment is cancelled or its status is Removed.', 'wp-base' ) ?>">
						<?php _e('Cancellation', 'wp-base' ) ?>
					</abbr>
					</span>
				</h3>
				<div class="inside">
					<table class="form-table">
						<?php wpb_setting_yn( 'send_cancellation' ) ?>
						<?php wpb_setting_yn( 'send_cancellation_bulk' ) ?>
						<tr>
							<th scope="row"><?php WpBConstant::echo_setting_name('cancellation_subject') ?></th>
							<td>
							<input value="<?php echo esc_attr( wpb_setting('cancellation_subject')); ?>" class="app-1" name="cancellation_subject" type="text" />
							<?php do_action( 'app_admin_email_settings_after_subject', 'cancellation' ) ?>
							</td>
						</tr>
						<?php
						$this->print_email_msg_field( 'cancellation' );
						do_action( 'app_admin_after_email', 'cancellation' );
						?>
					</table>
				</div>
			</div>

			<?php do_action( 'app_admin_after_email_cancellation' ) ?>

		</div><!-- Tabs -->

		<input type="hidden" name="action_app" value="save_email_settings" />
		<?php wp_nonce_field( 'update_app_settings', 'app_nonce' ) ?>
		<p class="submit">
			<input type="submit" class="button-primary" value="<?php _e('Save Email Settings', 'wp-base' ) ?>" />
		</p>
	</form>
</div>
	<?php
	}

	/**
	 * Render textarea or wp_editor for email messages
	 * @param $context		string		confirmation, pending, completed or cancellation
	 */
	public function print_email_msg_field( $context ) {
		$set = $context . '_message';
	?>
		<tr id="<?php echo str_replace( '_', '-', $set ) ?>">
			<th scope="row"><?php WpBConstant::echo_setting_name( $set ) ?></th>
			<td>
		<?php
			if ( wpb_setting('use_html') == 'yes' ) {
				wp_editor( wpb_setting( $set ), $set, wpb_editor_settings( __CLASS__ ) );
			} else { ?>
				<textarea class="app-1" name="<?php echo $context ?>_message"><?php echo esc_textarea( wpb_setting( $set ) ) ?></textarea>
		<?php } ?>
		<?php do_action( 'app_admin_email_settings_after_message', $context ) ?>
			</td>
		</tr>
	<?php
	}

	/**
	 * Contents of advanced tab
	 */
	private function print_advanced_tab(){
	?>
	<div id="poststuff" class="metabox-holder metabox-holder-advanced">
		<?php wpb_infobox( __('Some business models may need advanced settings. Here you can adjust them. These settings are enabled by Addons.', 'wp-base' ) ); ?>

		<?php do_action( 'app_admin_advanced_settings_after_info' ); ?>

		<form class="app-form" method="post" action="<?php echo wpb_add_query_arg( null, null )?>">
			<div id="tabs" class="app-tabs">
				<ul></ul>

				<?php do_action( 'app_advanced_settings' ); ?>

			</div><!--tabs -->

			<input type="hidden" name="action_app" value="save_advanced" />
			<?php wp_nonce_field( 'update_app_settings', 'app_nonce' ); ?>
			<p class="submit">
				<input type="submit" class="button-primary" value="<?php _e('Save Advanced Settings', 'wp-base' ) ?>" />
			</p>
		</form>
	</div>
	<?php
	}

}

	BASE('GlobalSettings')->add_hooks();
}
