<?php
/**
 * WPB Clients
 *
 * Creates clients list
 *
 * Copyright © 2018-2022 Hakan Ozevin
 * @author		Hakan Ozevin
 * @package     WC BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.9.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBClients' ) ) {

class WpBClients{

	/**
     * Clients page identifier
     */
	public $clients_page;

	/**
     * WC BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Admin page ID
     */
	const PAGE = 'app_clients';

	/**
     * Constructor
     */
	public function __construct(){
		$this->a = BASE();
	}

	/**
     * Add admin actions
     */
	public function add_hooks() {
		add_filter( 'app_user_fields', array( $this, 'user_fields' ) );
		add_action( 'app_menu_before_all', array( $this, 'add_menu' ), 18 );
		add_action( 'updated_user_meta', array( $this, 'add_hidden_col_check' ), 10, 4 );
		add_action( 'added_user_meta', array( $this, 'add_hidden_col_check' ), 10, 4 );
		add_action( 'admin_init', array( $this, 'change_status_bulk' ) );

		add_action( 'wp_ajax_inline_edit_client', array( $this, 'inline_edit' ) );
		add_action( 'wp_ajax_inline_edit_save_client', array( $this, 'inline_edit_save' ) );
		add_action( 'wp_ajax_app_populate_user_client', array( $this, 'populate_user' ) );
	}

	/**
	 * Add display_name to user fields
	 * @since 3.9.0
	 * @return array
	 */
	public function user_fields( $fields ) {
		if ( self::is_clients_page() ) {
			$fields[] = 'display_name';
		}

		return $fields;
	}

	/**
	 * Check if this is clients page
	 * @since 3.0
	 * @return false|string
	 */
	private static function is_clients_page() {
		$screen_id	= wpb_get_current_screen_id();

		if ( strpos( $screen_id, self::PAGE ) !== false ) {
			return $screen_id;
		} else if ( ! empty( $_POST['page'] ) && strpos( $_POST['page'], self::PAGE ) !== false ) {
			return $_POST['page'];
		}

		return false;
	}

	/**
     * Add menu and submenu page to main admin menu for admin
     */
	public function add_menu(){

		$this->clients_page = wpb_add_submenu_page('appointments', __('Clients','wp-base'), __('Clients','wp-base'), array(WPB_ADMIN_CAP,'manage_clients'), self::PAGE, array($this,'client_list'));
		add_filter( "manage_{$this->clients_page}_columns", array( $this, 'manage_columns' ) );
	}

	/**
	 * Query to check if user is member of blog
	 * @since 3.7.5.2
	 */
	private function multisite_query(){
		return is_multisite()
			  ? " AND ID IN (SELECT user_id FROM {$this->a->db->usermeta} AS usermeta2
				WHERE users.ID=usermeta2.user_id AND meta_key='".$this->a->db->get_blog_prefix()."capabilities' AND meta_value<>'') "
			  : "";
	}

	/**
	 * Define an array of allowed columns in clients page
	 * Called after addons loaded
	 */
	public function get_allowed_columns() {
		$allowed = array( 'delete','id','client','first_name','last_name','email','phone','address','city','zip','nof_bookings','revenue' );
		return apply_filters( 'app_clients_allowed_columns', $allowed, '' );
	}

	/**
     * Add some columns as hidden to Bookings page if there is no selection before
     */
	private function default_hidden_columns( ) {
		return apply_filters( 'app_clients_hidden_columns', array( 'first_name','last_name','phone','address','city','zip' ) );
	}

	/**
	 * Get hidden columns for the table
	 * @since 3.0
	 * @return array
	 */
	private function get_hidden_columns(){
		if ( ! $page = self::is_clients_page() ) {
			return array();
		}

		$hidden_columns = get_user_option( 'manage'.$page.'columnshidden' );

		if ( empty( $hidden_columns['hidden_check'] ) ) {
			$hidden_columns = $this->default_hidden_columns();
		}

		return $hidden_columns;
	}

	/**
	 * Add hidden_check option
	 * @since 3.0
	 */
	public function add_hidden_col_check( $meta_id, $object_id, $meta_key, $_meta_value ) {
		if ( ! $page = self::is_clients_page() ) {
			return;
		}

		if ( 'manage'.$page.'columnshidden' != $meta_key ) {
			return;
		}

		remove_action( 'updated_user_metadata', array( $this, 'add_hidden_col_check' ) );
		remove_action( 'added_user_metadata', array( $this, 'add_hidden_col_check' ) );

		$meta_value = array_merge( (array)$_meta_value, array( 'hidden_check' => 1 ) );
		update_metadata( 'user', $object_id, $meta_key, $meta_value );

		add_action( 'updated_user_metadata', array( $this, 'add_hidden_col_check' ), 10, 4 );
		add_action( 'added_user_metadata', array( $this, 'add_hidden_col_check' ), 10, 4 );

	}

	/**
	 * Add hide checkboxes to admin screen options
	 * Output is fed to manage_{screen}_columns filter
	 * @param $arr is irrelevant
	 * @since 2.0
	 * @return array
	 */
	public function manage_columns( $arr ) {

		if ( ! is_admin() || empty( $this->clients_page ) || wpb_get_current_screen_id( ) != $this->clients_page ) {
			return;
		}

		$args = array();

		foreach ( $this->get_allowed_columns() as $col ) {
			if ( 'delete' == $col || 'client' == $col ) {
				continue;
			}

			if ( 'id' == $col ) {
				$args['id']	= __( 'ID', 'wp-base' );
			}

			if ( 'nof_bookings' == $col ) {
				$args['nof_bookings']	= $this->a->get_text('nof_bookings');
			} else if ( 'revenue' == $col ) {
				$args[ $col ] = __( 'Revenue', 'wp-base' );
			} else {
				$args[ $col ] = $this->a->get_text($col);
			}
		}

		return $args;
	}

	/**
	 * Helper to pass page and tab variables
	 */
	public static function print_fields(){
		if ( ! empty( $_GET['tab'] ) ) { ?>
			<input type="hidden" name="tab" value="<?php echo wpb_clean( $_GET['tab'] ) ?>"/>
		<?php }
		if ( is_admin() ) { ?>
			<input type="hidden" name="page" value="<?php echo self::PAGE ?>"/>
		<?php }
	}

	/**
	 * Creates the list for Appointments admin page
	 * Also used by Front End Management addon
	 */
	public function client_list() {

		global $type, $post, $current_user;

		wpb_admin_access_check();

		wpb_add_action_footer( $this );

	?>
	<div class="wrap app-page wp-clearfix">
		<h2 class="app-dashicons-before dashicons-businessperson"><?php echo __('Clients','wp-base'); ?>
			<a href="javascript:void(0)" class="add-new-h2 add-new-client"><?php _e('Add New', 'wp-base') ?></a>
			<img class="add-new-waiting" style="display:none;" src="<?php echo admin_url('images/wpspin_light.gif')?>" alt="">
		</h2>

		<div class="app-manage-row first-row">
			<div class="app-manage-first-column">
				<?php // $this->status_links_ul( $type ); ?>
			</div>

			<div class="app-manage-second-column">
				<?php $this->search_form() ?>
			</div>
		</div>

		<div class="app-manage-row third-row">
			<div class="app-manage-first-column">
				<?php $this->status_change_form() ?>
			</div>

		<?php
		# Pagination
		if ( is_admin() ) {
			$paged = empty($_GET['paged']) ? 1 : absint( $_GET['paged'] );
		} else {
			$paged = get_query_var( 'paged' ) ? absint( get_query_var( 'paged' ) ) : 1;
		}

		$rpp 		= wpb_setting( 'records_per_page', 20 ); # Records per page
		$startat	= ($paged - 1) * $rpp;
		$users		= $this->get_admin_clients( $startat, $rpp );
		$total		= $this->get_clients_total();

		wpb_paginate_links( $total, $paged, $rpp );

		?>
		</div>
		<?php
		$this->display_table( $users );
		?>
	</div> <!-- wrap -->
		<?php
	}

	/**
	 * Query to check if user has client role
	 * @since 3.7.5.2
	 */
	private function cap_query(){
		return " AND ID IN (SELECT user_id FROM {$this->a->db->usermeta} AS usermeta2
				WHERE users.ID=usermeta2.user_id AND meta_key='".$this->a->db->get_blog_prefix()."capabilities' AND meta_value LIKE '%wpb_client%') "
		;
	}

	/**
	 * Build mysql query and return results for clients
	 */
	private function get_admin_clients( $startat, $num ) {

		if ( ! empty( $_GET['app_s'] ) && ! empty( $_GET['stype'] ) ) {
			$stype 	= esc_sql( $_GET['stype'] );
			$s 		= apply_filters( 'app_search', esc_sql( $_GET['app_s'] ), $stype );

			switch ( $stype ) {
				case 'client_id':
								$add = " AND users.ID={$s}";
								break;
				case 'name':	$add = " AND user_login LIKE '%{$s}%' OR user_nicename LIKE '%{$s}%'
										 OR ID IN ( SELECT user_id FROM {$this->a->db->usermeta} AS usermeta
										 WHERE users.ID=usermeta.user_id AND meta_value LIKE '%{$s}%'
										 AND (meta_key='app_full_name' OR meta_key='app_first_name' OR meta_key='app_last_name') ) ";
								break;
				case 'email':	$add = " AND user_email LIKE '%{$s}%'
										 OR ID IN ( SELECT user_id FROM {$this->a->db->usermeta} AS usermeta
										 WHERE users.ID=usermeta.user_id AND meta_value LIKE '%{$s}%'
										 AND meta_key='app_email' ) ";
								break;
				case 'phone':	$add = " AND ID IN ( SELECT user_id FROM {$this->a->db->usermeta}
										 WHERE meta_key='app_phone' AND meta_value LIKE '%{$s}%' ) ";
								break;
				case 'address':	$add = " AND ID IN ( SELECT user_id FROM {$this->a->db->usermeta}
										 WHERE meta_key='app_address' AND meta_value LIKE '%{$s}%' ) ";
								break;
				case 'city':	$add = " AND ID IN ( SELECT user_id FROM {$this->a->db->usermeta}
										 WHERE meta_key='app_city' AND meta_value LIKE '%{$s}%' ) ";
								break;
				case 'postcode':
								$add = " AND ID IN ( SELECT user_id FROM {$this->a->db->usermeta}
										 WHERE meta_key='app_zip' AND meta_value LIKE '%{$s}%' ) ";
								break;
				default:		$add = ''; break;
			}
		} else {
			$add = '';
		}

		$select 	= "SELECT SQL_CALC_FOUND_ROWS * FROM {$this->a->db->users} AS users ";
		$order_by 	= "ORDER BY ID DESC";
		$limit 		= $this->a->db->prepare( "LIMIT %d,%d", $startat, $num );

		$sql = $select. "WHERE ID IN (SELECT user_id FROM {$this->a->db->usermeta} AS usermeta2
			WHERE users.ID=usermeta2.user_id
			AND meta_key='".$this->a->db->get_blog_prefix()."capabilities'
			AND meta_value LIKE '%wpb_client%') {$add} {$order_by} {$limit}";

		return $this->a->db->get_results( $sql );
	}

	/**
	 * Get total number from previous query
	 * @return integer
	 */
	private function get_clients_total( ) {
		return $this->a->db->get_var( "SELECT FOUND_ROWS();" );
	}

	/**
	 * Helper to produce search form
	 * @return string
	 */
	private function search_form(){
		$stype = ! empty( $_GET['stype'] ) ? wpb_clean( $_GET['stype'] ) : '';
		$s = ! empty( $_GET['app_s'] ) ? wpb_clean( $_GET['app_s'] ) : '';
		$s = apply_filters( 'app_clients_search', $s, $stype );
	?>
	<form id="app-search-form" method="get" action="" class="search-form">
		<?php self::print_fields() ?>
		<input type="hidden" value="1" name="app_or_fltr" />
		<input type="search" value="<?php echo esc_attr( $s ); ?>" name="app_s"/>

		<?php $add_class = $stype === 'app_id' ? 'class="app-option-selected"' : ''; ?>
		<select name="stype" <?php echo $add_class ?>>
			<option value="name" <?php selected( $stype, 'name' ); ?>><?php _e('Client Name','wp-base'); ?></option>
			<option value="client_id" <?php selected( $stype, 'client_id' ); ?>><?php _e('Client ID','wp-base'); ?></option>
			<option value="email" <?php selected( $stype, 'email' ); ?>><?php _e('Email','wp-base'); ?></option>
			<option value="phone" <?php selected( $stype, 'phone' ); ?>><?php _e('Phone','wp-base'); ?></option>
			<option value="address" <?php selected( $stype, 'address' ); ?>><?php _e('Address','wp-base'); ?></option>
			<option value="city" <?php selected( $stype, 'city' ); ?>><?php _e('City','wp-base'); ?></option>
			<option value="postcode" <?php selected( $stype, 'postcode' ); ?>><?php _e('Postcode','wp-base'); ?></option>
			<?php do_action( 'app_clients_search_options', $stype ) ?>
		</select>

		<input type="submit" class="button app-search-button" value="<?php _e('Search','wp-base'); ?>" />
	</form>
	<?php
	}

	/**
	 * Helper to produce status change form
	 * @return string
	 */
	private function status_change_form(){
		global $post;
	?>
	<form id="app-bulk-change-form" method="post" action="" >
		<select name="app_new_status">
			<option value=""><?php _e( 'Bulk status change', 'wp-base' ); ?></option>
			<option value="delete"><?php _e( 'Remove', 'wp-base' ) ?></option>
		</select>
		<?php self::print_fields() ?>
		<input type="hidden" value="<?php if ( isset( $post->ID ) ) echo $post->ID; else echo 0; ?>" name="page_id" />
		<input type="hidden" value="app_client_status_change" name="action_app" />
		<input type="hidden" value="1" name="app_client_status_change" />
		<?php wp_nonce_field( 'update_app_settings', 'app_nonce' ); ?>
		<input type="submit" class="button app-change-status-btn" value="<?php _e('Change','wp-base'); ?>" />
	</form>
	<?php
	}

	/**
	 * Change status of clients from admin pge
	 */
	public function change_status_bulk(){
		if ( !( isset( $_POST['app_client_status_change'] ) && $_POST['app_new_status'] && isset( $_POST['app'] ) && is_array( $_POST['app'] ) ) ) {
			return;
		}

		$changed	= 0;
		$stat		= $_POST['app_new_status'];

		foreach ( $_POST['app'] as $uid ) {
			$client = new WpB_Client( $uid );
			if ( 'delete' == $_POST['app_new_status'] && $client->remove() ) {
				$changed++;
			}
		}

		if ( $changed ) {
			wpb_notice( sprintf( __( '%d records have been changed', 'wp-base' ), $changed ) );
		}
	}

	/**
	 * Helper function for render clients table
	 */
	public function display_table( $users ) {

		$columns	= apply_filters( 'app_clients_default_columns','id,client,first_name,last_name,email,phone,address,city,zip,nof_bookings,revenue');
		$cols		= explode( ',', wpb_sanitize_commas( $columns ) );
		$cols		= array_map( "strtolower", $cols );
		array_unshift( $cols, 'delete' );

		$ret  = apply_filters( 'app_clients_before_table', '<form class="app-form" method="post" >', $cols );
		$ret .= '<table %HIDDENCOLUMNS%	id="app-clients" class="app-list-table wp-list-table widefat app-clients dt-responsive display striped dataTable">';
		$ret .= '<thead>';

		/* hf : Common for header-footer */
		$this->colspan = 0;
		$hf = $this->table_head( $cols );

		$ret .= $hf. '</thead>';
		// Remove id from foot
		$ret .= '<tfoot>' . preg_replace( '/<th id="(.*?)"/is','<th ', $hf ). '</tfoot>';
		$ret = str_replace(
				'%HIDDENCOLUMNS%',
				'data-hide_boxes="'.implode( ',', array_diff( $this->get_allowed_columns(), $this->used_cols ) ).'"',
				$ret
			);

		$ret .= '<tbody>';
		$ret  = apply_filters( 'app_clients_after_table', $ret, $cols );
		echo $ret;

		$ret = '';

		if ( $users ) {
			remove_filter( 'gettext', array( 'WpBCustomTexts', 'global_text_replace' ) );

			foreach ( $users as $r ) {

				$classes = array( 'app-tr' );

				$ret .= '<tr id="app-tr-'.$r->ID.'" class="'.implode( ' ', $classes ).'">';

				foreach( $cols as $col ) {

					if ( ! in_array( $col, $this->get_allowed_columns() ) ) {
						continue;
					}

					$hidden			= in_array( $col, $this->get_hidden_columns() ) ? " hidden" : "";
					$col_primary	= in_array( $col, array('client' ) ) ? " column-primary" : "";
					$col_check		= in_array( $col, array( 'delete' ) ) ? " check-column" : "";

					$ret .= '<td class="column-'.$col. $hidden. $col_primary. $col_check. '">';
					$ret .= $this->table_cell( $col, $r, $cols );
					$ret .= '</td>';
				}
				$ret .= '</tr>';
			}
			echo $ret;

		} else {
			?>
			<tr class="alternate app-tr">
				<td colspan="<?php echo $this->colspan; ?>" scope="row"><?php _e( 'No matching records have been found.','wp-base' ); ?></td>
			</tr>
	<?php }	?>
		</tbody>
	</table>
	<?php
		do_action( 'app_admin_clients_form' );
	?>
</form>
	<?php
	}

	/**
	 * Prepare header for clients table
	 * @since 3.0
	 */
	private function table_head( $cols ) {

		$hf = '<tr>';
		foreach( $cols as $col ) {
			if ( ! in_array( $col, $this->get_allowed_columns() ) ) {
				continue;
			}

			$this->used_cols[] 	= $col; // Used for js to hide not allowed columns. Hidden columns should also be here!
			$hidden 			= in_array( $col, $this->get_hidden_columns() ) ? " hidden" : "";
			$col_primary 		= in_array( $col, array( 'client' ) ) ? " column-primary" : "";
			$col_check 			= in_array( $col, array( 'delete' ) ) ? " check-column" : "";

			if ( ! $hidden ) {
				$this->colspan++;
			}

			$hf .= '<th id="'.$col.'" class="manage-column column-'.$col. $hidden. $col_primary. $col_check. '">';

			switch ($col) {
				case 'delete':		$hf .= '<input type="checkbox" class="app-no-save-alert" />';
									break;
				case 'id':			$hf .= __( 'ID', 'wp-base' );
									break;

				case 'client':		$hf .= __( 'Client', 'wp-base' );
									break;

				case 'nof_bookings':
									$hf .= $this->a->get_text('nof_bookings');
									break;

				case 'revenue':		$hf .= __( 'Revenue', 'wp-base' );
									break;

				case $col:			$hf .= '<span>'. apply_filters( 'app_clients_column_title', str_replace( ':', '', $this->a->get_text($col)), $col) .'</span>'; // E.g. address, phone
									break;
				default:			break;
			}
			$hf .= '</th>';

		}
		$hf .= '</tr>';

		return $hf;
	}

	/**
	 * Prepare a single cell in clients table
	 * @since 3.0
	 */
	private function table_cell( $col, $r, $cols ) {
		$ret		= '';
		$client		= new WpB_Client( $r->ID );

		switch ( $col ) {
			case 'delete':			$ret .= '<input type="checkbox" class="app-no-save-alert" name="app[]" value="'. $r->ID .'" />';
									$ret .= '<div class="locked-indicator"><span class="locked-indicator-icon" aria-hidden="true"></span></div>';
									break;

			case 'id': 				$ret .= '<span class="span_user_id">'.$r->ID .'</span>'; break;
			case 'client':			$ret .= self::table_cell_client( $client ); break;
			case 'first_name':		$ret .= $client->get_first_name(); break;
			case 'last_name':		$ret .= $client->get_last_name(); break;
			case 'full_name':
			case 'email':
			case 'phone':
			case 'city':
			case 'address':
			case 'zip':
			case 'state':
			case 'country':			$ret .= $client->get_meta( $col ); break;
			case 'revenue':			$ret .= wpb_format_currency( $client->get_revenue()/100 ); break;
			case 'nof_bookings':	$ret .= $client->get_nof_bookings(); break;
			default:				$ret .= apply_filters( 'app_clients_add_cell', '', $col, $r, $cols ); break;
		}

		return $ret;
	}

	/**
	 * Prepare client display name cell content
	 * @since 3.0
	 * @return string
	 */
	private static function table_cell_client( $client ) {
		$ret  = $client->get_image_html( 32 );
		$ret .= '<div class="user-inner">';
		$ret .= '<span class="app-client-name">'. $client->get_display_name() . '</span>';
		$ret .= '</div>';
		$ret .= '<div class="row-actions">';
		$ret .= '<a href="javascript:void(0)" class="app-inline-edit-client" title="'.__('Click to edit user','wp-base').'">';
		$ret .= __('Edit', 'wp-base');
		$ret .= '</a>';
		$ret .= '</div>';

		return $ret;
	}

	/**
	 * Bring user data when user is selected in Select Client
	 * @since 2.0
	 */
	public function populate_user() {
		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array( 'error' => $this->a->get_text('unauthorised') ) ) );
		}

		$user_id = !empty( $_POST['user_id'] ) ? wpb_clean( $_POST['user_id'] ) : 0;

		$r = $user_id ? BASE('User')->get_app_userdata( 0, $user_id ) : array();
		$r['display_name'] = ! empty( $r['display_name'] )
							 ? $r['display_name']
							 : ( ! empty( $r['name'] ) ? $r['name'] : '' );

		$r = apply_filters( 'app_inline_edit_populate_user', $r, $user_id );

		if ( isset( $r['error'] ) || ! is_array( $r ) ) {
			wp_send_json( array( 'error' => __( 'Unknown or empty user ID', 'wp-base' ) ) );
		} else {
			wp_send_json( array( $r, 'info' => $this->client_info( new WpB_Client( $user_id ) ) ) );
		}
	}

	/**
	 * List of user fields
	 */
	public static function get_fields(){
		return apply_filters( 'app_client_fields', array('display_name', 'first_name', 'last_name', 'email', 'phone', 'address', 'city', 'zip') );
	}

	/**
	 * Edit or create appointments on admin side
	 */
	public function inline_edit( $echo = false, $colspan = 0 ) {

		if ( ! $echo ) {
			if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
				die( json_encode( array( 'error' => $this->a->get_text('unauthorised') ) ) );
			}
		}

		# What if such a user does not exist?
		$user_id	= isset( $_REQUEST['user_id'] ) ? wpb_clean( $_REQUEST['user_id'] ) : 0;
		$client		= new WpB_Client( $user_id );
		$js_id		= $user_id ? $user_id : uniqid();
		$_colspan 	= ! empty( $_POST['col_len'] ) ? wpb_clean( $_POST['col_len'] ) : ( $colspan ? $colspan : 7 );

		$html  = '';
		$html .= '<tr class="inline-edit-row inline-edit-row-post quick-edit-row-post '.($user_id ? "" : "inline-edit-row-add-new").'">';
		$html .= '<td colspan="'.$_colspan.'" class="colspanchange">';

		$html .= '<fieldset>';
		$html .= '<div class="inline-edit-col">';

		if ( ! $user_id ) {

			$html .= $this->client_info_html( $client, true ); # Add as hidden

			$html .= wpb_wrap_field( 'user', __('Select or Register User as Client', 'wp-base'),
				BASE('User')->app_dropdown_users( apply_filters( 'app_inline_edit_new_client_args',
					array(
						'show_option_all'	=> __('Register new user as client','wp-base'),
						'echo'				=> 0,
						'show'				=> 'display_name',
						'add_email' 		=> true,
						'name'				=> 'user',
						'class'				=> 'app_users_client',
						'id'				=> 'app_users_client_'.$js_id,
						'exclude'			=> wpb_get_clients( array( 'fields' => 'ID' ) ),
					),
					$client )
				),
				__( 'Selecting a user from the list will add Client role to the existing WP user. Registering new user will create a new WP user.', 'wp-base')
			);

			$html .= wpb_wrap_field( 'user_login', 'User Login',
					'<input type="text" name="user_login" class="ptitle" autocomplete="off" value="">');
			$html .= wpb_wrap_field( 'password', 'Password',
					'<input type="password" name="user_pass" class="ptitle" autocomplete="off" value="'.wp_generate_password().'">');

		} else {
			$html .= $this->client_info_html( $client );
		}

		$html .= '</fieldset>';
		$html .= '</div>';

		$html .= '<fieldset class="">';
		$html .= '<div class="inline-edit-col">';

		foreach ( self::get_fields() as $f ) {

			$field_name = 'name' === $f ? 'cname' : $f;

			if ( 'email' == $f ) {
				$value = $client->get_email();
			} else if ( 'name' == $f || 'display_name' == $f ) {
				$value = $client->get_display_name();
			} else if ( 'first_name' == $f ) {
				$value = $client->get_first_name();
			} else if ( 'last_name' == $f ) {
				$value = $client->get_last_name();
			} else {
				$value = $client->get_meta( $f );
			}

			if ( $user_id && wpb_is_demo() ) {
				if ( 'email' === $f ) {
					$value = 'email@example.com';
				} else if ( 'phone' === $f ) {
					$value = '0123456789';
				} else {
					$value = 'Demo '. wpb_get_field_name( $f );
				}
			}

			if ( $user_id && 'email' == $f ) {
				$ronly = ' readonly';
				$help = __( 'Readonly field. To edit email, use WordPress user page.');
			} else {
				$ronly = $help = '';
			}

			$html .= wpb_wrap_field( $f, wpb_get_field_name( $f ),
					'<input type="text" name="'.$field_name.'" class="ptitle" autocomplete="'.$f.'" value="'.esc_attr( $value ).'"'.$ronly.' />',
					$help
			);
		}

		/* UDF fields */
		$html  = apply_filters( 'app_clients_inline_edit_user_fields', $html, $client, $this );

		$html .= '</div>';

		/* Logs */
		// $html .= '<hr />';
		// $html .= '<div class="inline-edit-col">';
		// $html .= wpb_wrap_field( 'logs', __('Logs', 'wp-base'), self::log_output( $client ) );
		// $html .= '</div>';

		$html .= '</fieldset>';

		/* Credits */
		$html  = apply_filters( 'app_clients_inline_edit_before_save', $html, $client, $this );

		$html .= '<div style="clear:both"></div>';

	/* SAVE and OTHER ACTIONS */
		$html .= '<div class="inline-edit-save wp-clearfix">';
		$html .= '<input type="hidden" name="user_id" value="'.$user_id.'" />';
		$html .= '<a href="javascript:void(0)" title="'.__('Cancel', 'wp-base').'" class="button-secondary cancel alignleft">'.__('Cancel','wp-base').'</a>';
		$html .= '<img class="waiting alignleft" style="display:none;" src="'.admin_url('images/wpspin_light.gif').'" alt="">';
		$html .= '<span class="error alignleft" style="display:none"></span>';
		$html .= '<button class="button-primary save alignright">'.( $user_id ? __('Update','wp-base') : __('Add as Client','wp-base') ).'</button>';

		$html .= '</div>';
		$html .= '</td>';
		$html .= '</tr>';

		if ( $echo ) {
			echo $html;
		} else {
			die( json_encode( array(
				'result'	=> $html,
				'id'		=> $js_id,
			) ) );
		}
	}

	/**
	 * Client avatar, ID, etc info
	 * @param $hidden 	bool	Add as hidden if new
	 */
	private function client_info_html( $client, $hidden = false ){
		$info = $this->client_info( $client );
		ob_start();
?>
<div class="app-user-info" <?php echo ($hidden ? 'style="display:none"' : '')?>>

	<div class="app-user-avatar app-client-avatar left">
		<?php echo $info['avatar']; ?>
	</div>

	<div class="app-user-id app-client-id right">
		<?php echo $info['id']; ?>
	</div>

	<div class="app-user-main left">
		<span class="app-user-item app-user-name app-client-display-name">
			<?php echo $info['display-name']; ?>
		</span>
		<span class="app-user-item app-start-date app-client-since">
			<?php echo $info['since']; ?>
		</span>
		<span class="app-user-item app-edit-user app-client-edit">
			<?php echo $info['edit']; ?>
		</span>
	</div>
</div>
<?php
		return ob_get_clean();
	}

	/**
	 * Details about client
	*/
	private function client_info( $client ) {
		return array(
			'id'			=> '#'. $client->get_ID(),
			'user_id'		=> $client->get_ID(),
			'avatar'		=> $client->get_image_html(),
			'display-name'	=> $client->get_display_name(),
			'since'			=> $client->start_date() ? sprintf( __( 'Client since %s', 'wp-base'), $client->start_date() ) : '',
			'edit'			=> '<a href="'. admin_url( 'user-edit.php?user_id='. $client->get_ID() ) .'" target="_blank">' . __( 'WordPress User Profile', 'wp-base' ) .'</a>',
		);
	}

	private static function log_output( $client ) {
		$out = '';

		$logs = $client->get_log();

		if ( ! empty( $log ) ) {
			foreach ( $logs as $log ) {
				$out .= '<p>'. $log . '</p>';
			}
		} else {
			$out .= '<p>'. __( 'None' , 'wp-base' ) . '</p>';
		}

		return $out;
	}

	/**
	 * Save submitted client data
	 */
	public function inline_edit_save(){
		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array('error' => $this->a->get_text('unauthorised') ) ) );
		}

		if ( wpb_is_demo() ) {
			die( json_encode( array( 'error' => __('Changes cannot be saved in DEMO mode!', 'wp-base' ) ) ) );
		}

		$user_data = json_decode( wp_unslash( $_POST['app_user_data'] ), true );

		if ( ! is_array( $user_data ) ) {
			die( json_encode( array( 'error' => __('Invalid user data', 'wp-base' ) ) ) );
		}

		$changed = false;
		$user_id = ! empty( $_POST['user_id'] ) ? wpb_clean( $_POST['user_id'] ) : 0;

		if ( ! $user_id ) {
			$this->register( $user_data );
			exit;
		}

		$client		= new WpB_Client( $user_id );
		$old_client = clone $client;
		$new_client	= false;

		if ( ! $client->is_client() ) {
			$client->add();
			$changed = $new_client = true;
		}

		foreach ( self::get_fields() as $f ) {
			if ( isset( $user_data[ $f ] ) ) {
				if ( $client->update_meta( $f, $user_data[ $f ] ) ) {
					$changed = true;
				}
			}
		}

		if ( ! empty( $_POST['timezone_check'] ) && $client->update_timezone( $_POST['timezone'] ) ) {
			$changed = true;
		}

		if ( ! empty( $_POST['lang_check'] ) && $client->update_lang( $_POST['lang'] ) ) {
			$changed = true;
		}
		
		# Save UDF
		if ( apply_filters( 'app_clients_inline_edit_save', false, $client, $old_client ) ) {
			$changed = true;
		}		

		if ( $changed ) {
			$client = new WpB_Client( $user_id ); # Update cache
			$r = array(
				'result'		=> $new_client ? __( 'Client added.', 'wp-base' ) : __( 'Changes saved.', 'wp-base' ),
				'collapse'		=> 'yes' == wpb_setting( 'admin_edit_collapse' ) ? 1 : 0,
				'client'		=> $client->get_display_name(),
				'new_client'	=> $new_client ? $client->get_ID() : false,
			);

			foreach ( self::get_fields() as $f ) {
				if ( 'display_name' == $f ) {
					$r[ $f ] = $client->get_display_name();
				} else if ( 'first_name' == $f ) {
					$r[ $f ] = $client->get_first_name();
				} else if ( 'last_name' == $f ) {
					$r[ $f ] = $client->get_last_name();
				} else {
					$r[ $f ] = $client->get_meta( $f );
				}
			}
		} else {
			$r = array( 'no_change' => __( 'You did not make any changes...', 'wp-base' ) );
		}

		die( json_encode( $r ) );

	}

	/**
	 * Register a person as WP User and assign as client manually
	 */
	private function register( $user_data ) {

		if ( ! is_email( $user_data['email'] ) ) {
			die( json_encode( array( 'error' => $this->a->get_text( 'mv_email_invalid' ) ) ) );
		} else if ( email_exists( $user_data['email'] ) ) {
			die( json_encode( array( 'error' => $this->a->get_text( 'mv_email_in_use' ) ) ) );
		}

		// Is the email on the Banned Email Domains list?
		// Note: This check only works on Multisite.
		if ( function_exists( 'is_email_address_unsafe' ) && is_email_address_unsafe( $user_data['email'] ) ) {
			die( json_encode( array( 'error' => $this->a->get_text( 'mv_email_banned' ) ) ) );
		}

		// Is the email on the Limited Email Domains list?
		// Note: This check only works on Multisite.
		$limited_email_domains = get_site_option( 'limited_email_domains' );
		if ( is_array( $limited_email_domains ) && empty( $limited_email_domains ) == false ) {
			$emaildomain = substr( $user_data['email'], 1 + strpos( $user_data['email'], '@' ) );
			if ( ! in_array( $emaildomain, $limited_email_domains ) ) {
				die( json_encode( array( 'error' => $this->a->get_text( 'mv_email_banned' ) ) ) );
			}
		}

		$user_data['user_email']	= $user_data['email'];
		$user_data['user_login']	= ! empty( $_POST['user_login'] ) ? $_POST['user_login'] : '';
		$user_data['user_pass']		= ! empty( $_POST['user_pass'] ) ? $_POST['user_pass'] : '';
		$result						= wp_insert_user( $user_data );
		$inserted					= $email_attempt = $email_sent = false;

		if ( is_wp_error( $result ) ) {
			die( json_encode( array( 'error' => $result->get_error_message() ) ) );
		} else if ( is_integer( $result ) ) {
			$user_id = $result;
		} else {
			die( json_encode( array( 'error' => $this->a->get_text('error') ) ) );
		}

		$client = new WpB_Client( $user_id );

		foreach ( self::get_fields() as $f ) {
			if ( isset( $user_data[ $f ] ) ) {
				if ( $client->update_meta( $f, $user_data[ $f ] ) ) {
					$inserted = true;
				}
			}
		}

		if ( $client->add() ) {
			$inserted = true;
		}

		if ( $inserted ) {
			$r = array( 'result' => __( 'Client added.', 'wp-base' ) );
		} else {
			$r = array( 'no_change' => __( 'Unexpected result. Maybe user was already a client. Check if client actually added', 'wp-base' ) );
		}

		die( json_encode( $r ) );
	}

	/**
	 * Add script to footer;
	 */
	public function footer(){
?>
<script type="text/javascript">
jQuery(document).ready(function($){
   /**
     * Add/edit/save clients
     */
    var WPB_Clients = {

        init: function () {
            var me = this;
            me.doBind();
        },

        doBind: function () {
            var me = this;
            var table = $("table.app-clients");

            $(".add-new-client").on("click", function () {
                me.addNew($(this));
            });

             table.on("click", ".app-inline-edit-client", function (e) {
                me.edit(e);
            });

			table.on("click", ".save", function (e) {
                me.save(e);
            });

            /* Cancel */
            table.on("click", ".cancel", function (e) {
                me.collapseRecord($(e.target));
            });

            /* Redraw multiselect */
            $(window).resize(function () {
                $.each($(".app_users_client, .app-select-multiple"), function (ignore, val) {
                    var $this = $(val);
                    if ($this.data().hasOwnProperty("echMultiselect")) {
                        $this.multiselect("refresh");
                    }
                });
            });

			if (!table.length) {
				return false;
			}
        },

        /**
         * Read all field values of a row (par)
         * @param par   object  Possibly a row in the table under which fields will be read
         */
        readData: function (par) {

            var app_user_data = {};
            $.each(_app_.user_fields, function (ignore, v) {
                app_user_data[v] = par.find("input[name='" + v + "']").val();
            });

            var postData = {
				wpb_ajax: true,
				user: par.find("select[name='user'] option:selected").val(),
				name: par.find("input[name='cname']").val(),
				app_user_data: JSON.stringify(app_user_data),
				user_id: par.find("input[name='user_id']").val(),
				user_login: par.find("input[name='user_login']").val(),
				user_pass: par.find("input[name='user_pass']").val(),
				timezone: par.find("select[name='timezone']").val(),
				timezone_check: par.find("[name='timezone_check']").val(),
				lang: par.find("select[name='lang']").val(),
				lang_check: par.find("input[name='lang_check']").val(),
				ajax_nonce: _app_.iedit_nonce
            };

            var udfs = {};
            $.each(_app_.udf_ids, function (ignore, v) {
                var field = par.find(".app-udf-field-entry-" + v);
                if (parseInt(field.length) > 0) {
                    if (field.hasClass("app-udf-checkbox")) {
                        udfs["udf_" + v] = field.is(":checked") ? 1 : 0;
                    } else {
                        udfs["udf_" + v] = field.val();
                    }
                }
            });
            postData = $.extend(postData, {
						udf_data: JSON.stringify(udfs)
                    });

            return postData;
        },

        /**
         * Add a new client record/row
         */
        addNew: function () {
            var me = this;
            $(".add-new-waiting").show();
            $.infoPanel("reading");
            var table = $("table.app-clients").first();
            var data = {
                wpb_ajax: true,
                action: "inline_edit_client",
                add_new: $.readGet("add_new"),
                col_len: parseInt(table.find("tr:first th:visible").length),
                user_id: 0,
                ajax_nonce: _app_.iedit_nonce
            };
            $.post(_app_.ajax_url, data, function (r) {
                $(".add-new-waiting").hide();
                if (r && r.error) {
                    window.alert(r.error);
                } else if (r) {
                    table.prepend(r.result);
                    me.configureMS(r.id);
                    me.configQtip(r.id);
                } else {
                    window.alert(_app_.con_error);
                }
            }, "json");
        },

        /**
         * Edit a client
         */
        edit: function (e) {
            var me = this;
            var $this = $(e.target);
            var par = $this.parents(".app-tr");
            var table = par.parents("table.app-clients");
            var user_id = par.find(".span_user_id").html();
            $.infoPanel("reading");
            $this.addClass("app-disabled-button");
            par.find(".waiting").css("visibility", "visible");
            var data = {
                wpb_ajax: true,
                action: "inline_edit_client",
                col_len: parseInt(table.find("tr:first th:visible").length),
                user_id: user_id,
                ajax_nonce: _app_.iedit_nonce
            };
            $.post(_app_.ajax_url, data, function (r) {
                par.find(".waiting").css("visibility", "hidden");
                if (r && r.error) {
                    window.alert(r.error);
                } else if (r) {
                    par.hide();
                    var iedit_row = r.result;
                    $(iedit_row).insertAfter(par);
                    me.configureMS(r.id);
                    me.configQtip(r.id);
					$(document).trigger("app-client-record-opened", r );
                } else {
                    window.alert(_app_.con_error);
                }
            }, "json");
        },

        /**
         * Save a client
         */
        save: function (e) {
            var me = this;
            var $this = $(e.target);
            var sPar = $this.parents(".inline-edit-row");
            var postData = me.readData(sPar);

            $this.attr("disabled", true);
            $.infoPanel("saving");
            sPar.find(".waiting").show();

            postData.action = "inline_edit_save_client";

            $.post(_app_.ajax_url, postData, function (r) {
                $this.attr("disabled", false);
                sPar.find(".waiting").hide();

                if (!r) {
                    window.alert(_app_.con_error);
                    return false;
                }
                var emailMsg = r.emailMsg ? " " + r.emailMsg : "";
                if (r.error) {
                    sPar.find(".error")
                    .html("<span class='app-error'>" + r.error + emailMsg + "</span>")
                    .show().delay(10000).fadeOut("slow");
                } else if (r.no_change) {
                    sPar.find(".error")
                    .html("<span class='app-b'>" + r.no_change + emailMsg + "</span>")
                    .show().delay(10000).fadeOut("slow");
                } else if (r.result) {
					me.sendSuccess(r, sPar, emailMsg, $this);
                } else if (emailMsg) {
                    sPar.find(".error")
                    .html("<span class='app-success'>" + emailMsg + "</span>")
                    .show().delay(10000).fadeOut("slow");
				}
            }, "json");
        },

        /**
         * Send success message
         */
		sendSuccess: function(r, sPar, emailMsg, $this) {
			var me = this;
            var appID = sPar.find("input[name='app_id']").val();
			sPar.find(".error")
			.html("<span class='app-success'>" + r.result + emailMsg + "</span>")
			.show().delay(10000).fadeOut("slow");

			var prevRow = sPar.prev(".app-tr");
			if (r.result_app_id) {
				var nid = r.result_app_id;
				sPar.find("input[name='app_id']").val(nid);
				if (nid !== appID) {
					sPar.find(".app_iedit_app_h").append(" (ID:" + nid + ")");
				}
				if (nid) {
					sPar.find("a.save").text(_app_.update_text);
				}
			}

			$.each(r, function (i, v) {
				if ( "client" === i ) {
					prevRow.find(".user-inner .app-client-name").text(v);
				} else {
					prevRow.find(".column-" + i).text(v);
				}
			});

			prevRow.addClass("ui-state-highlight");
			setTimeout(function () {
				prevRow.removeClass("ui-state-highlight");
			}, 10000);

			if (parseInt(r.collapse) > 0) {
				setTimeout(function () {
					me.collapseRecord($this, r);
				}, 500);
			}
		},

        /**
         * Populate userdata upon user selection
         */
        populateUser: function (me) {
            var sel_user = parseInt(me.val());
            var par = me.parents(".inline-edit-row");
			$(".app_iedit_user_login, .app_iedit_password").hide();
            if (sel_user === 0) {
                // Clear fields for unregistered user
				par.find(".app-user-info").hide();
                $.each(_app_.user_fields, function (ignore, v) {
                    par.find(".app_iedit_" + v + " input").val("");
                });
				$(".app_iedit_user_login, .app_iedit_password").show();
                return false;
            }
            $.infoPanel("reading");
            var data = {
                action: "app_populate_user_client",
                user_id: sel_user,
                ajax_nonce: _app_.iedit_nonce
            };
            $.post(_app_.ajax_url, data, function (r) {
                if (r && r.error) {
                    window.alert(r.error);
                } else if (r) {
                    $.each(r, function (i) {
						if ( "info" === i ) {
							$.each(r["info"], function (ix, vx) {
								par.find(".app-user-info .app-client-"+ix).html(vx);
								if ( "user_id" === ix ) {
									par.find("input[name='user_id']").val(vx);
								}
							});
							par.find(".app-user-info").show();
						} else {
							$.each(r[i], function (iy, vy) {
								par.find(".app_iedit_" + iy + " input").val(vy);
							});
						}
                    });
                } else {
                    window.alert(_app_.con_error);
                }
            }, "json");
        },

        /**
         * Initiates Multiselect for users
         */
        configureMS: function (id) {
            var me = this;
            var $this = $("#app_users_client_" + id);
            $this.multiselect({
                selectedList: 1,
                buttonWidth: "100%",
                classes: "app_users_client app-ms-small",
                close: function () {
                    me.populateUser($this);
                }
            }).multiselectfilter();

            $(".app-select-multiple").multiselect({
                selectedList: 5,
                buttonWidth: "100%",
                classes: "app-ms-small"
            }).multiselectfilter();
        },

        /**
         * Collapses record after cancel/save
         */
        collapseRecord: function (el, r) {
            var row = el.parents(".inline-edit-row");
            $(".app-inline-edit-client").removeClass("app-disabled-button");
            row.fadeOut(700, function () {
                row.remove();
            });
            row.prev(".app-tr").removeClass("app-locked").show("slow");

			if (r && r.new_client) {
				window.location.href = $.locationUrl() + "#app-tr-" + r.new_client;
				location.reload();
			} else {
				return;
			}
        },

        /**
         * Initiates qtip
         */
        configQtip: function () {
            var edit_row = $(".inline-edit-row");
            edit_row.find("[title][title!='']").each(function (ignore, val) {
                var $this = $(val);
                var title = $this.attr("title");
                var ttip = title ? title.replace(/●/g, "<br />").replace("/|/", "<br/>") : "";

                $this.qtip({
                    content: {
                        text: ttip
                    },
                    hide: _app_.qtipHide,
                    position: _app_.qtipPos,
                    style: _app_.qtipYellow
                });
            });

        }
    };
    WPB_Clients.init();

});
</script>
<?php
	}

}
	BASE('Clients')->add_hooks();
}