<?php
/**
 * WPB Admin Bookings
 *
 * Handles admin booking functions
 *
 * Copyright © 2018-2021 Hakan Ozevin
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       2.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBAdminBookings' ) ) {

class WpBAdminBookings{

	/**
     * Bookings page identifier
     */
	public $bookings_page;

	/**
     * Keeps monetary total values of bookings on the page
     */
	private $money_vars = array();

	/**
     * A list of used column names incl. hidden
     */
	private $used_cols = array();

	/**
     * Number of visible columns
     */
	private $colspan = 0;

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Name of user option for Hidden Columns
     */
	const HIDDEN_COLUMNS = 'app_manage_bookings_columnshidden';

	/**
     * Name of user option for Hidden Navbar
     */
	const HIDDEN_NAVBAR = 'app_manage_bookings_navbarhidden';

	/**
     * Constructor
     */
	public function __construct(){
		$this->a = BASE();
	}

	/**
     * Add admin actions
     */
	public function add_hooks() {

		add_action( 'app_menu_before_all', array( $this, 'add_menu' ), 15 );
		add_action( 'app_menu_for_worker', array( $this, 'add_menu_for_worker' ) );
		add_action( 'wp_ajax_app_save_hidden_columns_bookings', array( $this, 'save_hidden_columns' ) );
		add_action( 'wp_ajax_app_save_navbar_bookings', array( $this, 'save_navbar' ) );

		add_action( 'wp_ajax_inline_edit', array( $this, 'inline_edit' ) ); 						// Add/edit appointments
		add_action( 'wp_ajax_inline_edit_save', array( $this, 'inline_edit_save' ) ); 				// Save edits
		add_action( 'wp_ajax_update_inline_edit', array( $this, 'update_inline_edit' ) ); 			//
		add_action( 'wp_ajax_app_populate_user', array( $this, 'populate_user' ) );					// Populate user in New Booking
		add_action( 'wp_ajax_app_show_payment_in_tooltip', array( $this, 'show_payment_in_tooltip' ) );
		add_filter( 'heartbeat_received', array( $this, 'refresh_lock' ), 10, 3 );
		add_action( 'wp_ajax_app_delete_lock', array( $this, 'delete_lock' ) );
	}

	/**
     * Add menu and submenu page to main admin menu for admin
     */
	public function add_menu(){

		$this->bookings_page = add_submenu_page('appointments', __('Bookings','wp-base'), __('Bookings','wp-base'), WPB_ADMIN_CAP, "app_bookings", array($this,'appointment_list'));
	}

	/**
     * Add menu and submenu page to main admin menu for worker
     */
	public function add_menu_for_worker() {
		if ( empty( $this->bookings_page ) ) {

			add_menu_page('WPB Bookings', WPB_NAME, 'wpb_worker', 'appointments', array($this,'appointment_list'), 'dashicons-calendar', '25.672');
			add_submenu_page('appointments', __('Bookings','wp-base'), __('Bookings','wp-base'), 'manage_own_bookings', "appointments", array($this,'appointment_list'));
		}
	}

	/**
	 * Define an array of allowed columns in bookings page
	 * Called after addons loaded
	 */
	public static function get_allowed_columns() {
		$allowed = array( 'delete','client','id','created','created_by','email','phone','city','address','zip','date_time','end_date_time','location','service','worker','status','price','deposit','total_paid','balance' );
		return apply_filters( 'app_bookings_allowed_columns', $allowed, '' );
	}

	/**
     * Add some columns as hidden to Bookings page if there is no selection before
     */
	private static function default_hidden_columns( ) {
		$hidden_cols = array('created','created_by','email','phone','city','address','zip','note','date','day','time','end_date_time','location','location_note','price','deposit','total_paid','balance');
		return apply_filters( 'app_default_hidden_columns', $hidden_cols );
	}

	/**
	 * Get hidden columns for the table
	 * @since 3.0
	 * @return array
	 */
	private function get_hidden_columns(){
		$hidden_columns = get_user_option( self::HIDDEN_COLUMNS );

		if ( empty( $hidden_columns['hidden_check'] ) ) {
			$hidden_columns = self::default_hidden_columns();
		}

		return $hidden_columns;
	}

	/**
	 * Select element to display columns on the table
	 * @since 3.8.2
	 * @return string
	 */
	private function displayed_columns_selection(){
		$cols	= self::get_allowed_columns();
		$hidden = self::get_hidden_columns();

		$html = '<select size="16" multiple data-scope="bookings" class="app_displayed_columns app-ms-small" data-selectedtext="'.esc_attr( __( 'Columns', 'wp-base' ) ).'">';
		foreach ( $cols as $col ) {

			if ( 'delete' == $col ) {
				continue;
			}

			if ( 'id' == $col ) {
				$text = $this->a->get_text( 'app_id' );
			} else if ( 'udf_' == substr( $col, 0, 4 ) ) {
				$text = apply_filters( 'app_bookings_udf_column_title', '', $col );
			} else {
				$text = $this->a->get_text( $col );
			}

			$html .= '<option value="'.$col.'" '. selected( in_array( $col, $hidden ), false, false).'>'.$text.'</option>';
		}
		$html .= '</select>';

		return $html;
	}

	/**
	 * Ajax save hidden columns
	 * @since 3.8.2
	 * @return json
	 */
	public function save_hidden_columns(){
		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array('error' => $this->a->get_text('unauthorised') ) ) );
		}

		if ( wpb_is_demo() ) {
			die( json_encode( array( 'error' => __('Changes cannot be saved in DEMO mode!', 'wp-base' ) ) ) );
		}

		if ( ! empty( $_POST['hidden'] ) ) {
			update_user_option( get_current_user_id(), self::HIDDEN_COLUMNS, array_merge( $_POST['hidden'], array( 'hidden_check' => 1 ) ) );
		}

		wp_send_json_success();
	}

	/**
	 * Ajax save navbar
	 * @since 3.8.2
	 * @return json
	 */
	public function save_navbar(){
		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array('error' => $this->a->get_text('unauthorised') ) ) );
		}

		if ( wpb_is_demo() ) {
			die( json_encode( array( 'error' => __('Changes cannot be saved in DEMO mode!', 'wp-base' ) ) ) );
		}

		update_user_option( get_current_user_id(), self::HIDDEN_NAVBAR, ! empty( $_POST['hidden'] ) );

		wp_send_json_success();
	}

	/**
	 * Helper to pass page and tab variables
	 */
	public static function print_fields(){
		if ( ! empty( $_GET['tab'] ) ) { ?>
			<input type="hidden" name="tab" value="<?php echo wpb_clean( $_GET['tab'] ) ?>"/>
		<?php }
		if ( is_admin() ) { ?>
			<input type="hidden" name="page" value="app_bookings"/>
		<?php }
	}

	/**
	 * Creates the list for Appointments admin page
	 * Also used by Front End Management addon
	 */
	public function appointment_list(
								$return			= false,	# true means called by [app_manage]
								$only_own		= 0,
								$allow_delete	= 1,
								$count			= 0,
								$override		= 'auto',
								$status			= 'any',
								$columns		= '',
								$columns_mobile	= '',
								$add_export		= 0,
								$sel_location	= 0,
								$febm_cap 		= ''		# FEBM capability
							) {

		global $current_user;

		$args = compact( 'return', 'only_own', 'allow_delete', 'count', 'override', 'status', 'columns', 'columns_mobile', 'add_export', 'sel_location', 'febm_cap' );

		if ( ! (current_user_can( WPB_ADMIN_CAP ) || ( $return && current_user_can( $febm_cap ) ) ||
			($this->a->is_worker( $current_user->ID ) && 'yes' === wpb_setting('allow_worker_edit') && wpb_current_user_can( 'manage_own_bookings' ) )
			) ) {

			wp_die( __('You do not have sufficient permissions to access this page.','wp-base') );
		}

		// If worker allowed to edit his appointments, show only his.
		if ( wpb_is_admin() && ! current_user_can( WPB_ADMIN_CAP ) ) {
			$only_own = 1;
		}

		// Check if such a location exists
		$sel_location = $sel_location && $this->a->location_exists( $sel_location ) ? $sel_location : 0;
		$stat = explode( ',', wpb_sanitize_commas( $status ) );

		// Limit statuses acc to front end selection + Exclude invalid statuses
		if ( in_array( 'any', $stat, true ) ) {
			$statii = $this->a->get_statuses();
		} else {
			$statii = array_intersect_key( $this->a->get_statuses(), array_flip( $stat ) );
		}

		// $statii is all statuses, $stat is the selected status
		$statii = apply_filters( 'app_admin_statuses', $statii, $stat );

		/* Find which status is called */
		// wp_reset_vars( array('type') );
		$type = ! empty( $_GET['status'] ) ? wpb_clean( $_GET['status'] ) : '';

		if ( empty( $type ) ) {
			if ( 'all' == $status ) {
				$type = 'all';
			} else if ( in_array( 'paid', $stat ) || in_array( 'confirmed', $stat ) ) {
				$type = defined( 'WPB_SEPARATE_UPCOMING' ) && WPB_SEPARATE_UPCOMING ? current( $stat ) : 'active';
			} else if ( ! empty( $stat ) && in_array( current( $stat ), array_keys( $this->a->get_statuses() ) ) ) {
				$type = current( $stat );
			}
		}

		# Booking status preferred type
		$meta = 'app_admin_bookings_preferred_type';
		$pref = get_user_meta( $current_user->ID, $meta, true );
		if ( $pref && !( 'all' == $pref || 'active' == $pref || in_array( $pref, array_keys( $this->a->get_statuses() ) ) ) ) {
			delete_user_meta( $current_user->ID, $meta );
			$pref = '';
		}

		$type = empty( $type ) ? ( $pref ?: 'all' ) : $type;

		if ( $type != $pref ) {
			update_user_meta( $current_user->ID, $meta, $type );
		}

		# Get a sanitized filter params
		$filt = wpb_sanitize_search();
		$is_hidden = (bool)get_user_option( self::HIDDEN_NAVBAR );

		ob_start();

		do_action( 'app_admin_bookings_before_page_wrap', $args );
	?>
	<div class="wrap app-page wp-clearfix">
		<div id="app-open-navbar"<?php echo(! $is_hidden ? ' style="display:none"' : '')?>>
			<button title="<?php echo esc_attr( __( 'Open toolbar', 'wp-base' ) ) ?>" data-scope="bookings"></button>
		</div>

		<div id="app-navbar" class="app-control-wrap metabox-holder"<?php echo ($is_hidden ? ' style="display:none"' : '')?>>

			<div id="app-close-navbar">
				<button title="<?php echo esc_attr( __( 'Close toolbar', 'wp-base' ) ) ?>" data-scope="bookings"></button>
			</div>

			<div class="postbox app-controls">
				<div id="bookings-filter" class="tablenav top">
					<div class="app-actions actions">

						<div class="app-dash-title"><span><?php echo ($only_own ? BASE()->get_text('bp_bookings') : sprintf( __('%s Bookings','wp-base'), WPB_NAME ) ); ?></span></div>

						<?php $this->status_links_select( $statii, $stat, $type ); ?>

						<div class="app-mb0">
						<?php if ( wpb_eb() != 'only_events' ) { ?>
							<a href="javascript:void(0)" class="add-new-h2 add-new-booking"><?php echo (class_exists( 'WpBEvents' ) ? __('Add New Service Booking', 'wp-base') : __('Add New Booking', 'wp-base'))?></a>
						<?php }
							if ( class_exists( 'WpBEvents' ) ) { ?>
							<a href="javascript:void(0)" class="add-new-h2 add-new-event"><?php _e('Add New Event Booking', 'wp-base')?></a>
						<?php } ?>
						</div>

						<?php do_action( 'app_admin_bookings_before_status_links', $args ); ?>

						<div class="app-manage-second-column">
							<?php $this->search_form() ?>
						</div>
					</div>


					<?php do_action( 'app_admin_bookings_before_manage_rows', $args ); ?>

					<div class="app-actions actions">
						<div class="app-manage-first-column">
							<?php $this->sort_form( $filt, $type ) ?>
						</div>

						<div class="app-manage-second-column">
							<?php $this->filter_form( $type, $only_own, $sel_location ) ?>
						</div>

						<div class="app-manage-third-column">
							<form id="app-reset-form" method="get" action="<?php echo wpb_add_query_arg('page', 'app_bookings'); ?>" >
								<?php self::print_fields() ?>
								<input type="hidden" value="<?php echo $type?>" name="status" />
								<input type="hidden" value="1" name="app_filter_reset" />
								<input type="submit" class="button" value="<?php _e('Clear Filters','wp-base'); ?>" />
							</form>
						</div>
					</div>
				</div>
			</div>
		</div>

		<?php do_action( 'app_admin_bookings_before_status_form', $args ); ?>

		<div class="app-manage-row third-row app-mt">
			<div class="app-manage-first-column">
				<?php $this->status_change_form() ?>
			</div>

			<div class="app-manage-second-column app-crowded">
				<?php echo $this->displayed_columns_selection(); ?>
			</div>

		<?php
			# Pagination
			if ( is_admin() ) {
				$paged = empty( $_GET['paged'] ) ? 1 : absint( $_GET['paged'] );
			} else {
				$paged = get_query_var( 'paged' ) ? absint( get_query_var( 'paged' ) ) : 1;
			}

			$rpp 		= $count ?: wpb_setting( 'records_per_page', 20 ); # Records per page
			$startat	= ($paged - 1) * $rpp;
			$apps		= $this->get_admin_apps( $type, $startat, $rpp, $only_own, $sel_location );
			$total		= $this->get_apps_total( );

			wpb_paginate_links( $total, $paged, $rpp );
		?>

		</div>
		<?php

		do_action( 'app_admin_bookings_before_table', $args );

		$this->display_table( $apps, $columns, $columns_mobile, $allow_delete, $sel_location, $only_own, $override );

		do_action( 'app_admin_bookings_after_table', $args );

		if ( $add_export && BASE('EXIM') ) {
			BASE('EXIM')->export_csv_html( $this->a->get_text('export_csv'), $only_own );
		}

		?>
	</div> <!-- wrap -->
		<?php

		$c = ob_get_contents();
		ob_end_clean();

		if ( $return ) {
			return $c;
		} else {
			echo $c;
		}
	}

	/**
	 * Build mysql query and return results for bookings
	 */
	private function get_admin_apps( $type, $startat, $num, $only_own = 0, $sel_location = 0 ) {

		if( ! empty( $_GET['app_s'] ) && ! empty( $_GET['stype'] ) ) {
			$stype 	= esc_sql( $_GET['stype'] );
			$s 		= apply_filters( 'app_search', esc_sql( $_GET['app_s'] ), $stype );

			switch ( $stype ) {
				case 'app_id':	$sarr = array_filter( array_map( 'esc_sql', array_map( 'trim', explode( ' ', str_replace( ',', ' ', $s ) ) ) ) );
								$add = ! empty( $sarr ) ? " AND (ID='". implode( "' OR ID='", $sarr ) . "')" : "";
								break;
				case 'app_date':$add = " AND DATE(start)='{$s}' "; break;
				case 'name':	$add = " AND ( ( user IN ( SELECT ID FROM {$this->a->db->users} AS users
										 WHERE user_login LIKE '%{$s}%' OR user_nicename LIKE '%{$s}%'
										 OR ID IN ( SELECT user_id FROM {$this->a->db->usermeta} AS usermeta
										 WHERE users.ID=usermeta.user_id AND meta_value LIKE '%{$s}%'
										 AND (meta_key='app_first_name' OR meta_key='app_last_name')  ) ) )
										 OR ( user=0 AND ID IN ( SELECT object_id FROM {$this->a->meta_table}
										 WHERE meta_type='app' AND (meta_key='name' OR meta_key='first_name'
										 OR meta_key='last_name') AND meta_value LIKE '%{$s}%' )	)
										) ";
								break;
				case 'client_id':
								$add = " AND user={$s}";
								break;
				case 'email':	$add = " AND ( ( user IN ( SELECT ID FROM {$this->a->db->users} WHERE user_email LIKE '%{$s}%' ) )
										 OR ( user=0 AND ID IN ( SELECT object_id FROM {$this->a->meta_table}
										 WHERE meta_type='app' AND meta_key='email' AND meta_value LIKE '%{$s}%' ) )
										)";
								break;
				case 'phone':	$add = " AND ( ( user IN ( SELECT ID FROM {$this->a->db->users}
										 WHERE ID IN ( SELECT user_id FROM {$this->a->db->usermeta}
										 WHERE meta_key='app_phone' AND meta_value LIKE '%{$s}%' ) ) )
										 OR ( user=0 AND ID IN ( SELECT object_id FROM {$this->a->meta_table}
										 WHERE meta_type='app' AND meta_key='phone' AND meta_value LIKE '%{$s}%' ) )
										 ) ";
								break;
				case 'address':	$add = " AND ( ( user IN ( SELECT ID FROM {$this->a->db->users}
										 WHERE ID IN ( SELECT user_id FROM {$this->a->db->usermeta}
										 WHERE meta_key='app_address' AND meta_value LIKE '%{$s}%' ) ) )
										 OR ( user=0 AND ID IN ( SELECT object_id FROM {$this->a->meta_table}
										 WHERE meta_type='app' AND meta_key='address' AND meta_value LIKE '%{$s}%' ) )
										 ) ";
								break;
				case 'city':	$add = " AND ( ( user IN ( SELECT ID FROM {$this->a->db->users}
										 WHERE ID IN ( SELECT user_id FROM {$this->a->db->usermeta}
										 WHERE meta_key='app_city' AND meta_value LIKE '%{$s}%' ) ) )
										 OR ( user=0 AND ID IN ( SELECT object_id FROM {$this->a->meta_table}
										 WHERE meta_type='app' AND meta_key='city' AND meta_value LIKE '%{$s}%' ) )
										 ) ";
								break;
				case 'note':	$add = " AND ( ID IN ( SELECT object_id FROM {$this->a->meta_table}
										 WHERE meta_type='app' AND meta_key='note' AND meta_value LIKE '%{$s}%' ) ) ";
								break;
				case 'admin_note':
								$add = " AND ( ID IN ( SELECT object_id FROM {$this->a->meta_table}
										 WHERE meta_type='app' AND meta_key='admin_note' AND meta_value LIKE '%{$s}%' ) ) ";
								break;
				default:		$add = apply_filters( 'app_search_switch', '', $stype, $s ); break;
			}
		} else {
			$add = '';
		}

		# Get sanitized filter params
		$filt = wpb_sanitize_search();

		if ( ! isset( $_GET['app_or_fltr'] ) ) {
			// Filters
			if ( $filt['location_id'] ) {
				$add .= $this->a->db->prepare( " AND location=%d ", $filt['location_id'] );
			}

			if ( $filt['service_id'] ) {
				$add .= $this->a->db->prepare( " AND service=%d ", $filt['service_id'] );
			}

			// Allow filtering for unassigned provider
			if ( $filt['worker_id'] || "0" === (string)$filt['worker_id'] ) {
				$add .= $this->a->db->prepare( " AND worker=%d ", $filt['worker_id'] );
			}

			if ( $filt['monweek'] ) {
				// Year + Week
				if ( 'w' == substr( $filt['monweek'], 0, 1 ) ) {
					$mode = $this->a->start_of_week ? 7 : 4;
					$add .= $this->a->db->prepare( " AND YEARWEEK(DATE(start),%d)=%s ", $mode, substr( $filt['monweek'], 1 ) );
				} else {
					$year = substr( $filt['monweek'], 0, 4 );
					$month = substr( $filt['monweek'], 4, 2 );
					if ( $year && $month ) {
						$add .= $this->a->db->prepare( " AND YEAR(start)=%d AND MONTH(start)=%d ", $year, $month );
					}
				}
			}

			if ( $filt['balance'] ) {
				$total_paid = "IFNULL((SELECT SUM(transaction_total_amount) FROM {$this->a->transaction_table} AS tr WHERE tr.transaction_app_ID=app.ID),0)";

				if ( 'negative' == $filt['balance'] ) {
					$add .= " AND ( {$total_paid}/100 - IFNULL(price,0) - IFNULL(deposit,0) < 0 ) ";
				} else if ( 'positive' == $filt['balance'] ) {
					$add .= " AND ( {$total_paid}/100 - IFNULL(price,0) - IFNULL(deposit,0) > 0 ) ";
				} else if ( 'zero' == $filt['balance'] ) {
					$add .= " AND ( {$total_paid}/100 - IFNULL(price,0) - IFNULL(deposit,0) = 0 ) ";
				}
			}
		}

		if ( $only_own ) {
			if ( apply_filters( 'app_admin_apps_allow_unassigned', false ) ) {
				$sbw = $this->a->get_services_by_worker( get_current_user_id() );
				$services_in = $sbw ? implode( ',', array_keys( $sbw ) ) : WPB_HUGE_NUMBER;
				$add .= $this->a->db->prepare( " AND ( worker=%d OR (worker=0 AND service IN ($services_in) ) ) ", get_current_user_id() );
			} else {
				$add .= $this->a->db->prepare( " AND worker=%d ", get_current_user_id() );
			}
		}

		if ( $sel_location ) {
			$add .= $this->a->db->prepare( " AND location=%d ", $sel_location );
		}

		# Sanitize Order by
		$test = str_replace( array(' desc', ' inc', ' ' ), '', strtolower( $filt['order_by'] ) );
		$meta = 'app_admin_bookings_preferred_order';

		if ( $test && in_array( $test, array( 'id', 'start', 'id_desc', 'start_desc' ) ) ) {
			$order = str_replace( '_', ' ', $filt['order_by'] );
			update_user_meta( get_current_user_id(), $meta, $order );
		} else if ( $pref = get_user_meta( get_current_user_id(), $meta, true ) ) {
			# Order By preferred value
			$order = wpb_clean( $pref );
		} else {
			$order = "ID DESC";
		}

		$order 		= apply_filters( 'app_admin_apps_order_by', $order, $type, $add );
		$active		= "'". implode( "','", self::active_stats() ) . "'";
		$select 	= "SELECT SQL_CALC_FOUND_ROWS * FROM {$this->a->app_table} AS app ";
		$order_by 	= "ORDER BY $order";
		$limit 		= $this->a->db->prepare( "LIMIT %d,%d", $startat, $num );
		$add		= apply_filters( 'app_admin_apps_sql_add', $add, $type, $order_by, $limit );

		switch( $type ) {
			case 'all': 	$sql = $select. "WHERE 1=1 {$add} {$order_by} {$limit}"; break;
			case 'active':	$sql = $select. "WHERE status IN ({$active}) {$add} {$order_by} {$limit}"; break;
			case $type:		$sql = $select. "WHERE status IN ('".esc_sql( $type )."') {$add} {$order_by} {$limit}"; break;
			default:		$sql = $select. "WHERE 1=1 {$add} {$order_by} {$limit}"; break;
		}

		$sql = apply_filters( 'app_admin_apps_sql', $sql, $type, $add, $order_by, $limit );

		return $this->a->db->get_results( $sql, OBJECT_K );
	}

	/**
	 * Get total number from previous query
	 * @return integer
	 */
	private function get_apps_total( ) {
		return $this->a->db->get_var( "SELECT FOUND_ROWS();" );
	}

	/**
	 * Which statuses are considered as active
	 * @since 3.7.2.1
	 * @return array
	 */
	public static function active_stats(){
		return apply_filters( 'app_admin_apps_active_stats', array( 'confirmed', 'paid' ) );
	}

	/**
	 * Helper to produce status links as select for small screen
	 * @return string
	 */
	private function status_links_select( $statii, $stat, $type ){
	?>
	<div class="app-status-links-select">
		<select class="app-status-elm" onchange="if (this.value) window.location.href=''+this.value">
			<option disabled><?php _e('Select Status', 'wp-base') ?></option>
			<option value="<?php $this->status_links( 'all' ) ?>" <?php selected( $type, 'all' )?>><?php _e('All', 'wp-base') ?></option>
			<option value="<?php $this->status_links( 'active' ) ?>" <?php selected( $type, 'active' )?>><?php _e('Upcoming', 'wp-base') ?></option>
		<?php
		foreach( $statii as $status => $text ) {
			?><option value="<?php $this->status_links( $status ) ?>" <?php selected( $type, $status )?>><?php echo $text ?></option><?php
		}
	?></select>
	</div>
	<?php
	}

	/**
	 * Helper to produce status link
	 * @return string
	 */
	private function status_links( $type ) {
		echo esc_attr( wpb_add_query_arg( array( 'status' => $type, 'paged' => false, 'add_new' => false, 'cpy_from' => false ) ) );
	}

	/**
	 * Helper to produce search form
	 * @return string
	 */
	private function search_form(){
		$stype = ! empty( $_GET['stype'] ) ? wpb_clean( $_GET['stype'] ) : '';
		$s = ! empty( $_GET['app_s'] ) ? wpb_clean( $_GET['app_s'] ) : '';
		$s = apply_filters( 'app_search', $s, $stype );
	?>
	<form id="app-search-form" method="get" action="" class="search-form">
		<?php self::print_fields() ?>
		<input type="hidden" value="all" name="status" />
		<input type="hidden" value="1" name="app_or_fltr" />
		<input type="search" value="<?php echo esc_attr($s); ?>" name="app_s"/>

		<?php $add_class = $stype === 'app_id' ? 'class="app-option-selected"' : ''; ?>
		<select name="stype" <?php echo $add_class ?> title="<?php _e('Select which field to search. For appointment ID search, multiple IDs separated with comma or space is possible.','wp-base') ?>">
			<?php do_action( 'app_search_options_pre', $stype ) ?>
			<option value="app_id" <?php selected( $stype, 'app_id' ); ?> title="<?php _e('Multiple IDs separated with comma or space is possible','wp-base')?>"><?php _e('Booking ID','wp-base'); ?></option>
			<option value="app_date" <?php selected( $stype, 'app_date' ); ?>><?php _e('Booking Date','wp-base'); ?></option>
			<option value="name" <?php selected( $stype, 'name' ); ?>><?php _e('Client Name','wp-base'); ?></option>
			<option value="client_id" <?php selected( $stype, 'client_id' ); ?>><?php _e('Client ID','wp-base'); ?></option>
			<option value="email" <?php selected( $stype, 'email' ); ?>><?php _e('Email','wp-base'); ?></option>
			<option value="phone" <?php selected( $stype, 'phone' ); ?>><?php _e('Phone','wp-base'); ?></option>
			<option value="address" <?php selected( $stype, 'address' ); ?>><?php _e('Address','wp-base'); ?></option>
			<option value="city" <?php selected( $stype, 'city' ); ?>><?php _e('City','wp-base'); ?></option>
			<option value="note" <?php selected( $stype, 'note' ); ?>><?php _e('Note','wp-base'); ?></option>
			<option value="admin_note" <?php selected( $stype, 'admin_note' ); ?>><?php _e('Admin Note','wp-base'); ?></option>
			<?php do_action( 'app_search_options', $stype ) ?>
		</select>

		<input type="submit" class="button app-search-button" value="<?php _e('Search','wp-base'); ?>" />
	</form>
	<?php
	}

	/**
	 * Helper to produce filter form
	 * @return string
	 */
	private function filter_form( $type, $only_own, $sel_location ){
		# Get sanitized filter params
		$filt = wpb_sanitize_search();
	?>
	<form id="app-filter-form" class="app-filter-form" method="get" action="" >
		<?php
		do_action( 'app_admin_bookings_form_filter_pre' );

		switch($type) {
			case 'all':			$where = " WHERE 1=1 "; break;
			case 'running':		$where = " WHERE status IN ('running') "; break;
			case 'active':		$where = " WHERE status IN ('confirmed', 'paid') "; break;
			case 'pending':		$where = " WHERE status IN ('pending') "; break;
			case 'completed':	$where = " WHERE status IN ('completed') ";	break;
			case 'removed':		$where = " WHERE status IN ('removed') "; break;
			case 'reserved':	$where = " WHERE status IN ('reserved') "; break;
			default:			$where = $this->a->db->prepare( " WHERE status IN (%s) ", $type ); break;
		}

		$where = apply_filters( 'app_admin_apps_where', $where, $type );

		if ( $only_own )
			$where .= " AND worker='".get_current_user_id()."' ";

		if ( $sel_location )
			$where .= $this->a->db->prepare( " AND location=%d ", $sel_location );

		$add_class = $filt['monweek'] ? 'class="app-option-selected"' : '';
		?>
		<select name="app_monweek" <?php echo $add_class ?>>
			<option value=""><?php _e('Filter by month/week','wp-base'); ?></option>
			<optgroup label="<?php echo ucwords( $this->a->get_text('month') ) ?>">
				<?php wpb_month_options( $where ) ?>
			</optgroup>
			<optgroup label="<?php echo ucwords( $this->a->get_text('week') ) ?>">
				<?php wpb_week_options( $where ) ?>
			</optgroup>
		</select>
		<?php

		$add_class = $filt['balance'] ? 'class="app-option-selected"' : '';
		?>
		<select name="app_balance" <?php echo $add_class ?>>
			<option value=""><?php _e('Filter by balance','wp-base'); ?></option>
			<option value="negative" <?php selected( $filt['balance'],'negative' ) ?>><?php _e('Negative balance','wp-base'); ?></option>
			<option value="positive" <?php selected( $filt['balance'],'positive' ) ?>><?php _e('Positive balance','wp-base'); ?></option>
			<option value="zero" <?php selected( $filt['balance'], 'zero' ) ?>><?php _e('Zero balance','wp-base'); ?></option>
		</select>
		<?php

		$locations = $this->a->get_locations( 'name' );

		if ( $locations && ! $sel_location ) {
			$add_class = $filt['location_id'] ? 'class="app-option-selected"' : '';
		?>
			<select name="app_location_id" <?php echo $add_class ?>>
				<option value=""><?php _e('Filter by location','wp-base'); ?></option>
		<?php
			foreach ( $locations as $location ) {
				$selected = $filt['location_id'] == $location->ID ? " selected='selected' " : '';
				echo '<option '.$selected.' value="'.$location->ID.'">'. $this->a->get_location_name( $location->ID ) .'</option>';
			}
		?>
			</select>
		<?php }

			$add_class = $filt['service_id'] ? 'class="app-option-selected"' : '';
		?>
		<select name="app_service_id" <?php echo $add_class ?>>
			<option value=""><?php _e('Filter by service','wp-base'); ?></option>
			<?php
			$services = $sel_location ? $this->a->get_services_by_location( $sel_location, 'name' ) : $this->a->get_services( 'name' );

			foreach ( (array)$services as $service ) {

				if ( $this->a->is_package( $service->ID ) ) {
					continue;
				}

				$selected = $filt['service_id'] == $service->ID ? " selected='selected' " : '';
				echo '<option '.$selected.' value="'.$service->ID.'">'. $this->a->get_service_name( $service->ID ) .'</option>';
			}
			?>
		</select>
		<?php

		if ( ! $only_own ) {
			$workers = $this->a->get_workers( 'name' );
			$add_class = $filt['worker_id'] || "0" === (string)$filt['worker_id'] ? 'class="app-option-selected"' : '';
			?>
			<select name="app_worker_id" <?php echo $add_class ?>>
				<option value=""><?php _e('Filter by service provider','wp-base'); ?></option>
			<?php if ( $workers ) {  ?>
				<option value="0" <?php selected( $filt['worker_id'], 0 ) ?>><?php _e('Unassigned','wp-base'); ?></option>
			<?php
				foreach ( $workers as $worker ) {
					$selected = $filt['worker_id'] == $worker->ID ? " selected='selected' " : '';
					echo '<option '.$selected.' value="'.$worker->ID.'">'. $this->a->get_worker_name( $worker->ID ) .'</option>';
				}
			}
			?>
			</select>
		<?php } ?>

		<?php do_action( 'app_admin_bookings_form_filter' )  ?>

		<?php self::print_fields() ?>
		<input type="hidden" value="<?php echo $type?>" name="status" />
		<input type="submit" class="button" value="<?php _e('Filter','wp-base'); ?>" />
	</form>
	<?php
	}

	/**
	 * Helper to produce status change form
	 * @return string
	 */
	private function status_change_form(){
	?>
	<form id="app-bulk-change-form" method="post" action="" >
		<select name="app_new_status">
			<option value=""><?php _e('Bulk status change','wp-base'); ?></option>
			<?php foreach ( $this->a->get_statuses() as $value => $name ) {
					if ( 'running' == $value )
						continue;
				echo '<option value="'.$value.'">'.$name.'</option>';
			} ?>
		</select>
		<?php self::print_fields() ?>
		<input type="hidden" value="<?php if ( isset( $post->ID ) ) echo $post->ID; else echo 0; ?>" name="page_id" />
		<input type="hidden" value="app_status_change" name="action_app" />
		<input type="hidden" value="1" name="app_status_change" />
		<?php wp_nonce_field( 'update_app_settings', 'app_nonce' ); ?>
		<input type="submit" class="button app-change-status-btn" value="<?php _e('Change','wp-base'); ?>" />
	</form>
	<?php
	}

	/**
	 * Helper to produce sort form
	 * @param $filt		array	Filtered search parameters
	 * @return string
	 */
	private function sort_form( $filt, $type ) {
	?>
	<form id="app-sort-form" method="get" action="" >
		<select name="app_order_by">
			<option value=""><?php _e('Sort by','wp-base'); ?></option>
			<option value="start" <?php selected( $filt['order_by'], 'start' ); ?>><?php _e('App. date (01/01 &rarr; 31/12)','wp-base'); ?></option>
			<option value="start_DESC" <?php selected( $filt['order_by'], 'start_DESC' ); ?>><?php _e('App. date (31/12 &rarr; 01/01)','wp-base'); ?></option>
			<option value="ID" <?php selected( $filt['order_by'], 'ID' ); ?>><?php _e('Booking ID (1 &rarr; 99)','wp-base'); ?></option>
			<option value="ID_DESC" <?php selected( $filt['order_by'], 'ID_DESC' ); ?>><?php _e('Booking ID (99 &rarr; 1)','wp-base'); ?></option>
		</select>
		<?php self::print_fields() ?>
		<input type="hidden" value="<?php echo $type ?>" name="status" />
		<input type="hidden" value="<?php echo $filt['location_id'] ?>" name="app_location_id" />
		<input type="hidden" value="<?php echo $filt['service_id'] ?>" name="app_service_id" />
		<input type="hidden" value="<?php echo $filt['worker_id'] ?>" name="app_worker_id" />
		<input type="hidden" value="<?php echo $filt['monweek'] ?>" name="app_monweek" />
		<input type="hidden" value="<?php echo $filt['balance'] ?>" name="app_balance" />
		<input type="submit" class="button" value="<?php _e('Sort','wp-base'); ?>" />
	</form>
	<?php
	}

	/**
	 * Helper function for render bookings table
	 */
	public function display_table( $apps, $columns, $columns_mobile, $allow_delete, $sel_location, $only_own, $override ) {

		wpb_add_action_footer( $this );

		// Load defaults and sanitize columns
		if ( '' == trim($columns) ) {
			$columns = apply_filters( 'app_bookings_default_columns','client,id,created,created_by,email,phone,address,zip,city,country,date_time,end_date_time,location,service,worker,status,price,deposit,total_paid,balance');
		}

		$_columns	= wpb_is_mobile() && trim( $columns_mobile ) ? $columns_mobile : $columns;
		$cols		= explode( ',', wpb_sanitize_commas( $_columns ) );
		$cols		= array_map( "strtolower", $cols );
		array_unshift( $cols, 'delete' );

		$ret  = apply_filters( 'app_bookings_before_table', '<form class="app-form" method="post" ><div class="app-scrollable">', $cols );
		$ret .= '<table %HIDDENCOLUMNS% data-location="'.esc_attr($sel_location).'" data-only_own="'.esc_attr($only_own).'" data-override="'.esc_attr($override).'"
				class="wp-list-table widefat app-manage dt-responsive display '.( $apps && ! isset( $_GET['add_new'] ) ? 'dataTable' : '').'">';
		$ret .= '<thead>';

		/* hf : Common for header-footer */
		$this->colspan = 0;
		$hf = $this->table_head( $cols );

		$ret .= $hf. '</thead>';
		// Remove id from foot
		$ret .= '<tfoot>' . preg_replace( '/<th id="(.*?)"/is','<th ', $hf ). '</tfoot>';
		$ret = str_replace(
				'%HIDDENCOLUMNS%',
				'data-hide_boxes="'.implode( ',', array_diff( self::get_allowed_columns(), $this->used_cols ) ).'"',
				$ret
			);

		$ret .= '<tbody>';
		$ret  = apply_filters( 'app_bookings_table_tbody', $ret, $cols );
		echo $ret;

		$ret = '';
		
		$apps = apply_filters( 'app_bookings_table_before_apps', $apps );

		if ( $apps ) {

			remove_filter( 'gettext', array( 'WpBCustomTexts', 'global_text_replace' ) );

			# Preload app and user cache as a whole instead of calling one by one
			WpBMeta::update_meta_cache( 'app', wp_list_pluck( $apps, 'ID' ) );
			update_meta_cache( 'user', wp_list_pluck( $apps, 'user' ) );

			# Save app list results
			wp_cache_set( wpb_cache_prefix() . 'apps_in_listing', $apps );

			do_action( 'app_bookings_table_before_loop', $apps );
			
			foreach ( $apps as $r ) {

				$classes = array( 'app-tr' );

				if ( $r->parent_id ) {
					$classes[] = 'multi-app-child';
				}

				if ( $this->check_lock( $r->ID ) ) {
					$classes[] = 'app-locked';
				}

				$ret .= '<tr id="app-tr-'.$r->ID.'" class="'.implode( ' ', $classes ).'">';

				foreach( $cols as $col ) {

					if ( !in_array( $col, self::get_allowed_columns() ) ) {
						continue;
					}

					$hidden			= in_array( $col, $this->get_hidden_columns() ) ? ' hidden' : '';
					$col_primary	= in_array( $col, array('client' ) ) ? ' column-primary' : '';
					$col_check		= in_array( $col, array( 'delete' ) ) ? ' check-column' : '';
					$col_udf		= 'udf_' == substr( $col, 0, 4 ) ? ' column-udf' : '';

					$ret .= '<td class="column-'.$col. $hidden. $col_primary. $col_check. $col_udf. '">';
					$ret .= $this->table_cell( $col, $r, $apps, $cols );
					$ret .= '</td>';
				}
				$ret .= '</tr>';
			}
			echo $ret;

		} else {
			?>
			<tr class="alternate app-tr">
				<td colspan="<?php echo $this->colspan; ?>" scope="row"><?php _e('No matching bookings have been found.','wp-base'); ?></td>
			</tr>
	<?php }	?>
		</tbody>
	</table>
		<?php
		# Only for "Removed" tab
		if ( isset( $_GET["status"] ) && 'removed' == $_GET["status"] ) {
			global $post;
			if ( $allow_delete && wpb_admin_access_check( 'delete_bookings', false ) ) {
		?>
			<p>
			<input type="submit" id="delete_removed" class="button-secondary" value="<?php _e('Permanently Delete Selected Records', 'wp-base' ) ?>" title="<?php _e('Clicking this button permanently deletes selected records', 'wp-base' ) ?>" />
			<?php wp_nonce_field( 'delete_or_reset', 'app_delete_nonce' ); ?>
			<input type="hidden" value="<?php if ( isset( $post->ID ) ) echo $post->ID; else echo 0; ?>" name="page_id" />
			<input type="hidden" name="delete_removed" value="delete_removed" />
			<input type="hidden" name="action_app" value="delete_removed" />
			</p>
		<?php }
		}

		do_action( 'app_admin_apps_form' );

		echo apply_filters( 'app_bookings_after_table', '</div></form>', $cols );

	}

	/**
	 * Adds monetary total values to footer so that they can be read by tooltip
	 * @since 2.0
	 */
	public function footer(){
		foreach ( $this->money_vars as $name => $money_var ) {
			echo '<div id="'.$name.'-tt" style="display:none">';
			printf( __( '%s totals on this page:', 'wp-base' ), ucfirst( $this->a->get_text( $name ) ) );
			echo '<br/>';
			foreach( $money_var as $client => $var ) {
				if ( is_numeric( $client ) && $client > 0 ) {
					$userdata = BASE('User')->get_app_userdata( 0, $client, true );
					$display = isset($userdata['name']) ? $userdata['name'] : ( isset($userdata['email']) ? $userdata['email'] : $client);
				} else if ( is_email( $client ) ) {
					$display = $client;
				} else {
					$display = $client;
				}

				echo $display .': '. wpb_format_currency( $var ) .'<br/>';
			}

			echo '</div>';
		}
	}

	/**
	 * Prepare header for bookings table
	 * @since 3.0
	 */
	private function table_head( $cols ) {
		$hf = '<tr>';
		foreach( $cols as $col ) {

			if ( ! in_array( $col, self::get_allowed_columns() ) ) {
				continue;
			}

			$this->used_cols[] 	= $col; // Used for js to hide not allowed columns. Hidden columns should also be here!
			$hidden 			= in_array( $col, $this->get_hidden_columns() ) ? " hidden" : "";
			$col_primary 		= in_array( $col, array( 'client' ) ) ? " column-primary" : "";
			$col_check 			= in_array( $col, array( 'delete' ) ) ? " check-column" : "";

			if ( ! $hidden ) {
				$this->colspan++;
			}

			$hf .= '<th id="'.$col.'" class="manage-column column-'.$col. $hidden. $col_primary. $col_check. '">';

			switch ($col) {
				case 'delete':		$hf .= '<input type="checkbox" class="app-no-save-alert" />';
									break;
				case 'id':			$hf .= $this->a->get_text('app_id');
									break;
				case 'provider':
				case 'worker':		$hf .= $this->a->get_text('provider');
									break;

				case $col:			if ( 'udf_' == substr( $col, 0, 4 ) )
										$hf .= apply_filters('app_bookings_udf_column_title','',$col);
									else
										$hf .= '<span>'. apply_filters( 'app_bookings_column_title', str_replace( ':', '', $this->a->get_text($col)), $col) .'</span>'; // E.g. status, address, phone
									break;
				default:			break;
			}
			$hf .= '</th>';

		}
		$hf .= '</tr>';

		return $hf;
	}

	/**
	 * Prepare a single cell in bookings table
	 * @since 3.0
	 */
	private function table_cell( $col, $r, $apps, $cols ) {
		$client		= ! empty( $r->user ) ? $r->user : ( ! empty($r->email) ? $r->email : BASE('User')->get_client_name( $r->ID, $r, true, 22, true ) );
		$is_daily	= $this->a->is_daily( $r->service );
		$booking	= new WpB_Booking( $r->ID );

		$ret = '';

		switch ( $col ) {
			case 'delete':			$ret .= '<input type="checkbox" class="app-no-save-alert" name="app[]" value="'. $r->ID .'" />';
									$ret .= '<div class="locked-indicator"><span class="locked-indicator-icon" aria-hidden="true"></span></div>';
									break;

			case 'id': 				$ret .= $this->table_cell_ID( $r ); break;
			case 'client':			$ret .= $this->table_cell_client( $r, $apps ); break;
			case 'created':			$ret .= date_i18n( $this->a->dt_format, strtotime( $r->created ) ); break;
			case 'created_by':		$ret .= $this->created_by( $r->ID ); break;
			case 'location':		$ret .= $this->a->get_location_name( $r->location ); break;
			case 'location_address':$ret .= wpb_get_location_meta( $r->location, 'address' ); break;
			case 'service':			$ret .= apply_filters( 'app_bookings_service_name', wpb_cut( $this->a->get_service_name( $r->service ) ), $r ); break;
			case 'provider':
			case 'worker':			$ret .= wpb_cut( $this->a->get_worker_name( $r->worker ) ); break;
			case 'price':
									$price = apply_filters( 'app_bookings_price', (! empty( $r->price ) ? (float)$r->price : 0), $booking );
									$this->add_money_var( 'price', $client, $price );
									$ret .= wpb_format_currency( $price, false, true );
									break;
			case 'deposit':
									$deposit = apply_filters( 'app_bookings_deposit', (! empty( $r->deposit ) ? (float)$r->deposit : 0), $booking );
									$this->add_money_var( 'deposit', $client, $deposit );
									$ret .= wpb_format_currency( $deposit, false, true );
									break;
			case 'total_paid':
									$paid = self::get_paid_by_app( $r->ID );
									$paid = apply_filters( 'app_bookings_total_paid', $paid/100, $booking );
									$this->add_money_var( 'total_paid', $client, $paid );
									$ret .= wpb_format_currency( $paid, false, true );
									break;
			case 'balance':
									$paid = self::get_paid_by_app( $r->ID );
									$balance = $paid/100 - (float)$r->price - (float)$r->deposit;
									$balance = apply_filters( 'app_bookings_balance', $balance, $booking );
									$this->add_money_var( 'balance', $client, $balance );
									$ret .= wpb_format_currency( $balance, false, true );
									break;
			case 'email':
			case 'phone':
			case 'city':
			case 'address':
			case 'zip':
			case 'country':
			case 'note':			$ret .= wpb_get_app_meta( $r->ID, $col ); break;
			case 'date_time':		$ret .= $this->table_cell_dt( $r, $is_daily ); break;
			case 'date':			$ret .= $this->table_cell_dt( $r, true ); break;
			case 'day':				$ret .= date_i18n( "l", strtotime( $r->start ) ); break;
			case 'time':			$ret .= date_i18n( $this->a->time_format, strtotime( $r->start ) ); break;
			case 'end_date_time':	$ret .= $this->table_cell_dt( $r, $is_daily, true ); break;
			case 'end_date':		$ret .= $this->table_cell_dt( $r, true, true ); break;
			case 'status':			$ret .= apply_filters( 'app_bookings_status', $this->table_cell_status( $r ), $r ); break;
			default:				$ret .= apply_filters( 'app_bookings_add_cell', '', $col, $r, $cols ); break;
		}

		return $ret;
	}

	/**
	 * Get total paid amount or a booking (Multiplied by x100)
	 * @param $app_id	integer		Booking ID
	 * @since 3.5.7
	 * @return integer
	 */
	private static function get_paid_by_app( $app_id ) {
		$trs = wpb_get_all_transactions();

		if ( isset( $trs[ $app_id ] ) && isset( $trs[ $app_id ]->total_paid ) ) {
			return $trs[ $app_id ]->total_paid;
		} else {
			return 0;
		}
	}

	/**
	 * Prepare date/time cell content
	 * @param $just_date	bool		if true only date, if false date+time
	 * @param $end			bool		If true "end", if false "start"
	 * @since 3.0
	 * @return string
	 */
	private function table_cell_dt( $r, $just_date = false, $end = false ) {
		$format = $just_date ? $this->a->date_format : $this->a->dt_format;
		$var	= $end ? $r->end : $r->start;

		return '<span title="'. date_i18n("l", strtotime($var)). '">'. date_i18n( $format, strtotime( $var ) ). '</span>';
	}

	/**
	 * Prepare ID cell content
	 * @since 3.0
	 * @return string
	 */
	private function table_cell_ID( $r ) {
		$ret = '<span class="span_app_ID">'. apply_filters( 'app_ID_text', $r->ID, $r ) .'</span>';
		if ( 'reserved' != $r->status && ! $r->parent_id ) {
			$href = wpb_add_query_arg( array( 'add_new' => 1, 'cpy_from' => $r->ID, 'app_timestamp' => false, 'app_worker' => false ) );
			$ret .= '<div><input type="button" data-href="'.$href.'"';
			$ret .= ' class="rebook-button button-secondary button-petit" value="'.__('Rebook','wp-base').'" /></div>';
		}

		return $ret;
	}

	/**
	 * Prepare client cell content
	 * @since 3.0
	 * @return string
	 */
	private function table_cell_client( $r, $apps ) {
		$lock_holder = $this->check_lock( $r->ID );

		if ( $lock_holder ) {
			$lock_holder   = get_userdata( $lock_holder );
			$locked_avatar = get_avatar( $lock_holder->ID, 18 );
			/* translators: %s: User's display name. */
			$locked_text = esc_html( sprintf( __( '%s is editing', 'wp-base' ), $lock_holder->display_name ) );
		} else {
			$locked_avatar = '';
			$locked_text   = '';
		}

		$ret = '<div class="locked-info"><span class="locked-avatar">' . $locked_avatar . '</span> <span class="locked-text">' . $locked_text . "</span></div>\n";

		$ret .= '<div class="user-inner">';

		$is_multi		= BASE('Multiple')->is_active();
		$client_name 	= '<span class="app-client-name">'.
						  BASE('User')->get_client_name( $r->ID, $r, true, 22, true ).
						  apply_filters( 'app_bookings_add_text_after_client', '', $r->ID, $r ).
						  '</span>';
		$link 			= wpb_add_query_arg( array('status'=>'all','app_or_fltr'=>1,'app_s'=>$r->ID,'stype'=>'app_id') );

		if ( ! $is_multi ) {
			$ret .= $client_name;
		} else if ( !$r->parent_id ) {
			if ( $children = BASE('Multiple')->get_children( $r->ID, $apps ) ) {
				$title = esc_attr( sprintf( __( 'Parent of %s', 'wp-base' ), '#'. implode( ', #', wp_list_pluck( $children, 'ID' ) ) ) );
				$ret .= '<a href="'.$link.'"><span class="dashicons dashicons-migrate app-mr5" title="'.$title.'"></span></a>'.$client_name;
			} else {
				$ret .= '<span class="dashicons app-mr5"></span>'.$client_name;
			}
		} else {
			$ret .= '<a href="'.$link.'"><span class="dashicons dashicons-migrate reverse app-mr5" title="'.sprintf( __( 'Child of #%d', 'wp-base' ), $r->parent_id ).'"></span></a><span class="app-inner-child">' . $client_name . '</span>';
		}

		$ret .= '<span class="booking-info">'.apply_filters( 'app_ID_text', $r->ID, $r ) .'</span>';
		$ret .= '<span class="booking-info">'.apply_filters( 'app_bookings_service_name', wpb_cut( $this->a->get_service_name( $r->service ) ), $r ) .'</span>';
		$ret .= '<span class="booking-info">'.date_i18n( $this->a->dt_format, strtotime( $r->start ) ) .'</span>';
		$ret .= '</div>';

		$dashicon = 'reserved' == $r->status ? 'dashicons-editor-expand' : ( $is_multi ? 'dashicons-edit' : 'dashicons-edit app-pl0' );
		$ret .= '<div class="row-actions"><span class="dashicons '.$dashicon.'"></span>';

		$ret .= '<a href="javascript:void(0)" class="app-inline-edit'.($lock_holder ? ' app-takeover': '').'" title="'.__('Click to edit booking','wp-base').'">';

		if ( 'reserved' == $r->status ) {
			$ret .= __('Details', 'wp-base');
		} else {
			$ret .= __('Edit', 'wp-base');
		}

		$ret .= '</a>';

		$ret  = apply_filters( 'app_bookings_table_cell_client_after_actions', $ret, $r );

		$ret .= '</div>';

		return $ret;
	}

	/**
	 * Prepare status cell content
	 * @since 3.0
	 * @return string
	 */
	private function table_cell_status( $r ) {
		$app = $r;
		$ret = '';
		$booking = new WpB_Booking( $app->ID );

		if ( 'paid' == $app->status ) {
			if ( is_admin() ) {
			 $ret .= '<a href="'.admin_url('admin.php?page=app_transactions&amp;type=past&amp;stype=app_id&amp;app_s=').$r->ID.'" title="'.__('To view the transaction, click this link','wp-base').'"><span class="app-status">'. $this->a->get_status_name( 'paid' ) . '</span></a>';
			} else {
				$ret .= '<span class="app-status">'.$this->a->get_status_name( 'paid' ).'</span>';
			}
		} else if ( 'pending' == $app->status ) {
			if ( empty( $app->price ) || empty( $app->payment_method ) || 'manual-payments' == $app->payment_method ) {
				$ret .= $this->a->get_text('pending_approval');
			} else {
				$ret .= $this->a->get_text('pending_payment');
			}
		} else {
			$ret .= '<span class="app-status">'.$this->a->get_status_name( $app->status ).'</span>';
		}

		if ( 'removed' === $app->status && $reason = $booking->why_removed() ) {
			$ret .= '<span class="app-abandon">['. $reason .']</span>';
		}

		return $ret;
	}

	/**
	 * Add monetary variable to its field
	 * @since 3.0
	 */
	private function add_money_var( $field, $client, $val ) {
		$this->money_vars[ $field ][ $client ] = isset( $this->money_vars[ $field ][ $client ] )
												 ? $this->money_vars[ $field ][ $client ] + $val
												 : $val;
	}

	/**
	 * Edit or create appointments on admin side
	 */
	public function inline_edit( $echo = false, $colspan = 0 ) {

		if ( ! $echo ) {
			if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) )
				die( json_encode( array( 'error' => $this->a->get_text('unauthorised') ) ) );
		}

		$_colspan 		= ! empty( $_POST['col_len'] ) ? wpb_clean( $_POST['col_len'] ) : ( $colspan ? $colspan : 7 );
		$safe_format	= $this->a->safe_date_format();
		$step			= 60 * $this->a->get_min_time();

		$app_id			= isset( $_REQUEST["app_id"] ) ? wpb_clean( $_REQUEST["app_id"] ) : 0;
		$booking		= new WpB_Booking( $app_id );
		$controller 	= $this->prepare( $booking );
		$bid			= $booking->get_ID();
		$js_id			= $bid ? $bid : $booking->get_uniqid(); 	// If this is a new appt, generate an id to be used by javascript

		wpb_delete_app_meta( $bid, '_edit_lock' );
		$this->set_lock( $bid );

		$html  = '';
		$html .= '<tr class="inline-edit-row inline-edit-row-post quick-edit-row-post '.($bid ? "" : "inline-edit-row-add-new").'">';
		$html .= '<td colspan="'.$_colspan.'" class="colspanchange">';
		
		$html = apply_filters( 'app_inline_edit_before_fields', $html, $booking, $this );

	/* LEFT COLUMN */
		$html .= '<fieldset class="inline-edit-col-left">';
		$html .= '<div class="inline-edit-col">';
		$html .= '<h4 class="app_iedit_client_h">'.__('CLIENT', 'wp-base').'</h4>';

		/* user */
		$html .= wpb_wrap_field( 'user', __('User', 'wp-base'),
				apply_filters( 'app_inline_edit_users_html', BASE('User')->app_dropdown_users( apply_filters( 'app_inline_edit_users_args',
					array(
						'show_option_all'	=> __('Not registered user','wp-base'),
						'echo'				=> 0,
						'add_email' 		=> true,
						'selected'			=> $booking->get_user() ?: get_user_meta( get_current_user_id(), 'app_preferred_user', true ),
						'name'				=> 'user',
						'class'				=> 'app_users',
						'id'				=> 'app_users_'.$js_id
					),
					$booking )
				), $booking )
			);

		/* Client fields */
		foreach ( $this->a->get_user_fields() as $f ) {
			$field_name = 'name' === $f ? 'cname' : $f;
			$_bid = $booking->get_source() ? $booking->get_source() : $bid;
			$value = wpb_get_app_meta( $_bid, $f );
			if ( wpb_is_demo() ) {
				if ( 'email' === $f )
					$value = 'email@example.com';
				else if ( 'phone' === $f )
					$value = '0123456789';
				else
					$value = 'Demo '. $f;
			}

			$html .= wpb_wrap_field( $f, wpb_get_field_name($f),
					'<input type="text" name="'.$field_name.'" class="ptitle" autocomplete="'.$f.'" value="'.$value.'" />'
			);
		}

		/* UDF fields */
		$html  = apply_filters( 'app_inline_edit_user_fields', $html, $booking, $controller );

		$html .= '</div>';
		$html .= '</fieldset>';

	/* CENTER COLUMN */
		$html .= '<fieldset class="inline-edit-col-center">';
		$html .= '<div class="inline-edit-col">';

		$html .= '<h4 class="app_iedit_lsw_h">';
		$html .= $booking->is_event() ? __('LOCATION - EVENT', 'wp-base') : __('LOCATION - SERVICE - PROVIDER', 'wp-base');
		$html .= '</h4>';

		/* Locations */
		$html .= wpb_wrap_field( 'location', __('Location', 'wp-base'), $controller->select_location( ) );

		/* Services */
		$html .= wpb_wrap_field( 'service',
			$booking->is_event() ? __('Event', 'wp-base') : __('Service', 'wp-base'),
			$controller->select_service( !$app_id )
		);

		/* Recurring or similar */
		$html  = apply_filters( 'app_inline_edit_after_service', $html, $booking, $controller );

		/* Workers */
		if ( !$booking->is_event() ) {
			$html .= wpb_wrap_field( 'worker', __('Provider', 'wp-base'), $controller->select_worker( ) );
		}

		/* Pricing */
		$html .= $this->price_fields_html( $booking );

		$html .= '</div>';

		$html = apply_filters( 'app_inline_edit_after_price_fields', $html, $booking, $this );

		$html .= '</fieldset>';

	/* RIGHT COLUMN */
		$html .= '<fieldset class="inline-edit-col-right">';
		$html .= '<div class="inline-edit-col">';
		$html .= $this->appt_fields_html( $booking );
		$html .= '</div>';
		$html .= '</fieldset>';

		$html .= '<div style="clear:both"></div>';

	/* SAVE and OTHER ACTIONS */
		$html .= '<div class="inline-edit-save wp-clearfix">';
		$html .= $this->actions_html( $booking );
		$html .= '</div>';

		$html = apply_filters( 'app_inline_edit_after_actions', $html, $booking, $this );

		$html .= '</td>';
		$html .= '</tr>';

		if ( $echo ) {
			echo $html;
		} else {
			die( json_encode( array(
				'result'	=> $html,
				'id'		=> $js_id,
				'locked'	=> wpb_get_app_meta( $booking->get_ID(), 'unlocked' ) ? 1 : 0,
			) ) );
		}
	}

	private function controller_order_by( $booking = null ) {
		return apply_filters( 'app_admin_controller_order_by', 'name', $booking );
	}

	/**
	 * Initiate controller object
	 * Config booking object for new booking and rebook
	 * @return object	WpB_Controller object
	 */
	private function prepare( &$booking ) {

		if ( $booking->get_ID() ) {
			$controller = new WpB_Controller( $booking, self::controller_order_by( $booking ) );
		} else {
			if ( ! empty( $_REQUEST['add_new_event'] ) ) {
				$booking->set_as_event();
			}

			/* Rebook */
			if ( isset( $_REQUEST['add_new'] ) && isset( $_REQUEST['cpy_from'] ) && $app = wpb_get_app( wpb_clean( $_REQUEST['cpy_from'] ) ) ) {
				$booking = new WpB_Booking( $app->ID );
				$booking->set_source( $app->ID );
				$booking->set_ID( 0 );
				if ( ! empty( $_POST['sel_location'] ) ) {
					$booking->set_location( wpb_clean( $_POST['sel_location'] ) );
				}
				$booking->set_status( 'confirmed' );
				$booking->set_payment_method('');

				$controller = new WpB_Controller( $booking, self::controller_order_by( $booking ) );
			} else { /* Add New */
				$controller = $this->init_controller();

				$booking->set_location( $controller->get_location() );
				$booking->set_service( $controller->get_service() );
				$booking->set_worker( $controller->get_worker() );
			}

			$duration = wpb_get_duration( $booking ) * 60;
			$calendar = new WpB_Calendar( $booking );

			// If selected thru Calendar page add a booking
			if ( ! empty( $_REQUEST['app_timestamp'] ) ) {
				$timestamp = wpb_clean( $_REQUEST['app_timestamp'] );
				$booking->set_start( $timestamp );
				$booking->set_end( $timestamp + $duration );
			} else if ( $booking->is_event() ) {
				$booking = apply_filters( 'app_admin_bookings_prepare_event', $booking );
				$controller->set_service( $booking->get_service() );
			} else {
				$first		= false;
				$max_days	= min( $this->a->get_upper_limit(), 365 );
				$today		= strtotime( 'today', $this->a->_time );

				for ( $d = $today; $d < $today + $max_days *DAY_IN_SECONDS; $d = $d + DAY_IN_SECONDS ) {
					$out = $calendar->find_slots_in_day( $d, 1 );
					if ( count( $out ) ) {
						$first = key( $out );
						break;
					}
				}

				if ( $first ) {
					$booking->set_start( $first );
				} else if ( $booking->is_daily() ) {
					$booking->set_start( strtotime( 'tomorrow' ) );
				} else {
					$step = 60 * $this->a->get_min_time();
					$booking->set_start( $step*(intval( $this->a->_time/$step ) + 1 ) );
				}

				$booking->set_end( wpb_strtotime( $booking->get_start() ) + $duration );
			}

			$slot = new WpB_Slot( $booking );
			$booking->set_price( $slot->get_price( ) );
			$booking->set_deposit( $slot->get_deposit( ) );
		}

		return $controller;
	}

	/**
	 * Instantiate a Controller object
	 * @since 3.0
	 * @return WpB_Controller object
	 */
	private function init_controller( ) {
		$updated        = ! empty( $_POST['updated'] ) ? wpb_clean( $_POST['updated'] ) : ''; # If either of lsw updated, we see it here, e.g. 'worker'
		$hier_set		= wpb_setting( 'lsw_priority', WPB_DEFAULT_LSW_PRIORITY );
		$hier_forced	= 'SLW';

		$forced_loc		= ! empty( $_POST['sel_location'] ) ? wpb_clean( $_POST['sel_location'] ) : 0;
		$sel_location 	= ! empty( $_POST['location'] ) ? wpb_clean( $_POST['location'] ) : 0;

		if ( $forced_loc ) {
			$sel_location = $forced_loc;
			$hier_forced = str_replace( 'L', '', $hier_set ) == 'SW' ? 'LSW' : 'LWS';
		} else if ( $sel_location && 'location' == $updated ) {
			$hier_forced = str_replace( 'L', '', $hier_set ) == 'SW' ? 'LSW' : 'LWS';
		}

		$sel_service = ! empty( $_POST['service'] ) ? wpb_clean( $_POST['service'] ) : 0;
		if ( $sel_service && 'service' == $updated ) {
			$hier_forced = str_replace( 'S', '', $hier_set ) == 'LW' ? 'SLW' : 'SWL';
		}

		$only_own = ! empty( $_POST['only_own'] ) ? esc_sql( $_POST['only_own'] ) : false;
		if ( $only_own ) {
			$sel_worker = get_current_user_id();
			$hier_forced = str_replace( 'W', '', $hier_set ) == 'LS' ? 'WLS' : 'WSL';
		} else if ( ! empty( $_POST['worker'] ) ) {
			$sel_worker = wpb_clean( $_POST['worker'] );
			if ( 'worker' == $updated )
				$hier_forced = str_replace( 'W', '', $hier_set ) == 'LS' ? 'WLS' : 'WSL';
		} else {
			$sel_worker = 0;
		}

		return new WpB_Controller( new WpB_Norm( $sel_location, $sel_service, $sel_worker ), self::controller_order_by(), $hier_forced, ! empty( $_REQUEST['add_new_event'] ) || ! empty( $_REQUEST['is_event'] ) );
	}

	/**
	 * Create html for start time
	 * @since 3.0
	 * @return string
	 */
	private function start_time_html( &$booking, $lsw_changed = false ) {

		if ( $booking->is_gcal_event() ) {
			return '<input type="text" class="app-admin-time" name="start_time" value="'.$booking->time().'">';
		}

		$slot = new WpB_Slot( $booking );
		$bid  = $booking->get_ID();

		$start_time_ex 	= $booking->time();
		$start_time 	= isset( $_POST['start_time'] ) ? wpb_clean( $_POST['start_time'] ) : $start_time_ex;
        $start_date_ts 	= isset( $_POST['start_date'] ) ? wpb_strtotime( $_POST['start_date'] ) :  wpb_strtotime( $booking->date() );
		$date_changed 	= $start_date_ts != wpb_strtotime( $booking->date() );

		if ( $bid ) {
			$old_slot = clone $slot;
			$old_slot->update_start_end( $start_date_ts + wpb_from_military( wpb_to_military( $start_time ), true ) );
			$virtual = array( $bid => BASE('Multiple')->create_virtual( $bid, $old_slot ) );

			if ( $to_change = apply_filters( 'app_inline_edit_bookings_to_change', $bid, $booking ) ) {
				$slot->exclude( $to_change );
			}

		} else {
			$virtual = false;
		}

		$selected_t 	= 0;
		$found 			= $found_2nd = false;
		$time_options 	= array();
		$step 			= 60 * $this->a->get_min_time();

		$html  = '';
		$html .= '<select class="app-admin-time" name="start_time" '.($booking->is_daily() ? 'disabled="disabled"' : '').'>';

		for ( $t = 0; $t < 3600*24; $t = $t + $step ) {

			$t_disp = wpb_secs2hours( $t );
			$s = $d = '';

			if ( $this->is_strict_check() && 'removed' != $booking->get_status() ) {

				$changed = $lsw_changed || $date_changed || $t_disp != $start_time_ex;

				$slot->update_start_end( $start_date_ts + $t );

				if ( $changed && $slot->why_not_free( '', $virtual ) ) {
					$d = ' disabled="disabled"';
				} else if ( ! $selected_t ) {
					$selected_t = $t;
				}
			}

			if ( $t_disp == $start_time && !$d ) {
				if ( $found ) {
					$found_2nd = true;
				}
				$found		= true;
				$selected_t = $t;
				$s 			= " selected='selected'";
			}

			$time_options[ $t_disp ] = $s.$d;
		}

		# Build options
		foreach ( $time_options as $t_disp => $sd ) {
			if ( $found_2nd && $t_disp != $start_time )
				$sd = '';
			$html .= '<option value="'.$t_disp. '"'.$sd.'>';
			$html .= $t_disp;
			$html .= '</option>';
		}

		$html .= '</select>';

		# Resulting start timestamp
		$start_ts = $start_date_ts + $selected_t;

		$booking->set_start( $start_ts );
		$booking->set_end( $start_ts + 60*wpb_get_duration( $booking ) );

		return $html;
	}

	/**
	 * Create html for end time
	 * @param $selected_t	integer		Selected start time
	 * @since 3.0
	 * @return string
	 */
	private function end_time_html( $booking ) {

		if ( $booking->is_gcal_event() ) {
			return '<input type="text" class="app-admin-time" name="end_time" value="'.$booking->end_time().'">';
		}

		$step			= 60 * $this->a->get_min_time();
		$date_time_ts	= wpb_strtotime( $booking->get_start() );
		$selected_t		= $date_time_ts - DAY_IN_SECONDS*intval( $date_time_ts/DAY_IN_SECONDS );

		$html  = '';
		$html .= '<select class="app-admin-time" name="end_time" '.($booking->is_daily() ? 'disabled="disabled"' : '').'>';

		for ( $t = 0; $t < 3600*24; $t = $t + $step ) {
			$t_disp = wpb_secs2hours( $t );
			if ( $t == ( $selected_t + wpb_get_duration( $booking ) * 60 ) ) {
				$s = " selected='selected'";
				$d = '';
			} else {
				$s = '';
				$d = $this->is_strict_check() ? ' disabled="disabled"' : '';
			}

			$html .= '<option value="'.$t_disp.'"'.$s.$d.'>';
			$html .= $t_disp;
			$html .= '</option>';
		}
		$html .= '</select>';

		return $html;
	}

	/**
	 * Helper for price fields
	 */
	private function price_fields_html( $booking ) {
		$bid 	= $booking->get_ID();
		$js_id	= $bid ? $bid : $booking->get_uniqid();

		$html  = '';
		$html .= '<h4 class="app_iedit_price_h">'.sprintf( __('PRICING - PAYMENT (%s)', 'wp-base'), wpb_format_currency( ) ).'</h4>';

		/* Selected payment method - Don't show for a new app */
		if ( $bid ) {
			$html .= wpb_wrap_field( 'payment_method',
				__('Method', 'wp-base'),
				$booking->payment_method_name(),
				__('This is the payment method selected at checkout. ●It does not indicate that payment has realised','wp-base')
			);
		}

		/* Lock */
		$checked = $bid ? checked( null, wpb_get_app_meta( $bid, 'unlocked' ), false ) : '';
		$html .= wpb_wrap_field( 'lock',
					__('Locked', 'wp-base'),
					'<input type="checkbox" name="locked" '.$checked.' />'.'<input type="hidden" name="locked_check" value="1" />',
					__('If locked, editing any other field will not change price and deposit','wp-base')
				);

		/* Price */
		$price_readonly = !$bid || wpb_get_app_meta( $bid, 'unlocked' ) ? '' : ' readonly="readonly"';
		$html .= wpb_wrap_field( 'price', __('Price', 'wp-base'),
					'<input type="text" name="price" class="ptitle" '.
					'value="'.wpb_format_currency( $booking->get_price(), false, true ).'" '.$price_readonly.'/>',
					apply_filters( 'app_admin_booking_price_tooltip', '', $booking ) 
				);

		/* Refundable deposit */
		$html .= wpb_wrap_field( 'deposit', __('Deposit', 'wp-base'),
					'<input type="text" name="deposit" class="ptitle" '.
					'value="'.wpb_format_currency( $booking->get_deposit(), false, true).'" '.$price_readonly.'/>'
				);

		$html = apply_filters( 'app_inline_edit_after_deposit', $html, $booking, $this );

		$html .= '<div class="app-hr" ></div>';

		/* Total due */
		$due = apply_filters( 'app_inline_edit_total_due', $booking->get_price() + $booking->get_deposit(), $booking );

		$html .= wpb_wrap_field( 'due', '<strong>'.__('Total due', 'wp-base').'</strong>',
					'<input type="text" name="total_due" class="ptitle" '.
					'value="'.wpb_format_currency( $due, false, true ).'" readonly="readonly" />'
				);

		$html .= '<div class="app-hr" ></div>';

		/* Payment */
		$paid = $booking->get_total_paid( );
		$payment_text = $paid && wpb_admin_access_check( 'manage_transactions', false )
						? '<abbr id="app-payment-ttip-'.$js_id.'" class="app-payment-ttip">'. __('Payment', 'wp-base' ) .'</abbr>'
						: __('Payment', 'wp-base');

		$html .= wpb_wrap_field( 'payment', $payment_text,
					'<input type="text" name="payment" value="'.wpb_format_currency( $paid, false, true ).'" readonly="readonly" />'
				);

		$html .= '<div class="app-hr app-thick" ></div>';

		/* Balance */
		$balance = apply_filters( 'app_inline_edit_balance', $paid - $booking->get_price() - $booking->get_deposit(), $booking );

		$html .= wpb_wrap_field( 'balance', '<strong>'.__('Balance', 'wp-base').'</strong>',
					'<input type="text" name="balance" '.
					'value="'.wpb_format_currency( $balance, false, true ).'" readonly="readonly" />'
					);

		$html = apply_filters( 'app_inline_edit_price_fields_inside', $html, $booking, $this );

		return $html;
	}

	/**
	 * Helper for appointment fields
	 */
	private function appt_fields_html( $booking ) {
		$bid 		= $booking->get_ID();
		$js_id		= $bid ? $bid : $booking->get_uniqid();
		$safe_format = $this->a->safe_date_format();

		$name = $bid ? sprintf( __( 'BOOKING (%d)', 'wp-base' ), $bid ) : __( 'NEW BOOKING', 'wp-base' );
		$name = apply_filters( 'app_appointment_name_in_bookings', $name, $booking );

		$html  = '<h4 class="app_iedit_app_h '. ($bid ? '' : 'app_blink').'">'. $name .'</h4>' ;
		$html  = apply_filters( 'app_inline_edit_appointment_fields_before', $html, $booking );
		
		/* Parent */
		if ( BASE('Multiple')->is_active() ) {
			$par_text = esc_html( __( 'Parent', 'wp-base' ) );
			if ( $booking->get_parent_id() ) {
				$parent 		= wpb_get_app( $booking->get_parent_id() );
				$parent_date 	= $parent ? sprintf( __( 'starting at <b>%s</b>', 'wp-base' ), mysql2date( $this->a->dt_format, $parent->start ) ) : '';
				$parent_id 		= $parent ? $parent->ID : '';
				$par_text 		= $parent ? '<abbr title="'. esc_attr( $parent_date ) .'">'. $par_text . '</abbr>' : $par_text;
			} else {
				$parent_id = $parent_date = '';
			}

			$html .= wpb_wrap_field( 'parent', $par_text,
						'<input type="text" name="parent_id" value="'.$parent_id.'"/>'
					);
		}
		
		/* Created - Don't show for a new app */
		if ( $bid ) {
			$html .= wpb_wrap_field( 'created_by', __('Created by', 'wp-base'), $this->created_by_select( $booking ) );
			$html .= wpb_wrap_field( 'created', __('Created at', 'wp-base'), date_i18n( $this->a->dt_format, strtotime( $booking->get_created() ) ) );
		}

		/* Client IP and las update - Don't show for a new app */
		if ( $bid ) {
			$html .= wpb_wrap_field( 'last_update', __('Last update', 'wp-base'), $this->last_updated( $booking ) );
			$html .= wpb_wrap_field( 'client_ip', __('Client IP', 'wp-base'), ( $booking->get_client_ip() ?: '&nbsp;' ) );
		}

		$html  = apply_filters( 'app_inline_edit_appointment_fields', $html, $booking, $this );

		$bdays	= $this->blocked_days( $booking );
		$b		= $bdays ? htmlspecialchars( wp_json_encode( $bdays ) ) : '';

		/* Start */
		$html .= '<label class="app_iedit_start app_iedit_time">';
		$html .= '<span class="title">';
		$html .= __('Start', 'wp-base');
		$html .= '</span>';

		if ( $booking->is_event() ) {
			$html .= apply_filters( 'app_inline_edit_start', '<input name="event_start" class="event-dt" value="'.$booking->client_dt().'" readonly>', $booking );
		} else {
			$html .= '<input type="text" id="start_date_'.$js_id.'" name="start_date" class="datepicker" size="12" '.
						'value="'.date( $safe_format, strtotime( $booking->get_start() ) ).'" />';
			$html .= '<input type="hidden" class="blocked-days" data-blocked="'.$b.'" />';
			$html .= $this->start_time_html( $booking );
		}
		$html .= '</label>';

		/* End */
		if ( !$booking->is_event() ) {
			$disabled	= $booking->is_daily() ? ' disabled="disabled"' : '';
			$readonly	= $this->is_strict_check() ? ' readonly="readonly"' : '';

			$html .= '<label class="app_iedit_end app_iedit_time">';
			$html .= '<span class="title">';
			$html .= __('End', 'wp-base');
			$html .= '</span>';
			$html .= '<input type="text" id="end_date_'.$js_id.'" name="end_date" class="datepicker" size="12" '.
					'value="'.date( $safe_format, strtotime( $booking->get_end() ) ).'" '.$disabled.$readonly.'/>';
			$html .= $this->end_time_html( $booking );
			$html .= '</label>';
		}

		/* Client Note */
		$html .= '<label class="app_iedit_note app_iedit_textarea">';
		$html .= '<span class="title">'.wpb_get_field_name('note').'<em class="app-helptip" title="'.esc_attr( __('This is the note submitted by the client','wp-base') ).'"></em></span>';
		$html .= '<textarea name="note">';
		$html .= esc_textarea( wpb_get_app_meta( $bid, 'note' ) );
		$html .= '</textarea>';
		$html .= '</label>';

		/* Admin Note */
		$html .= '<label class="app_iedit_admin_note app_iedit_textarea">';
		$html .= '<span class="title">'.__('Admin Note', 'wp-base').'<em class="app-helptip" title="'.esc_attr( __('This is only visible to admins, e.g. in order to write internal notes','wp-base') ).'"></em></span>';
		$html .= '<textarea name="admin_note">';
		$html .= esc_textarea( wpb_get_app_meta( $bid, 'admin_note' ) );
		$html .= '</textarea>';
		$html .= '</label>';

		/* Status */
		$html .= wpb_wrap_field( 'status', __('Status', 'wp-base'), $this->status_selection( $booking ) );

		$html  = apply_filters( 'app_inline_edit_appointment_fields_after', $html, $booking, $this );

		return $html;
	}

	/**
	 * Helper to create status selection
	 */
	private function status_selection( $booking ) {
		$html = '<select name="status">';

		foreach ( (array)$this->a->get_statuses() as $status => $status_name ) {

			if ( 'running' == $status ) {
				continue;
			}

			$html .= '<option value="'.$status.'" '.selected( $booking->get_status(), $status, false ).'>'. esc_html( $status_name ) . '</option>';
		}

		$html .= '</select>';

		return $html;
	}

	/**
	 * Helper for buttons, email boxes
	 */
	private function actions_html( $booking ) {
		$bid 	= $booking->get_ID();
		$status = $booking->get_status();

		$html  = '';
		$html .= '<h4 class="app_iedit_actions">'.__('ACTIONS', 'wp-base').'</h4>';
		$html .= '<input type="hidden" name="app_id" value="'.$bid.'" />';
		$html .= '<input type="hidden" class="is_event" value="'.(int)$booking->is_event().'" />';
		$html .= '<input type="hidden" name="parent_id" value="'.$booking->get_parent_id().'" />';
		
		$html = apply_filters( 'app_inline_edit_before_buttons', $html, $booking, $this );
		
		$html .= '<a href="javascript:void(0)" title="'.__('Cancel', 'wp-base').'" class="button-secondary cancel alignleft">'.__('Cancel','wp-base').'</a>';
		$html .= '<img class="waiting alignleft" style="display:none;" src="'.admin_url('images/wpspin_light.gif').'" alt="">';
		$html .= '<span class="error alignleft" style="display:none"></span>';

		if ( ! empty( $_POST['taken_over'] ) ) {
			$html .= '<input type="hidden" name="taken_over_at" value="'.$this->a->_time.'" />';
		}

		if ( 'reserved' == $status ) {
			$button_cl = 'app-disabled-button';
			$title = __('Reserved bookings cannot be edited here. Edit them in your GCal.', 'wp-base');
		} else {
			$button_cl = 'save';
			$title = __('Click to save or update', 'wp-base');
		}
		$save_text = $bid ? __('Update','wp-base') : __('Save','wp-base');
		$html .= '<button title="'.$title.'" class="button-primary '.$button_cl.' alignright">'.$save_text.'</button>';

		/* emails */
		$html .= '<div class="alignright app-iedit-email-actions">';
		$html .= '<label class="app_iedit_send_mail">';
		$html .= '<span class="title">'.($bid ? __('(Re)Send email:','wp-base') : __('Send email:','wp-base')).'</span>'.
				'&nbsp;&nbsp;&nbsp;&nbsp;';
		$html .= '</label>';

		/* Confirmation email */
		$_true = apply_filters( 'app_inline_edit_email_checkbox_default', true, $booking );
		$d = $bid && in_array( $status, array( 'pending', 'removed', 'completed' ) ) ? ' disabled="disabled"' : '';
		$html .= wpb_wrap_field( 'send_confirm', '&nbsp;',
				'<input type="checkbox" name="resend" '.checked( (bool)($bid && !$d && $_true), true, false ).' value="1" '.$d.' />&nbsp;' .__('Confirmed','wp-base') . '&nbsp;&nbsp;'
				);

		/* Pending email */
		$d = ! $bid || in_array( $status, array( 'confirmed', 'paid', 'removed', 'completed' ) ) ? ' disabled="disabled"' : '';
		$html .= wpb_wrap_field( 'send_pending', '&nbsp;',
					'<input type="checkbox" name="send_pending" value="1" '.$d.' />&nbsp;' .__('Pending','wp-base') . '&nbsp;&nbsp;'
				);

		/* Completed email */
		$d = ! $bid || in_array( $status, array( 'confirmed', 'paid', 'pending', 'removed' ) ) ? ' disabled="disabled"' : '';
		$html .= wpb_wrap_field( 'send_completed', '&nbsp;',
				'<input type="checkbox" name="send_completed" value="1" '.$d.' />&nbsp;' .__('Completed','wp-base') . '&nbsp;&nbsp;'
				);

		/* Cancellation email */
		$d = ! $bid || in_array( $status, array( 'confirmed', 'paid', 'pending', 'completed' ) ) ? ' disabled="disabled"' : '';
		$html .= wpb_wrap_field( 'send_cancel', '&nbsp;',
				'<input type="checkbox" name="send_cancel" value="1" '.$d.' />&nbsp;' . __('Cancelled','wp-base')
				);

		$html .= '</div>';

		$html = apply_filters( 'app_inline_edit_email_actions_after', $html, $booking, $this );

		return $html;
	}

	/**
	 * Bring user data when user is selected in New Bookings
	 * @since 2.0
	 */
	public function populate_user() {
		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array( 'error' => $this->a->get_text('unauthorised') ) ) );
		}

		$user_id = ! empty( $_POST['user_id'] ) ? wpb_clean( $_POST['user_id'] ) : 0;

		$r = $user_id ? BASE('User')->get_app_userdata( 0, $user_id ) : array();
		$r = apply_filters( 'app_inline_edit_populate_user', $r, $user_id );

		if ( isset( $r['error'] ) || !is_array( $r ) ) {
			wp_send_json( array( 'error' => __( 'Unknown or empty user ID', 'wp-base' ) ) );
		} else {
			wp_send_json( $r );
		}
	}

	/**
	 * Dynamically show payment in qtip content
	 * @since 2.0
	 */
	public function show_payment_in_tooltip() {
		if ( empty( $_POST['app_id'] ) ) {
			wp_send_json( array( 'result' => __('Unexpected error','wp-base') ) );
		}

		if ( ! wpb_admin_access_check( 'manage_transactions', false ) ) {
			wp_send_json( array( 'result' => $this->a->get_text( 'unauthorised' ) ) );
		}

		$transactions = BASE('Transactions')->get_admin_transactions( 0, 100, wpb_clean( $_POST['app_id'] ) );

		ob_start();
		BASE('Transactions')->display_table( $transactions, true ); // Get short table
		$result = preg_replace( '%<tfoot>(.*?)</tfoot>%is', '', ob_get_contents() ); // Remove footer
		ob_end_clean();

		wp_send_json( array( 'result' => $result ) );
	}

	/**
	 * Find who made the booking in the first place
	 * @since 3.0
	 * @return string
	 */
	private function created_by( $booking ) {
		$booking	= $booking instanceof WpB_Booking ? $booking : new WpB_Booking( $booking );
		$meta		= $booking->created_by();

		if ( $meta && $who = get_user_by( 'id', $meta ) ) {
			$name = wpb_is_worker( $meta ) ? $this->a->get_worker_name( $meta ) : BASE('User')->get_name( $meta );

			$tt = $name;

			if ( $email = get_user_meta( $meta, 'app_email', true ) ) {
				$tt = $tt . WPB_TT_SEPARATOR . $email;
			}

			if ( $phone = get_user_meta( $meta, 'app_phone', true ) ) {
				$tt = $tt . WPB_TT_SEPARATOR . $phone;
			}

			if ( current_user_can( 'edit_users' ) ) {
				$created_by = '<a href="'. admin_url("user-edit.php?user_id="). $meta . '" target="_blank" title="'.$tt.'">'. $name . '</a>';
			} else {
				$created_by = $name;
			}
		} else {
			$created_by = 'reserved' == $booking->get_status() ? __( 'GCal', 'wp-base' ) : __( 'Client', 'wp-base' );
		}

		return $created_by;
	}

	/**
	 * Select created by
	 * @since 3.7.3.1
	 * @return string
	 */
	private function created_by_select( $booking ) {
		$booking	= $booking instanceof WpB_Booking ? $booking : new WpB_Booking( $booking );
		$created_by	= $booking->get_created_by();

		$pholder = 'reserved' == $booking->get_status() ? __( 'GCal', 'wp-base' ) : __( 'Client', 'wp-base' );

		$html  = '<select name="created_by" class="created_by">';
		$html .= '<option value="">'. $pholder .'</option>';

		foreach ( (array)get_users( array( 'role' => 'wpb_admin' ) ) as $user ) {

			if ( $created_by && $user->ID == $created_by && get_user_by( 'id', $created_by ) ) {
				$name = wpb_is_worker( $created_by ) ? $this->a->get_worker_name( $created_by ) : BASE('User')->get_name( $created_by );
				$selected = ' selected="selected"';
			} else {
				$name = '';
				$selected = '';
			}

			$html .= '<option value="'.$user->ID.'"'.$selected.'>'. ($name ?: $user->display_name) . '</option>';
		}

		$html .= '</select>';

		return $html;
	}

	/**
	 * Find who last updated the booking
	 * @since 3.5.6
	 * @return string
	 */
	private function last_updated( $booking ) {
		$meta = $booking->updated_by();

		if ( $meta && $who = get_user_by( 'id', $meta ) ) {
			$name = wpb_is_worker( $meta ) ? $this->a->get_worker_name( $meta ) : $who->display_name;

			$tt = $name;

			if ( $email = get_user_meta( $meta, 'app_email', true ) ) {
				$tt = $tt . WPB_TT_SEPARATOR . $email;
			}

			if ( $phone = get_user_meta( $meta, 'app_phone', true ) ) {
				$tt = $tt . WPB_TT_SEPARATOR . $phone;
			}

			if ( current_user_can( 'edit_users' ) ) {
				$updated_by = '<a href="'. admin_url("user-edit.php?user_id="). $meta . '" target="_blank" title="'.$tt.'">'. $name . '</a>';
			} else {
				$updated_by = $name;
			}
		} else {
			$updated_by = 'reserved' == $booking->get_status() ? __( 'GCal', 'wp-base' ) : __( 'unknown', 'wp-base' );
		}

		$gcal_update	= wpb_strtotime( $booking->get_gcal_updated() );
		$reg_update		= $booking->updated_at( true );
		$updated_at		= $reg_update >= $gcal_update ? $booking->updated_at() : date_i18n( $this->a->dt_format, $gcal_update );

		return ($updated_at ? sprintf( __( '%s by %s', 'wp-base' ), $updated_at, $updated_by ) : '&nbsp;');
	}

	/**
	 * Helper to find blocked days
	 * @since 3.0
	 * @return array
	 */
	public function blocked_days( $booking ) {
		if ( ! $this->is_strict_check() || $booking->is_event() || 'removed' == $booking->get_status() ) {
			return array();
		}

		$start_date_ts	= strtotime( $booking->date() );
		$first_day_ts	= wpb_first_of_month( $start_date_ts );
		$last_day_ts	= wpb_last_of_month( $start_date_ts, 1 );
		$allowed_days 	= array();
		$calendar		= new WpB_Calendar( $booking );

		$calendar->setup( array( 'hard_limit' => 2.0, 'display' => 'minimum', 'admin' => false ) );

		for ( $d = $first_day_ts; $d < $last_day_ts; $d = $d + DAY_IN_SECONDS ) {

			if ( $calendar->is_hard_limit_exceeded() ) {
				break;
			}

			if ( $calendar->slot( $d, $d + DAY_IN_SECONDS )->why_not_free( 'quick' ) ) {
				continue;
			}

			$allowed_days[] = date( 'Y-m-d', $d );
		}

		$all_days 	= array();

		for ( $d = $first_day_ts; $d < $last_day_ts; $d = $d + DAY_IN_SECONDS ) {
			$all_days[] = date( 'Y-m-d', $d );
		}

		return array_values( array_diff( $all_days, $allowed_days ) );
	}

	/**
	 * Modify menus as location, service, provider, start date changes
	 * @since 2.0
	 */
	public function update_inline_edit() {

		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array( 'error' => $this->a->get_text('unauthorised') ) ) );
		}

		add_filter( 'app_seats_no_pax_filter', '__return_true' );

		$controller 	= $this->init_controller();
		$app_id 		= isset( $_REQUEST['app_id'] ) ? wpb_clean( $_REQUEST['app_id'] ) : 0; # Can be 0 for a new booking
        $booking 		= new WpB_Booking( $app_id );
		$old_booking	= clone $booking;
		$seats			= apply_filters( 'app_bookings_seats', (! empty($_POST['app_seats']) ? wpb_clean( $_POST['app_seats'] ) : 1), $booking, $controller );
		$lsw_changed 	= !( $booking->get_location() == $controller->get_location() &&
							 $booking->get_service() == $controller->get_service() &&
							 $booking->get_worker() == $controller->get_worker() &&
							 $seats != $booking->get_seats()
							);

		$booking->set_location( $controller->get_location() );
		$booking->set_service( $controller->get_service() );
		$booking->set_worker( $controller->get_worker() );
		$booking->set_seats( $seats );

		$out = array(
			'locations_sel'		=> $controller->select_location( ),
			'services_sel'		=> $controller->select_service( ! $app_id ),
			'workers_sel'		=> $controller->select_worker( ),
		);

		if ( $this->is_strict_check() ) {
			$out = array_merge( $out, array(
					'start_time_sel'	=> $this->start_time_html( $booking, $lsw_changed ),
					'end_date'			=> date_i18n( $this->a->safe_date_format(), wpb_strtotime( $booking->get_end() ) ),
					'end_time_sel'		=> $this->end_time_html( $booking ),
					'blocked_days'		=> $booking->date() != $old_booking->date()
										   ? $this->blocked_days( $booking )
										   : false,
			));
		}

		# Update price and deposit only if unlocked
		if ( isset( $_POST['locked_check'] ) && empty( $_POST['locked'] ) ) {
			$slot	= new WpB_Slot( $booking );
			$pax	= ! empty( $_POST['app_seats'] ) ? max( 1, wpb_clean( $_POST['app_seats'] ) ) : $slot->get_pax();
			$out	= array_merge( $out, array( 'price' => $pax * $slot->get_price(), 'deposit' => $pax * $slot->get_deposit() ) );
		}

		$out = apply_filters( 'app_inline_edit_update', $out, $booking, $old_booking, $controller, $this );

		die( json_encode( $out ) );
	}

	/**
	 * Check if strict check is asked
	 * return bool	 "true" means capabilities limited
	 * @since 2.0
	 */
	public function is_strict_check(){
		$override = isset( $_POST['override'] ) ? wpb_clean( $_POST['override'] ) : 'auto';

		if ( ! $override ) {
			return true;
		} else if ( 1 == $override ) {
			return false;
		} else if ( 'auto' == $override && 'yes' == wpb_setting('strict_check') ) {
			return true;
		}

		return false;
	}

	/**
	 * Save edited or new appointment on admin side
	 */
	public function inline_edit_save() {

		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array('error' => $this->a->get_text('unauthorised') ) ) );
		}

		if ( wpb_is_demo() ) {
			die( json_encode( array( 'error' => __('Changes cannot be saved in DEMO mode!', 'wp-base' ) ) ) );
		}

		if ( ! $this->a->service_exists( wpb_clean( $_POST['service'] ) ) ) {
			die( json_encode( array( 'error' => __('Service does not exist!', 'wp-base' ) ) ) );
		}

		$booking 		= new WpB_Booking( wpb_clean( $_POST['app_id'] ) );
		$old_booking	= clone $booking;
		
		if ( ! current_user_can( WPB_ADMIN_CAP ) && ! empty( $_POST['worker'] ) && $_POST['worker'] != get_current_user_id() ) {
			die( json_encode( array('error' => __('You are not allowed to edit other provider\'s booking', 'wp-base' ) ) ) );
		}		

		if ( ! $booking->get_ID() ) {
			$booking->set_created( $this->a->_time );
		} else if ( $locked_by_id = $this->check_lock( $booking->get_ID() ) ) {
			$locked_by	= get_userdata( $locked_by_id );
			$name		= isset( $locked_by->display_name ) ? $locked_by->display_name : $locked_by_id;
			die( json_encode( array( 'error' => sprintf( __('Record has been taken over by %s. It could not be saved.', 'wp-base' ), $name ) ) ) );
		}

		$booking->set_user( wpb_clean( ! empty( $_POST['user'] ) ? $_POST['user'] : 0 ) );
		$booking->set_location( wpb_clean( $_POST['location'] ) );
		$booking->set_service( wpb_clean( $_POST['service'] ) );
		$booking->set_worker( wpb_clean( $_POST['worker'] ) );
		$booking->set_price( wpb_sanitize_price( $_POST['price'] ) );
		$booking->set_deposit( wpb_sanitize_price( $_POST['deposit'] ) );
		$booking->set_status( wpb_clean( $_POST['status'] ) );
		$booking->set_parent_id( (int)$_POST['parent_id'] );

		if ( $booking->is_event() ) {
			if ( strpos( $_POST['event_start'], '|' ) !== false ) {
				list( $start, $end ) = explode( '|', $_POST['event_start'] );
				$booking->set_start( $start );
				$booking->set_end( $end );
			}
		} else {
			if ( $booking->is_daily() ) {
				$booking->set_start( date( 'Y-m-d', strtotime( $_POST['start_date'] ) ) );
				$booking->set_end( strtotime( $_POST['start_date'] ) + wpb_get_duration( $booking ) *60 );
			} else {
				$start_time = isset( $_POST['start_time'] ) ? wpb_to_military( $_POST['start_time'] ) : '00:00';
				if ( isset( $_POST['end_time'] ) ) {
					$end_ts = strtotime( $_POST['end_date']. " " . wpb_to_military( $_POST['end_time'] ) );
				} else {
					list( $hours, $mins ) = explode( ':', $start_time );
					$end_secs = $hours *60*60 + $mins *60 + wpb_get_duration( $booking ) *60;
					$end_ts = strtotime( $_POST['end_date'] ) + $end_secs;
				}

				$booking->set_start( strtotime( $_POST['start_date']. " " . $start_time ) );
				$booking->set_end( $end_ts );
			}
		}

		$booking = apply_filters( 'app_inline_edit_save_data', $booking, $old_booking );

		if ( strtotime( $booking->get_start() ) > strtotime( $booking->get_end() ) ) {
			die( json_encode( array( 'error' => __('Booking start time cannot be later than end time!', 'wp-base' ) ) ) );
		}

		/* Strict check - Disabled if in removed, test or completed status */
		if ( $this->is_strict_check() && !in_array( $booking->get_status(), array( 'removed', 'completed', 'test' ) ) ) {
			$booking->check_update( $old_booking );
		}

		do_action( 'app_inline_edit_save_before_save', $booking, $old_booking, $this );

		$update_result = $insert_result = null;
		$changed	= false;

		if ( $booking->get_ID() ) {
			$update_result = $this->update( $booking, $old_booking );
		} else {
			$insert_result = $this->insert( $booking, $old_booking );
		}

		if ( $update_result || $insert_result ) {
			wpb_flush_cache();
			$this->a->update_appointments();
		}

		if ( isset( $_POST['created_by'] ) ) {
			if ( $booking->set_created_by( wpb_clean( $_POST['created_by'] ) ) ) {
				$changed = true;
			}
		}

		# Default is locked. We are saving unlock selection.
		if ( isset( $_POST['locked_check'] ) && empty( $_POST['locked'] ) ) {
			if ( $booking->update_meta( 'unlocked', true ) ) {
				$changed = true;
			}
		} else if ( $booking->delete_meta( 'unlocked' ) ) {
			$changed = true;
		}

		if ( isset( $_POST['admin_note'] ) ) {
			if ( $booking->update_meta( 'admin_note', $_POST['admin_note'] ) ) {
				$changed = true;
			}
		}

		if ( $userdata = $this->save_userdata( $booking ) ) {
			$changed = true;
		}

		$changed = apply_filters( 'app_inline_edit_save', $changed, $booking, $old_booking, $this );

		if ( $update_result ) {
			do_action( 'app_inline_edit_updated', $booking, $old_booking, $this );
		} else if ( $insert_result ) {
			do_action( 'app_inline_edit_new_booking', $booking, $old_booking, $this );
		}

		$email_sent = $this->send_email( $booking );

		$new_results = apply_filters( 'app_inline_edit_save_result', array_merge( $userdata,
			array(
				'result_app_id'	=>	$booking->get_ID(),
				'user'			=>	$this->client_text( $booking ),
				'date_time'		=>	mysql2date( $this->a->dt_format, $booking->get_start() ),
				'end_date_time'	=>	mysql2date( $this->a->dt_format, $booking->get_end() ),
				'location'		=>	$this->a->get_location_name( $booking->get_location() ),
				'service'		=>	apply_filters( 'app_bookings_service_name_result', wpb_cut( $this->a->get_service_name( $booking->get_service() ) ), $booking ),
				'worker'		=>	$this->a->get_worker_name( $booking->get_worker() ),
				'status'		=>	$this->stat_text( $booking ),
				'price'			=>	wpb_format_currency( $booking->get_price(), false, true ),
				'deposit'		=>	wpb_format_currency( $booking->get_deposit(), false, true ),
				'total_paid'	=>	wpb_format_currency( $booking->get_total_paid(), false, true ),
				'balance'		=>	wpb_format_currency( $booking->get_balance(), false, true ),
				'created_by'	=>	$this->created_by( $booking ),
				'collapse'		=>	'yes' == wpb_setting( 'admin_edit_collapse' ) ? 1 : 0,
				'new_booking'	=>	$insert_result,
				'client_name'	=>	BASE('User')->get_client_name( $booking->get_ID() ). apply_filters( 'app_bookings_add_text_after_client', '', $booking->get_ID(), null ),
		) ), $booking, $old_booking );

		if ( $update_result === false || $insert_result === false ) {
			$result = array( 'result' => __('Record could not be saved!', 'wp-base' ) );
		} else if ( $insert_result ) {
			$result = array_merge( array( 'result' => __('New appointment successfully saved.', 'wp-base' ) ), $new_results );
		} else if ( $update_result || $changed ) {
			$result = array_merge( array( 'result' => __('Changes saved.', 'wp-base' ) ), $new_results );
		} else {
			$result = array( 'no_change' => __('You did not make any changes...', 'wp-base' ) );
		}

		if ( $_POST["resend"] || $_POST["send_pending"] || $_POST["send_completed"] || $_POST["send_cancel"] ) {
			if ( $email_sent ) {
				$result = array_merge( array( 'emailMsg' => __('Email has been sent.', 'wp-base' ) ), $result );
			} else {
				$result = array_merge( array( 'emailMsg' => __('Email could NOT be sent!', 'wp-base' ) ), $result );
			}
		}

		if ( $update_result || $insert_result || $changed ) {
			$booking->update_meta( '_edit_last', $this->a->_time .':'. get_current_user_id() );
			$booking->delete_meta( '_edit_lock' );
		}

		$result = apply_filters( 'app_inline_edit_save_result_final', $result, $booking, $old_booking );

		die( json_encode( $result ) );
	}

	/**
	 * Do update
	 * @param $booking		object	WPB_Booking object having latest data
	 * @param $old_booking	object	WpB_Booking object having data prior to update
	 * @return	bool
	 */
	private function update( $booking, $old_booking ) {
		global $current_user;

		$booking = apply_filters( 'app_inline_edit_update_pre', $booking, $old_booking );

		$update_result = $this->a->db->update(
			$this->a->app_table,
			$booking->get_save_data(),
			array( 'ID' => $booking->get_ID() )
		);

		if ( $update_result && $booking->get_status() != $old_booking->get_status() ) {
			$this->a->log( sprintf(
				__('Status changed from %1$s to %2$s by %3$s for booking ID:%4$d','wp-base'),
				$old_booking->get_status(),
				$booking->get_status(),
				$current_user->user_login,
				$booking->get_ID()
			));
		}

		if ( $update_result && $booking->get_start() != $old_booking->get_start() ) {
			$this->a->log( sprintf(
				__('Start time of the booking changed from %1$s to %2$s by %3$s for booking ID:%4$d','wp-base'),
				date_i18n( $this->a->dt_format, wpb_strtotime( $old_booking->get_start() ) ),
				date_i18n( $this->a->dt_format, wpb_strtotime( $booking->get_start() ) ),
				$current_user->user_login,
				$booking->get_ID()
			));
		}
		
		return $update_result;
	}

	/**
	 * Do insert
	 * @param $booking		object	WPB_Booking object having latest data
	 * @param $old_booking	object	WpB_Booking object having data prior to update
	 * @return	bool
	 */
	private function insert( $booking, $old_booking ) {
		$booking = apply_filters( 'app_inline_edit_insert_pre', $booking, $old_booking );

		if ( $insert_result = $this->a->db->insert( $this->a->app_table, $booking->get_save_data() ) ) {

			$booking->set_ID( $this->a->db->insert_id );

			# Who made the booking? Record only if BOB
			if ( get_current_user_id() != $booking->get_user() ) {
				$booking->add_meta( 'booking_on_behalf', get_current_user_id() );
			}
		}

		return $insert_result;
	}

	/**
	 * Save admin edited client values
	 * @return array
	 */
	public function save_userdata( $booking ) {
		$user_data = json_decode( wp_unslash( $_POST['app_user_data'] ), true );

		if ( ! is_array( $user_data ) ) {
			return array();
		}

		$result = array();
		$user_data['name'] = wpb_clean( $_POST['name'] );	// Exceptional, because field name is "cname"

		foreach( $this->a->get_user_fields() as $f ) {
			if ( $booking->update_meta( $f, $user_data[ $f ] ) ) {
				$result[ $f ] = $user_data[ $f ];
			}
		}

		if ( $booking->update_meta( 'note', $_POST['note'] ) ) {
			$result['note'] = wpb_clean( $_POST['note'] );
		}

		return $result;
	}

	/**
	 * Send email
	 * @param $booking		object	WPB_Booking object having latest data
	 * @return	bool
	 */
	private function send_email( $booking ) {
		$sent = false;

		if ( $_POST['resend'] && in_array( $booking->get_status(), array( 'confirmed', 'paid' ) ) ) {
			$sent = $this->a->send_email( $booking->get_ID(), 'confirmation', true );
		}

		if ( $_POST['send_pending'] && in_array( $booking->get_status(), array( 'pending' ) ) ) {
			$sent = $this->a->send_email( $booking->get_ID(), 'pending', true );
		}

		if ( $_POST['send_completed'] && in_array( $booking->get_status(), array( 'completed' ) ) ) {
			$sent = $this->a->send_email( $booking->get_ID(), 'completed', true );
		}

		if ( $_POST['send_cancel'] && in_array( $booking->get_status(), array( 'removed' ) ) ) {
			$sent = $this->a->send_email( $booking->get_ID(), 'cancellation', true );
		}

		return $sent;
	}

	/**
	 * Helper for human readable client name
	 * @param $booking		object	WPB_Booking object having latest data
	 * @return	string
	 */
	private function client_text( $booking ) {
		$text = '';

		if ( ! $booking->get_parent_id() ) {
			if ( $booking->get_ID() ) {
				$text = BASE('User')->get_client_name( $booking->get_ID(), null, true );
			} else if ( ! empty( $_POST['name'] ) ) {
				$text = wpb_clean( $_POST['name'] );
			}
		}

		return $text;
	}

	/**
	 * Helper for human readable status
	 * @param $booking		object	WPB_Booking object having latest data
	 * @return	string
	 */
	private function stat_text( $booking ) {
		if ( $_stat = $booking->get_status() ) {
			$stats = $this->a->get_statuses();
			if ( isset( $stats[ $_stat ] ) ) {
				$stat_text = $stats[ $_stat ];
			} else {
				$stat_text = $this->a->get_text('not_defined');
			}
		} else {
			$stat_text = __('None yet','wp-base');
		}

		return $stat_text;
	}

	/**
	 * Set lock for the booking record
	 * @param $bid		integer		Booking ID
	 * @since 3.5.6
	 * @return	string
	 */
	private function set_lock( $bid ) {
		if ( ! $bid ) {
			return false;
		}

		$user_id = get_current_user_id();
		if ( 0 == $user_id ) {
			return false;
		}

		$now  = $this->a->_time;
		$lock = "$now:$user_id";

		wpb_update_app_meta( $bid, '_edit_lock', $lock );

		return array( $now, $user_id );
	}

	/**
	 * Check if booking record is locked
	 * @param $bid		integer		Booking ID
	 * @since 3.5.6
	 * @return int|false|null   ID of the user with lock. False if the post does not exist, post is not locked,
	 *                   		the user with lock does not exist. null if the post is locked by current user.
 	 */
	private function check_lock( $bid ) {
		if ( ! $bid ) {
			return false;
		}

		$lock = wpb_get_app_meta( $bid, '_edit_lock' );

		if ( ! $lock ) {
			return false;
		}

		$lock = explode( ':', $lock );
		$time = $lock[0];

		if ( isset( $lock[1] ) ) {
			$user = $lock[1];
		} else if ( $maybe_elast =  wpb_get_app_meta( $bid, '_edit_last' ) ) {
			$maybe_elast = explode( ':', $maybe_elast );
			$user = isset( $maybe_elast[1] ) ? $maybe_elast[1] : 0;
		} else {
			$user = 0;
		}

		if ( ! get_userdata( $user ) ) {
			return false;
		}

		$time_window = apply_filters( 'wp_check_post_lock_window', 150 );
		$time_window = apply_filters( 'app_check_booking_lock_window', $time_window );

		if ( $time && $time > $this->a->_time - $time_window && $user != get_current_user_id() ) {
			return $user;
		}

		if ( $user == get_current_user_id() ) {
			return null;
		}

		return false;
	}

	/**
	 * Check lock status on the booking record and refresh the lock
	 *
	 * @since 3.5.6
	 *
	 * @param array  $response  The Heartbeat response.
	 * @param array  $data      The $_POST data sent.
	 * @param string $screen_id The screen id.
	 * @return array The Heartbeat response.
	 */
	public function refresh_lock( $response, $data, $screen_id ) {
		if ( ! array_key_exists( 'app-check-locked', $data ) ) {
			return $response;
		}

		$received	= $data['app-check-locked'];
		$checked	= array();

		if ( empty ( $received ) || !is_array( $received )  || ! wpb_admin_access_check( 'manage_bookings', false ) ) {
			return $response;
		}

		foreach ( $received as $key ) {

			$bid = absint( substr( $key, 7 ) );

			$user_id = $this->check_lock( $bid );
			$user    = get_userdata( $user_id );

			if ( null === $user_id ) {
				$editing = ! empty( $data['app-refresh-locked'] ) ? $data['app-refresh-locked'] : array();
				if ( in_array( 'app-tr-'. $bid, $editing ) ) {
					if ( $new_lock = $this->set_lock( $bid ) ) {
						$checked[ $bid ]['new_lock'] = implode( ':', $new_lock );
					}
				}
			} else if ( $user ) {
				$send = array(
					/* translators: %s: User's display name. */
					'text' => sprintf( __( '%s is editing', 'wp-base' ), $user->display_name ),
				);

				$avatar = get_avatar( $user->ID, 18 );
				if ( $avatar ) {
					if ( preg_match( "|src='([^']+)'|", $avatar, $matches ) ) {
						$send['avatar_src'] = $matches[1];
					}
				}

				$checked[ $bid ] = $send;
			}

			$response['app-check-locked'] = $checked;
		}

		return $response;
	}

	/**
	 * Delete lock when a record is closed
	 * @since 3.5.6
	 */
	public function delete_lock() {
		if ( ! check_ajax_referer( 'inline_edit', 'ajax_nonce', false ) ) {
			die( json_encode( array( 'error' => $this->a->get_text('unauthorised') ) ) );
		}

		$app_id		= ! empty( $_POST['app_id'] ) ? wpb_clean( $_POST['app_id'] ) : 0;
		$user_id	= $this->check_lock( $app_id );

		if ( null === $user_id ) {
			wpb_delete_app_meta( $app_id, '_edit_lock' );
		}

		wp_send_json_success();
	}

}
	BASE('AdminBookings')->add_hooks();
}