<?php
/**
 * WPB Admin
 *
 * Handles admin pages and includes methods for admin side
 *
 * Copyright © 2018-2021 Hakan Ozevin
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       2.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBAdmin' ) ) {

class WpBAdmin extends WpBFront {

	/**
     * Constructor
     */
	public function __construct() {
		parent::__construct();
	}

	/**
     * Add admin side actions
     */
	public function add_hooks_admin() {

		include_once( WPBASE_PLUGIN_DIR . '/includes/constant-data.php' );
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/dashboard.php' );
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/bookings.php' );
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/clients.php' );
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/monetary-settings.php' );
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/display-settings.php' );
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/display-advanced.php' );
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/global-settings.php' );
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/help.php' );
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/services-list.php' );
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/toolbar.php' );
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/transactions.php' );
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/welcome.php' );
		include_once( WPBASE_PLUGIN_DIR . '/includes/admin/tinymce.php' );
		include_once( WPBASE_PLUGIN_DIR . '/includes/lib/plugin-updater.php' );

		add_action( 'admin_head', array( $this, 'menu_order_count' ) );									// Add number of pending bookings
		add_action( 'admin_print_scripts', array( $this, 'admin_print_scripts' ) );						// Print some js codes to header
		add_action( 'admin_menu', array(  $this, 'admin_init' ) ); 										// Creates admin settings window. Correct action, because screen names are generated just before this hook
		add_filter( 'plugin_action_links_' . WPB_PLUGIN_BASENAME, array( $this, 'add_action_links' ));	// Add Settings link
		add_action( 'wp_dashboard_setup', array( $this, 'add_app_counts') );							// Add app counts
		add_filter( 'manage_pages_columns', array( $this, 'manage_pages_add_column' ) );				// Add column to Manage Pages
		add_action( 'manage_pages_custom_column', array( $this, 'manage_pages_column_content'), 10, 2 );// Add content tp column td
		add_action( 'wp_ajax_delete_log', array( $this, 'delete_log' ) ); 								// Clear log
		add_action( 'wp_ajax_nopriv_app_cron', array( $this, 'run_cron' ) ); 							// Cron job
		add_action( 'wp_ajax_nopriv_app_cron_log', array( $this, 'run_cron_log' ) );					// Cron job with log
	}

	/**
	 * Do cron job
	 * @since 3.7.7.1
	 */
	public function run_cron( $log = false ){

		$this->update_appointments();

		do_action( 'app_cron' );

		if ( $log ) {
			$this->log( sprintf( __( 'Cron job finished at %s', 'wp-base' ), date_i18n( $this->dt_format, $this->_time ) ) );
		}
	}

	/**
	 * Do cron job with log
	 * @since 3.7.7.1
	 */
	public function run_cron_log(){
		$this->run_cron( true );
	}

	/**
	 * Add settings link to Plugins page
	 * @since 2.0
	 */
	public function add_action_links( $links ) {
		$wpb_links = array(
			'<a href="' . admin_url( 'admin.php?page=app_settings' ) . '">'.__('Settings', 'wp-base' ).'</a>',
			'<a href="'.WPB_DEMO_WEBSITE.'" target="_blank">'.__('Demo', 'wp-base' ).'</a>',
			'<a href="' . admin_url( 'admin.php?page=app_help' ) . '">'.__('Help', 'wp-base' ).'</a>',
		 );
		return array_merge( $links, $wpb_links );
	}

	/**
	 * Add app status counts widget on admin Dashboard
	 * @since 2.0
	 */
	public function add_app_counts() {
		wp_add_dashboard_widget( 'app_dashboard_widget', WPB_NAME, array( $this, 'add_app_counts_' ) );
	}
	public function add_app_counts_() {

		global $wpdb;

		echo '<ul>';
		$num_running = $wpdb->get_var("SELECT COUNT(ID) FROM " . $this->app_table . " WHERE status='running' " );
        $num = number_format_i18n( $num_running );
		$text = _n( 'Booking In Progress', 'Bookings In Progress', intval( $num_running ) );
		if ( current_user_can( WPB_ADMIN_CAP ) ) {
			$text = "<a href='admin.php?page=app_bookings&status=running'>$num $text</a>";
		}
		echo '<li><span class="dashicons dashicons-backup" style="margin-right:5px"></span>';
		echo $text;
		echo '</li>';

		$num_active = $wpdb->get_var("SELECT COUNT(ID) FROM " . $this->app_table . " WHERE status='paid' OR status='confirmed' " );
        $num = number_format_i18n( $num_active );
        $text = _n( 'Upcoming Booking', 'Upcoming Bookings', intval( $num_active ) );
        if ( current_user_can( WPB_ADMIN_CAP ) ) {
            $text = "<a href='admin.php?page=app_bookings&status=active'>$num $text</a>";
        }
		echo '<li><span class="dashicons dashicons-clock" style="margin-right:5px"></span>';
		echo $text;
        echo '</li>';

		$num_pending = wpb_pending_count();
        $num = number_format_i18n( $num_pending );
		$text = _n( 'Pending Booking', 'Pending Bookings', intval( $num_pending ) );
		if ( current_user_can( WPB_ADMIN_CAP ) ) {
			$text = "<a href='admin.php?page=app_bookings&status=pending'>$num $text</a>";
		}
		echo '<li><span class="dashicons dashicons-calendar-alt" style="margin-right:5px"></span>';
		echo $text;
		echo '</li>';
		do_action( 'app_dashboard_widget_item' );
		echo '</ul>';

		echo '<p><span>';
		printf( __( 'You are using %1$s version %2$s', 'wp-base'), '<a href="https://wp-base.com" target="_blank">'. WPB_NAME.'</a>', $this->version );
		echo '</span></p>';

		echo '<form method="get" action="'.admin_url('admin.php').'">';
		echo '<p class="submit">';
		echo '<input type="submit" class="button button-primary" value="'.__('Add New Booking', 'wp-base' ).'">';
		echo '<input type="hidden" name="page" value="app_bookings">';
		echo '<input type="hidden" name="add_new" value="1">';
		echo '</p>';
		echo '</form>';
	}

	/**
	 * Adds a column to Manage Pages
	 * @since 2.0
	 */
	public function manage_pages_add_column( $columns ) {
		return
			array_slice( $columns, 0, 3, true ) +
			array( "wp_base" => "WP BASE" ) +
			array_slice( $columns, 3, count( $columns ) - 1, true );
	}

	/**
	 * Adds column td content to Manage Pages
	 * @since 2.0
	 */
	public function manage_pages_column_content( $column_name, $post_id ) {
		if ( 'wp_base' == $column_name ) {
			$post = get_post( $post_id );
			if ( strpos( $post->post_content, '[app_' ) !== false )
				echo '<span class="dashicons dashicons-calendar" title="'.__('This page includes WP BASE shortcode', 'wp-base' ) .'"></span>';
		}
	}

	/**
	 * Print codes to header
	 * @since 2.0
	 */
	public function admin_print_scripts() {
?>
<style type="text/css">
#toplevel_page_appointments li:not(:last-child) a[href^="admin.php?page=app_transactions"]:after,
#toplevel_page_appointments li:not(:last-child) a[href^="admin.php?page=app_settings"]:after,
#toplevel_page_appointments li:not(:last-child) a[href^="admin.php?page=app_vendor_settings"]:after {
    border-bottom: 1px solid hsla(0,0%,100%,.2);
    display: block;
    float: left;
    margin: 13px -15px 8px;
    content: "";
    width: calc(100% + 26px);
}
</style>
<?php if ( ! WPB_DEV ) { ?>
<style type="text/css">
#toplevel_page_appointments li a[href^="admin.php?page=app_addons"], 
#toplevel_page_appointments li a[href^="admin.php?page=app_addons"]:hover {
    color: #FFFFFF !important;
    font-weight: 600;
}
#toplevel_page_appointments li:nth-last-child(2) {
    background-color: #1da867;
}
#wpadminbar #wp-admin-bar-wpbase-app_addons a {
	background-color: #1da867;
    color: #fff;
	font-weight: 600;
    margin-top: 5px;
}
</style>
<?php }

		if ( ! wpb_is_admin_page() ) {
			return;
		}

		if ( ! wpb_is_admin_page('calendars') || ! isset( $_GET['tab'] ) || 'daily' != $_GET['tab'] ) {
			echo '<style>'. $this->inline_styles() .'</style>';
		} else {
			wpb_fix_font_colors();
		}

		if ( $css = trim( wpb_setting("additional_css_admin") ) ) {
			echo '<style type="text/css">';
			echo $css;
			echo '</style>';
		}
	}

	/**
	 * Adds the order processing count to the menu.
	 * @since 3.6.0
	 */
	public function menu_order_count() {
		global $submenu;

		if ( isset( $submenu['appointments'] ) ) {

			// Add count if user has access.
			if ( apply_filters( 'app_include_pending_count_in_menu', true ) && current_user_can( WPB_ADMIN_CAP ) ) {
				$order_count = wpb_pending_count();

				if ( $order_count ) {
					foreach ( $submenu['appointments'] as $key => $menu_item ) {
						if ( 0 === strpos( $menu_item[0], _x( 'Bookings', 'Admin menu name', 'wp-base' ) ) ) {
							$submenu['appointments'][ $key ][0] .= ' <span class="awaiting-mod update-plugins count-' . esc_attr( $order_count ) . '"><span class="processing-count">' . number_format_i18n( $order_count ) . '</span></span>'; // WPCS: override ok.
							break;
						}
					}
				}
			}
		}
	}

	/**
	 * Admin pages init stuff, save settings
	 *
	 */
	public function admin_init() {

		do_action( 'app_menu_before_all' ); // e.g. Bookings page
		do_action( 'app_submenu_before_business' );
		wpb_add_submenu_page('appointments', __('Business Settings', 'wp-base' ), __('Business Settings', 'wp-base' ), array(WPB_ADMIN_CAP,'manage_locations','manage_services','manage_working_hours','manage_extras'), "app_business", array($this,'business'));
		do_action( 'app_submenu_after_business' );
		do_action( 'app_submenu_before_tools' ); // e.g. Global Settings page
		wpb_add_submenu_page('appointments', __('Tools', 'wp-base' ), __('Tools', 'wp-base' ), array(WPB_ADMIN_CAP,'manage_tools'), "app_tools", array($this,'tools'));
		do_action( 'app_submenu_after_tools' ); // e.g. Help page

		if ( 'yes' == wpb_setting('allow_worker_edit') && $this->is_worker( get_current_user_id() ) ) {
			do_action( 'app_menu_for_worker' );
		}

		if ( isset( $_REQUEST["action_app"] ) && wpb_is_demo() ) {
			wpb_notice( 'demo' );
			return;
		}

		# Use this action for addons, depending classes, etc to save settings with $_GET
		do_action( 'app_save_settings_with_get' );

		// If there is no save request, nothing to do
		if ( ! isset( $_POST['action_app'] ) ) {
			return;
		}

		$this->save_settings();

		# Use this action for addons, depending classes, etc to save settings with $_POST
		do_action( 'app_save_settings' );
	}

	/**
	 * Admin save settings
	 */
	public function save_settings() {

		if ( isset( $_POST['app_nonce'] ) && !wp_verify_nonce( $_POST['app_nonce'], 'update_app_settings' ) ) {
			wpb_notice( 'unauthorised', 'error' );
			return;
		}

		$flush_cache = false;
		$result = $updated = $inserted = false;

		# Change status bulk
		if ( $this->change_status_bulk() ) {
			$flush_cache = true;
		}

		# Delete removed appointments
		if ( isset( $_POST['app_delete_nonce'] ) && wp_verify_nonce($_POST['app_delete_nonce'],'delete_or_reset') &&
				isset( $_POST['delete_removed'] ) && 'delete_removed' == $_POST['delete_removed'] ) {

			wpb_admin_access_check( 'delete_bookings' );

			if ( $this->delete_removed() )
				$flush_cache = true;
		}

		# Reset settings
		if ( isset( $_POST['app_delete_nonce'] ) && wp_verify_nonce($_POST['app_delete_nonce'],'delete_or_reset') &&
				isset( $_POST['restore_defaults'] ) && 'restore_defaults' == $_POST['restore_defaults'] ) {

			wpb_admin_access_check( 'manage_tools' );

			if ( defined( 'WPB_DISABLE_RESET' ) && WPB_DISABLE_RESET ) {
				wp_die( $this->get_text('unauthorised') );
			} else if ( isset( $_POST['reset_only_templates_check'] ) && ! empty( $_POST['reset_only_templates'] ) ) {
				update_option( 'wp_base_options', array_merge( wpb_setting(), WpBConstant::defaults( 'only_templates' ) ) );
				$message = sprintf( __('Email and SMS templates have been restored by user %s', 'wp-base' ), BASE('User')->get_name() );
				wpb_notice( $message );
				$this->log( $message );
			} else {
				delete_option( 'wp_base_business_options' );
				delete_option( 'wp_base_udfs' );
				
				if ( isset( $_POST['reset_custom_texts_check'] ) && ! empty( $_POST['reset_custom_texts'] ) ) {
					delete_option( 'wp_base_texts' );
					delete_option( 'wp_base_replace_texts' );
				}

				if ( isset( $_POST['reset_templates_check'] ) && ! empty( $_POST['reset_templates'] ) ) {
					delete_option( 'wp_base_options' );
					update_option( 'wp_base_options', WpBConstant::defaults() );
				} else {
					$old_templates = array_intersect_key( wpb_setting(), WpBConstant::defaults( 'only_templates' ) );
					update_option( 'wp_base_options', array_merge( WpBConstant::defaults( ), $old_templates ) );
				}

				wpb_flush_cache();
				wpb_notice( 'restored' );
				# Check and regenerate admin toolbar
				wpb_rebuild_menu();
				$this->log( sprintf( __('Default settings have been restored by user %s', 'wp-base' ), BASE('User')->get_name() ) );
			}

			return;
		}

		// Reset database
		if ( isset( $_POST['app_delete_nonce'] ) && wp_verify_nonce($_POST['app_delete_nonce'],'delete_or_reset')
			&& isset( $_POST['reset_db'] ) && 'reset_db' == $_POST['reset_db'] ) {

			wpb_admin_access_check( 'manage_tools' );

			if ( defined( 'WPB_DISABLE_RESET' ) && WPB_DISABLE_RESET ) {
				wp_die( $this->get_text('unauthorised') );
			} else {
				$this->reset_db();
			}

			return;
		}

		if ( $flush_cache ) {
			wpb_flush_cache();
		}
	}

	/**
	 * Truncate WP BASE tables
	 * Since 3.0
	 */
	private function reset_db(){
		if ( isset( $_POST['reset_payments_check'] ) && ! empty( $_POST['reset_payments'] ) ) {
			$tables = array(
				$this->transaction_table,
			);
		} else {
			$tables = array(
				$this->app_table,
				$this->meta_table,
				$this->transaction_table,
				$this->locations_table,
				$this->services_table,
				$this->workers_table,
				$this->wh_w_table,
				$this->wh_s_table,
				$this->wh_a_table,
			);
		}

		$tables = apply_filters( 'app_reset_tables', $tables );

		$error = '';

		foreach ( $tables as $table ) {
			# As a safety measure, allow only truncating of tables having {prefix}base_ in the name
			if ( strpos( $table, $this->db->prefix .'base_' ) !== false ) {
				$result = $this->db->query( 'TRUNCATE ' . $table );
				if ( ! $result ) {
					$error .= sprintf( __('Error truncating table: %s <br />', 'wp-base'), $table );
				}
			}
		}
		
		$this->db->query( "DELETE FROM {$this->db->usermeta} WHERE meta_key='app_vendor' OR meta_key='app_vendor_status'" );
		
		foreach ( wpb_get_clients() as $user_id => $user ) {
			$user->remove_role( 'wpb_client' );
		}
		
		wpb_rebuild_menu();
		BASE('WH')->new_install( );

		if ( ! $error ) {
			$this->log( sprintf( __('Database tables truncated by user %s', 'wp-base' ), BASE('User')->get_name() ) );
			wpb_notice( 'reset_db' );
		} else {
			$error = rtrim( $error, "<br />" );
			$this->log( $error );
			wpb_notice( 'reset_error' );
		}

		wpb_flush_cache();

		do_action( 'app_db_reset', $tables );
	}

	/**
	 * Delete removed app records
	 * $param to_delete: If array given deletes those apps provide that status match ('Cart')
	 * Since 2.0 - previously it was integrated in admin_init method
	 */
	public function delete_removed( $to_delete = false ) {
		if ( false === $to_delete ) {
			$to_delete = isset( $_POST['app'] ) && is_array( $_POST['app'] ) ? $_POST['app'] : array();
		}

		if ( empty( $to_delete ) ) {
			return false;
		}

		$q = '';
		foreach ( (array)$to_delete as $app_id ) {
			$q .= " ID='". esc_sql( $app_id ). "' OR";
		}

		$q = apply_filters( 'app_delete_removed_query', rtrim( $q, " OR" ) );
		$stat = isset($_POST["delete_removed"]) && 'delete_removed' == $_POST["delete_removed"] ? "removed" : "cart";
		do_action( 'app_delete_pre', 'removed', $to_delete );

		$result = $this->db->query( $this->db->prepare(
			"DELETE FROM " .$this->app_table.
			" WHERE (".$q.") AND status=%s", $stat )
		);

		if ( $result ) {

			$this->db->query( "UPDATE " .$this->transaction_table.
				" SET transaction_app_ID=0 WHERE (". str_replace( 'ID','transaction_app_ID', $q ).")" );

			$this->db->query( $this->db->prepare( "DELETE FROM " .$this->meta_table.
				" WHERE (". str_replace( 'ID','object_id', $q ).") AND meta_type=%s", 'app' )
			);

			if ( 'cart' != $stat ) {
				$userdata = BASE('User')->_get_userdata( get_current_user_id() );
				$user_login = isset( $userdata->user_login ) ? $userdata->user_login : esc_url_raw( $_SERVER['REMOTE_ADDR'] );
				wpb_notice( 'deleted' );
				$this->log( sprintf(
					__('Booking(s) with id(s): %1$s deleted by user %2$s', 'wp-base' ),
					implode( ', ', $to_delete ),
					$user_login )
				);
			}

			do_action( 'app_deleted', 'removed', $to_delete );

			return true;
		}
	}

	/**
	 * Bulk status change
	 * Since 2.0 - previously it was integrated in admin_init method
	 */
	public function change_status_bulk() {

		if ( !( isset( $_POST["app_status_change"] ) && $_POST["app_new_status"] && isset( $_POST['app'] ) && is_array( $_POST['app'] ) ) ) {
			return;
		}

		$q = '';
		foreach ( $_POST['app'] as $app_id ) {
			$q .= " ID='". esc_sql( $app_id ). "' OR";
		}

		$q 			= rtrim( $q, " OR" );
		$last_app 	= $this->get_app( $app_id ); // Saving the latest status will be enough

		if ( empty( $last_app->status ) ) {
			return;
		}

		$old_stat 	= $last_app->status;
		$new_stat 	= $_POST["app_new_status"];

		// Make a status re-check here - It should be in status map
		if ( $new_stat == $old_stat || !array_key_exists( $new_stat, $this->get_statuses() ) ) {
			return;
		}

		if ( ! $this->db->query( $this->db->prepare( "UPDATE ". $this->app_table ." SET status=%s WHERE (".$q.")", $new_stat ) ) ) {
			return;
		}

		wpb_flush_cache();

		// Email on bulk change - Confirmation
		if ( 'yes' == wpb_setting('send_confirmation_bulk')
			&& ( ( 'confirmed' == $new_stat  && 'paid' != $old_stat ) || ( 'paid' == $new_stat && 'confirmed' != $old_stat ) ) ) {
			foreach ( $_POST['app'] as $app_id ) {
				$this->send_email( $app_id );
			}
		}

		// Email on bulk change - Pending
		if ( 'yes' == wpb_setting('send_pending_bulk') && 'pending' == $new_stat ) {
			foreach ( $_POST['app'] as $app_id ) {
				$this->send_email( $app_id, 'pending' );
			}
		}

		// Email on bulk change - Cancellation
		if ( 'yes' == wpb_setting('send_cancellation_bulk') && 'removed' == $new_stat ) {
			foreach ( $_POST['app'] as $app_id ) {
				$this->send_email( $app_id, 'cancellation' );
			}
		}

		wpb_notice( 'updated' );

		do_action( 'app_bulk_status_change', $new_stat, $_POST['app'] );

		$userdata = BASE('User')->_get_userdata( get_current_user_id() );

		$this->log( sprintf(
			__( 'Status of Booking(s) with id(s):%1$s changed to %2$s by user: %3$s', 'wp-base' ),
			implode( ', ', $_POST['app'] ),
			$new_stat,
			$userdata->user_login
		) );

		return true;
	}

	/**
	 * Admin business definitions HTML code
	 * @since 2.0
	 */
	public function business() {
	?>
<div class="wrap app-page app-business-settings">
	<h2 class="app-dashicons-before dashicons-store"><?php echo __( 'Business Settings', 'wp-base' ); ?></h2>
	<h3 class="nav-tab-wrapper"><?php

		$tab 	= ! empty( $_GET['tab'] ) ? wpb_clean( $_GET['tab'] ) : 'services';
		$tabhtml = array();

		foreach ( wpb_admin_tabs( 'business' ) as $stub => $title ) {
			$class = ( $stub == $tab ) ? ' nav-tab-active' : '';
			$tabhtml[] = '	<a href="'. esc_attr( admin_url( 'admin.php?page=app_business&tab=' . $stub ) ) .'" class="nav-tab'.$class.'" id="app_tab_'.$stub.'">'.$title.'</a>';
		}

		echo implode( "\n", $tabhtml );
		?>
	</h3>
	<div class="clear"></div>

		<?php switch( $tab ) {

			case $tab:	do_action( 'app_business_'.$tab.'_tab' ); break;
		}
	?>
</div><!-- Wrap -->
	<?php
	}

	/**
	 * Admin tools page HTML
	 * @since 2.0
	 */
	public function tools() {

		wpb_admin_access_check( 'manage_tools' );
	?>
<div class="wrap app-tools">
	<h2 class="app-dashicons-before dashicons-admin-tools"><?php echo __('Tools', 'wp-base' ); ?></h2>
	<h3 class="nav-tab-wrapper"><?php

		$tab 		= ( ! empty($_GET['tab']) ) ? wpb_clean( $_GET['tab'] ) : 'log';
		$tabhtml 	= array();

		foreach ( wpb_admin_tabs( 'tools' ) as $stub => $title ) {
			$class = ( $stub == $tab ) ? ' nav-tab-active' : '';
			$tabhtml[] = '	<a href="' . admin_url( 'admin.php?page=app_tools&amp;tab=' . $stub ) . '" class="nav-tab'.$class.'" id="app_tab_'.$stub.'">'.$title.'</a>';
		}

		echo implode( "\n", $tabhtml );
		?>
	</h3>
	<div class="clear"></div>
	<div id="poststuff" class="metabox-holder"><?php

		switch( $tab ) {
	/*******************************
	* Log tab
	********************************
	*/
		case 'log':	?>
		<div class="postbox">
			<div class="inside" style="word-wrap:break-word;" id="app_log">
			<?php
				if ( wp_is_writable( $this->uploads_dir ) ) {
					if ( file_exists( $this->log_file ) )
						echo nl2br( file_get_contents( $this->log_file ) );
					else
						echo __( 'There are no log records yet.', 'wp-base' );
				} else {
					echo __( 'Uploads directory is not writable.', 'wp-base' );
				}
				?>
			</div>
		</div>
		<table class="form-table">
			<tr>
				<th scope="row" >
				<input type="button" id="log_clear_button" class="button-primary" value="<?php _e('Clear Log File') ?>" title="<?php _e('Clicking this button deletes logs saved on the server') ?>" />
				</th>
			</tr>
		</table>
		<?php break;
	/*******************************
	* Reset tab
	********************************
	*/
		case 'reset' :
			wpb_admin_access_check( 'manage_tools' );

			$wp_nonce = wp_nonce_field( 'update_app_settings', 'app_nonce', true, false );
			wpb_infobox( sprintf( __('Here you can return settings and/or WP BASE related database tables to the point of first installation of the plugin, e.g. after you finished testing. To completely disable this function, add %s to wp-config.php.', 'wp-base'), '<code>define("WPB_DISABLE_RESET", true);</code>' ) );
		?>
		<div class="postbox">
			<div class="inside">
				<?php
				if ( defined( 'WPB_DISABLE_RESET' ) && WPB_DISABLE_RESET ) {
					_e( 'Disabled by WPB_DISABLE_RESET', 'wp-base' );
				} else {
				?>
				<table>
					<tr class="app_impex">
						<td class="app-b"><?php _e('Global Settings', 'wp-base' ) ?></td>
						<td class="app-c">
						<form id="restore_defaults_form" method="post">
							<?php echo $wp_nonce; ?>
							<input type="hidden" name="restore_defaults" value="restore_defaults" />
							<input type="hidden" name="action_app" value="restore_defaults" />
							<?php wp_nonce_field( 'delete_or_reset', 'app_delete_nonce' ); ?>
							<input type="submit" id="restore_defaults_button" class="button-secondary" value="<?php _e('Restore to default settings') ?>" title="<?php _e('Clicking this button will set all settings to their default values') ?>" />
							<br />
							<div class="app-chkbx">
								<input type="hidden" name="reset_custom_texts_check" value="yes" />
								<input type="checkbox" name="reset_custom_texts" value="yes" />&nbsp;
								<span class="description app-btm"><?php  _e('Also reset Custom Texts', 'wp-base' ) ?></span>
								<br />
								<input type="hidden" name="reset_templates_check" value="yes" />
								<input type="checkbox" name="reset_templates" value="yes" />&nbsp;
								<span class="description app-btm"><?php  _e('Also reset email and SMS templates', 'wp-base' ) ?></span>
								<br />
								<input type="hidden" name="reset_only_templates_check" value="yes" />
								<input type="checkbox" name="reset_only_templates" value="yes" />&nbsp;
								<span class="description app-btm"><?php  _e('Reset ONLY email and SMS templates', 'wp-base' ) ?></span>
							</div>							
						</form>
						</td>
					</tr>
					<tr class="app_impex">
						<td class="app-b"><?php _e('Database Tables (Bookings, Transactions and Business Settings)', 'wp-base' ) ?></td>
						<td class="app-c">
						<form id="reset_db_form" method="post">
							<?php echo $wp_nonce; ?>
							<input type="hidden" name="reset_db" value="reset_db" />
							<input type="hidden" name="action_app" value="reset_db" />
							<?php wp_nonce_field( 'delete_or_reset', 'app_delete_nonce' ); ?>
							<input type="submit" id="reset_db_button" class="button-secondary" value="<?php _e('Reset Database Tables') ?>" title="<?php _e('Clicking this button will clear all WP BASE related database tables.') ?>" />
							<br />
							<div class="app-chkbx">
								<input type="hidden" name="reset_payments_check" value="yes" />
								<input type="checkbox" name="reset_payments" value="yes" />&nbsp;
								<span class="description app-btm"><?php  _e('Reset ONLY Payment History', 'wp-base' ) ?></span>
							</div>
						</form>
						</td>
					</tr>
				</table>
				<?php } ?>
			</div>
		</div>
		<?php break;

		case $tab:	do_action( 'app_tools_'.$tab.'_tab' );
					break;

		} # End of switch
		?>
	</div>
</div>
	<?php
	}

	/**
	 * Delete log file
	 */
	public function delete_log(){
		if ( !check_ajax_referer( 'delete_log', 'ajax_nonce', false ) )
			die( json_encode( array( 'error' => $this->get_text('unauthorised') ) ) );

		if ( wpb_is_demo() ) {
			die( json_encode( array( 'error' => __( 'In DEMO mode log file cannot be deleted', 'wp-base' ) ) ) );
		}

		if ( file_exists( $this->log_file ) )
			unlink( $this->log_file );

		sleep( 1 );

		if ( file_exists( $this->log_file ) )
			die( json_encode( array( 'error' => __( 'Log file could not be deleted', 'wp-base' ) ) ) );

		die( json_encode( array( 'success' => 1 ) ) );
	}

}

	BASE('Admin')->add_hooks_admin();
	$GLOBALS['appointments'] = BASE('Admin');

} else {
	add_action( 'admin_notices', '_wpb_plugin_conflict_own' );
}

