<?php
/**
 * WPB Addons
 *
 * Manages loading of addons
 * @author		Hakan Ozevin
 * @package     WP BASE
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       2.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WpBAddons' ) ) {

class WpBAddons {

	/**
     * WP BASE Core + Front [+Admin] instance
     */
	protected $a = null;

	/**
     * Constructor
     */
	public function __construct() {
		$this->a = BASE();
	}

	/**
     * Add actions and filters
     */
	public function add_hooks() {
		add_action( 'app_submenu_after_tools', array( $this, 'add_submenu' ), 10 );				// Add submenu item
		add_action( 'app_save_settings_with_get', array( $this, 'save_settings' ), 16 );			// Save settings
		add_action( 'app_loaded', array( $this, 'load_addons' ), 20 );
		add_action( 'app_installed', array( $this, 'installed' ) );
		add_action( 'activate_plugin', array( $this, 'plugin_activated_deactivated' ), 10, 2 );
		add_action( 'deactivate_plugin', array( $this, 'plugin_activated_deactivated' ), 10, 2 );

		add_action( 'admin_init', array( $this, 'admin_init' ) );
		add_filter( 'appointments_tools_tabs', array( $this, 'add_tab' ), 30 );						// Add tab to Tools
		add_action( 'admin_init', array( $this, 'display_notice' ) );
		add_action( 'admin_init', array( $this, 'display_success' ) );
		add_action( 'admin_init', array( $this, 'handle_license_activation' ) );
		add_action( 'update_option', array( $this, 'sanitize_license' ), 10, 3 );
	}

	public function admin_init() {
		if ( ! WPB_DEV ) {
			return;
		}

		if ( !class_exists( 'WPB_Plugin_Updater' ) ) {
			include_once( WPBASE_PLUGIN_DIR . "/includes/lib/plugin-updater.php" );
		}

		$data = get_file_data( WPBASE_PLUGIN_FILENAME, wpb_default_headers(), 'plugin' );

		$wpb_updater = new WPB_Plugin_Updater( WPBASE_URL, WPBASE_PLUGIN_FILENAME, array(
			'version'   	=> $data['Version'],
			'license'   	=> trim( get_option( 'wpb_license_key_'. $data['ID'] ) ),
			'item_id' 		=> $data['ID'],
			'Category' 		=> $data['Category'],
			'author'    	=> $data['Author'],
			'wp_override'	=> true,
			)
		);
	}

	/**
     * List all files in addons directory. Only file name with php extension
	 * @return array
     */
	public function get_all_addons() {
		$identifier = wpb_cache_prefix() . 'all_addons';
		$all_addons = wp_cache_get( $identifier );

		if ( false === $all_addons ) {

			$all = glob( WPB_ADDON_DIR  .'/' . '*.php' );
			$all = $all ? $all : array();
			$all_addons = array();
			foreach ($all as $path) {
				$all_addons[] = pathinfo($path, PATHINFO_FILENAME);
			}

			wp_cache_set( $identifier, $all_addons );
		}

		return apply_filters( 'app_all_addons', $all_addons );
	}

	/**
     * List all extensions (Addons + Plugins) who are registered themselves.
	 * @return array
     */
	public function get_all_active_extensions(){
		return apply_filters( 'app_active_extensions', false );
	}

	/**
     * List info of the addon
	 * @return array
     */
	public function get_addon_info( $addon ) {
		$addon = str_replace( '/', '_', $addon );
		$path = WPB_ADDON_DIR ."/" . "{$addon}.php";

		return apply_filters( 'app_addon_info', get_file_data( $path, wpb_default_headers(), 'plugin' ), $addon );
	}

	/**
     * Return name of the addon
	 * @return string
     */
	public function get_addon_name( $addon ) {
		$info = $this->get_addon_info( $addon );
		return isset( $info['Name'] ) ? str_replace( 'WP BASE Addon: ','', $info['Name']) : '';
	}

	/**
     * Check if an addon is active
     */
	public function is_active( $addon ) {
		if ( $this->is_disabled( $addon ) || in_array( $addon, wpb_setting('deactivated_addons', array() ) ) ) {
			$result = false;
		} else {
			$result = true;
		}

		return apply_filters( 'app_addon_is_active', $result, $addon );
	}

	/**
     * Check if an addon is disabled by an external plugin
	 * External plugin is expected to return false for not to disable and
	 * A reason (string) to disable
     */
	public function is_disabled( $addon ) {
		
		return apply_filters( 'app_disable_addon', false, $addon );
	}

	/**
     * Allow addons to be deactivated at installation
     */
	public function installed(){
		if ( defined( 'WPB_ADDONS_DEACTIVATED_AT_INSTALL' ) && WPB_ADDONS_DEACTIVATED_AT_INSTALL ) {
			$options = wpb_setting();
			$deact = empty( $options['deactivated_addons'] ) ? array() : $options['deactivated_addons'];
			foreach ( $this->get_all_addons( ) as $addon ) {
				$deact[] = $addon;
			}

			$options['deactivated_addons'] = array_unique( $deact );
			$this->a->update_options( $options );
		}
	}

	/**
     * Load all files in addons directory
	 * This happens after "plugins loaded". Therefore a WP plugin with the same class name of an addon has priority
	 * Such a plugin prevents loading of an addon by wpb_disable_addon function
     */
	public function load_addons( ) {

		$all_addons = apply_filters( 'app_addons_to_load', $this->get_all_addons() );

		foreach ( (array)$all_addons as $addon ){
			if ( $this->is_active( $addon ) ) {
				
				# Allow forced disabling of gcal in case of emergency
				if ( 'google-calendar' == $addon && defined( 'WPB_DISABLE_GCAL' ) ) {
					continue;
				}

				include_once WPB_ADDON_DIR ."/" . $addon . ".php";
			}
		}

		do_action( 'app_addons_loaded' );

		add_action( 'app_tools_licenses_tab', array( $this, 'license_settings' ) );			// Create tab HTML
	}

	/**
     * Add submenu
     */
	public function add_submenu() {
		add_submenu_page( 'appointments', __('Addons','wp-base'), __('Addons','wp-base'), WPB_ADMIN_CAP, "app_addons", array( $this,'settings'));
	}

	/**
     * Check if Advanced Features Extension is activated or deactivated
     */
	public function plugin_activated_deactivated( $wp_plugin, $network_wide ) {
		if ( strpos( $wp_plugin, 'advanced-features' ) === false ) {
			return;
		}

		$this->rebuild_shortcodes();
	}

	/**
     * Recreate list of shortcodes
     */
	public function rebuild_shortcodes() {
		include_once( WPBASE_PLUGIN_DIR . '/includes/constant-data.php' );
		update_option( 'wp_base_shortcodes', array_keys( WpBConstant::shortcode_desc() ) );
	}

	/**
	 * Save settings
	 */
	public function save_settings() {
		
		if ( ( !isset( $_REQUEST['update_addons'] ) || !isset( $_REQUEST['addon'] ) ) && !isset( $_POST['optimize_addons'] ) ) {
			return;
		}

		if ( !wp_verify_nonce( $_REQUEST['app_nonce'], 'update_addons' ) ) {
			wpb_notice( 'unauthorised', 'error' );
			return;
		}

		$options 		= wpb_setting();
		$buss_options 	= $this->a->get_business_options();
		$deact 			= empty( $options['deactivated_addons'] ) ? array() : $options['deactivated_addons'];
		$switched 		= '';
		$all_false 		= array(
			'action_app'	=> false,
			'app_nonce' 	=> false,
			'addon' 		=> false,
			'deactivated' 	=> false,
			'activated' 	=> false,
			'changed_addons'=> false
		);

		// Optimize addons
		if ( isset( $_POST["optimize_addons"] ) ) {

			// Annual
			if( empty( $buss_options['alt_schedules'] ) )
				$deact[] = 'seasonal-working-hours';

			// Front end management
			// Time zones
			$posts_pages = $this->a->get_app_pages();
			$management_used = $timezones_used = $modular_used = false;

			// Modular shortcodes
			$all_shortcodes = wpb_shortcodes();
			$compact_shortcodes = array( 'app_book', 'app_list', 'app_theme', 'app_is_mobile', 'app_is_not_mobile' );
			$modular_shortcodes = array_diff( $all_shortcodes, $compact_shortcodes );

			if ( !empty( $posts_pages ) ) {
				foreach ( $posts_pages as $post ) {

					if ( !$management_used && strpos( $post->post_content, '[app_manage' ) !== false )
						$management_used = true;

					if ( !$modular_used && preg_match_all( '/'. get_shortcode_regex( $modular_shortcodes ) .'/s', $post->post_content, $matches ) )
						$modular_used = true;

					if ( !$timezones_used && strpos( $post->post_content, '[app_timezone' ) !== false )
						$timezones_used = true;

					if ( $management_used && $timezones_used && $modular_used )
						break;
				}
			}

			// Also check widget contents
			$app_widgets = get_option( 'widget_appointments_shortcode' );

			if ( is_array( $app_widgets ) ) {
				foreach ( $app_widgets as $widget ) {
					if ( empty( $widget['content'] ) || !empty( $widget['disable'] ) )
						continue;
					if ( !$management_used && strpos( $widget['content'], '[app_manage' ) !== false )
						$management_used = true;
					if ( !$modular_used && preg_match_all( '/'. get_shortcode_regex( $modular_shortcodes ) .'/s', $post->post_content, $matches ) )
						$modular_used = true;
					if ( !$timezones_used && strpos( $widget['content'], '[app_timezone' ) !== false )
						$timezones_used = true;
				}
			}

			if ( !$management_used )
				$deact[] = 'front-end-booking-management';

			if ( !$timezones_used && 'yes' != wpb_setting('enable_timezones') )
				$deact[] = 'timezones';

			// BuddyPress
			global $bp;
			if ( !is_object( $bp ) )
				$deact[] = 'buddypress';

			// Coupons
			if( empty( $buss_options['coupons'] ) )
				$deact[] = 'coupons';

			// Custom pricing
			if ( !trim( wpb_setting('advanced_pricing_total_price') )
				&& !trim( wpb_setting('advanced_pricing_deposit') )
				&& empty( $buss_options['ep_rules'] ) )
				$deact[] = 'custom-variable-pricing';

			// Export/import - If admin capability set to be different than admin, deactivate
			if ( defined( 'WPB_ADMIN_CAP' ) && WPB_ADMIN_CAP != 'manage_options' )
				$deact[] = 'export-import';

			// Extras
			if( empty( $buss_options['extras'] ) )
				$deact[] = 'extras';

			// Events
			if ( !wpb_setting('event_post_type') )
				$deact[] = 'event-bookings';

			// Front end edit
			if( 'yes' != wpb_setting('allow_edit') && 'yes' != wpb_setting('allow_worker_edit') )
				$deact[] = 'front-end-edit';

			// GCal
			if( 'yes' != wpb_setting('gcal_allow_worker') && 'none' == wpb_setting('gcal_api_mode') )
				$deact[] = 'gcal';

			// Extended Service, Group Bookings and Quotas
			$esf_not_used = $quotas_not_used = $group_not_used = true;
			$services = $this->a->get_services();

			foreach ( $services as $service ) {

				$limits = wpb_get_service_meta( $service->ID, 'limits' );
				if ( !empty( $limits ) && is_array( $limits ) && array_filter( $limits ) )
					$esf_not_used = false;

				if ( $esf_not_used && ($service->padding || $service->break_time || $service->internal || $service->deposit) )
					$esf_not_used = false;

				if ( $quotas_not_used && (wpb_get_service_meta( $service->ID, 'weekday_quota' ) || wpb_get_service_meta( $service->ID, 'weekend_quota' )
					|| wpb_get_service_meta( $service->ID, 'weekly_quota' ) || wpb_get_service_meta( $service->ID, 'any_quota' )) )
					$quotas_not_used = false;

				if ( $group_not_used ) {
					$meta = wpb_get_service_meta( $service->ID, 'group_bookings' );
					if ( !empty( $meta['enabled'] ) )
						$group_not_used = false;
				}

				if ( !$esf_not_used && !$quotas_not_used && !$group_not_used )
					break;
			}

			if ( $esf_not_used && 'no' == wpb_setting('time_slot_calculus_legacy') && !$this->a->get_nof_categories() )
				$deact[] = 'extended-service-features';

			if ( $quotas_not_used )
				$deact[] = 'quotas';

			if ( $group_not_used )
				$deact[] = 'group-bookings';

			// Locations
			if ( !$this->a->get_nof_locations() )
				$deact[] = 'locations';

			// Multilanguage
			if ( !wpb_setting( 'site_langs' ) )
				$deact[] = 'multi-language';

			// Packages
			$pack_not_used = true;
			if ( BASE('Packages') ) {
				foreach ( (array)$services as $service ) {
					if ( BASE('Packages')->is_package( $service->ID ) ) {
						$pack_not_used = false;
						break;
					}
				}
			}
			if ( $pack_not_used )
				$deact[] = 'packages';

			// Payment gateways
			$allowed = !empty( $options['gateways']['allowed'] ) ? (array)$options['gateways']['allowed'] : array();
			foreach ( $this->get_all_addons( ) as $addon ) {
				if( substr( $addon, 0, 16 ) != "payment-gateway-" )
					continue;
				if ( !in_array( str_replace( "payment-gateway-", '', $addon ), $allowed ) )
					$deact[] = $addon;
			}

			// Recurring appointments
			$rec_not_used = true;
			foreach( (array)$services as $service ) {
				$meta = wpb_get_service_meta( $service->ID, 'recurring' );
				if ( !empty( $meta['enabled'] ) ) {
					$rec_not_used = false;
					break;
				}
			}
			if ( $rec_not_used )
				$deact[] = 'recurring-appointments';

			// Reminder emails
			if( 'yes' != wpb_setting('send_reminder') && 'yes' != wpb_setting('send_reminder_worker')
				&& 'yes' != wpb_setting('send_dp_reminder') && 'yes' != wpb_setting('send_follow_up') )
				$deact[] = 'reminder-follow-up-emails';

			// Service providers
			if ( !$this->a->get_nof_workers() )
				$deact[] = 'service-providers';

			// Shopping Cart
			if ( 'yes' != wpb_setting('use_cart') )
				$deact[] = 'shopping-cart';

			// SMS
			if ( !wpb_setting('sms_service') )
				$deact[] = 'sms';

			// Test appointments
			if ( !$this->a->db->get_var( "SELECT COUNT(ID) FROM " . $this->a->app_table . " WHERE status='test'" ) )
				$deact[] = 'test-bookings';

			// UDF
			if ( !get_option( 'wp_base_udfs' ) )
				$deact[] = 'user-defined-fields';

			// Variable durations
			$var_not_used = true;
			foreach( (array)$services as $service ) {
				foreach ( array( 'time_variant_durations', 'selectable_durations' ) as $tool ) {
					$meta = wpb_get_service_meta( $service->ID, $tool );
					if ( !empty( $meta['enabled'] ) ) {
						$var_not_used = false;
						break 2;
					}
				}
			}
			if ( $var_not_used )
				$deact[] = 'variable-durations';

			// Waiting List
			$wl_not_used = true;
			foreach( (array)$services as $service ) {
				$meta = wpb_get_service_meta( $service->ID, 'waiting_list' );
				if ( !empty( $meta['enabled'] ) ) {
					$wl_not_used = false;
					break;
				}
			}
			if ( $wl_not_used )
				$deact[] = 'waiting-list';

			// WooCommerce
			if ( 'yes' != wpb_setting('wc_enabled') )
				$deact[] = 'woocommerce';

			$deact = array_unique( $deact );

			// Pro Addon
			$total_count 		= $this->get_all_addons() ? count( $this->get_all_addons() ) : 0;
			$deactivated_count 	= $deact ? count( $deact ) : 0;

			if ( !$modular_used && ($total_count - 1 == $deactivated_count || 1 == count( wpb_active_gateways() ) ) )
				$deact[] = 'advanced-features';

			$options['deactivated_addons'] = array_unique( $deact );

			foreach ( $deact as $filename ) {
				if ( $this->is_active( $filename ) )
					$switched .= $this->get_addon_name($filename) . ', ';
			}
			$switched = urlencode( rtrim( $switched, ', ' ) );

			if( !empty( $deact ) && $this->a->update_options( $options ) ) {
				wp_redirect( wpb_add_query_arg( array_merge( $all_false, array( 'deactivated' => 1, 'changed_addons' => $switched ) ) ) );
				exit;
			} else {
				wp_redirect( wpb_add_query_arg( $all_false ) );
				exit;
			}
		}

		if ( isset( $_POST["addon"] ) && is_array( $_POST["addon"] ) && !empty( $_POST['app_new_status'] ) ) {
			$new_status = wpb_clean( $_POST["app_new_status"] );
			foreach ( $_POST["addon"] as $filename ) {
				//deactivate
				if ( 'deactivate' == $new_status ) {
					if ( $this->is_active( $filename ) ) {
						$deact[] = $filename;
						$switched .= $this->get_addon_name($filename) . ', ';
						$deactivated = true;
					}
				} else if ( 'activate' == $new_status ) {
					if ( !$this->is_active( $filename ) ) {
						$key = array_search( $filename, $deact );
						if( $key !== false ){
							unset( $deact[$key] );
							$switched .= $this->get_addon_name($filename) . ', ';
							$deactivated = false;
						}
					}
				}
			}
			$switched = urlencode( rtrim( $switched, ', ' ) );
		} else if ( isset( $_GET["addon"] ) ) {
			$filename = wpb_clean( $_GET["addon"] );

			if ( $this->is_active( $filename ) ) {
				$deact[] = $filename;
				$deactivated = true;
				$switched = urlencode( $this->get_addon_name($filename) );
			} else {
				$key = array_search( $filename, $deact );
				if( $key !== false ){
					unset( $deact[$key] );
					$deactivated = false; // Means activated
					$switched = urlencode( $this->get_addon_name($filename) );
				}
			}
		}

		$options['deactivated_addons'] = array_unique( $deact );

		if( $this->a->update_options( $options ) ) {
			wpb_flush_cache();

			$this->rebuild_shortcodes();

			if ( !empty( $deactivated ) )
				wp_redirect( wpb_add_query_arg( array_merge( $all_false, array( 'deactivated' => 1, 'changed_addons' => $switched ) ) ) );
			else
				wp_redirect( wpb_add_query_arg( array_merge( $all_false, array( 'activated' => 1, 'changed_addons' => $switched ) ) ) );
		} else {
			wp_redirect( wpb_add_query_arg( $all_false ) );
		}
	}

	/**
	 * Admin settings HTML code
	 */
	public function settings() {

		wpb_admin_access_check( 'manage_addons' );

		?>
<div class="wrap app-page" id="wpb-addons">
	<h2 class="app-dashicons-before dashicons-admin-plugins"><?php echo __('Addons','wp-base'); ?></h2>
	<div id="poststuff" class="metabox-holder"><?php

	if ( ! WPB_DEV || wpb_is_demo() ) {
		$this->showcase();
	} else {
		if ( !empty( $_GET['changed_addons'] ) )
			$text = wpb_clean( urldecode( $_GET['changed_addons'] ) );
		else if ( isset( $_GET['addon'] ) )
			$text = $this->get_addon_name( wpb_clean( $_GET['addon'] ) );
		else $text = false;

		$count = $text ? count( explode(',', $text ) ) : 0;

		if ( (isset( $_GET['changed_addons'] ) && $_GET['changed_addons']) || ( isset( $_GET['addon'] ) && $_GET['addon'] ) ) {
			if ( isset( $_GET['deactivated'] ) && $_GET['deactivated'] )
				echo '<div class="updated fade"><p><b>[WP BASE]</b> '. sprintf( __('%1$d addon(s) deactivated: %2$s.','wp-base'), $count, $text ).'</p></div>';
			else if ( isset( $_GET['activated'] ) && $_GET['activated'] )
				echo '<div class="updated fade"><p><b>[WP BASE]</b> '. sprintf( __('%1$d addon(s) activated: %2$s.','wp-base'), $count, $text ).'</p></div>';
		}

		wpb_desc_infobox( 'addon' );

		$wp_nonce = wp_create_nonce( 'update_addons' );
		?>
		<form method="post" action="<?php echo wpb_add_query_arg( array( 'page'=>'app_addons', 'addon'=>false, 'app_nonce'=>$wp_nonce ) ); ?>" >
			<div class="alignright actions" style="margin:6px 0 4px;">
				<input type="hidden" value="app_addons" name="page" />
				<input type="hidden" value="<?php if ( isset( $post->ID ) ) echo $post->ID; else echo 0; ?>" name="page_id" />
				<input type="hidden" value="1" name="optimize_addons" />
				<input type="hidden" value="optimize_addons" name="action_app" />
				<input type="hidden" name="app_nonce" value="<?php echo $wp_nonce?>">
				<input type="submit" class="button" value="<?php _e('Deactivate Unused Addons','wp-base'); ?>" title="<?php _e('Checks your settings and deactivates any addon that has not set up. Use this only after you finished configuring your website.','wp-base'); ?>" />
			</div>
		</form>
		<form method="post" action="<?php echo wpb_add_query_arg( array( 'page'=>'app_addons', 'addon'=>false, 'app_nonce'=>$wp_nonce ) ); ?>" >
			<div class="alignleft actions" style="margin:6px 0 4px;">
				<input type="hidden" value="app_addons" name="page" />
				<input type="hidden" value="<?php if ( isset( $post->ID ) ) echo $post->ID; else echo 0; ?>" name="page_id" />
				<input type="hidden" value="1" name="update_addons" />
				<input type="hidden" value="update_addons" name="action_app" />
				<input type="hidden" name="app_nonce" value="<?php echo $wp_nonce?>">
				<select name="app_new_status" style='float:none;'>
					<option value=""><?php _e('Bulk actions','wp-base'); ?></option>
					<option value="activate" class="hide-if-no-js"><?php _e('Activate','wp-base'); ?></option>
					<option value="deactivate" class="hide-if-no-js"><?php _e('Deactivate','wp-base'); ?></option>
				</select>
				<input type="submit" class="button" value="<?php _e('Apply','wp-base'); ?>" />
			</div>

			<table class="wp-list-table widefat plugins app-addons">
				<thead>
				<tr>
					<th style="width:3%; text-align:left;" class="column-delete manage-column check-column"><input type="checkbox" /></th>
					<th style="width:22%; text-align:left;"><?php _e( 'Addon', 'wp-base' ) ?></th>
					<th style="width:75%; text-align:left;"><?php _e( 'Description', 'wp-base' ) ?></th>
				</tr>
				</thead>
			<?php
			foreach ( $this->get_all_addons() as $filename ) {
				$addon_data = $this->get_addon_info( $filename );
				if ( !$addon_data["Name"] )
					continue;

				$url = wpb_add_query_arg( array( 'action_app'=>1, 'app_nonce'=>$wp_nonce, 'page'=>'app_addons', 'update_addons'=>1, 'addon'=>$filename, 'changed_addons'=>false ) );
				?>
				<tr <?php if ( $this->is_active( $filename) ) echo "class='active'"; else echo "class='inactive'";?>>
					<td class="column-delete check-column app-check-column">
						<input type="checkbox" name="addon[]" value="<?php echo $filename;?>" />
					</td>
					<td class="plugin-title">
						<?php
						echo '<strong>'. str_replace( 'WP BASE Addon: ', '', $addon_data["Name"] ) . '</strong>';
						echo '<div class="row-actions-visible">';

						if ( $this->is_active( $filename ) ) {
							echo '<a href="'.$url.'">'. __('Deactivate', 'wp-base'). '</a>';
							do_action( 'app_addon_settings_link', $filename );
						} else {
							if ( $reason = $this->is_disabled( $filename ) )
								echo $reason;
							else
								echo '<a href="'.$url.'">'. __('Activate', 'wp-base'). '</a>';
						}

						do_action( 'app_addon_help_link', $filename );
						echo '</div>';
						?>
					</td>
					<td class="column-description desc">
						<?php
							echo '<div class="plugin-description">'. $addon_data["Description"] . '</div>';
							echo '<div class="second plugin-version-author-uri">'. __('Version', 'wp-base'). ' ' . $addon_data["Version"];
							if ( $addon_data["Author"] )
								echo ' | By ' . $addon_data["Author"];
							echo '</div>';
						?>
					</td>
				</tr>
		<?php } ?>
			</table>
		</form>
	<?php } ?>
	</div>
</div>
	<?php
	}

	private function showcase(){
		include_once( WPBASE_PLUGIN_DIR . '/includes/constant-data.php' );
		$desc = WpBConstant::addon_showcase_desc();
		wpb_infobox( $desc[0] );

?>		<div id="wpb-addon-showcase"><?php
			echo wp_kses_post( $this->get_feed( ) );
?>		</div><?php
	}

	private function get_feed( ) {
		$id		= 'wpbase_addons_feed';
		$id2	= 'wpbase_addons_feed_backup';
		$cache	= get_transient( $id );

		if ( false === $cache ) {
			$url = 'https://wp-base.com/?feed=wpbaseaddons';

			$feed = wp_remote_get( esc_url_raw( $url ), array( 'sslverify' => false, 'timeout' => 20 ) );

			if ( is_wp_error( $feed ) || ! isset( $feed['body'] ) || strpos( $feed['body'], 'wpb-addon-title' ) === false ) {
				if ( ! $cache = get_transient( $id2 ) ) {
					$cache = '<div class="error"><p>' . __( 'There was an error retrieving the addons list from the server. Please try again later.', 'wp-base' ) . '</div>';
				}
			} else {
				$cache = wp_remote_retrieve_body( $feed );
				set_transient( $id, $cache, 3600 );
				set_transient( $id2, $cache, 7200 );
			}
		}

		return $cache;
	}

	/**
	 * Add tabs to Tools
	 * @uses appointments_tools_tabs filter
	 * @return string
	 */
	public function add_tab( $tabs ) {
		$tabs['licenses']	= __('Licenses', 'wp-base');
		return $tabs;
	}

	/**
	 * License settings HTML code
	 */
	public function license_settings() {

		wpb_admin_access_check( 'manage_licenses' );

		$extensions = $this->get_all_active_extensions();
		wpb_infobox( __('To activate a license on this website, enter the provided license key to the related field below and click Activate button.', 'wp-base' ) );

	?>
	<div class="postbox">
		<div class="inside">
			<table class="app-license form-table striped">
				<thead>
					<tr>
						<th class="app-license-name" style="width:30%"><?php _e( 'Addon or Product Name', 'wp-base' ) ?></th>
						<th class="app-license-key" style="width:30%"><?php _e( 'License Key', 'wp-base' ) ?></th>
						<th class="app-license-status" style="width:15%"><?php _e( 'Status', 'wp-base' ) ?></th>
						<th class="app-license-actions" style="width:25%"><?php _e( 'Action', 'wp-base' ) ?></th>
					</tr>
				</thead>
		<?php
			if ( WPB_DEV ) {
				$file = WP_PLUGIN_DIR. '/'. WPB_PLUGIN_BASENAME;
				$data = get_file_data( $file, wpb_default_headers(), 'plugin' );
				$data['Name'] = WPB_NAME;
				$this->license_form( $file, $data );
			}

			foreach ( (array)$extensions as $file => $identifier ) {

				if ( empty( $file ) ) {
					continue;
				}

				$addon_path = realpath( WPB_ADDON_DIR );

				if ( $addon_path && strpos( $file, $addon_path ) !== false ) {
					continue;
				}

				$data = get_file_data( $file, wpb_default_headers(), 'plugin' );

				if ( empty( $data['ID'] ) ) {
					continue;
				}

				if ( ! empty( $data['Category'] ) && strtolower( $data['Category'] ) == 'free' ) {
					continue;
				}

				$this->license_form( $file, $data );

			} ?>
			</table>
		</div>
	</div>
	<?php
	}

	private function license_form( $file, $data ) {
		$name			= trim( str_replace( 'WP BASE Addon: ','', $data['Name'] ) );
		$license		= get_option( 'wpb_license_key_' . $data['ID'] );
		$status_option  = get_option( 'wpb_license_status_' . $data['ID'] );
		$status			= 'valid' === $status_option;
		$status_text	= $status ? '<span class="dashicons dashicons-yes" title="'.__( 'Valid and active', 'wp-base' ).'" ></span>' : '<span class="dashicons dashicons-no" title="'.__( 'Invalid, expired or inactive', 'wp-base' ).'" ></span>' ;
	?>	<form class="app-form app-license" method="post">
			<tr>
				<th class="app-license-name" scope="row" ><?php echo $name; ?></th>
				<td class="app-license-key">
					<input name="<?php echo 'wpb_license_key_' . $data['ID'] ?>" type="text" class="app-1" value="<?php echo $license; ?>" />
				</td>

				<td class="app-license-status">
					<span><?php echo $status_text ?></span>
				</td>

				<td class="app-license-actions">
					<input type="hidden" name="wpb_activation_file_name" value="<?php esc_attr_e( $file ) ?>" />
			<?php if ( ! $status ) { ?>
					<button class="activate" ><?php _e('Activate', 'wp-base' ) ?></button>
					<input type="hidden" name="activate" value="1" />
					<?php wp_nonce_field( 'wpb_license_activate_nonce', 'wpb_license_activate_nonce' ); ?>
			<?php } else { ?>
					<button class="deactivate" ><?php _e('Deactivate', 'wp-base' ) ?></button>
					<input type="hidden" name="deactivate" value="1" />
					<?php wp_nonce_field( 'wpb_license_deactivate_nonce', 'wpb_license_deactivate_nonce' ); ?>
			<?php } ?>
				</td>
			</tr>
		</form>
<?php
	}

	public function sanitize_license( $option, $old_value, $value ) {
		if ( 'wpb_license_key_' === substr( $option, 0, 16 ) ) {

			if ( $old_value && $old_value != $value ) {
				$id = str_replace( 'wpb_license_key_', '', $option );
				delete_option( 'wpb_license_status_' . $id );
			}
		}
	}

	public function display_notice( ) {
		if ( empty( $_GET['sl_message'] ) ) {
			return;
		}

		wpb_notice( wpb_clean( urldecode( $_GET['sl_message'] ) ), 'error' );
	}

	public function display_success( ) {
		if ( empty( $_GET['sl_success'] ) ) {
			return;
		}

		wpb_notice( wpb_clean( urldecode( $_GET['sl_success'] ) ) );
	}

	public function handle_license_activation() {

		if ( empty( $_POST['wpb_activation_file_name'] ) ) {
			return;
		}

		$base_url		= admin_url( 'admin.php?page=app_tools&tab=licenses' );
		$file			= wpb_clean( $_POST['wpb_activation_file_name'] );
		$data			= get_file_data( $file, wpb_default_headers(), 'plugin' );
		$name			= trim( str_replace( 'WP BASE Addon: ','', $data['Name'] ) );
		$option_name	= 'wpb_license_key_' . $data['ID'];

		if ( isset( $_POST['activate'] ) ) {

			if( ! check_admin_referer( 'wpb_license_activate_nonce', 'wpb_license_activate_nonce' ) ) {
				wpb_notice( 'unauthorised', 'error' );
				return;
			}
			
			$license = isset( $_POST[$option_name] ) && update_option( $option_name, trim($_POST[$option_name]) ) 
					   ? trim($_POST[$option_name]) 
					   : get_option( $option_name );
			
			if ( empty( $license ) ) {
				wp_redirect( wpb_add_query_arg( array( 'sl_activation' => 'false', 'sl_message' => urlencode( __('Empty license key','wp-base') ) ), $base_url ) );
				exit();
			}

			$api_params = array(
				'edd_action'=> 'activate_license',
				'license'	=> $license,
				'item_id'	=> $data['ID'],
				'url'		=> home_url(),
			);

			$response = wp_remote_post( WPBASE_URL, array( 'timeout' => 15, 'sslverify' => false, 'body' => $api_params ) );

			if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {

				if ( is_wp_error( $response ) ) {
					$message = $response->get_error_message();
				} else {
					$message = __( 'An error occurred, please try again', 'wp-base' );
				}

			} else {

				$license_data = json_decode( wp_remote_retrieve_body( $response ) );

				if ( false === $license_data->success ) {

					switch( $license_data->error ) {

						case 'expired' :

							$message = sprintf(
								__( 'Your license key expired on %s.', 'wp-base' ),
								date_i18n( $this->a->dt_format, strtotime( $license_data->expires, $this->a->_time ) )
							);
							break;

						case 'revoked' :

							$message = sprintf( __( 'This license key has been disabled. If you think this is a mistake, please contact us using our website %s', 'wp-base' ), WPBASE_URL );
							break;

						case 'missing' :

							$message = __( 'Invalid license', 'wp-base' );
							break;

						case 'invalid' :
						case 'site_inactive' :

							$message = __( 'Your license is not active for this URL', 'wp-base' );
							break;

						case 'item_name_mismatch' :

							$message = sprintf( __( 'This appears to be an invalid license key for %s', 'wp-base' ), $data['Name'] );
							break;

						case 'no_activations_left':

							$message = __( 'Your license key has reached its activation limit', 'wp-base' );
							break;

						default :

							$message = __( 'An error occurred, please try again', 'wp-base' );
							break;
					}
				}
			}

			if ( ! empty( $message ) ) {
				wp_redirect( wpb_add_query_arg( array( 'sl_activation' => 'false', 'sl_message' => urlencode( $message ) ), $base_url ) );
				exit();
			}

			update_option( 'wpb_license_status_'.$data['ID'], $license_data->license );
			$msg = sprintf( __( '%s license activated','wp-base' ), $name );
			wp_redirect( wpb_add_query_arg( array( 'sl_message' =>false, 'sl_success'=> urlencode( $msg ) ), $base_url ) );
			exit();
		}

		if ( isset( $_POST['deactivate'] ) ) {

			if( ! check_admin_referer( 'wpb_license_deactivate_nonce', 'wpb_license_deactivate_nonce' ) ) {
				wpb_notice( 'unauthorised', 'error' );
				return;
			}
			
			$license = isset( $_POST[$option_name] ) && update_option( $option_name, trim($_POST[$option_name]) ) 
					   ? trim($_POST[$option_name]) 
					   : get_option( $option_name );
			
			if ( empty( $license ) ) {
				delete_option( 'wpb_license_status_'.$data['ID'] );
				$msg = sprintf( __( '%s license cleared','wp-base' ), $name );
				wp_redirect( wpb_add_query_arg( array( 'sl_activation' => 'false', 'sl_message'=> false, 'sl_success' => urlencode( $msg ) ), $base_url ) );
				exit();
			}
			
			$api_params = array(
				'edd_action'=> 'deactivate_license',
				'license'	=> $license,
				'item_id'	=> $data['ID'],
				'url'		=> home_url(),
			);

			$response = wp_remote_post( WPBASE_URL, array( 'timeout' => 15, 'sslverify' => false, 'body' => $api_params ) );

			if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {

				if ( is_wp_error( $response ) ) {
					$message = $response->get_error_message();
				} else {
					$message = __( 'An error occurred, please try again', 'wp-base' );
				}

				wp_redirect( wpb_add_query_arg( array( 'sl_activation' => 'false', 'sl_message' => urlencode( $message ) ), $base_url ) );
				exit();
			}

			$license_data = json_decode( wp_remote_retrieve_body( $response ) );

			if ( $license_data->license == 'deactivated' ) {
				delete_option( 'wpb_license_status_'.$data['ID'] );
				$msg = sprintf( __( '%s license deactivated','wp-base' ), $name );
				$reply = array( 'sl_message' =>false, 'sl_success'=> urlencode( $msg ) );
			} else {
				$msg = sprintf( __( '%s license could not be deactivated. Reason: %s','wp-base' ), $name, $license_data->license );
				$reply = array( 'sl_message' => urlencode( $msg ), 'sl_success'=> false );
			}

			wp_redirect( wpb_add_query_arg( $reply, $base_url ) );
			exit();
		}

	}
}
	BASE('Addons')->add_hooks();

}


