<?php

namespace Api\Wpl\routes;

use Api\Wpl\managers\AboutManager;
use Api\Wpl\managers\ListingManager;
use Api\Wpl\managers\UserManager;
use Api\Wpl\models\core\Auth\AuthToken;
use Api\Wpl\models\core\Properties\Properties;
use Api\Wpl\models\core\Properties\PropertiesImageUpload;
use Api\Wpl\models\core\Properties\PropertiesMapper;
use Api\Wpl\models\core\Properties\PropertiesSave;
use Api\Wpl\models\core\Properties\PropertiesDelete;
use Api\Wpl\models\core\User\Wpluser;
use Api\Wpl\models\imported\Email\Email;
use Api\Wpl\models\imported\Email\Sender;
use Api\Wpl\models\imported\Idx\Idx;
use Api\Wpl\models\imported\Map\Map;
use Api\Wpl\models\imported\Notifications\CronManager;
use Api\Wpl\models\imported\Notifications\OneSignalSetting;
use Api\Wpl\models\imported\Twilio\Twilio;
use Api\Wpl\routes\api\Api;
use Api\Wpl\routes\api\CronApi;
use Api\Wpl\routes\api\FavoriteApi;
use Api\Wpl\routes\api\FlexApi;
use Api\Wpl\routes\api\SettingsApi;
use Api\Wpl\routes\api\PropertiesApi;
use Api\Wpl\routes\api\SearchApi;
use Api\Wpl\routes\api\UserApi;
use Api\Wpl\validation\AuthValidation;

use Api\Wpl\models\core\Auth\Auth;
use Api\Wpl\models\core\Database\Database as DB;

class route
{

    /**
     * @var string The namespace to add to the api call
     */
    private $namespace = 'wpl_api/v1';
    private $namespace_v2 = 'wpl_api/v2';
    private $namespace_v3 = 'wpl_api/v3';

    /**
     * @var ListingManager
     */
    private $listing;

    /**
     * @var UserManager
     */
    private $auth;

    /**
     * @var Properties
     */
    private $properties;


    /**
     * route constructor.
     *
     * @param ListingManager $listing
     * @param Properties $properties
     * @param UserManager $auth
     */
    public function __construct(
        ListingManager $listing,
        Properties $properties,
        UserManager $auth
    ) {
        $this->listing = $listing;
        $this->properties = $properties;
        $this->auth = $auth;
    }



    public function register_route($route, $method, $api, $action, $defaults = [], $namespace = null) {
        $parameters = Api::getParameters($api, $action);
        $args = [];
        foreach ($parameters as $parameter) {
            // ignore WP_REST_Request object
            if($parameter['type'] == 'WP_REST_Request') {
                continue;
            }
            if(strpos($parameter['type'] ?? '', '\Auth') !== false) {
                continue;
            }
            $arg = [
                'required' => $parameter['allowsNull'] == false,
                'description' => !empty($parameter['description']) ? $parameter['description'] : '',
            ];
            if(!empty($parameter['type'])) {
                $arg['type'] = $parameter['type'];
            }
            $args[$parameter['name']] = $arg;
        }
        $options = array(
            'methods' => $method,
            'callback' => function (\WP_REST_Request $request) use ($api, $action, $defaults) {
                return Api::runAction($api, $action, $defaults, $request);
            },
            'args' => $args,
            'permission_callback' => '__return_true',
        );
        if(!empty($namespace)) {
            register_rest_route($namespace, $route, $options);
            return;
        }
        register_rest_route($this->namespace_v2, $route, $options);
    }
    // Register our routes.
    public function register_routes_v2()
    {
        $this->register_route('/app-settings', 'GET', SettingsApi::class, 'appSettingsAction');
        $this->register_route('/crm-url', 'GET', SettingsApi::class, 'crmURLAction');
        $this->register_route('/listing_types', 'GET', FlexApi::class, 'listingTypesAction');
        $this->register_route('/flex/listing-types', 'GET', FlexApi::class, 'listingTypesAction');
        $this->register_route('/flex/property-types', 'GET', FlexApi::class, 'propertyTypesAction');
        $this->register_route('/flex/(?P<kind>[\d]+)/(?P<type>[\w]+)', 'GET', FlexApi::class, 'getFlexFieldsAction');
        $this->register_route('/get-properties-count', 'POST', PropertiesApi::class, 'getPropertiesCountAction');
        $this->register_route('/listings', 'POST', PropertiesApi::class, 'listingsAction');
        $this->register_route('/list-properties', 'POST', PropertiesApi::class, 'listPropertiesAction');
        $this->register_route('/list-properties-dynamic', 'POST', PropertiesApi::class, 'listPropertiesDynamicAction');
        $this->register_route('/add-listing', 'POST', PropertiesApi::class, 'addListingAction');
        $this->register_route('/edit-listing', 'POST', PropertiesApi::class, 'editListingAction');
        $this->register_route('/get-fields', 'GET', FlexApi::class, 'getFieldsAction');
        $this->register_route('/get-filter-fields', 'GET', FlexApi::class, 'getFilterFieldsAction');
        $this->register_route('/get-filter-fields-options/(?P<id>[\d]+)', 'GET', FlexApi::class, 'getFilterFieldsOptionsAction');
        $this->register_route('/get-user-fields', 'GET', FlexApi::class, 'getUserFieldsAction');
        $this->register_route('/property/(?P<id>[\d]+)/agent', 'GET', PropertiesApi::class, 'getAgentAction');
        $this->register_route('/property/(?P<id>[\d]+)/multi-agents', 'GET', PropertiesApi::class, 'getMultiAgentsAction');
        $this->register_route('/property/(?P<id>[\d]+)/items/(?P<item_type>[\w]+)(/*)(?P<item_cat>[\w]*)', 'GET', PropertiesApi::class, 'getItemsAction');
        $this->register_route('/property/(?P<id>[\d]+)/raw', 'GET', PropertiesApi::class, 'getRawInfoAction');
        $this->register_route('/property/(?P<id>[\d]+)/similar', 'GET', PropertiesApi::class, 'getSimilarAction');
        $this->register_route('/property/(?P<id>[\d]+)/prev-next', 'GET', PropertiesApi::class, 'getPrevNextAction');
        $this->register_route('/property/(?P<id>[\d]+)/kind', 'GET', PropertiesApi::class, 'getKindAction');
        $this->register_route('/categories/property', 'GET', PropertiesApi::class, 'getCategoriesAction', ['kind' => 0]);
        $this->register_route('/categories/neighborhood', 'GET', PropertiesApi::class, 'getCategoriesAction', ['kind' => 4]);
        $this->register_route('/categories/complex', 'GET', PropertiesApi::class, 'getCategoriesAction', ['kind' => 1]);
        $this->register_route('/property/kinds', 'GET', PropertiesApi::class, 'getKindsAction', ['table' => 'wpl_properties']);
        $this->register_route('/property/last-id', 'GET', PropertiesApi::class, 'getLastIdAction');
        $this->register_route('/property-listing-link', 'GET', PropertiesApi::class, 'getListingLinkAction');
        $this->register_route('/detailed', 'POST', PropertiesApi::class, 'detailedAction');
        $this->register_route('/get-location/(?P<type>[a-zA-Z0-9-_-]+)', 'GET', PropertiesApi::class, 'getLocationAction');
        $this->register_route('/version', 'GET', SettingsApi::class, 'versionAction');
        $this->register_route('/register', 'POST', UserApi::class, 'registerAction');
        $this->register_route('/login', 'POST', UserApi::class, 'loginAction');
        $this->register_route('/reset-password', 'POST', UserApi::class, 'resetPasswordAction');
        $this->register_route('/change-password', 'POST', UserApi::class, 'changePasswordAction');
        $this->register_route('/filter', 'GET', SearchApi::class, 'filterAction');
        $this->register_route('/is-favorite/(?P<id>[\d]+)/find', 'GET', FavoriteApi::class, 'isFavoriteFindAction');
        $this->register_route('/add-favorite', 'POST', FavoriteApi::class, 'addFavoriteAction');
        $this->register_route('/remove-favorite', 'POST', FavoriteApi::class, 'removeFavoriteAction');
        $this->register_route('/favorite-list', 'POST', FavoriteApi::class, 'favoriteListAction');
        $this->register_route('/favorite-get-pids/(?P<id>[\d]+)', 'GET', FavoriteApi::class, 'favoriteGetPidsAction');
        $this->register_route('/compare-list', 'GET', FavoriteApi::class, 'compareListAction');
        $this->register_route('/save-search-list/(?P<id>[\d]+)', 'GET', SearchApi::class, 'saveSearchListAction');
        $this->register_route('/add-search', 'POST', SearchApi::class, 'addSearchAction');
        $this->register_route('/search-list', 'POST', SearchApi::class, 'searchListAction');
        $this->register_route('/remove-save-search', 'POST', SearchApi::class, 'removeSaveSearchAction');
        $this->register_route('/sort-options', 'GET', FlexApi::class, 'sortOptionsAction');
        $this->register_route('/get-sort-options/(?P<kind>[\d]+)', 'GET', FlexApi::class, 'getSortOptionsAction');
        $this->register_route('/check-token', 'GET', SettingsApi::class, 'checkTokenAction');
        $this->register_route('/create-token', 'POST', SettingsApi::class, 'createTokenAction');
        $this->register_route('/reset-token', 'GET', SettingsApi::class, 'resetTokenAction');
        $this->register_route('/save-sms-setting', 'POST', SettingsApi::class, 'saveSmsSettingsAction');
        $this->register_route('/sms-setting', 'GET', SettingsApi::class, 'smsSettingsAction');
        $this->register_route('/save-smtp-setting', 'POST', SettingsApi::class, 'saveSmtpSettingsAction');
        $this->register_route('/send-mail', 'POST', SettingsApi::class, 'sendEmailAction');
        $this->register_route('/reset-sms-setting', 'GET', SettingsApi::class, 'resetSmsSettingsAction');
        $this->register_route('/save-map-setting', 'POST', SettingsApi::class, 'saveMapSettingsAction');
        $this->register_route('/map-settings', 'GET', SettingsApi::class, 'mapSettingsAction');
        $this->register_route('/get-settings', 'POST', SettingsApi::class, 'getSettingsAction');
        $this->register_route('/unit-types', 'GET', SettingsApi::class, 'unitTypesAction');
        $this->register_route('/price-units', 'GET', SettingsApi::class, 'priceUnitsAction');
        $this->register_route('/wpl-users-options', 'GET', SettingsApi::class, 'wplUsersOptionsAction');
        $this->register_route('/add-listing/files', 'POST', PropertiesApi::class, 'addFileAction');
        $this->register_route('/add-listing/remove-file', 'POST', PropertiesApi::class, 'removeFileAction');
        $this->register_route('/add-about', 'POST', SettingsApi::class, 'addAboutAction');
        $this->register_route('/get-about', 'GET', SettingsApi::class, 'getAboutAction');
        $this->register_route('/get-mail-setting', 'GET', SettingsApi::class, 'getMailSettingAction');
        $this->register_route('/save-onesignal', 'POST', SettingsApi::class, 'saveOneSignalAction');
        $this->register_route('/get-onesignal-setting', 'GET', SettingsApi::class, 'getOneSignalSettingAction');
        $this->register_route('/cron-jobs', 'GET', CronApi::class, 'cronJobsAction');
        $this->register_route('/delete-listings', 'POST', PropertiesApi::class, 'deleteListingsAction');
        $this->register_route('/agents', 'GET', UserApi::class, 'agentsAction');
        $this->register_route('/agent-list', 'POST', UserApi::class, 'agentListAction');
        $this->register_route('/list-currency', 'GET', FlexApi::class, 'listCurrencyAction');
        $this->register_route('/list-area-unit', 'GET', FlexApi::class, 'listAreaUnitAction');
        $this->register_route('/edit-profile', 'POST', UserApi::class, 'editProfileAction');
        $this->register_route('/user/(?P<id>[\d]+)/raw', 'GET', UserApi::class, 'getRawInfoAction');
        $this->register_route('/users', 'GET', UserApi::class, 'getUsersAction');
        $this->register_route('/user-info', 'POST', UserApi::class, 'userInfoAction');
        $this->register_route('/user-exists', 'POST', UserApi::class, 'userExistsAction');
        $this->register_route('/delete-account', 'POST', UserApi::class, 'deleteAccountAction');
        $this->register_route('/deactivate-account', 'POST', UserApi::class, 'deactivateAccountAction');
        $this->register_route('/check-access', 'POST', UserApi::class, 'checkAccessAction');
        $this->register_route('/check-addon/(?P<addon_name>[a-zA-Z0-9-_-]+)', 'GET', SettingsApi::class, 'wplCheckAddonAction');
        $this->register_route('/get-wpl-setting', 'POST', SettingsApi::class, 'getWplSettingAction');
        $this->register_route('/property/(?P<id>[\d]+)/visited', 'GET', PropertiesApi::class, 'propertyVisitedAction');
        $this->register_route('/property-image', 'POST', PropertiesApi::class, 'getPropertyImage');
        $this->register_route('/property/qrcode', 'POST', PropertiesApi::class, 'getPropertyQRCodeAction');

        //======================================
        if (\wpl_global::check_addon('crm')) {
            require_once dirname(__FILE__) . "/zapier.php";
        }
    }

    public function register_routes_v3()
    {
        $this->register_route('/app-settings', 'GET', SettingsApi::class, 'appSettingsAction', [], $this->namespace_v3);
        $this->register_route('/listing_types', 'GET', FlexApi::class, 'listingTypesAction', [], $this->namespace_v3);
        $this->register_route('/flex/listing-types', 'GET', FlexApi::class, 'listingTypesAction', [], $this->namespace_v3);
        $this->register_route('/flex/property-types', 'GET', FlexApi::class, 'propertyTypesAction', [], $this->namespace_v3);
        $this->register_route('/flex/(?P<kind>[\d]+)/(?P<type>[\w]+)', 'GET', FlexApi::class, 'getFlexFieldsAction', [], $this->namespace_v3);
        $this->register_route('/listings', 'POST', PropertiesApi::class, 'listingsV3Action', [], $this->namespace_v3);
        $this->register_route('/list-properties', 'POST', PropertiesApi::class, 'listPropertiesAction', [], $this->namespace_v3);
        $this->register_route('/list-properties-dynamic', 'POST', PropertiesApi::class, 'listPropertiesDynamicAction', [], $this->namespace_v3);
        $this->register_route('/add-listing', 'POST', PropertiesApi::class, 'addListingAction', [], $this->namespace_v3);
        $this->register_route('/edit-listing', 'POST', PropertiesApi::class, 'editListingAction', [], $this->namespace_v3);
        $this->register_route('/get-fields', 'GET', FlexApi::class, 'getFieldsAction', [], $this->namespace_v3);
        $this->register_route('/property/(?P<id>[\d]+)/agent', 'GET', PropertiesApi::class, 'getAgentAction', [], $this->namespace_v3);
        $this->register_route('/property/(?P<id>[\d]+)/items/(?P<item_type>[\w]+)(/*)(?P<item_cat>[\w]*)', 'GET', PropertiesApi::class, 'getItemsAction', [], $this->namespace_v3);
        $this->register_route('/property/(?P<id>[\d]+)/raw', 'GET', PropertiesApi::class, 'getRawInfoAction', [], $this->namespace_v3);
        $this->register_route('/property/(?P<id>[\d]+)/similar', 'GET', PropertiesApi::class, 'getSimilarAction', [], $this->namespace_v3);
        $this->register_route('/property/(?P<id>[\d]+)/prev-next', 'GET', PropertiesApi::class, 'getPrevNextAction', [], $this->namespace_v3);
        $this->register_route('/property/(?P<id>[\d]+)/kind', 'GET', PropertiesApi::class, 'getKindAction', [], $this->namespace_v3);
        $this->register_route('/categories/property', 'GET', PropertiesApi::class, 'getCategoriesAction', ['kind' => 0], $this->namespace_v3);
        $this->register_route('/categories/complex', 'GET', PropertiesApi::class, 'getCategoriesAction', ['kind' => 1], $this->namespace_v3);
        $this->register_route('/property/kinds', 'GET', PropertiesApi::class, 'getKindsAction', ['table' => 'wpl_properties'], $this->namespace_v3);
        $this->register_route('/property/last-id', 'GET', PropertiesApi::class, 'getLastIdAction', [], $this->namespace_v3);
        $this->register_route('/detailed', 'POST', PropertiesApi::class, 'detailedAction', [], $this->namespace_v3);
        $this->register_route('/preview', 'POST', PropertiesApi::class, 'previewAction', [], $this->namespace_v3);
        $this->register_route('/version', 'GET', SettingsApi::class, 'versionAction', [], $this->namespace_v3);
        $this->register_route('/register', 'POST', UserApi::class, 'registerAction', [], $this->namespace_v3);
        $this->register_route('/login', 'POST', UserApi::class, 'loginAction', [], $this->namespace_v3);
        $this->register_route('/reset-password', 'POST', UserApi::class, 'resetPasswordAction', [], $this->namespace_v3);
        $this->register_route('/change-password', 'POST', UserApi::class, 'changePasswordAction', [], $this->namespace_v3);
        $this->register_route('/filter', 'GET', SearchApi::class, 'filterAction', [], $this->namespace_v3);
        $this->register_route('/is-favorite/(?P<id>[\d]+)/find', 'GET', FavoriteApi::class, 'isFavoriteFindAction', [], $this->namespace_v3);
        $this->register_route('/add-favorite', 'POST', FavoriteApi::class, 'addFavoriteAction', [], $this->namespace_v3);
        $this->register_route('/remove-favorite', 'POST', FavoriteApi::class, 'removeFavoriteAction', [], $this->namespace_v3);
        $this->register_route('/favorite-list', 'POST', FavoriteApi::class, 'favoriteListAction', [], $this->namespace_v3);
        $this->register_route('/compare-list', 'POST', FavoriteApi::class, 'compareListAction', [], $this->namespace_v3);
        $this->register_route('/add-search', 'POST', SearchApi::class, 'addSearchAction', [], $this->namespace_v3);
        $this->register_route('/search-list', 'POST', SearchApi::class, 'searchListAction', [], $this->namespace_v3);
        $this->register_route('/remove-save-search', 'POST', SearchApi::class, 'removeSaveSearchAction', [], $this->namespace_v3);
        $this->register_route('/sort-options', 'GET', FlexApi::class, 'sortOptionsAction', [], $this->namespace_v3);
        $this->register_route('/check-token', 'GET', SettingsApi::class, 'checkTokenAction', [], $this->namespace_v3);
        $this->register_route('/create-token', 'POST', SettingsApi::class, 'createTokenAction', [], $this->namespace_v3);
        $this->register_route('/reset-token', 'GET', SettingsApi::class, 'resetTokenAction', [], $this->namespace_v3);
        $this->register_route('/save-sms-setting', 'POST', SettingsApi::class, 'saveSmsSettingsAction', [], $this->namespace_v3);
        $this->register_route('/sms-setting', 'GET', SettingsApi::class, 'smsSettingsAction', [], $this->namespace_v3);
        $this->register_route('/save-smtp-setting', 'POST', SettingsApi::class, 'saveSmtpSettingsAction', [], $this->namespace_v3);
        $this->register_route('/send-mail', 'POST', SettingsApi::class, 'sendEmailAction', [], $this->namespace_v3);
        $this->register_route('/reset-sms-setting', 'GET', SettingsApi::class, 'resetSmsSettingsAction', [], $this->namespace_v3);
        $this->register_route('/save-map-setting', 'POST', SettingsApi::class, 'saveMapSettingsAction', [], $this->namespace_v3);
        $this->register_route('/map-settings', 'GET', SettingsApi::class, 'mapSettingsAction', [], $this->namespace_v3);
        $this->register_route('/get-settings', 'POST', SettingsApi::class, 'getSettingsAction', [], $this->namespace_v3);
        $this->register_route('/unit-types', 'GET', SettingsApi::class, 'unitTypesAction', [], $this->namespace_v3);
        $this->register_route('/price-units', 'GET', SettingsApi::class, 'priceUnitsAction', [], $this->namespace_v3);
        $this->register_route('/wpl-users-options', 'GET', SettingsApi::class, 'wplUsersOptionsAction', [], $this->namespace_v3);
        $this->register_route('/add-listing/files', 'POST', PropertiesApi::class, 'addFileAction', [], $this->namespace_v3);
        $this->register_route('/add-listing/remove-file', 'POST', PropertiesApi::class, 'removeFileAction', [], $this->namespace_v3);
        $this->register_route('/add-about', 'POST', SettingsApi::class, 'addAboutAction', [], $this->namespace_v3);
        $this->register_route('/get-about', 'GET', SettingsApi::class, 'getAboutAction', [], $this->namespace_v3);
        $this->register_route('/get-mail-setting', 'GET', SettingsApi::class, 'getMailSettingAction', [], $this->namespace_v3);
        $this->register_route('/save-onesignal', 'POST', SettingsApi::class, 'saveOneSignalAction', [], $this->namespace_v3);
        $this->register_route('/get-onesignal-setting', 'GET', SettingsApi::class, 'getOneSignalSettingAction', [], $this->namespace_v3);
        $this->register_route('/cron-jobs', 'GET', CronApi::class, 'cronJobsAction', [], $this->namespace_v3);
        $this->register_route('/delete-listings', 'POST', PropertiesApi::class, 'deleteListingsAction', [], $this->namespace_v3);
        $this->register_route('/agents', 'GET', UserApi::class, 'agentsAction', [], $this->namespace_v3);
        $this->register_route('/list-currency', 'GET', FlexApi::class, 'listCurrencyAction', [], $this->namespace_v3);
        $this->register_route('/list-area-unit', 'GET', FlexApi::class, 'listAreaUnitAction', [], $this->namespace_v3);
        $this->register_route('/edit-profile', 'POST', UserApi::class, 'editProfileAction', [], $this->namespace_v3);
        $this->register_route('/user/(?P<id>[\d]+)/raw', 'GET', UserApi::class, 'getRawInfoAction', [], $this->namespace_v3);
        $this->register_route('/user-info', 'POST', UserApi::class, 'userInfoAction', [], $this->namespace_v3);
        $this->register_route('/user-exists', 'POST', UserApi::class, 'userExistsAction', [], $this->namespace_v3);
        $this->register_route('/delete-account', 'POST', UserApi::class, 'deleteAccountAction', [], $this->namespace_v3);
        $this->register_route('/deactivate-account', 'POST', UserApi::class, 'deactivateAccountAction', [], $this->namespace_v3);
        $this->register_route('/check-addon/(?P<addon_name>[a-zA-Z0-9-_-]+)', 'GET', SettingsApi::class, 'wplCheckAddonAction', [], $this->namespace_v3);
        $this->register_route('/get-wpl-setting', 'POST', SettingsApi::class, 'getWplSettingAction', [], $this->namespace_v3);
        $this->register_route('/property/(?P<id>[\d]+)/visited', 'GET', PropertiesApi::class, 'propertyVisitedAction', [], $this->namespace_v3);
        $this->register_route('/property-image', 'POST', PropertiesApi::class, 'getPropertyImage', [], $this->namespace_v3);
        $this->register_route('/property/qrcode', 'POST', PropertiesApi::class, 'getPropertyQRCodeAction', [], $this->namespace_v3);
    }
    public function register_routes()
    {
        $this->register_routes_v2();
        $this->register_routes_v3();
        register_rest_route($this->namespace, '/version', array(
            'methods' => 'GET',
            'callback' => function () {
                wp_send_json_success(array(
                    'plugin_name' => WPL_API_PLUGIN,
                    'plugin_version' => WPLAPI_VERSION
                ));
            },
            'permission_callback' => '__return_true',
        ));

        register_rest_route($this->namespace, '/check-token', array(
            'methods' => 'GET',
            'callback' => function () {
                AuthToken::checkToken(wp_get_referer());
            },
            'permission_callback' => '__return_true',
        ));

        register_rest_route($this->namespace, '/create-token', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {

                if (!email_exists($request->get_param('email'))) {
                    wp_send_json_error(array(
                        'message' => 'Email does not exist.'
                    ));
                }

                $author_obj = get_user_by('email', $request->get_param('email'));

                if (!$author_obj->allcaps['administrator']) {
                    wp_send_json_error(array(
                        'message' => 'Only administrators can access this feature.'
                    ));
                }

                return (new AuthToken($request))
                    ->create();
            },
            'permission_callback' => '__return_true',
            'args' => array(
                'email' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'The user\'s email address',
                    'format' => 'email'
                ),
                'username' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'The client\'s username',
                ),
                'password' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'The client\'s password',
                )
            )
        ));

        register_rest_route($this->namespace, '/reset-token', array(
            'methods' => 'GET',
            'callback' => function () {
                return AuthToken::remove();
            },
            'permission_callback' => array($this, 'permissionCheck'),
        ));

        register_rest_route($this->namespace, '/get-fields', array(
            'methods' => 'GET',
            'callback' => function () {
                $kind = 0;
                $propertyId = $_GET['id'];
                $property = \wpl_property::get_property_raw_data($propertyId);
                $categories = \wpl_flex::get_categories(1, $kind);
                $return = [];
                $items = [];
                if(!empty($propertyId)) {
                    $items = \wpl_items::get_items($property['id'], '', $property['kind'], '', 1);
                }

                foreach ($categories as $category) {
                    if(in_array($category->id, [9, 11, 31])) {
                        continue;
                    }
                    $fields = \wpl_property::get_pwizard_fields($category->id, $kind, 1);
                    if (!empty($fields)) {
                        $filtered = [];
                        foreach ($fields as $field) {
                            if($field->id >= 3000) {
                                continue;
                            }
                            if(in_array($field->id, [4, 5, 310, 311, 312, 405, 171, 41, 42, 43, 45, 51, 52, 56, 57])) {
                                continue;
                            }
                            if($field->type == 'separator') {
                                continue;
                            }
                            $units = [];
                            if (in_array($field->type, array('price', 'volume', 'area', 'length'))) {
                                if ($field->type == 'price') $units = \wpl_units::get_units(4, 1, " AND `type`='4' AND `enabled`>='1'");
                                if ($field->type == 'volume') $units = \wpl_units::get_units(3, 1, " AND `type`='3' AND `enabled`>='1'");
                                if ($field->type == 'area') $units = \wpl_units::get_units(2, 1, " AND `type`='2' AND `enabled`>='1'");
                                if ($field->type == 'length') $units = \wpl_units::get_units(1, 1, " AND `type`='1' AND `enabled`>='1'");
                            }
                            $filteredUnits = [];
                            foreach ($units as $unit) {
                                $filteredUnits[] = [
                                    'id' => $unit['id'],
                                    'name' => $unit['name'],
                                    'add_before' => empty($unit['add_before']) ? false : true,
                                ];
                            }
                            if($field->type == 'rooms'){
                                $rooms = \wpl_room_types::get_room_types();
                                $field->options = ['values' => []];
                                foreach($rooms as $room) {
                                    $field->options['values'][] = [
                                        'key' => $room['id'],
                                        'value' => $room['name'],
                                        'enabled' => $room['enabled'],
                                    ];
                                }
                            }
                            $field->options = is_array($field->options) ? $field->options : json_decode($field->options ?? '', true);
                            if(in_array($field->type, ['textarea']) || array_key_exists( 'if_zero', $field->options)) {
                                $field->options = null;
                            }
                            if(in_array($field->type, ['select'])) {
                                $field->options =  [
                                    'values' => array_values($field->options['params']),
                                ];
                            }
                            if($field->type == 'listings') {
                                $listings = \wpl_global::get_listings();
                                $field->options = ['values' => []];
                                foreach ($listings as $listing) {
                                    $field->options['values'][] = [
                                        'value' => __($listing['name'], 'real-estate-listing-realtyna-wpl'),
                                        'key' => $listing['id'],
                                    ];
                                }
                            }
                            if($field->type == 'property_types') {
                                $property_types = \wpl_global::get_property_types();
                                $field->options = ['values' => []];
                                foreach ($property_types as $property_type) {
                                    $field->options['values'][] = [
                                        'value' => __($property_type['name'], 'real-estate-listing-realtyna-wpl'),
                                        'key' => $property_type['id'],
                                    ];
                                }
                            }
                            if($field->type == 'feature') {
                                $field->options['values'] =  array_values($field->options['values']);
                            }
                            $value = $property[$field->table_column];
                            if($field->id == 53) {
                                $value = $property['googlemap_lt'] . ',' . $property['googlemap_ln'];
                            }
                            $gallery = [];
                            if($field->id == 300 && !empty($items['gallery'])) {
                                $base = wp_upload_dir();
                                $url = $base['baseurl'] . '/WPL/' . $property['id'] . '/';
                                foreach ($items['gallery'] as $image) {
                                    $image_url = $url . $image->item_name;
                                    if($image->item_cat == 'external') {
                                        $image_url = $image->item_extra3;
                                    }
                                    $gallery[] = array(
                                        'type' => 'image',
                                        'title' => $image->item_name,
                                        'url' => $image_url,
                                    );
                                }
                                foreach ($items['video'] as $video) {
                                    $re = '/src(.*)(?<link>http([^ \'"]*))/m';
                                    $str = $video->item_extra2;
                                    preg_match_all($re, $str, $matches, PREG_SET_ORDER, 0);
                                    $gallery[] = array(
                                        'type' => 'video',
                                        'title' => $video->item_name,
                                        'url' => $matches[0]['link'],
                                    );
                                }
                            }
                            /*if($field->id == 567 && !empty($items['video'])){

                            }*/
                            $filtered[] = [
                                'id' => $field->id,
                                'mandatory' => $field->mandatory,
                                'name' => $field->name,
                                'type' => $field->type,
                                'units' => $filteredUnits,
                                'options' => $field->options,
                                'table_name' => $field->table_name,
                                'table_column' => $field->table_column,
                                'listing_specific' => $field->listing_specific,
                                'property_type_specific' => $field->property_type_specific,
                                'value' => $value,
                                'value_unit' => $property[$field->table_column . '_unit'] == null ? '' : $property[$field->table_column . '_unit'],
                                'gallery' => $gallery,
                            ];
                        }
                        $return[] = [
                            'category' => $category,
                            'fields' => $filtered,
                        ];
                    }
                }
                wp_send_json_success($return);
            },
            'permission_callback' => array($this, 'permissionCheck'),
        ));

        register_rest_route($this->namespace, '/save-sms-setting', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                Twilio::save(
                    $request
                );
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'AccountSID' => array(
                    'required' => true,
                ),
                'TokenID' => array(
                    'required' => true,
                ),
                'SenderNumber' => array(
                    'required' => true,
                )
            )
        ));

        register_rest_route($this->namespace, '/sms-setting', array(
            'methods' => 'GET',
            'callback' => function () {
                Twilio::getSettings();
            },
            'permission_callback' => array($this, 'permissionCheck'),
        ));

        register_rest_route($this->namespace, '/save-smtp-setting', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                (new Email())
                    ->SaveSmtpSettings($request);
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'smtpHost' => array(
                    'required' => true,
                ),
                'smtpPort' => array(
                    'required' => true,
                ),
                'smtpSecure' => array(
                    'required' => false,
                ),
                'smtpUsername' => array(
                    'required' => true,
                ),
                'smtpPassword' => array(
                    'required' => true,
                )
            )
        ));

        register_rest_route($this->namespace, '/send-mail', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                wp_mail('david.m@realtyna.net', 'hi', 'bye');
                (new Sender())
                    ->sendEmail($request);
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'email' => array(
                    'required' => true,
                    'format' => 'email'
                ),
                'agent_email' => array(
                    'required' => true,
                    'format' => 'email'
                ),
                'user' => array(
                    'required' => true,
                ),
                'phone' => array(
                    'required' => true,
                ),
                'text' => array(
                    'required' => true,
                ),
                'listing_id' => array(
                    'required' => true,
                )
            )
        ));

        register_rest_route($this->namespace, '/reset-sms-setting', array(
            'methods' => 'GET',
            'callback' => function () {
                Twilio::delete();
            },
            'permission_callback' => array($this, 'permissionCheck'),
        ));

        register_rest_route($this->namespace, '/save-map-setting', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                Map::saveMapSettings(
                    $request
                );
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'latitude' => array(
                    'required' => true,
                ),
                'longitude' => array(
                    'required' => true,
                ),
                'zoom' => array(
                    'required' => true,
                )
            )
        ));

        register_rest_route($this->namespace, '/map-settings', array(
            'methods' => 'GET',
            'callback' => function () {
                Map::getMapSetting();
            },
            'permission_callback' => array($this, 'permissionCheck'),
        ));

        register_rest_route($this->namespace, '/register', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $this->auth
                    ->registration($request);
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'first_name' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'The client\'s first name',
                ),
                'last_name' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'The client\'s last name',
                ),
                'username' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'The client\'s username',
                ),
                'email' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'The user\'s email address',
                    'format' => 'email'
                ),
                'phone' => array(
                    'required' => true,
                    'type' => 'string'
                ),
                'password' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'The client\'s password',
                ),
            )
        ));

        register_rest_route($this->namespace, '/login', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {

                $this->auth
                    ->login($request);
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'email' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'The user\'s email address',
                    'Format' => 'email'
                ),
                'password' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'The client\'s password',
                ),
                'remember' => array(
                    'required' => false,
                    'type' => 'int'
                )
            )
        ));

        register_rest_route( $this->namespace, '/reset-password', array(
            'methods'             => 'POST',
            'callback'            => function ( \WP_REST_Request $request ) {
                $usermail = trim($request->get_param('usermail') ?? '');

                if(strpos($usermail, '@')) $user_data = \wpl_users::get_user_by('email', $usermail);
                else $user_data = \wpl_users::get_user_by('login', $usermail);

                if(!$user_data) {
                    wp_send_json_error(array(
                        'message' => 'Invalid username or email.'
                    ));
                    return;
                }

                $errors = new \WP_Error();

                /**
                 * Fires before errors are returned from a password reset request.
                 */
                do_action('lostpassword_post');

                if($errors->get_error_code()) {
                    wp_send_json_error(array(
                        'message' => $errors->get_error_message()
                    ));
                    return;
                }

                $user_login = $user_data->user_login;

                $allow = apply_filters('allow_password_reset', true, $user_data->ID);
                if(!$allow or ($allow and is_wp_error($allow))){
                    wp_send_json_error(array(
                        'message' => 'Password reset is not allowed for this user.'
                    ));
                    return;
                }

                $hashed = substr(uniqid(rand(), true), 4, 4);

                /** Set hashed key in database **/
                $query = "UPDATE `#__users` SET `user_activation_key`='$hashed' WHERE `user_login`='$user_login'";
                $result = \wpl_db::q($query, 'UPDATE');
                if (!$result) {
                    wp_send_json_error(array(
                        'message' => __('Error Occurred.', 'real-estate-listing-realtyna-wpl')
                    ));
                    return;
                }
                /** Trigger event for sending notification **/
                $parameters = array('user_activation_key'=>$hashed, 'user_id'=>$user_data->ID);
                _wpl_import('libraries.notifications.notifications');
                $notification = new \wpl_notifications('email');

                $notification->prepare(101, array());
                $user = \wpl_users::get_user($user_data->ID);
                $notification->replacements = $notification->set_replacements(array(
                    'name' => isset($user->data->wpl_data) ? $user->data->wpl_data->first_name : $user->data->display_name,
                    'link' => $hashed,
                ));
                $notification->rendered_content = $notification->render_notification_content();
                $notification->rendered_content = str_replace('Please click on this link for resetting your password.', 'Your reset code is', $notification->rendered_content);
                $notification->recipients = $notification->set_recipients(array($user->data->user_email));
                $notification->send();

                wp_send_json_success(array(
                    'message' => __('An email was sent to you to complete the password reset.', 'real-estate-listing-realtyna-wpl')
                ));
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args'                => array(
                'usermail' => array(
                    'required'    => true,
                    'type'        => 'string',
                    'description' => 'username or email',
                ),
            )
        ));
        register_rest_route($this->namespace, '/change-password', array(
            'methods'             => 'POST',
            'callback'            => function ( \WP_REST_Request $request ) {
                $usermail = trim($request->get_param('usermail') ?? '');

                if(strpos($usermail, '@')) $user_data = \wpl_users::get_user_by('email', $usermail);
                else $user_data = \wpl_users::get_user_by('login', $usermail);

                if(!$user_data) {
                    wp_send_json_error(array(
                        'message' => 'Invalid username or email.'
                    ));
                    return;
                }
                $key = trim($request->get_param('code') ?? '');
                $user_id = \wpl_users::validate_activation_key($key);
                if($user_data->ID != $user_id) {
                    wp_send_json_error(array(
                        'message' => 'Invalid code.'
                    ));
                }

                $password = urldecode($request->get_param('password'));

                /** Password is short **/
                if(strlen($password) < 6) {
                    wp_send_json_error(array(
                        'message' => __('Password should be more than 6 characters!', 'real-estate-listing-realtyna-wpl')
                    ));
                    return;
                }


                \wpl_users::set_password($user_id, $password);
                \wpl_users::update_user_option($user_id, 'default_password_nag', false, true);

                /** Empty key in database **/
                $query = "UPDATE `#__users` SET `user_activation_key`='' WHERE `ID`='$user_id'";
                \wpl_db::q($query, 'UPDATE');

                $user_data = \wpl_users::get_user($user_id);
                /** Trigger event for sending notification **/
                $parameters = array('user_id'=>$user_id, 'password'=>$password);
                \wpl_events::trigger('user_password_reset', $parameters);
                wp_send_json_success(array(
                    'message' => __('Password changed successfully.', 'wpl-api')
                ));
            },
            'permission_callback' => array( $this, 'permissionCheck' ),
            'args'                => array(
                'code'        => array(
                    'required' => true,
                    'type'     => 'int'
                ),
                'usermail'       => array(
                    'required'    => true,
                    'type'        => 'string',
                    'description' => 'username or email',
                ),
                'password'    => array(
                    'required'    => true,
                    'type'        => 'string',
                    'description' => 'The client\'s password',
                ),
            )
        ) );

        register_rest_route($this->namespace, '/listing_types', array(
            'methods' => 'GET',
            'callback' => function () {
                $this->listing
                    ->getListingTypes(true);
            },
            'permission_callback' => array($this, 'permissionCheck'),
        ));

        register_rest_route($this->namespace, '/sort-options', array(
            'methods' => 'GET',
            'callback' => function () {
                $this->listing
                    ->getSortOptions();
            },
            'permission_callback' => array($this, 'permissionCheck'),
        ));

        register_rest_route($this->namespace, '/sort-listings', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $this->properties
                    ->sortListings($request);
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'sort_id' => array(
                    'required' => true,
                    'type' => 'integer'
                ),
                'page' => array(
                    'required' => true,
                    'type' => 'integer'
                )
            )
        ));

        register_rest_route($this->namespace, '/listings', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                header('Content-Type: application/json');
                $this->listing
                    ->listings($request);
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'listing_type' => array(
                    'required' => true,
                    'type' => 'integer'
                ),
                'property_type' => array(
                    'required' => false
                ),
                'swlat' => array(
                    'required' => true,
                ),
                'swlng' => array(
                    'required' => true,
                ),
                'nelat' => array(
                    'required' => true,
                ),
                'nelng' => array(
                    'required' => true,
                ),
                'minbedroom' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'maxbedroom' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'minbathroom' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'maxbathroom' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'minprice' => array(
                    'required' => false
                ),
                'maxprice' => array(
                    'required' => false
                ),
                'minsqft' => array(
                    'required' => false
                ),
                'maxsqft' => array(
                    'required' => false
                ),
                'appliances' => array(
                    'required' => false
                ),
                'tags' => array(
                    'required' => false
                ),
                'neighborhoods' => array(
                    'required' => false
                ),
                'listing_id' => array(
                    'required' => false
                )
            ),
        ));

        register_rest_route($this->namespace, '/listings2', array(
            'methods' => 'POST',
            'callback' => function () {
                $_REQUEST['limit'] = $_POST['limit'] = 500;
                $result = $this->getListProperties();
                unset($result['info']);
                foreach ($result['properties'] as &$property) {
                    $property['image'] = '';
                    if (!empty($property['images'])) {
                        $property['image'] = $property['images'][0];
                    }
                    unset($property['images']);
                }
                wp_send_json_success($result);
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(),
        ));

        register_rest_route($this->namespace, '/list-properties2', array(
            'methods' => 'POST',
            'callback' => function () {
                wp_send_json_success($this->getListProperties());
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(),
        ));

        register_rest_route($this->namespace, '/list-properties3', array(
            'methods' => 'POST',
            'callback' => function () {
                \_wpl_import('libraries.addon_save_searches');
                $query = "SELECT `id`, `kind`, `criteria`, `last_notify_date` FROM `#__wpl_addon_save_searches` WHERE 1 AND `enabled`>='1' and user_id = 3 AND `notify_mode`='after_adding_property' ORDER BY `creation_date` DESC";
                $searches = \wpl_db::select($query, 'loadAssocList');

                $save_searches = new \wpl_addon_save_searches();

                $pid = 111944;
                foreach ($searches as $search) {
                    $search_id = $search['id'];

                    /** Update last_notify_date **/
                    $save_searches->update(array('last_notify_date' => date('Y-m-d H:i:s')), $search_id);

                    $save_searches->kind = $search['kind'];
                    $save_searches->criteria = json_decode($search['criteria'] ?? '', true);
                    $save_searches->last_notify_date = $search['last_notify_date'];
                    $news = $save_searches->search('after_adding_property');

                    if (!count($news)) continue;
                    $pids = array_keys($news);
                    if (!in_array($pid, $pids)) continue;

                    $reordered = array();
                    $reordered[$pid] = $news[$pid];
                    foreach ($news as $key => $item) {
                        if ($key != $pid) {
                            $reordered[$key] = $item;
                        }
                    }
                    \wpl_events::trigger('save_searches_new_results', array('id' => $search_id, 'new_results' => $reordered));
                }
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(),
        ));

        register_rest_route($this->namespace, '/edit-listing', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {

                $property = $this->properties->getBylistingId(
                    $request, true
                );

                $params = $request->get_params();
                $rs_data = json_decode(file_get_contents('php://input') ?? '', true);
                if ($_REQUEST['features']) $params['features'] = explode(',', $_REQUEST['features']);
                if ($rs_data['features']) $params['features'] = explode(',', $_REQUEST['features']);

                $params['address'] = [
                    'county' => $params['county'],
                    'state' => $params['state'],
                    'city' => $params['city'],
                    'street' => $params['street'],
                    'street_number' => $params['street_number'],
                    'street_suffix' => $params['street_suffix'],
                    'zipcode' => $params['zipcode'],
                ];

                $property = array_merge($property, $params);

                $property = new PropertiesMapper($property);

                (new PropertiesSave($property))->save(true);
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'listing_id' => array(
                    'required' => true,
                    'type' => 'integer'
                ),
                'user_id' => array(
                    'required' => true,
                    'type' => 'integer'
                )
            )
        ));

        register_rest_route($this->namespace, '/add-listing2', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $params = $request->get_params();
                $property_id = \wpl_property::create_property_default($params['user_id']);
                $this->editProperty($property_id, $params);
                wp_send_json_success(array(
                    'message'    => __('Property successfully added.', 'wpl-api'),
                    'listing_id' => $property_id
                ));
            },
            'permission_callback' => array($this, 'permissionCheck'),
        ));

        register_rest_route($this->namespace, '/edit-listing2', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $params = $request->get_params();
                $property_id = $params['id'];
                $this->editProperty($property_id, $params);
                wp_send_json_success(array(
                    'message'    => __('Property successfully edited.', 'wpl-api'),
                    'listing_id' => $property_id
                ));
            },
            'permission_callback' => array($this, 'permissionCheck'),
        ));

        register_rest_route($this->namespace, '/add-listing', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $params = $request->get_params();
                $rs_data = json_decode(file_get_contents('php://input') ?? '', true);
                if ($_REQUEST['features']) $params['features'] = explode(',', $_REQUEST['features']);
                if ($rs_data['features']) $params['features'] = explode(',', $_REQUEST['features']);

                $params['address'] = [
                    'county' => $params['county'],
                    'state' => $params['state'],
                    'city' => $params['city'],
                    'street' => $params['street'],
                    'street_number' => $params['street_number'],
                    'street_suffix' => $params['street_suffix'],
                    'zipcode' => $params['zipcode'],
                ];

                $insertData = new PropertiesMapper(
                    $params
                );

                (new PropertiesSave($insertData))->save();
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'listing_type' => array(
                    'required' => true,
                    'type' => 'string'
                ),
                'property_type' => array(
                    'required' => true,
                    'type' => 'string'
                ),
                'description' => array(
                    'required' => true,
                    'type' => 'string'
                ),
                'price' => array(
                    'required' => true,
                    'type' => 'number',
                ),
                'price_unit' => array(
                    'required' => true,
                    'type' => 'string'
                ),
                'price_period' => array(
                    'required' => false,
                    'type' => 'string'
                ),
                'bedrooms' => array(
                    'required' => false,
                    'type' => 'number'
                ),
                'bathrooms' => array(
                    'required' => false,
                    'type' => 'number'
                ),
                'half_bathrooms' => array(
                    'required' => false,
                    'type' => 'number'
                ),
                'square_footage' => array(
                    'required' => false,
                    'type' => 'number'
                ),
                'square_footage_unit' => array(
                    'required' => false,
                    'type' => 'string'
                ),
                'lot_area' => array(
                    'required' => false,
                    'type' => 'number'
                ),
                'lot_area_unit' => array(
                    'required' => false,
                    'type' => 'string'
                ),
                'year_built' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'address' => array(
                    'required' => true,
                    'type' => 'array',
                ),
                'features' => array(
                    'required' => false,
                    'type' => 'array',
                ),
                'image' => array(
                    'required' => false,
                    'type' => 'array',
                )
            )
        ));

        register_rest_route($this->namespace, '/add-listing/files', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {

                $properties = new PropertiesImageUpload();
                $properties->uploadImages(
                    $request->get_file_params(),
                    $request->get_param('listing_id')
                );
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'listing_id' => array(
                    'required' => true,
                    'type' => 'integer'
                )
            )
        ));

        /**
         * Zapier route
         *
         */
        register_rest_route($this->namespace, '/add-new-listing', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $params = $request->get_params();
                /**
                 * David.M
                 * at this time we don't support complex
                 */
                if(empty($params['kind'])) $params['kind'] = 0;
                $property = [];
                foreach($params as $column => $value) {
                    $property[$column] = ['wpl_value' => $value];
                }
                if(array_key_exists('confirmed', $params) == false) {
                     $property['confirmed'] = ['wpl_value' => 1];
                }


                $founded = \wpl_property::pid($params['ref_id'], 'ref_id');
                $pids = \wpl_property::import([$property], 'ref_id', null, 'zapier');
                \wpl_property::finalize($pids[0], empty($founded) ? 'add' : 'edit');

                return array ('status' => 'success', 'data' => ['property_id' => $pids[0]]);
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array()
        ));

        register_rest_route($this->namespace, '/add-listing/remove-file', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {

                global $wpdb;
                $params = $request->get_params();
                //$userId = (new Auth)->getUserIdByToken();
                $uid = (isset($params['user_id']) AND !empty($params['user_id'])) ? $params['user_id'] : '';
                $pid = (isset($params['listing_id']) AND !empty($params['listing_id'])) ? $params['listing_id'] : '';
                $url = (isset($params['url']) AND !empty($params['url'])) ? $params['url'] : '';
                if ($wpdb->get_row("SELECT * FROM {$wpdb->base_prefix}wpl_properties WHERE id='$pid' and user_id='$uid' and confirmed = 1 AND finalized = 1 AND deleted = 0 AND expired = 0")) {
                    $where = array(
                        'parent_id' => $pid,
                        'item_extra3' => $url
                    );
                    $res = $wpdb->update("{$wpdb->base_prefix}wpl_items", array('enabled' => 0), $where);
                    if ($res) {
                        return array('success' => true, 'data' => array('message' => 'the file successfully deleted.'));
                    } else {
                        return array('success' => false, 'data' => array('message' => 'deleting the image failed.'));
                    }
                } else {
                    return array('success' => false, 'data' => array('message' => 'listing not found.'));
                }
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'listing_id' => array(
                    'required' => true,
                    'type' => 'integer'
                )
            )
        ));


        register_rest_route($this->namespace, '/list-properties', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $this->listing
                    ->propertiesList($request);
            },
            //'permission_callback' => array( $this, 'permissionCheck' ),
            'permission_callback' => '__return_true',
            'args' => array(
                'listing_type' => array(
                    'required' => true,
                    'type' => 'integer'
                ),
                'property_type' => array(
                    'required' => false
                ),
                'page' => array(
                    'required' => true,
                    'type' => 'integer'
                ),
                'swlat' => array(
                    'required' => false,
                ),
                'swlng' => array(
                    'required' => false,
                ),
                'nelat' => array(
                    'required' => false,
                ),
                'nelng' => array(
                    'required' => false,
                ),
                'minbedroom' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'maxbedroom' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'minbathroom' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'maxbathroom' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'minprice' => array(
                    'required' => false
                ),
                'maxprice' => array(
                    'required' => false
                ),
                'minsqft' => array(
                    'required' => false
                ),
                'maxsqft' => array(
                    'required' => false
                ),
                'appliances' => array(
                    'required' => false
                ),
                'tags' => array(
                    'required' => false
                ),
                'neighborhoods' => array(
                    'required' => false
                ),
                'listing_id' => array(
                    'required' => false
                )
            )
        ));

        register_rest_route($this->namespace, '/filter', array(
            'methods' => 'GET',
            'callback' => function () {
                $this->listing
                    ->settings();
            },
            'permission_callback' => array($this, 'permissionCheck')
        ));

        register_rest_route($this->namespace, '/filter2', array(
            'methods' => 'GET',
            'callback' => function (\WP_REST_Request $request) {
                $userId = $request->get_param('user_id');
                $widgetId = 1;
                $widgetInfo = get_option('widget_wpl_search_widget');
                $instance = $widgetInfo[$widgetId];
                $search = new \wpl_search_widget();
                $rendered = $search->render_search_fields($instance, $widgetId);
                $return = [];
                foreach ($rendered as $key => $field) {
                    $searchOption = $instance['data'][$key];
                    $type = $field['field_data']['type'];
                    $options = [];
                    $unit = '';

                    if ($type == 'locations') {
                        continue;
                    }
                    $minValue = '';
                    $maxValue = '';
                    $step = '';
                    switch ($type) {
                        case 'select':
                            foreach ($field['field_options']['params'] as $option) {
                                if ($option['enabled']) {
                                    $options[] = [
                                        'name' => __($option['value'], 'real-estate-listing-realtyna-wpl'),
                                        'value' => $option['key'],
                                    ];
                                }
                            }
                            break;
                        case 'listings':
                            $listings = \wpl_global::get_listings();
                            $current_user = \wpl_users::get_wpl_user($userId);
                            $availables = (isset($current_user->maccess_lrestrict_plisting) and $current_user->maccess_lrestrict_plisting and trim($current_user->maccess_listings_plisting ?? '', ', ') != '') ? explode(',', trim($current_user->maccess_listings_plisting ?? '', ', ')) : array();
                            foreach ($listings as $listing) {
                                // Skip if listing type is not allowed for this user
                                if (is_array($availables) and count($availables) and !in_array($listing['id'], $availables)) continue;
                                $options[] = [
                                    'name' => __($listing['name'], 'real-estate-listing-realtyna-wpl'),
                                    'value' => $listing['id'],
                                ];
                            }
                            break;
                        case 'price':
                            $unit = '$';
                        case 'number':
                            $minMaxStep = explode(',', $field['search_options']);
                            if (!empty($minMaxStep)) {
                                $minValue = $minMaxStep[0];
                                $maxValue = $minMaxStep[1];
                                $step = $minMaxStep[2];
                            }
                            break;
                        case 'feature':
                            $featureOptions = json_decode($field['field_data']['options'] ?? '', true);
                            foreach ($featureOptions['values'] as $featureOption) {
                                $options[] = [
                                    'name' => __($featureOption['value'], 'real-estate-listing-realtyna-wpl'),
                                    'value' => $featureOption['key'],
                                ];
                            }
                            break;
                        case 'area':
                            $extoptions = explode(',', $searchOption['extoption']);
                            $min_value = $extoptions[0];
                            $max_value = $extoptions[1];
                            $division = $extoptions[2];
                            $default_unit_id = $extoptions[3];

                            $condition = '';
                            if ($default_unit_id) $condition .= " AND `id`='" . $default_unit_id . "'";

                            $units = \wpl_units::get_units(2, 1, $condition);
                            $unit = count($units) == 1 ? $units[0]['name'] : '';
                            $minValue = $min_value;
                            $maxValue = $max_value;
                            $step = $division;
                            break;
                        case 'property_types':
                            $property_types = \wpl_global::get_property_types();

                            foreach ($property_types as $property_type) {
                                $options[] = [
                                    'name' => __($property_type['name'], 'real-estate-listing-realtyna-wpl'),
                                    'value' => $property_type['id'],
                                ];
                            }
                    }
                    $return[] = [
                        'name' => $field['field_data']['name'],
                        'show_type' => $searchOption['type'],
                        'type' => $type,
                        'unit' => $unit,
                        'min' => intval($minValue),
                        'max' => intval($maxValue),
                        'step' => intval($step),
                        'key' => $field['field_data']['table_column'],
                        'options' => $options,
                    ];
                }

                wp_send_json_success($return);
                /*				print_r($return);
                //				print_r($instance);
                                print_r($rendered);
                                die;*/
            },
            'permission_callback' => array($this, 'permissionCheck')
        ));

        register_rest_route($this->namespace, '/add-favorite', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $this->properties
                    ->AddFavorite($request);
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'user_id' => array(
                    'required' => true,
                    'type' => 'integer'
                ),
                'listing_id' => array(
                    'required' => true,
                    'type' => 'string'
                )
            )
        ));

        register_rest_route($this->namespace, '/remove-favorite', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $this->properties
                    ->removeFavorite($request);
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'user_id' => array(
                    'required' => true,
                    'type' => 'integer'
                ),
                'listing_id' => array(
                    'required' => true,
                    'type' => 'string'
                )
            )
        ));

        register_rest_route($this->namespace, '/search', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $this->listing->search(
                    $request
                );
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'swlat' => array(
                    'required' => true,
                ),
                'swlng' => array(
                    'required' => true,
                ),
                'nelat' => array(
                    'required' => true,
                ),
                'nelng' => array(
                    'required' => true,
                ),
                'property_type' => array(
                    'required' => true
                ),
                'listing_type' => array(
                    'required' => true
                ),
                'minbedroom' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'maxbedroom' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'minbathroom' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'maxbathroom' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'minprice' => array(
                    'required' => false
                ),
                'maxprice' => array(
                    'required' => false
                ),
                'minsqft' => array(
                    'required' => false
                ),
                'maxsqft' => array(
                    'required' => false
                ),
                'appliances' => array(
                    'required' => false
                ),
                'tags' => array(
                    'required' => false
                ),
                'neighborhoods' => array(
                    'required' => false
                )
            )
        ));

        register_rest_route($this->namespace, '/detailed', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $this->properties->getBylistingId(
                    $request
                );
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'listing_id' => array(
                    'required' => true,
                    'type' => 'integer'
                )
            )
        ));

        register_rest_route($this->namespace, '/detailed2', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                wp_send_json_success($this->getProperty($request->get_param('listing_id')));
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'listing_id' => array(
                    'required' => true,
                    'type' => 'integer'
                )
            )
        ));

        register_rest_route($this->namespace, '/polygon-search', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $this->properties->pointInPolygon(
                    $request
                );
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'points' => array(
                    'required' => false,
                )
            )
        ));

        register_rest_route($this->namespace, '/mls/(?P<id>[a-zA-Z0-9-]+)', array(
            'methods' => 'GET',
            'callback' => function (\WP_REST_Request $request) {
                $this->properties->getByMlsId(
                    $request
                );
            },
            'permission_callback' => array($this, 'permissionCheck'),
        ));

        register_rest_route($this->namespace, '/favorite-list', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $this->properties->GetFavoriteList(
                    $request
                );
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'user_id' => array(
                    'required' => true,
                    'type' => 'integer'
                ),
                'page' => array(
                    'required' => true,
                    'type' => 'integer'
                ),
            )
        ));

        register_rest_route($this->namespace, '/add-search', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $this->properties->AddSaveSearch(
                    $request
                );
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'user_id' => array(
                    'required' => true,
                    'type' => 'integer'
                ),
                'name' => array(
                    'required' => true
                ),
            )
        ));

        register_rest_route($this->namespace, '/search-list', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $this->properties->ListSaveSearch(
                    $request
                );
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'user_id' => array(
                    'required' => true,
                    'type' => 'integer'
                ),
            )
        ));

        register_rest_route($this->namespace, '/remove-save-search', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                $this->properties->RemoveSaveSearch(
                    $request
                );
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'user_id' => array(
                    'required' => true,
                    'type' => 'integer'
                ),
                'name' => array(
                    'required' => true
                )
            )
        ));

        register_rest_route($this->namespace, '/add-about', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                (new AboutManager())
                    ->update($request);
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'company_name' => array(
                    'required' => true,
                ),
                'company_address' => array(
                    'required' => true
                ),
                'company_phone' => array(
                    'required' => true
                ),
                'company_about' => array(
                    'required' => true
                )
            )
        ));

        register_rest_route($this->namespace, '/get-about', array(
            'methods' => 'GET',
            'callback' => function () {
                (new AboutManager())
                    ->getAbout();
            },
            'permission_callback' => array($this, 'permissionCheck')
        ));

        register_rest_route($this->namespace, '/get-mail-setting', array(
            'methods' => 'GET',
            'callback' => function () {
                (new Email())
                    ->GetSmtpSettings();
            },
            'permission_callback' => array($this, 'permissionCheck')
        ));

        register_rest_route($this->namespace, '/save-onesignal', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                (new OneSignalSetting($request))
                    ->save();
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'appId' => array(
                    'required' => true
                ),
                'AuthKey' => array(
                    'required' => true
                )
            )
        ));

        register_rest_route($this->namespace, '/save-realtyfeed', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                (new Idx($request))
                    ->save();
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'token' => array(
                    'required' => true
                )
            )
        ));

        register_rest_route($this->namespace, '/realtyfeeds', array(
            'methods' => 'GET',
            'callback' => function () {
                (new Idx())
                    ->import();
            },
            'permission_callback' => '__return_true',
        ));

        register_rest_route($this->namespace, '/get-onesignal-setting', array(
            'methods' => 'GET',
            'callback' => function () {
                (new OneSignalSetting())
                    ->getSettings();
            },
            'permission_callback' => array($this, 'permissionCheck')
        ));

        register_rest_route($this->namespace, '/cron-jobs', array(
            'methods' => 'GET',
            'callback' => function () {
                (new CronManager())
                    ->everyMinuteAction();
            },
            'permission_callback' => '__return_true',
        ));

        register_rest_route($this->namespace, '/wpl-users', array(
                'methods' => 'GET',
                'callback' => function () {
                    (new Wpluser)
                        ->getList();
                },
                'permission_callback' => array($this, 'permissionCheck')

            )
        );

        register_rest_route($this->namespace, '/crm-contacts', array(
                'methods' => 'GET',
                'callback' => function () {
                    (new Wpluser)
                        ->getCRMContacts();
                },
                'permission_callback' => array($this, 'permissionCheck')
            )
        );

        register_rest_route($this->namespace, '/check-auth', array(
                'methods' => 'GET',
                'callback' => function () {
                    (new Wpluser)
                        ->getUserData();
                },
                'permission_callback' => array($this, 'permissionCheck')

            )
        );

        register_rest_route($this->namespace, '/delete-listings', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                global $wpdb;
                $pid = (isset($_REQUEST['listing_id']) AND !empty($_REQUEST['listing_id'])) ? $_REQUEST['listing_id'] : '';
                $uid = (isset($_REQUEST['user_id']) AND !empty($_REQUEST['user_id'])) ? $_REQUEST['user_id'] : '';
                $property = $wpdb->get_row("SELECT id FROM {$wpdb->base_prefix}wpl_properties WHERE id = '$pid' AND user_id = '$uid'", ARRAY_A);
                if ($property) {
                    $data = new PropertiesMapper(
                        $request->get_params()
                    );

                    (new PropertiesDelete($data))->delete();
                } else {
                    return array('success' => false, 'data' => array('message' => 'listing not found.'));
                }
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'listing_id' => array(
                    'required' => false,
                    'type' => 'string'
                ),
                'mls_id' => array(
                    'required' => false,
                    'type' => 'string'
                )
            )
        ));

        register_rest_route($this->namespace, '/set-agreement', array(
            'methods' => 'POST',
            'callback' => function () {
                global $wpdb;
                $userId = (new Auth)->getUserIdByToken();
                $pid = intval($_REQUEST['pid']);
                $property_id = $wpdb->get_row("SELECT id FROM {$wpdb->base_prefix}wpl_properties WHERE id = '$pid'", ARRAY_A);
                $res = [];
                if ($property_id) {
                    update_user_meta($userId, 'agreement_' . $pid, '1');
                    $res = array(
                        'message' => 'successfully updated',
                        'code' => '1',
                        'success' => true
                    );
                } else {
                    $res = array(
                        'message' => 'property not found',
                        'code' => '0',
                        'success' => false
                    );
                }
                return $res;
            },
            'permission_callback' => array($this, 'permissionCheck'),
        ));

        register_rest_route($this->namespace, '/read-agreement', array(
            'methods' => 'GET',
            'callback' => function () {
                global $wpdb;
                //$agreement = $wpdb->get_row("SELECT setting_name, setting_value, title FROM {$wpdb->base_prefix}wpl_settings WHERE wp_wpl_settings.setting_name = 'membership_agreement_html'",ARRAY_A);
                $agreement = $wpdb->get_row("SELECT setting_name, setting_value, title FROM {$wpdb->base_prefix}wpl_settings WHERE wp_wpl_settings.setting_name = 'agreement_text_" . substr(ICL_LANGUAGE_CODE, 0, 2) . "'", ARRAY_A);
                switch ($_GET['type']) {
                    case 'html':
                        break;
                    case 'text':
                    default:
                        foreach ($agreement as $k => $v) {
                            $agreement[$k] = strip_tags($v ?? '');
                        }
                        break;
                }
                $res = array(
                    'message' => 'successfully read',
                    'code' => '1',
                    'success' => true,
                    'data' => $agreement
                );
                return $res;
            },
            'permission_callback' => array($this, 'permissionCheck'),
        ));

        register_rest_route($this->namespace, '/check-agreement', array(
            'methods' => 'POST',
            'callback' => function () {
                global $wpdb;
                $userId = (new Auth)->getUserIdByToken();
                $pid = intval($_REQUEST['pid']);
                $agreement = get_user_meta($userId, 'agreement_' . $pid);
                $res = [];
                if ($agreement) {
                    $property_id = $wpdb->get_row("SELECT id FROM {$wpdb->base_prefix}wpl_properties WHERE id = '$pid'", ARRAY_A);
                    if ($property_id) {
                        $res = array(
                            'message' => 'successfully updated',
                            'code' => '1',
                            'success' => true
                        );
                    } else {
                        $res = array(
                            'message' => 'listing not found',
                            'code' => '0',
                            'success' => false
                        );
                    }
                } else {
                    $res = array(
                        'message' => 'agreement not found',
                        'code' => '0',
                        'success' => false
                    );
                }
                return $res;
            },
            'permission_callback' => array($this, 'permissionCheck'),
        ));

        register_rest_route($this->namespace, '/full-detail', array(
            'methods' => 'POST',
            'callback' => function () {
                global $wpdb;
                $pid = (isset($_REQUEST['listing_id']) AND !empty($_REQUEST['listing_id'])) ? $_REQUEST['listing_id'] : '';
                $property = $wpdb->get_row("SELECT  p.*,
                                                    (SELECT name FROM {$wpdb->base_prefix}wpl_units WHERE type = 2 AND enabled = 1 AND id = p.lot_area_unit) as lot_area_unit2,
                                                    (SELECT name FROM {$wpdb->base_prefix}wpl_units WHERE type = 4 AND enabled = 1 AND id = p.price_unit) as price_unit2,
                                                    (SELECT name FROM {$wpdb->base_prefix}wpl_listing_types WHERE enabled = 1 AND id = p.listing) as listing_type,
                                                    (SELECT name FROM {$wpdb->base_prefix}wpl_property_types WHERE enabled = 1 AND id = p.property_type) as property_type
                                            FROM {$wpdb->base_prefix}wpl_properties p
                                            INNER JOIN {$wpdb->base_prefix}wpl_listing_types lt ON lt.id = p.listing
                                            INNER JOIN {$wpdb->base_prefix}wpl_property_types pt ON pt.id = p.property_type
                                            WHERE p.id = '$pid' AND p.deleted = 0 AND p.expire_days = 0 AND p.confirmed = 1 AND p.finalized = 1", ARRAY_A);
                if ($property) {
                    $agent = $wpdb->get_row("SELECT id, first_name, last_name, company_name, company_address, website, main_email, secondary_email, tel, fax, mobile FROM {$wpdb->base_prefix}wpl_users WHERE id = $property[user_id]", ARRAY_A);
                    $property['lot_area_unit'] = html_entity_decode($property['lot_area_unit2']);
                    $property['country'] = __($property['location1_name'], 'wpl-api');
                    $property['state'] = __($property['location2_name'], 'wpl-api');
                    $property['county'] = __($property['location3_name'], 'wpl-api');
                    $property['city'] = __($property['location4_name'], 'wpl-api');
                    $property['price_unit'] = $property['price_unit2'];
                    $property['property_title'] = $property['field_313'];
                    //$property['property_page_title']=$property['field_312'];
                    $property['description'] = $property['field_308'];


                    switch (ICL_LANGUAGE_CODE) {
                        case 'ar':
                            $property['property_page_title'] = $property['field_312_ar'];
                            //$property['description_ar']=$property['textsearch_ar'];
                            break;
                        case 'en':
                            $property['property_page_title'] = $property['field_312_en_us'];
                            //$property['description_en']=$property['textsearch_en_us'];
                            break;
                        default:
                            $property['property_page_title'] = $property['field_312'];
                            break;
                    }

                    /*foreach ($agent as $k => $value) {
                        if (in_array($k,['membership_name','membership_id','membership_type','index','access_propertywizard','access_propertyshow','access_propertylisting','access_profilewizard','access_confirm','access_propertymanager','access_delete','access_public_profile','access_change_user','access_receive_notifications','maccess_num_prop','maccess_num_feat','maccess_num_hot','maccess_num_pic','maccess_show_in_register','maccess_period','maccess_price','maccess_price_unit','maccess_lrestrict','maccess_listings','maccess_ptrestrict','maccess_property_types','maccess_renewable','maccess_renewal_price','maccess_renewal_price_unit','maccess_upgradable','maccess_upgradable_to','maccess_recurring','maccess_short_description','maccess_long_description','maccess_wpl_color','maccess_private','maccess_fpc','maccess_crm','maccess_lrestrict_pshow','maccess_ptrestrict_pshow','maccess_listings_pshow','maccess_property_types_pshow','maccess_lrestrict_plisting','maccess_ptrestrict_plisting','maccess_listings_plisting','maccess_property_types_plisting','access_clone','access_propertystats','maccess_direct_contact','maccess_direct_contact_user_id','crm_cookies'])) unset($agent[$k]);
                    }*/

                    $property['agent'] = $agent;

                    unset($property['lot_area_unit2'], $property['field_313'], $property['textsearch_ar'], $property['textsearch_en_us'], $property['finalized'], $property['price_unit2'], $property['listing'], $property['field_312_ar'], $property['field_312_en_us'], $property['field_312'], $property['field_308'], $property['field_7']);
                    ksort($property);
                    $res = array('success' => true, 'data' => $property);
                } else {
                    $res = array('success' => false, 'message' => 'Property not found.');
                }
                $res = array_map('array_filter', $res);
                $res = array_filter($res);

                return $res;
            },
            'permission_callback' => array($this, 'permissionCheck'),
        ));

        register_rest_route($this->namespace, '/agents', array(
                'methods' => 'GET',
                'callback' => function () {
                    global $wpdb;
                    $search = (isset($_GET['id']) AND !empty($_GET['id'])) ? " AND {$wpdb->prefix}users.id=" . intval($_GET['id']) : '';
                    $search = (isset($_GET['public']) AND !empty($_GET['public'])) ? $search . " AND {$wpdb->prefix}wpl_users.access_public_profile=" . intval($_GET['public']) : $search;
                    $search = (isset($_GET['email']) AND !empty($_GET['email'])) ? $search . " AND ({$wpdb->prefix}users.user_email='" . $_GET['email'] . "' OR {$wpdb->prefix}wpl_users.main_email='" . $_GET['email'] . "')" : $search;
                    $ret["success"] = true;
                    $query = "SELECT first_name, last_name, company_name company, tel, main_email, profile_picture, {$wpdb->prefix}users.id id FROM {$wpdb->prefix}users INNER JOIN {$wpdb->prefix}wpl_users ON {$wpdb->prefix}wpl_users.id = {$wpdb->prefix}users.ID ";
                    if(!empty($search)) {
                        $query .= "WHERE  $search";
                    }
                    $ret['data']['agents'] = $wpdb->get_results($query, ARRAY_A);

                    $agents = [];
                    foreach ($ret['data']['agents'] as $k => $rec) {
                        $image = trim($rec['profile_picture'] ?? '');
                        if (isset($image) AND !empty($image)) {
                            $rec['profile_picture'] = get_site_url() . "/wp-content/uploads/WPL/users/" . $rec['id'] . "/" . $image;
                        } else {
                            $rec['profile_picture'] = '';
                        }
                        $ret['data']['agents'][$k] = $rec;
                        if (empty($rec['company'])) $ret['data']['agents'][$k]['company'] = '';
                    }

                    return $ret;
                },
                'permission_callback' => '__return_true',
            )
        );

        register_rest_route($this->namespace, '/list-currency', array(
                'methods' => 'GET',
                'callback' => function () {
                    global $wpdb;
                    $res = $wpdb->get_results("SELECT * FROM `{$wpdb->base_prefix}wpl_units` WHERE `type` = 4 and enabled = 1;", ARRAY_A);
                    $ret = array('success' => true, 'data' => $res);
                    return $ret;
                },
                'permission_callback' => '__return_true',
            )
        );

        register_rest_route($this->namespace, '/list-area-unit', array(
                'methods' => 'GET',
                'callback' => function () {
                    global $wpdb;
                    $res = $wpdb->get_results("SELECT * FROM `{$wpdb->base_prefix}wpl_units` WHERE `type` = 2;", ARRAY_A);

                    foreach ($res as $k => $item) {
                        $res[$k]['name'] = __(html_entity_decode($item['name']), 'wpl-api');
                    }

                    $ret = array('success' => true, 'data' => $res);
                    return $ret;
                },
                'permission_callback' => '__return_true',
            )
        );

        register_rest_route($this->namespace, '/edit-profile', array(
            'methods' => 'POST',
            'callback' => function (\WP_REST_Request $request) {
                global $wpdb;
                $info = $request->get_body_params();
                $userId = $info['user_id'];
                $first_name = $info['first_name'];
                $last_name = $info['last_name'];
                $email = $info['email'];
                $phone = $info['phone'];

                $user = get_user_by('ID', $userId);
                if (!$user) {
                    wp_send_json_error(array(
                        'message' => __('User not found', 'wpl-api')
                    ));
                }

                if (strtolower($email) != strtolower($user->user_email) && email_exists($email)) {
                    wp_send_json_error(array(
                        'message' => __('The Email is already registered. Try a different Email address.', 'wpl-api')
                    ));
                }
                $wplUser = DB::getRow("SELECT * FROM {$wpdb->base_prefix}wpl_users WHERE id= ?", array($userId));
                if ($wplUser && $wplUser['tel'] != $phone && DB::getRow("SELECT * FROM {$wpdb->base_prefix}wpl_users WHERE tel= ? and id != ?", array($phone, $userId))) {
                    wp_send_json_error(array(
                        'message' => __('The Phone is already registered. Try a different Phone number.', 'wpl-api')
                    ));
                }

                try {
                    $path = WP_CONTENT_DIR . '/uploads/WPL/users/' . $userId;
                    if (!is_dir($path)) {
                        mkdir($path, 0755, true);
                    }
                    $file = $request->get_file_params();
                    if (!empty($file['image']) && !move_uploaded_file($file['image']['tmp_name'], $path . '/profile.jpg')) {
                        return array(
                            "success" => false,
                            "data" => array(
                                "message" => "Uploading the file has been failed."
                            )
                        );
                    }
                    wp_update_user(array(
                        'ID' => $userId,
                        'user_email' => $email,
                        'display_name' => $first_name . ' ' . $last_name,
                    ));

                    update_user_meta($userId, 'phone_number', $phone);
                    update_user_meta($userId, 'first_name', $first_name);
                    update_user_meta($userId, 'last_name', $last_name);

                    DB::update("{$wpdb->base_prefix}wpl_users", array('first_name' => $first_name, 'last_name' => $last_name, 'tel' => $phone), "id = $userId");
                    if (!empty($file['image'])) {
                        DB::update("{$wpdb->base_prefix}wpl_users", array('profile_picture' => 'profile.jpg'), "id = $userId");
                    }
                    if(\wpl_global::check_addon('crm')) {
                        DB::update("{$wpdb->base_prefix}wpl_addon_crm_contacts", array('first_name' => $first_name, 'last_name' => $last_name, 'tel' => $phone, 'email' => $email), "user_id = $userId");
                    }

                    return array(
                        "success" => true,
                        "data" => array(
                            "message" => __('Profile updated.', 'wpl-api')
                        )
                    );
                } catch (\Exception $e) {
                    return array(
                        "success" => false,
                        "data" => array(
                            "message" => $e->getMessage()
                        )
                    );
                }
            },
            'permission_callback' => array($this, 'permissionCheck'),
            'args' => array(
                'user_id' => array(
                    'required' => true,
                    'type' => 'integer',
                    'description' => 'The client\'s ID',
                ),
                'first_name' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'The client\'s first name',
                ),
                'last_name' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'The client\'s last name',
                ),
                'email' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'The user\'s email address',
                    'format' => 'email'
                ),
                'phone' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'The user\'s phone number',
                ),
            )
        ));

        register_rest_route( $this->namespace, '/user-info', array(
                'methods'             => 'POST',
                'callback'            => function (\WP_REST_Request $request ) {
                    $user = $this->auth->getUserInfoById($request->get_param( 'user_id' ));
                    if(!$user) {
                        return wp_send_json_error( array(
                            'message' => 'User not found.'
                        ) );
                    }
                    wp_send_json_success($user);
                },
                'permission_callback' => array($this, 'permissionCheck')

            )
        );
    }

    public function permissionCheck($request)
    {
        $_POST = $request->get_params();
        $_REQUEST = $request->get_params();
        return (true === AuthValidation::validate()) ? true : false;
    }
}