<?php


namespace Api\Wpl\routes\api;


use Api\Wpl\routes\libs\AuthRequired;
use Api\Wpl\routes\libs\AuthAdmin;
use Firebase\JWT\JWT;

class UserApi extends Api
{

    public function registerAction(string $email, string $username, string $first_name, string $last_name, string $phone, ?string $password)
    {
        if (email_exists($email)) {
            throw new \Exception('The Email is already registered. Try a different Email address.');
        }
        $membership_id = -1;
        $user_type = 3;

        if (\wpl_users::username_exists($username)) {
            throw new \Exception('Username already exists.');
        }
        if (\wpl_users::email_exists($email)) {
            throw new \Exception('Email exists.');
        }
        $errors = new \WP_Error();

        /**
         * Fires before errors are returned from a register request.
         */
        do_action('register_post', $username, $email, $errors);

        if ($errors->get_error_code()) {
            throw new \Exception($errors->get_error_message());
        }

        $membership_data = \wpl_users::get_membership($membership_id);
        $user_type_data = \wpl_users::get_user_type($user_type);
        $password_generated = false;
        if (empty($password)) {
            $password_generated = true;
            $password = \wpl_global::generate_password(8);
        }
        $result = \wpl_users::insert_user(array('user_login' => $username, 'user_email' => $email, 'user_pass' => $password, 'first_name' => $first_name, 'last_name' => $last_name));
        if (is_wp_error($result)) {
            throw new \Exception($result->get_error_message());
        }

        $user_id = $result;

        \wpl_users::add_user_to_wpl($user_id);

        // Trigger event for sending notification
        \wpl_events::trigger('user_registered', array('password' => $password, 'user_id' => $user_id, 'mobile' => $phone));

        // If default membership is paid don't assign user to that membership
        $default_membership_id = !trim($membership_data->maccess_price ?? '') ? $membership_id : $user_type_data->default_membership_id;

        $default_membership_data = \wpl_users::get_membership($default_membership_id);
        if (trim($default_membership_data->maccess_price ?? '')) $default_membership_id = -1;

        \wpl_users::update('wpl_users', $user_id, 'mobile', $phone);

        // change membership of user to default membership
        \wpl_users::change_membership($user_id, $default_membership_id);

        // Update User
        \wpl_users::update('wpl_users', $user_id, 'mobile', $phone);

        // Trigger event
        \wpl_events::trigger('after_user_registered', array('id' => $user_id));
        if ($password_generated) {
            return array(
                'message' => __('User registered. Please check your email for password.', 'wpl-api')
            );
        }
        return array(
            'message' => __('User registered.', 'wpl-api')
        );
    }

    private function _loginByFacebook($accessToken)
    {
        $res = file_get_contents('https://graph.facebook.com/me?fields=id,name,email&access_token=' . $accessToken);
        $res = json_decode($res ?? '', true);
        if (empty($res['id'])) {
            return false;
        }
        $userId = $res['id'];
        $email = $res['email'];
        $name = $res['name'];
        $found_users = \wpl_users::get_wpl_users("AND (`facebook_uid` = '{$userId}' OR `user_email` = '" . $email . "')");
        if (empty($found_users)) {
            $user_id = \wpl_users::register_user($email, $email);
            \wpl_users::add_user_to_wpl($user_id);
            \wpl_users::change_membership($user_id);
            \wpl_users::update('users', $user_id, 'display_name', $name);
            \wpl_users::update('wpl_users', $user_id, 'facebook_uid', $userId);
            return \wpl_users::get_user($user_id);
        }
        return \wpl_users::get_user(array_pop($found_users)->ID);
    }

    private function _loginByGoogle($accessToken)
    {
        $res = file_get_contents('https://www.googleapis.com/oauth2/v1/userinfo?access_token=' . $accessToken);
        $res = json_decode($res ?? '', true);
        if (empty($res['id'])) {
            $res = file_get_contents('https://oauth2.googleapis.com/tokeninfo?id_token=' . $accessToken);
            $res = json_decode($res ?? '', true);
            if (empty($res['sub'])) {
                return false;
            }
            $res['id'] = $res['sub'];
        }
        $userId = $res['id'];
        $email = $res['email'];
        $name = $res['name'];
        $found_users = \wpl_users::get_wpl_users("AND (`user_email` = '" . $email . "')");
        if (empty($found_users)) {
            $user_id = \wpl_users::register_user($email, $email);
            \wpl_users::add_user_to_wpl($user_id);
            \wpl_users::change_membership($user_id);
            \wpl_users::update('users', $user_id, 'display_name', $name);
            return \wpl_users::get_user($user_id);
        }
        return \wpl_users::get_user(array_pop($found_users)->ID);
    }

    private function _loginByApple($token, $fullName)
    {
        $res = json_decode(base64_decode(str_replace('_', '/', str_replace('-', '+', explode('.', $token)[1] ?? '' ))), true);
        if (empty($res['email'])) {
            return false;
        }
        $email = $res['email'];
        $name = $fullName;
        $found_users = \wpl_users::get_wpl_users("AND (`user_email` = '" . $email . "')");
        if (empty($found_users)) {
            $user_id = \wpl_users::register_user($email, $email);
            \wpl_users::add_user_to_wpl($user_id);
            \wpl_users::change_membership($user_id);
            \wpl_users::update('users', $user_id, 'display_name', $name);
            return \wpl_users::get_user($user_id);
        }
        return \wpl_users::get_user(array_pop($found_users)->ID);
    }

    public function loginAction(string $loginType, ?string $accessToken, ?string $email, ?string $password, ?string $fullname)
    {
        if (!in_array($loginType, ['realtyna', 'facebook', 'google', 'apple'])) {
            throw new \Exception('Invalid login type.');
        }

        if (in_array($loginType, ['facebook', 'google', 'apple'])) {
            if (empty($accessToken)) {
                throw new \Exception('access token required.');
            }
            if ($loginType == 'facebook') {
                $user = $this->_loginByFacebook($accessToken);
            } elseif ($loginType == 'google') {
                $user = $this->_loginByGoogle($accessToken);
            } elseif ($loginType == 'apple') {
                $user = $this->_loginByApple($accessToken, $fullname);
            }
            if (empty($user)) {
                throw new \Exception('access token invalid.');
            }
        } elseif (in_array($loginType, ['realtyna'])) {
            if (empty($email) || empty($password)) {
                throw new \Exception('email and password required.');
            }
            $authData = array(
                'user_login' => $email,
                'user_password' => $password,
                'remember' => 1,
            );

            // disable event
            \wpl_events::$wpl_events = array();
            $user = wp_signon($authData, false);
        }

        if (empty($user) || is_wp_error($user)) {
            throw new \Exception('Authentication failed.');
        }

        if (empty($user->roles)) {
            throw new \Exception('User deactivated');
        }

        wp_set_current_user($user->ID, $authData);
        $user = wp_get_current_user();

        $wpl_user = \wpl_users::get_wpl_user($user->ID);
        $first_name = $wpl_user->first_name;
        if (empty($first_name)) {
            $first_name = get_user_meta($user->ID, 'first_name', true);
        }

        $last_name = $wpl_user->last_name;
        if (empty($last_name)) {
            $last_name = get_user_meta($user->ID, 'last_name', true);
        }

        $mobile = $wpl_user->mobile;
        if (empty($mobile)) {
            $mobile = get_user_meta($user->ID, 'phone_number', true);
        }

        if (\wpl_global::check_addon('crm')) {
            \_wpl_import('libraries.event_handlers.addon_crm');

            \wpl_events_crm::user_login(array(
                array(
                    'user_id' => $user->ID,
                    'params' => array('device' => 'mobile')
                ),
            ));
        }

        $token = [
            "iss" => get_bloginfo('url'),
            'data' => [
                'user_id' => get_current_user_id(),
            ]
        ];
        $access_token = JWT::encode($token, SECRET_KEY_JWT);
        return array(
            'message' => __('Authentication passed successfully.', 'wpl-api'),
            'access_token' => $access_token,
            'user_id' => get_current_user_id(),
            'name' => $first_name . ($last_name ? ' ' . $last_name : ''),
            'first_name' => $first_name,
            'last_name' => $last_name,
            'email' => $user->user_email,
            'username' => $user->user_login,
            'profile_image' => $wpl_user->profile_picture ? get_site_url() . "/wp-content/uploads/WPL/users/" . $wpl_user->id . "/" . $wpl_user->profile_picture : '',
            'phone' => $mobile,
            'is_admin' => !empty($user->allcaps['administrator']),
        );
    }


    public function userExistsAction(string $email) {
        return email_exists($email) > 0;
    }


    public function resetPasswordAction(string $usermail)
    {
        if (strpos($usermail ?? '', '@')) $user_data = \wpl_users::get_user_by('email', $usermail);
        else $user_data = \wpl_users::get_user_by('login', $usermail);

        if (!$user_data) {
            throw new \Exception('Invalid username or email.');
        }

        $errors = new \WP_Error();

        /**
         * Fires before errors are returned from a password reset request.
         */
        do_action('lostpassword_post');

        if ($errors->get_error_code()) {
            throw new \Exception($errors->get_error_message());
        }

        $user_login = $user_data->user_login;

        $allow = apply_filters('allow_password_reset', true, $user_data->ID);
        if (!$allow or ($allow and is_wp_error($allow))) {
            throw new \Exception('Password reset is not allowed for this user.');
        }

        $hashed = substr(uniqid(rand(), true), 4, 4);

        /** Set hashed key in database **/
        $query = "UPDATE `#__users` SET `user_activation_key`='$hashed' WHERE `user_login`='$user_login'";
        $result = \wpl_db::q($query, 'UPDATE');
        if (!$result) {
            throw new \Exception('Error Occurred.');
        }
        /** Trigger event for sending notification **/
        _wpl_import('libraries.notifications.notifications');
        $notification = new \wpl_notifications('email');

        $notification->prepare(101, array());
        $user = \wpl_users::get_user($user_data->ID);
        $notification->replacements = $notification->set_replacements(array(
            'name' => isset($user->data->wpl_data) ? $user->data->wpl_data->first_name : $user->data->display_name,
            'link' => $hashed,
        ));
        $notification->rendered_content = $notification->render_notification_content();
        $notification->rendered_content = str_replace('Please click on this link for resetting your password.', 'Your reset code is', $notification->rendered_content);
        $notification->recipients = $notification->set_recipients(array($user->data->user_email));
        $notification->send();

        return array(
            'message' => __('An email was sent to you to complete the password reset.', 'real-estate-listing-realtyna-wpl')
        );
    }

    public function changePasswordAction(string $usermail, string $code, string $password)
    {
        if (strpos($usermail ?? '', '@')) $user_data = \wpl_users::get_user_by('email', $usermail);
        else $user_data = \wpl_users::get_user_by('login', $usermail);

        if (!$user_data) {
            wp_send_json_error(array(
                'message' => 'Invalid username or email.'
            ));
            return;
        }
        $user_id = \wpl_users::validate_activation_key($code);
        if ($user_data->ID != $user_id) {
            wp_send_json_error(array(
                'message' => 'Invalid code.'
            ));
        }

        $password = urldecode($password);

        /** Password is short **/
        if (strlen($password) < 6) {
            throw new \Exception('Password should be more than 6 characters!');
        }


        \wpl_users::set_password($user_id, $password);
        \wpl_users::update_user_option($user_id, 'default_password_nag', false, true);

        /** Empty key in database **/
        $query = "UPDATE `#__users` SET `user_activation_key`='' WHERE `ID`='$user_id'";
        \wpl_db::q($query, 'UPDATE');

        $user_data = \wpl_users::get_user($user_id);
        /** Trigger event for sending notification **/
        $parameters = array('user_id' => $user_id, 'password' => $password);
        \wpl_events::trigger('user_password_reset', $parameters);
        return array(
            'message' => __('Password changed successfully.', 'wpl-api')
        );
    }

    public function agentsAction(?int $id, ?string $email)
    {
        $search = !empty($id) ? " AND #__wpl_users.id=" . intval($id) : '';
        $search = !empty($email) ? $search . " AND (#__wpl_users.main_email='" . $email . "')" : $search;
        $ret["success"] = true;
        $query = "SELECT first_name, last_name, company_name company, tel, main_email, profile_picture, id FROM #__wpl_users where id > 0 AND blog_id = '" . get_current_blog_id() . "'  AND `access_public_profile` = '1' AND `expired` = '0' ";
        if (!empty($search)) {
            $query .= " $search";
        }

        $ret['data']['agents'] = \wpl_db::select($query, 'loadAssocList');

        foreach ($ret['data']['agents'] as $k => $rec) {
            $image = trim($rec['profile_picture'] ?? '');
            if (isset($image) and !empty($image)) {
                $rec['profile_picture'] = get_site_url() . "/wp-content/uploads/WPL/users/" . $rec['id'] . "/" . $image;
            } else {
                $rec['profile_picture'] = '';
            }
            $rec['properties'] = \wpl_db::select("select count(*) from #__wpl_properties where kind = 0 and finalized = 1 and deleted = 0 and confirmed = 1 and expired = 0 and user_id = " . $rec['id'], 'loadResult');
            $ret['data']['agents'][$k] = $rec;
            if (empty($rec['company'])) $ret['data']['agents'][$k]['company'] = '';
        }

        return $ret['data'];
    }

    public function editProfileAction(\WP_REST_Request $request, AuthRequired $auth, string $first_name, string $last_name, string $email, string $phone)
    {

        $user = get_user_by('ID', $auth->getUserId());
        if (!$user) {
            throw new \Exception('User not found.');
        }

        if (strtolower($email) != strtolower($user->user_email) && email_exists($email)) {
            throw new \Exception('The Email is already registered. Try a different Email address.');
        }
        $wplUser = \wpl_users::get_wpl_user($auth->getUserId());

        if ($wplUser && $wplUser->mobile != $phone) {
            $found = \wpl_db::select("SELECT * FROM #__wpl_users WHERE mobile= '$phone' and id != '{$auth->getUserId()}'", 'loadObject');
            if ($found) {
                throw new \Exception('The Phone is already registered. Try a different Phone number.');
            }
        }

        $path = WP_CONTENT_DIR . '/uploads/WPL/users/' . $auth->getUserId();
        if (!is_dir($path)) {
            mkdir($path, 0755, true);
        }
        $file = $request->get_file_params();
        if (!empty($file['image']) && !move_uploaded_file($file['image']['tmp_name'], $path . '/profile.jpg')) {
            return array(
                "success" => false,
                "data" => array(
                    "message" => "Uploading the file has been failed."
                )
            );
        }
        wp_update_user(array(
            'ID' => $auth->getUserId(),
            'user_email' => $email,
            'display_name' => $first_name . ' ' . $last_name,
        ));

        update_user_meta($auth->getUserId(), 'phone_number', $phone);
        update_user_meta($auth->getUserId(), 'first_name', $first_name);
        update_user_meta($auth->getUserId(), 'last_name', $last_name);

        \wpl_db::update('wpl_users', array('first_name' => $first_name, 'last_name' => $last_name, 'mobile' => $phone), 'id', $auth->getUserId());
        if (!empty($file['image'])) {
            \wpl_users::update('wpl_users', $auth->getUserId(), 'profile_picture', 'profile.jpg');
        }
        if (\wpl_global::check_addon('crm')) {
            \wpl_db::update('wpl_addon_crm_contacts', array('first_name' => $first_name, 'last_name' => $last_name, 'mobile' => $phone, 'email' => $email), 'user_id', $auth->getUserId());
        }

        \wpl_users::remove_thumbnails($auth->getUserId());

        return array(
            "message" => __('Profile updated.', 'wpl-api')
        );
    }

    public function userInfoAction(AuthRequired $auth)
    {
        $user = get_user_by('ID', $auth->getUserId());
        if (!$user) {
            throw new \Exception('User not found.');
        }
        $wpl_user = \wpl_users::get_wpl_user($user->ID);

        $first_name = $wpl_user->first_name ? $wpl_user->first_name : get_user_meta($user->ID, 'first_name', true);
        $last_name = $wpl_user->last_name ? $wpl_user->last_name : get_user_meta($user->ID, 'last_name', true);
        $mobile = $wpl_user->mobile ? $wpl_user->mobile : get_user_meta($user->ID, 'phone_number', true);

        return array(
            'user_id' => $user->ID,
            'profile_image' => $wpl_user->profile_picture ? get_site_url() . "/wp-content/uploads/WPL/users/" . $wpl_user->id . "/" . $wpl_user->profile_picture : '',
            'name' => $first_name . ($last_name ? ' ' . $last_name : ''),
            'first_name' => $first_name,
            'last_name' => $last_name,
            'email' => $user->user_email,
            'username' => $user->user_login,
            'phone' => $mobile,
            'is_admin' => !empty($user->allcaps['administrator'])
        );
    }

    public function deleteAccountAction(AuthRequired $auth, string $comment)
    {
        $userId = $auth->getUserId();
        \wpl_db::q("UPDATE `#__wpl_properties` SET deleted = 1 where user_id = '$userId'");
        update_user_meta($userId, 'wpl_api_delete_reason', $comment);
        wp_delete_user($userId);
        return array(
            "message" => __('Account deleted.', 'wpl-api')
        );
    }

    public function deactivateAccountAction(AuthRequired $auth)
    {
        $userId = $auth->getUserId();
        $user = new \WP_User($userId);
        $user->set_role('none');
        return array(
            "message" => __('Account deactivated.', 'wpl-api')
        );
    }

    public function getRawInfoAction(AuthAdmin $auth, \WP_REST_Request $request)
    {
        $userId         = $request['id'];
        $wplUser        = \wpl_users::get_wpl_user($userId);
        $wplUserRender  = \wpl_users::full_render($userId, \wpl_users::get_pshow_fields(), NULL, array(), true);

        $wplUser->profile_picture_extra = array(
            'url'   => isset($wplUserRender['profile_picture']['url']) ? $wplUserRender['profile_picture']['url'] : '',
            'path'  => isset($wplUserRender['profile_picture']['path']) ? $wplUserRender['profile_picture']['path'] : '',
            'name'  => isset($wplUserRender['profile_picture']['name']) ? $wplUserRender['profile_picture']['name'] : '',
        );

        $wplUser->company_logo_extra = array(
            'url'   => isset($wplUserRender['company_logo']['url']) ? $wplUserRender['company_logo']['url'] : '',
            'path'  => isset($wplUserRender['company_logo']['path']) ? $wplUserRender['company_logo']['path'] : '',
            'name'  => isset($wplUserRender['company_logo']['name']) ? $wplUserRender['company_logo']['name'] : '',
        );

        $wplUser->profile_link = \wpl_users::get_profile_link($userId);

        return $wplUser;
    }

    public function getUsersAction(?string $membershipName, ?int $page = 1, ?int $limit = 20)
    {
        $offset = ($page - 1) * $limit;
        if ($offset < 0) $offset = 0;

        $query = "SELECT * FROM `#__wpl_users` WHERE `id` > 0 AND membership_name = '$membershipName' LIMIT $offset,$limit;";

        return \wpl_db::select($query);
    }

    public function agentListAction(\WP_REST_Request $request)
    {
        _wpl_import('libraries.pagination');

        $attributes = $request->get_param('attributes');

        if (!empty($attributes['wplpage'])) \wpl_request::setVar('wplpage', $attributes['wplpage']);
        $limit = !empty($attributes['limit']) ? $attributes['limit'] : 8;
        $order = !empty($attributes['wplorder']) ? $attributes['wplorder'] : "asc";

        $where = \wpl_db::create_query($attributes);
        $query = "FROM `#__users` AS u INNER JOIN `#__wpl_users` AS wpl ON u.ID = wpl.id WHERE 1 {$where} ORDER BY wpl.id {$order} ";

        $total = \wpl_db::select("SELECT count(*) $query", 'loadResult');
        $pagination = \wpl_pagination::get_pagination($total, $limit);
        $query .= $pagination->limit_query;
        $agents = \wpl_db::select("SELECT * {$query}");

        // Multisite
        $multisite = (function_exists('is_multisite') and is_multisite());

        $wpl_agents = array();

        foreach ($agents as $agent) {
            // User is not exists in WordPress
            if (!\wpl_users::is_wp_user($agent->id)) continue;

            // User is not Member of Current Blog
            if ($multisite and !is_user_member_of_blog($agent->id)) continue;

            $wpl_agents[$agent->id] = \wpl_users::full_render($agent->id, \wpl_users::get_pshow_fields(), NULL, array(), true);
        }

        $pages = ceil($pagination->total_pages / $pagination->limit);

        return [
            'agents'        => $wpl_agents,
            'pagination'    => $pagination,
            'info'          => [
                'current_page'  => $pagination->page,
                'next_page'     => $pagination->page == $pages ? $pages : $pagination->page + 1,
                'previous_page' =>  $pagination->page == 1 ? 1 : $pagination->page - 1,
                'total_page'    => $pages,
            ],
        ];
    }

    public function checkAccessAction(\WP_REST_Request $request)
    {
        $attributes = $request->get_param('attributes');
        $access = $attributes['access'];
        $owner_id = isset($attributes['owner_id']) ? $attributes['owner_id'] : 0;
        $user_id = isset($attributes['user_id']) ? $attributes['user_id'] : '';

        /** get current user id **/
        if (trim($user_id) == '') $user_id = \wpl_users::get_cur_user_id();

        // Admin user has access to anything
        if (\wpl_users::is_administrator($user_id)) return true;

        $user_data = \wpl_users::get_wpl_data($user_id);

        /** user is registered in WordPress but not in WPL so we choose guest user for accesses **/
        if (!$user_data) $user_data = \wpl_users::get_wpl_data(0);

        if ($access == 'edit') {
            if (\wpl_users::is_broker($user_id)) {
                _wpl_import('libraries.addon_brokerage');

                $brokerage = new \wpl_addon_brokerage();
                if ($brokerage->is_in_brokerage($owner_id, $user_id)) return true;
            }

            if ($owner_id == $user_id) return true;
        } elseif ($access == 'add') {
            $num_prop_limit = $user_data->maccess_num_prop;
            $num_prop = \wpl_users::get_users_properties_count($user_id);

            if (\wpl_users::is_part_of_brokerage($user_id)) {
                $broker = \wpl_users::get_broker($user_id);

                $num_prop_limit = min($user_data->maccess_num_prop, $broker->maccess_num_prop);
                if ($num_prop_limit == '-1') $num_prop_limit = $broker->maccess_num_prop;

                $num_prop = \wpl_users::get_users_properties_count($broker->id);
            }

            if ($num_prop_limit == '-1') return true; # unlimited
            if ($num_prop_limit <= $num_prop and !\wpl_users::is_administrator($user_id)) return false;
            else return true;
        } elseif ($access == 'delete') {
            if (\wpl_users::is_broker($user_id)) {
                _wpl_import('libraries.addon_brokerage');

                $brokerage = new \wpl_addon_brokerage();
                if ($brokerage->is_in_brokerage($owner_id, $user_id)) return true;
            }

            if ($user_data->access_delete and $owner_id == $user_id) return true;
        } elseif ($access == 'confirm') {
            if (\wpl_users::is_broker($user_id)) {
                _wpl_import('libraries.addon_brokerage');

                $brokerage = new \wpl_addon_brokerage();
                if ($brokerage->is_in_brokerage($owner_id, $user_id)) return true;
            }

            if ($user_data->access_confirm and $owner_id == $user_id) return true;
        } else {
            return isset($user_data->{'access_' . $access}) ? $user_data->{'access_' . $access} : 0;
        }

        return false;
    }
}
