<?php


namespace Api\Wpl\routes\api;

use Api\Wpl\models\imported\Email\Email;
use Api\Wpl\routes\libs\Auth;
use Api\Wpl\routes\libs\AuthAdmin;
use Api\Wpl\routes\libs\AuthRequired;
use \Firebase\JWT\JWT;

class SettingsApi extends Api
{
    public function versionAction()
    {
        return [
            'plugin_name' => WPL_API_PLUGIN,
            'plugin_version' => WPLAPI_VERSION
        ];
    }

    public function appSettingsAction(Auth $auth) {
        $requiredAuth = null;
        if($auth->userId) {
            $requiredAuth = new AuthRequired();
            $requiredAuth->userId = $auth->userId;
        }
        return [
            'sort-options' => (new FlexApi())->sortOptionsAction(),
            'map-settings' => $this->mapSettingsAction(),
            'listing-types' => (new FlexApi())->listingTypesAction(),
            'user-info' => $requiredAuth ? (new UserApi())->userInfoAction($requiredAuth) : (object) null,
        ];
    }

    public function checkTokenAction()
    {
        $token = get_option('wpl_api_token');
        if (!$token) {
            throw new \Exception('Token not found');
        }
        return $token;
    }

    public function createTokenAction(AuthAdmin $auth, $email, $username, $password)
    {
        if (!email_exists($email)) {
            throw new \Exception('Email does not exist.');
        }

        $author_obj = get_user_by('email', $email);

        if (!$author_obj->allcaps['administrator']) {
            throw new \Exception('Only administrators can access this feature.');
        }

        $user = wp_authenticate($username, $password);
        if (is_wp_error($user)) {
            throw new \Exception('username or password is incorrect');
        }

        // Make an array for the JWT Payload
        $token = [
            "iss" => get_bloginfo('url'),
            'data' => [
                'email' => $email
            ]
        ];

        // generate JSON web token and store as variable
        $jwt = JWT::encode($token, SECRET_KEY_JWT);
        update_option('wpl_api_token', $jwt);

        return array(
            'token' => get_option('wpl_api_token')
        );
    }

    public function resetTokenAction(AuthAdmin $auth)
    {
        $result = delete_option('wpl_api_token');
        if (!$result) {
            throw new \Exception('Error occurred!');
        }
        return array(
            'message' => __('Token has been removed.', 'wpl-api')
        );
    }

    public function saveSmsSettingsAction(AuthAdmin $auth, string $AccountSID, string $TokenID, string $SenderNumber)
    {
        //  if the option exist
        if (get_option('wpl_api_sms_settings')) {
            throw new \Exception('Sms settings already registered.');
        }

        $settings = array(
            'AccountSID' => $AccountSID,
            'TokenID' => $TokenID,
            'SenderNumber' => $SenderNumber
        );
        $result = add_option('wpl_api_sms_settings', $settings, '', 'yes');

        //  if the option does not exist
        if (!$result) {
            throw new \Exception('Error occurred!');
        }
        return array('message' => 'SMS Settings are saves successfully.');
    }

    public function smsSettingsAction(AuthAdmin $auth)
    {
        $settings = get_option('wpl_api_sms_settings');
        if (!$settings) {
            throw new \Exception('First you should add sms settings.');
        }
        return $settings;
    }

    public function saveSmtpSettingsAction(AuthAdmin $auth, string $smtpHost, string $smtpPort, string $smtpSecure, string $smtpUsername, string $smtpPassword)
    {
        $smtpSettings = array(
            'smtpHost' => $smtpHost,
            'smtpPort' => $smtpPort,
            'smtpSecure' => $smtpSecure,
            'smtpUsername' => $smtpUsername,
            'smtpPassword' => $smtpPassword
        );
        $result = update_option('wpl_api_smtp_setting', $smtpSettings);
        if (!$result) {
            throw new \Exception('The record already exists. Please modify your entry.');
        }
        return array(
            'message' => 'SMTP Settings are saves successfully.'
        );
    }

    public function sendEmailAction(int $listing_id, string $user, ?string $phone, string $email, string $text, string $agent_email)
    {
        $url = \wpl_property::get_property_link([], $listing_id);

        $message = "There is a new request for info about a listing.\r\n";
        $message .= " \r\n";
        $message .= "Listing: $url \r\n";
        $message .= " \r\n";
        $message .= "Full Name: $user \r\n";
        $message .= "Phone Number: $phone \r\n";
        $message .= "Email: $email \r\n";
        $message .= " \r\n";
        $message .= "Message: $text";

        $result = \wp_mail($agent_email, 'New Listing Information Request', $message);
        if (!$result) {
            throw new \Exception('Message could not be sent.');
        }
        return array(
            'message' => __('Message has been sent.', 'wpl-api')
        );
    }

    public function resetSmsSettingsAction(AuthAdmin $auth)
    {
        $result = delete_option('wpl_api_sms_settings');
        if (!$result) {
            throw new \Exception('SMS settings is not registered yet.');
        }
        return array(
            'message' => __('SMS settings are reset successfully.', 'wpl-api')
        );
    }

    public function saveMapSettingsAction(AuthAdmin $auth, string $latitude, string $longitude, int $zoom)
    {
        $settings = array(
            'latitude' => $latitude,
            'longitude' => $longitude,
            'zoom' => $zoom
        );
        $result = update_option('wpl_api_map_options', $settings, 'yes');
        if (!$result) {
            throw new \Exception('The record already exists. Please modify your entry.');
        }
        return array(
            'message' => __('Map Settings are saved successfully.', 'wpl-api')
        );
    }

    public function mapSettingsAction()
    {

        $settings = get_option('wpl_api_map_options');
        if (!$settings) {
            throw new \Exception('First you should save map settings.');
        }
        return $settings;
    }

    public function addAboutAction(AuthAdmin $auth, string $company_name, string $company_address, string $company_phone, string $company_about)
    {
        $about = array(
            'company_name' => $company_name,
            'company_address' => $company_address,
            'company_phone' => $company_phone,
            'company_about' => $company_about,
        );
        $result = update_option('wpl_api_about', $about, 'yes');
        if (!$result) {
            throw new \Exception('The record already exists. Please modify your entry.');
        }

        return array(
            'message' => 'About is created successfully.'
        );
    }

    public function getAboutAction()
    {
        $about = get_option('wpl_api_about');
        if (!$about) {
            throw new \Exception('The record is not found.');
        }
        return $about;
    }

    public function getMailSettingAction(AuthAdmin $auth)
    {
        $result = get_option('wpl_api_smtp_setting');
        if (!$result) {
            throw new \Exception('The record does not exists.');
        }
        return $result;
    }

    public function saveOneSignalAction(AuthAdmin $auth, string $appId, string $AuthKey)
    {
        $settings = array(
            'appId' => $appId,
            'AuthKey' => $AuthKey,
            'content' => ["en" => 'A new property is added'],
            'headings' => ["en" => 'New Property'],
        );
        $result = update_option('wpl_api_onesignal_options', $settings, 'yes');

        if (!$result) {
            throw new \Exception('The record already exists. Please modify your entry.');
        }
        return array('message' => 'OneSignal Settings are saved successfully.');
    }

    public function getOneSignalSettingAction(AuthAdmin $auth)
    {
        $settings = get_option('wpl_api_onesignal_options');
        if (!$settings) {
            throw new \Exception('First you should save OneSignal settings.');
        }
        return $settings;
    }

    public function getSettingsAction(string $category)
    {

        $settings = \wpl_global::get_settings($category);
        return $settings;
    }

    public function unitTypesAction()
    {
        // Unit Types
        $unit_types = \wpl_units::get_unit_types();

        return $unit_types;
    }

    public function priceUnitsAction()
    {
        // Price Units
        $price_units = \wpl_units::get_units(4);

        return $price_units;
    }

    public function wplUsersOptionsAction()
    {
        // Users
        $wpl_users = \wpl_users::get_wpl_users();

        return $wpl_users;
    }

    public function wplCheckAddonAction(\WP_REST_Request $request)
    {

        $addon_name = $request->get_param('addon_name') ?? '';
        if (trim($addon_name) == '') return false;
        $addon_name = strtolower($addon_name);
        return \wpl_global::check_addon($addon_name);
    }

    public function getWplSettingAction($setting_name, $category = '')
    {
        return \wpl_global::get_setting($setting_name, $category);
    }

    public function crmURLAction()
    {
        $url = '';

        if (\wpl_global::check_addon('crm')) {
            _wpl_import('libraries.addon_crm');
            $crm = new \wpl_addon_crm();

            $url = $crm->URL('dashboard');
        }

        return $url;
    }

    public function saveLicenseAction(AuthAdmin $auth)
    {

        if(!isset($_POST['email'])){
            throw new \Exception('Email is required!');
        }

        if(!is_email($_POST['email'])){
            throw new \Exception('Email is not valid!');
        }
        if(!isset($_POST['license'])){
            throw new \Exception('License is required!');
        }


        $email           = $_POST['email'];
        $license         = $_POST['license'];
        $domain          = $_SERVER['SERVER_NAME'];
        $license_api_url = 'https://realtyna.com/wp-json/license_checker/checkuser?email=' . $email . '&license=' . $license . '&domain=' . $domain;
        $response_api    = wp_remote_get($license_api_url);
        $response_body   = json_decode(wp_remote_retrieve_body($response_api) ?? '', true);
        if ( ! empty($response_body['license_data'])) {
            $product_status = $response_body['license_data']['product_status'];
            $user_status    = $response_body['license_data']['user_status'];
            $product_id     = $response_body['license_data']['product_id'];
            $order_status   = $response_body['order_status'];
            if ($product_id != $this->lead_product_id) {
                throw new \Exception('This license is not valid for this product!');
            } elseif ($order_status == 'completed' && $product_status == 0 && $user_status == 0) {
                $license_validation = true;
                $license_data_arr   = array(
                    'email'              => $email,
                    'license_validation' => $license_validation,
                    'license_data'       => $response_body['license_data'],
                );
                update_option('rlty_license_' . $product_id, $license_data_arr);

                return array(
                    'message' => __('License is valid, Reloading...', 'wpl-api')
                );
            } else {
                throw new \Exception('License not is valid.');
            }
        } else {
            throw new \Exception('License not is valid.');
        }
    }

    public function getLicenseAction(AuthAdmin $auth)
    {
        $settings = get_option('rlty_license_' . $this->lead_product_id);
        if ( ! $settings) {
            throw new \Exception('First you should save License settings.');
        }
        $response = [
            'email' => $settings['email'],
            'license_data' => $settings['license_data']['license'],
        ];
        return $response;
    }
}
