<?php

namespace Api\Wpl\models\core\Database;

use Api\Wpl\models\AbstractModel;

class Database extends AbstractModel {

	/**
	 * This holds list of Database instance that are connected
	 */
	static protected $connected = array();

	/**
	 * This will hold both connection objects (instances of PDO).
	 */
	static protected $con = array();

	/**
	 * Function is called whenever action with Database is made. Idea is that no actual connection is made until first query is called
	 * @return void
	 */
	public static function ensureConnected( $accessType ) {

		$wpdb = new AbstractModel();
		//if is in connected array, means connection is made. nothing to do here
		if ( in_array( $accessType, self::$connected ) ) {
			return;
		}

		//now we create a new PDO object through PDOWrapper.
		self::$con[ $accessType ] = new Pdo(
			"mysql:host={$wpdb->wpdb->dbhost}; dbname={$wpdb->wpdb->dbname}; charset=" . DB_CHARSET,
			$wpdb->wpdb->dbuser,
			$wpdb->wpdb->dbpassword
		);

		//make so that every error throws an exception
		self::$con[ $accessType ]->setAttribute( \PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION );

		//add this to list of connections
		self::$connected[] = $accessType;

	}

	/**
	 * Similar to getRows, but after loading rows, it returns only the first one.
	 * If no rows are found, then returns false
	 *
	 * @param string $query select query
	 * @param mixed[] $values list of values to put in place of ?
	 *
	 * @return first row or false if none is found
	 */
	public static function getRow( $query, $values = array(), $accessType = 'read' ) {
		//make sure is connected
		self::ensureConnected( $accessType );

		//get list of rows based on query
		$rows = self::getRows( $query, $values, $accessType );

		//if for some reason rows is false (error) or results are zero,

		//return false to client
		if ( ! $rows || sizeof( $rows ) == 0 ) {
			return false;
		}

		//else return the first from rows
		return $rows[0];
	}

	/**
	 * This query returns list of rows or empty array if nothing is found
	 *
	 * @param string $query select query with ? in place of values
	 * @param mixed[] $values list of mixed values that sholud be put in place of ? in query
	 * @param string $accessType which Database instance to use
	 *
	 * @return mixed list of rows returned from Database
	 */
	public static function getRows( $query, $values = array(), $accessType = 'read' ) {

		//make sure is connected
		self::ensureConnected( $accessType );

		//if values is not an array, then make an array and put that value in it.
		if ( ! is_array( $values ) ) {
			$values = array( $values );
		}

		//here we prepare query based on PDO needs
		$prepared = self::$con[ $accessType ]->prepare( $query );

		//here we execute by passing the values to put instead of ? marks
		$prepared->execute( $values );

		//here we get list of rows matched. Not the associative array flag.
		//that is our method of choice
		$rows = $prepared->fetchAll( \PDO::FETCH_ASSOC );

		if ( ! $rows || sizeof( $rows ) == 0 ) {
			return false;
		}

		//now return rows to whoever asked
		return $rows;
	}

	/**
	 *  function inserts data into Database
	 * use it like this DB::insert("user", array("name"=>'Joe'));
	 *
	 * @param string $tableName name of table to insert into
	 * @param assoc $data associative array of colums and values to insert
	 * @param string $accessType which Database instance to use
	 */
	static public function insert( $tableName, $data, $accessType = 'write' ) {
		//make sure is connected
		self::ensureConnected( $accessType );

		//here we combine column names
		$columnNamesJoined = "`" . implode( "`,`", array_keys( $data ) ) . "`";

		//here we do about the same for data, however instead of values themselves, we put question marks.
		//so now its like (?,?,?);
        $placeholders = array_fill( 0, sizeof( $data ), "?" );

        $columns = array_keys( $data );
        foreach ($columns as $index => $column) {
            if($column == 'geopoints'){
                $placeholders[$index] = 'PointFromText(?, 4326)';
            }
        }
		$questionMarksJoined = implode( ",", $placeholders);
		//here we strip the associative keys from array and name it values
		$values = array_values( $data );

		//here we set up the query for inserting
		$query = "insert into $tableName ($columnNamesJoined) values ($questionMarksJoined)";
		//prepare the query. Notice the write label

		$prepared = self::$con[ $accessType ]->prepare( $query );

		//execute the query with values. Now data is being inserted
		$prepared->execute( $values );

		//save the last id
		$lastInsertId = self::$con[ $accessType ]->lastInsertId();

		return $lastInsertId;

	}

	/**
	 * This updates row or rows in database based on conditions
	 *
	 * @param string $tableName name of table
	 * @param assoc $data list of key value pairs with fields to update
	 * @param string $where condition on what to update
	 * @param mixed[] $whereValues list of values to put in query
	 * @param string $accessType which database instance to use
	 *
	 * @return void
	 */
	static public function update( $tableName, $data, $where = "0", $whereValues = array(), $accessType = 'write' ) {

		//make sure is connected
		self::ensureConnected( $accessType );

		//here we will put parts like "fieldName = ?", so that they can be later
		//combined with commas.. that is how update query is required to be, so that is
		//what we make it
		$updateParts = array();
		//go through each column, value pair
		foreach ( $data as $fieldName => $fieldValue ) {
            if($fieldName == 'geopoints'){
                $updateParts[] = "`$fieldName` = PointFromText(?, 4326)";
            }else{
                //add to list of parts
                $updateParts[] = "`$fieldName` = ?";
            }

		}

		//now implode it into updateSet...
		$updateSet = implode( ", ", $updateParts );

		//here we get rid of data associative keys. then save those values in values
		$values = array_values( $data );

		//in case a non-array is passed as whereValues, put it in an array
		if ( ! is_array( $whereValues ) ) {
			$whereValues = array( $whereValues );
		}

		//here we add the primary value to list of values, because, as you see lower, we add one more question mark at end of query
		//note that this value must be last in list
		$values = array_merge( $values, $whereValues );

		//we combine the string query
		$query = "update $tableName set $updateSet where $where";

		//here we prepare it as PDO requires it. notice the write label
		$prepared = self::$con[ $accessType ]->prepare( $query );
		//here we execute the update function
		$prepared->execute( $values );
	}

	/**
	 * This function is to simply execute query. perhaps delete, or complicated update. Any response is ignored
	 * THIS SHOULD BE USED ONLY ON RARE CASES... whenever possible use insert, update, delete functions
	 *
	 * @param string $query select query
	 * @param mixed[] $values values to put in select query in place of ?
	 * @param string $accessType which Database instance to use
	 *
	 * @return void
	 */
	static public function exec( $query, $values = array(), $accessType = 'write' ) {

		//make sure is connected
		self::ensureConnected( $accessType );

		//if value is only one, then no array is necessary
		if ( ! is_array( $values ) ) {
			$values = array( $values );
		}

		//first we prepare it based on PDO requirements. Notice the write label on connection
		$prepared = self::$con[ $accessType ]->prepare( $query );

		//execute the prepared query with values given
		$prepared->execute( $values );

	}

	/**
	 * This uses the delete function. note that this can mean delete one or thousand of records.
	 * if there is limit, place it in where string
	 *
	 * @param string $tableName name of table where delete should be made
	 * @param string $where what should be deleted
	 * @param mixed[] $values values to put in select query in place of ?
	 * @param string $accessType which Database instance to use
	 *
	 * @return void
	 */
	static public function delete( $tableName, $where = "0", $values = array(), $accessType = 'write' ) {

		//make sure is connected
		self::ensureConnected( $accessType );

		//if value is only one, then no array is necessary
		if ( ! is_array( $values ) ) {
			$values = array( $values );
		}

		if ( $where == "" ) {
			$wherePart = "";
		} else {
			$wherePart = "where $where";
		}

		//prepare query
		$query = "delete from $tableName $wherePart";

		//do the delete action
		self::exec( $query, $values, $accessType );
	}
}