<?php

namespace Zprint\Admin;

use Zprint\Plugin;
use Zprint\Printer;

class PrintDialog
{
    public function __construct()
    {
        add_action('wp_ajax_zprint_reprint', [$this, 'processAjaxCallPrint']);
    }

    public function processAjaxCallPrint()
    {
        if (wp_verify_nonce($_GET['_wpnonce'], 'reprint')) {
            if (
                $order_id = filter_var($_GET['order_id'], FILTER_VALIDATE_INT)
            ) {
                Printer::reprintOrder(
                    $order_id,
                    array_map('intval', $_GET['location'])
                );
                $this->finishPrint($order_id);
                exit();
            } elseif (isset($_GET['bulk_ids'])) {
                $bulk_ids = json_decode($_GET['bulk_ids']);

                foreach ($bulk_ids as $bulk_id) {
                    Printer::reprintOrder(
                        intval($bulk_id),
                        array_map('intval', $_GET['location'])
                    );
                }

                $this->finishPrint();
                exit();
            }
        }
        wp_die();
    }

    private function finishPrint(int $order_id = 0)
    {
        session_start();
        $_SESSION['zprint_reprint'] = true;
        session_write_close();

        if (0 !== $order_id && $_GET['pagenow'] === 'post.php') {
            header(
                'Location: ' .
                    admin_url("post.php?post={$order_id}&action=edit")
            );
        } else {
            header('Location: ' . admin_url('edit.php?post_type=shop_order'));
        }
    }

    public static function render(
        array $locations,
        int $post_id,
        string $type,
        array $bulk_ids = []
    ) {
        $post_id = 0 === $post_id ? 'bulk' : $post_id;
        $dialog_id = 'zprint-dialog-' . $post_id;

        add_action('admin_footer', function () use (
            $dialog_id,
            $locations,
            $post_id,
            $type,
            $bulk_ids
        ) {
            global $pagenow; ?>
			<div id="<?= $dialog_id ?>" style="display: none;">
				<div class="zprint-dialog">
					<div class="zprint-dialog__logo">
						<img class="zprint-dialog__logo-emblem"
								 src="<?= Plugin::getUrl('assets/logo.png') . '?v=2' ?>"
								 alt="<?= esc_attr__(
             'Print Manager',
             'Print-Google-Cloud-Print-GCP-WooCommerce'
         ) ?>">
						<div class="zprint-dialog__logo-title">
								<?= __('Print Manager', 'Print-Google-Cloud-Print-GCP-WooCommerce') ?>
						</div>
					</div>
					<h3 class="zprint-dialog__title">
						<?= empty($bulk_ids)
          ? esc_html__(
                  'Print Order',
                  'Print-Google-Cloud-Print-GCP-WooCommerce'
              ) .
              ' #' .
              $post_id
          : esc_html__(
                  'Print Orders',
                  'Print-Google-Cloud-Print-GCP-WooCommerce'
              ) .
              array_reduce(
                  $bulk_ids,
                  function ($result, $post_id) {
                      return $result .
                          (empty($result) ? ' #' . $post_id : ', #' . $post_id);
                  },
                  ''
              ) ?>
					</h3>
					<form class="zprint-dialog__form" action="<?= admin_url('admin-ajax.php') ?>">
						<?php wp_nonce_field('reprint'); ?>
						<input type="hidden" name="action" value="zprint_reprint">
						<input type="hidden" name="pagenow" value="<?= $pagenow ?>">
						<input type="hidden" name="type" value="<?= $type ?>">
						<input type="hidden" name="order_id" value="<?= $post_id ?>">
						<?php if ($bulk_ids) { ?>
							<input type="hidden" name="bulk_ids" value="<?= wp_json_encode($bulk_ids) ?>">
						<?php } ?>
						<label for="location-<?= $post_id ?>">
							<?= esc_html__('Select Location', 'Print-Google-Cloud-Print-GCP-WooCommerce') ?>
						</label>
						<br>
						<select class="zprint-dialog__select" name="location[]" id="location-<?= $post_id ?>" multiple>
							<?php
       $i = 0;
       foreach ($locations as $location) { ?>
								<option <?= 0 === $i ? 'selected' : '' ?>
									value="<?= $location['id'] ?>">
										<?= $location['title'] ?>
								</option>
								<?php $i++;}?>
						</select>
						<div class="zprint-dialog__btn-row">
							<input type="submit" class="button button-primary zprint-dialog__btn" value="Print">
						</div>
					</form>
				</div>
			</div>
			<?php
        });
        return $dialog_id;
    }
}
