<?php
/**
 * Plugin Name: Five Star Restaurant Reviews
 * Plugin URI:  https://www.fivestarplugins.com/plugins/five-star-restaurant-reviews/
 * Description: Restaurant reviews made easy. Add and display reviews on your restaurant site using SEO friendly schema markup.
 * Version: 	2.3.3
 * Author:		Five Star Plugins
 * Author URI:  https://www.fivestarplugins.com
 * License:		GPLv3
 * License URI:	http://www.gnu.org/licenses/gpl-3.0.html
 *
 * Text Domain:	good-reviews-wp
 * Domain Path:	/languages/
 */

if ( ! defined( 'ABSPATH' ) )
	exit;

if ( !class_exists( 'grfwpInit' ) ) {
class grfwpInit {

	// Any data that needs to be passed from PHP to our JS files 
	public $front_end_php_js_data = array();

	/**
	 * The single instance of this class
	 */
	private static $instance;

	/**
	 *  WP_Query arguments when retrieving reviews
	 */
	public $args = array();

	/**
	 * Details about the object  being reviews for schema.org markup
	 */
	public $reviewed = array();

	/**
	 * IDs of any printed review sets, to distinguish between multiple
	 * widgest or sets of reviews on a single page
	 */
	public $ids = array();

	/**
	 * Create or retrieve the single instance of the class
	 *
	 * @since 0.1
	 */
	public static function instance() {

		if ( !isset( self::$instance ) ) {

			self::$instance = new grfwpInit();

			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Initialize the plugin and register hooks
	 */
	public function init() {

		// Common strings
		define( 'GRFWP_TEXTDOMAIN', 'good-reviews-wp' ); // keep for back compat
		define( 'GRFWP_VERSION', '2.3.3' );
		define( 'GRFWP_PLUGIN_DIR', untrailingslashit( plugin_dir_path( __FILE__ ) ) );
		define( 'GRFWP_PLUGIN_URL', untrailingslashit( plugins_url( basename( plugin_dir_path( __FILE__ ) ), basename( __FILE__ ) ) ) );
		define( 'GRFWP_PLUGIN_FNAME', plugin_basename( __FILE__ ) );
		define( 'GRFWP_TEMPLATE_DIR', 'templates' );
		define( 'GRFWP_REVIEW_POST_TYPE', 'grfwp-review' );
		define( 'GRFWP_REVIEW_CATEGORY', 'grfwp-category' );

		// Initialize the plugin
		add_action( 'plugins_loaded', array( $this, 'plugin_loaded_action_hook' ) );

		add_action( 'init', array( $this, 'load_textdomain' ) );



		// Load helper class
		require_once( GRFWP_PLUGIN_DIR . '/includes/Helper.class.php' );

		// Load custom post types
		require_once( GRFWP_PLUGIN_DIR . '/includes/CustomPostTypes.class.php' );
		$this->cpts = new grfwpCustomPostTypes();

		// Load deactivation survey
		require_once( GRFWP_PLUGIN_DIR . '/includes/DeactivationSurvey.class.php' );
		new grfwpDeactivationSurvey();

		// Load review ask
		require_once( GRFWP_PLUGIN_DIR . '/includes/ReviewAsk.class.php' );
		new grfwpReviewAsk();

		// Handle setting up AJAX interactions
		require_once( GRFWP_PLUGIN_DIR . '/includes/Ajax.class.php' );
		$this->ajax = new grfwpAJAX();

		// Handle setting up exports
		require_once( GRFWP_PLUGIN_DIR . '/includes/ExportHandler.class.php' );
		$this->exports = new grfwpExportHandler();

		// Load the settings page
		require_once( GRFWP_PLUGIN_DIR . '/includes/Settings.class.php' );
		$this->settings = new grfwpSettings(); 

		// Load permissions
		require_once( GRFWP_PLUGIN_DIR . '/includes/Permissions.class.php' );
		$this->permissions = new grfwpPermissions();

		// Load plugin dashboard
		require_once( GRFWP_PLUGIN_DIR . '/includes/Dashboard.class.php' );
		new grfwpDashboard();

		// Add the admin menu
		add_action( 'admin_menu', array( $this, 'add_menu_page' ) );

		// Load template functions
		require_once( GRFWP_PLUGIN_DIR . '/includes/template-functions.php' );

		// Load blocks
		require_once( GRFWP_PLUGIN_DIR . '/includes/Blocks.class.php' );
		new grfwpBlocks();

		// Load code to integrate with other plugins
		require_once( GRFWP_PLUGIN_DIR . '/includes/Integrations.class.php' );
		new grfwpIntegrations();

		// Load About Us page
		require_once( GRFWP_PLUGIN_DIR . '/includes/AboutUs.class.php' );
		new grfwpAboutUs();
		
		// Register assets
		add_action( 'wp_enqueue_scripts', array( $this, 'register_assets' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
		add_action( 'wp_footer', array( $this, 'assets_footer' ), 2 );

		// Add front-end AJAX URL
		add_action( 'wp_head', 'fsp_add_frontend_ajax_url' );

		// Reword the title placeholder text for a review post type
		add_filter( 'enter_title_here', array( $this, 'rename_review_title' ) );

		// Order review posts in admin screen by menu order
		add_filter('pre_get_posts', array( $this, 'admin_order_posts' ) );

		// Transform review $content variable to output review
		add_filter( 'the_content', array( $this, 'append_to_content' ) );

		// Register the widget
		add_action( 'widgets_init', array( $this, 'register_widgets' ) );

		// Add links to plugin listing
		add_filter('plugin_action_links', array( $this, 'plugin_action_links' ), 10, 2);

		// Load header area
		add_action( 'admin_notices', array( $this, 'display_header_area') );

		// Handle the helper notice
		add_action( 'admin_notices', array( $this, 'maybe_display_helper_notice' ) );
		add_action( 'wp_ajax_grfwp_hide_helper_notice', array( $this, 'hide_helper_notice' ) );
	}

	public function add_menu_page() {
		add_menu_page(
			__( 'Restaurant Reviews', 'good-reviews-wp' ),
			__( 'Restaurant Reviews', 'good-reviews-wp' ),
			'manage_options',
			'grfwp-restaurant-reviews',
			'',
			'dashicons-star-empty',
			50.35
		);
	}

	public function display_header_area() {
		global $grfwp_controller, $post;

		$screen = get_current_screen();
		$screenID = $screen->id;
		$screenBase = $screen->base;
		$screenPostType = $screen->post_type;
		$settings = get_option( 'grfwp-settings' );

		if ( $screenID != 'restaurant-reviews_page_grfwp-settings' && $screenID != 'restaurant-reviews_page_grfwp-dashboard' && $screenID != 'restaurant-reviews_page_grfwp-about-us' && $screenPostType != 'grfwp-review' ) {return;}

		if ( ! $grfwp_controller->permissions->check_permission( 'premium' ) || get_option("GRFWP_Trial_Happening") == "Yes" ) {
			?>
			<div class="grfwp-dashboard-new-upgrade-banner">
				<div class="grfwp-dashboard-banner-icon"></div>
				<div class="grfwp-dashboard-banner-buttons">
					<a class="grfwp-dashboard-new-upgrade-button" href="https://www.fivestarplugins.com/license-payment/?Selected=GRFWP&Quantity=1&utm_source=grfwp_admin&utm_content=banner" target="_blank">UPGRADE NOW</a>
				</div>
				<div class="grfwp-dashboard-banner-text">
					<div class="grfwp-dashboard-banner-title">
						GET FULL ACCESS WITH OUR PREMIUM VERSION
					</div>
					<div class="grfwp-dashboard-banner-brief">
						A responsive thumbnail layout, an image-based layout, as well as advanced styling options
					</div>
				</div>
			</div>
			<?php
		}
		
		?>
		<div class="grfwp-admin-header-menu">
			<h2 class="nav-tab-wrapper">
			<a id="grfwp-dash-mobile-menu-open" href="#" class="menu-tab nav-tab"><?php _e("MENU", 'good-reviews-wp'); ?><span id="grfwp-dash-mobile-menu-down-caret">&nbsp;&nbsp;&#9660;</span><span id="grfwp-dash-mobile-menu-up-caret">&nbsp;&nbsp;&#9650;</span></a>
			<a id="dashboard-menu" href='admin.php?page=grfwp-dashboard' class="menu-tab nav-tab <?php if ($screenID == 'restaurant-reviews_page_grfwp-dashboard') {echo 'nav-tab-active';}?>"><?php _e("Dashboard", 'good-reviews-wp'); ?></a>
			<a id="reviews-menu" href='edit.php?post_type=grfwp-review' class="menu-tab nav-tab <?php if ($screenPostType == 'grfwp-review' && $screenBase == 'edit') {echo 'nav-tab-active';}?>"><?php _e("Reviews", 'good-reviews-wp'); ?></a>
			<a id="categories-menu" href='edit-tags.php?taxonomy=grfwp-category&post_type=grfwp-review' class="menu-tab nav-tab <?php if ($screenPostType == 'grfwp-review' && $screenBase == 'edit-tags') {echo 'nav-tab-active';}?>"><?php _e("Categories", 'good-reviews-wp'); ?></a>
			<a id="options-menu" href='admin.php?page=grfwp-settings' class="menu-tab nav-tab <?php if ($screenID == 'restaurant-reviews_page_grfwp-settings') {echo 'nav-tab-active';}?>"><?php _e("Settings", 'good-reviews-wp'); ?></a>
			</h2>
		</div>
		<?php
	}

	/**
	 * Flush the rewrite rules when this plugin is activated to update with
	 * custom post types
	 * @since 0.1
	 */
	public function rewrite_flush() {
		$this->cpts->load_cpts();
		flush_rewrite_rules();
	}

	/**
	 * Allow third-party plugins to interact with the plugin, if necessary
	 * @since 2.2.0
	 */
	public function plugin_loaded_action_hook() {

		do_action( 'grfwp_initialized' );
	}

	/**
	 * Load the plugin textdomain for localistion
	 * @since 0.0.1
	 */
	public function load_textdomain() {
		load_plugin_textdomain( 'good-reviews-wp', false, plugin_basename( dirname( __FILE__ ) ) . "/languages/" );

		// Backwards compatibility
		load_plugin_textdomain( 'grfwpdomain', false, plugin_basename( dirname( __FILE__ ) ) . "/languages/" );
	}

	/**
	 * Register stylesheet
	 * @since 0.0.1
	 */
	public function register_assets() {
		wp_register_style( 'gr-reviews', GRFWP_PLUGIN_URL . '/assets/css/style.css', '1.0' );
		wp_register_script( 'gr-reviews', GRFWP_PLUGIN_URL . '/assets/js/frontend.js', array( 'jquery' ), false, true  );
		wp_register_script( 'gr-pagination', GRFWP_PLUGIN_URL . '/assets/js/pagination.js', array( 'jquery' ), false, true  );
	}

	/**
	 * Print out any PHP data needed for our JS to work correctly
	 * @since 2.2.0
	 */
	public function assets_footer() {

		if ( empty( $this->front_end_php_js_data ) ) { return; }

		$print_variables = array();

		foreach ( (array) $this->front_end_php_js_data as $variable => $values ) {
			$print_variables[ $variable ] = grfwpHelper::escape_js_recursive( $values );
		}

		foreach ( $print_variables as $variable => $values ) {

			echo "<script type='text/javascript'>\n";
			echo "/* <![CDATA[ */\n";
			echo 'var ' . esc_attr( $variable ) . ' = ' . wp_json_encode( $values ) . "\n";
			echo "/* ]]> */\n";
			echo "</script>\n";
		}
	}

	/**
	 * Adds a variable to be passed to our front-end JS
	 * @since 2.2.0
	 */
	public function add_front_end_php_data( $handle, $variable, $data ) {

		$this->front_end_php_js_data[ $variable ] = $data;
	}

	/**
	 * Enqueue the admin-only CSS
	 * @since 1.1
	 */
	public function enqueue_admin_assets(){
		global $post;

		wp_enqueue_script( 'grfwp-helper-notice', GRFWP_PLUGIN_URL . '/assets/js/helper-install-notice.js', array( 'jquery' ), GRFWP_VERSION, true );
			wp_localize_script(
				'grfwp-helper-notice',
				'grfwp_helper_notice',
				array( 'nonce' => wp_create_nonce( 'grfwp-helper-notice' ) )
			);

			wp_enqueue_style( 'grfwp-helper-notice', GRFWP_PLUGIN_URL . '/assets/css/helper-install-notice.css', array(), GRFWP_VERSION );

		wp_add_inline_style( 'wp-admin', '#adminmenu #menu-posts-grfwp-review .wp-menu-image:before { content: "\f313"; }' );

		$screen = get_current_screen();

		if ( $screen->post_type == 'grfwp-review' || $screen->id == 'restaurant-reviews_page_grfwp-settings' || $screen->id == 'restaurant-reviews_page_grfwp-about-us' ) {

			wp_enqueue_style( 'grfwp-admin-css', GRFWP_PLUGIN_URL . '/assets/css/admin.css', array(), GRFWP_VERSION );
			wp_enqueue_script( 'grfwp-spectrum-js', GRFWP_PLUGIN_URL . '/assets/js/spectrum.js', array( 'jquery' ), '', true );
			wp_enqueue_script( 'grfwp-admin-js', GRFWP_PLUGIN_URL . '/assets/js/admin.js', array( 'jquery' ), GRFWP_VERSION, true );
			$grfwp_admin_data = array(
				'enable_datepicker'	=> $this->settings->get_setting( 'grfwp-enable-datepicker' ),
				'date_format' 		=> $this->settings->get_setting( 'grfwp-date-format' ),
				'nonce' 			=> wp_create_nonce( 'grfwp-admin-js' ),
			);
			wp_localize_script( 'grfwp-admin-js', 'grfwp_php_add_data', $grfwp_admin_data );
		}

		if ( $screen->post_type == 'grfwp-review' ) {

			wp_enqueue_script( 'jquery-ui-core', array( 'jquery' ) );
			wp_enqueue_script( 'jquery-ui-datepicker', array( 'jquery' ) );
			wp_enqueue_style( 'grfwp-jquery-ui-datepicker-css', GRFWP_PLUGIN_URL . '/assets/css/jquery-ui/jquery-ui.css', array(), GRFWP_VERSION );
		}
	}

	/**
	 * Reword the title placeholder text for a review post type
	 * @since 0.1
	 */
	public function rename_review_title( $title ){
		 $screen = get_current_screen();

		 if  ( $screen->post_type == GRFWP_REVIEW_POST_TYPE ) {
			  $title = __( 'Enter reviewer here', 'good-reviews-wp' );
		 }

		 return $title;
	}

	/**
	 * Order the reviews by menu order parameter in the admin interface
	 * @since 0.1
	 */
	public function admin_order_posts( $query ) {

		if( ( is_admin() && $query->is_admin ) && $query->get( 'post_type' ) == GRFWP_REVIEW_POST_TYPE ) {

			// Don't override an existing orderby setting. This prevents other
			// orderby options from breaking.
			if ( !$query->get( 'orderby' ) ) {
				$query->set( 'orderby', 'menu_order' );
				$query->set( 'order', 'ASC' );
			}
		}

		return $query;
	}

	/**
	 * Run callback on every element in array recursively
	 *
	 * Used to sanitize all values in an array
	 * @since 0.1
	 */
	public static function array_filter_recursive( $arr, $callback ) {
		foreach ( $arr as &$value ) {
			if ( is_array( $value ) ) {
				$value = grfwpInit::array_filter_recursive( $value, $callback );
			}
		}
		return array_filter( $arr, $callback );
	}

	/**
	 * Tranform an array of CSS classes into an HTML attribute
	 * @since 0.1
	 */
	public function format_classes( $classes ) {
		if ( count( $classes ) ) {
			return ' class="' . join(" ", $classes) . '"';
		}
	}

	/**
	 * Process review query arguments
	 * @since 0.1
	 */
	public function get_query_args( $args ) {

		// Set and filter defaults
		$this->args = array(
			'posts_per_page' 	=> $this->settings->get_setting( 'grfwp-reviews-per-page' ),
			'post_type' 		=> GRFWP_REVIEW_POST_TYPE,
			'orderby' 			=> 'menu_order',
			'order' 			=> 'ASC',
			'cycle'				=> false,
			'excerpt'			=> false,
			'paged'				=> 1
		);
		$this->args = apply_filters( 'grfwp_query_args_defaults', $this->args );

		if ( !empty( $args['review'] ) ) {
			$this->args['p'] = $args['review'];
			unset( $this->args['posts_per_page'] );
		}

		if ( !empty( $args['category'] ) ) {
			$this->args[GRFWP_REVIEW_CATEGORY] = $args['category'];
		}

		if ( !empty( $args['random'] ) ) {
			$this->args['orderby'] = 'rand';
		}

		if ( !empty( $args['limit'] ) ) {
			$this->args['posts_per_page'] = $args['limit'];
		}

		if ( !empty( $args['cycle'] ) ) {
			$this->args['cycle'] = $args['cycle'];
		}

		if ( !empty( $args['excerpt'] ) ) {
			$this->args['excerpt'] = $args['excerpt'];
		}

		if ( !empty( $args['meta_query'] ) ) {
			$this->args['meta_query'] = $args['meta_query'];
		}
		
		if ( !empty( $args['meta_key'] ) ) {
			$this->args['meta_key'] = $args['meta_key'];
		}
		
		if ( !empty( $args['meta_value'] ) ) {
			$this->args['meta_value'] = $args['meta_value'];
		}

		if ( !empty( $args['paged'] ) ) {
			$this->args['paged'] = $args['paged'];
		}

		$this->args = apply_filters( 'grfwp_query_args', $this->args );
	}

	/**
	 * Retrieve schema.org details for the item being reviews
	 * @note $args is in place for future compatibility, for instance if support
	 *		for multiple venues is added
	 * @since 0.1
	 */
	public function get_reviewed_item( $args = array() ) {

		// Set and filter defaults
		// @todo use schema from settings
		$this->reviewed = array(
			'name'			=> esc_attr( get_bloginfo( 'name' ) ),
			'url'			=> esc_attr( get_bloginfo( 'url' ) ),
			'description'	=> esc_attr( get_bloginfo( 'description' ) ),
			'schema'		=> 'Thing',
		);

		$this->reviewed = apply_filters( 'grfwp_reviewed_defaults', $this->reviewed );

		$this->reviewed = array_merge( $this->reviewed, $args );

		$this->reviewed = apply_filters( 'grfwp_reviewed_values', $this->reviewed );
	}

	/**
	 * Transform review $content variable to output review
	 * @since 0.1
	 */
	function append_to_content( $content ) {
		global $post;

		if ( !in_the_loop() || !is_main_query() || is_search() || GRFWP_REVIEW_POST_TYPE !== $post->post_type ) {
			return $content;
		}

		// Allow overrides to disable the automatic append to content filter
		if ( !apply_filters( 'grfwp_append_to_content', true ) ) {
			return $content;
		}

		// We must disable this filter while we're rendering the review in order
		// to prevent it from falling into a recursive loop with each review's
		// content.
		remove_action( 'the_content', array( $this, 'append_to_content' ) );

		$args = array(
			'review'	=> $post->ID,
		);
		$args = apply_filters( 'grfwp_post_content_args', $args );

		$content = grfwp_print_reviews( $args );

		// Restore this filter
		add_action( 'the_content', array( $this, 'append_to_content' ) );

		return $content;
	}

	/**
	 * Register the widgets
	 * @since 0.0.1
	 */
	public function register_widgets() {
		require_once( GRFWP_PLUGIN_DIR . '/includes/WP_Widget.ReviewsWidget.class.php' );
		register_widget( 'grfwpWidgetReviews' );
	}

	/**
	 * Add links to the plugin listing on the installed plugins page
	 * @since 0.0.1
	 */
	public function plugin_action_links( $links, $plugin ) {
		global $grfwp_controller;
		
		if ( $plugin == GRFWP_PLUGIN_FNAME ) {

			if ( ! $grfwp_controller->permissions->check_permission( 'premium' ) ) {

				array_unshift( $links, '<a class="grfwp-plugin-page-upgrade-link" href="https://www.fivestarplugins.com/license-payment/?Selected=GRFWP&Quantity=1&utm_source=wp_admin_plugins_page" title="' . __( 'Try Premium', 'good-reviews-wp' ) . '" target="_blank">' . __( 'Try Premium', 'good-reviews-wp' ) . '</a>' );
			}

			$links['help'] = '<a href="' . GRFWP_PLUGIN_URL . '/docs" title="' . __( 'View the help documentation for Business Profile', 'good-reviews-wp' ) . '">' . __( 'Help', 'good-reviews-wp' ) . '</a>';
		}

		return $links;

	}

	public function maybe_display_helper_notice() {
		global $grfwp_controller;
	
		if ( empty( $grfwp_controller->permissions->check_permission( 'premium' ) ) ) { return; }
	
		if ( is_plugin_active( 'fsp-premium-helper/fsp-premium-helper.php' ) ) { return; }
	
		if ( get_transient( 'fsp-helper-notice-dismissed' ) ) { return; }
	
		?>
	
		<div class='notice notice-error is-dismissible grfwp-helper-install-notice'>
				
			<div class='grfwp-helper-install-notice-img'>
				<img src='<?php echo GRFWP_PLUGIN_URL . '/lib/simple-admin-pages/img/options-asset-exclamation.png' ; ?>' />
			</div>
	
			<div class='grfwp-helper-install-notice-txt'>
				<?php _e( 'You\'re using the Five-Star Restaurant Reviews premium version, but the premium helper plugin is not active.', 'good-reviews-wp' ); ?>
				<br />
				<?php echo sprintf( __( 'Please re-activate the helper plugin, or <a target=\'_blank\' href=\'%s\'>download and install it</a> if the plugin is no longer installed to ensure continued access to the premium features of the plugin.', 'good-reviews-wp' ), 'https://www.fivestarplugins.com/2021/12/23/requiring-premium-helper-plugin/' ); ?>
			</div>
	
			<div class='grfwp-clear'></div>
	
		</div>
	
		<?php 
	}
	
	public function hide_helper_notice() {
	
		// Authenticate request
		if ( ! check_ajax_referer( 'grfwp-helper-notice', 'nonce' ) or ! current_user_can( 'manage_options' ) ) {
				
			wp_send_json_error(
				array(
					'error' => 'loggedout',
					'msg' => sprintf( __( 'You have been logged out. Please %slogin again%s.', 'good-reviews-wp' ), '<a href="' . wp_login_url( admin_url( 'admin.php?page=grfwp-dashboard' ) ) . '">', '</a>' ),
				)
			);
		}
	
		set_transient( 'fsp-helper-notice-dismissed', true, 3600*24*7 );
	
		die();
	}

}
} // endif;

// Global instance
$grfwp_controller = grfwpInit::instance();

// Flush the rewrite rules for the custom post types
register_activation_hook( __FILE__, array( $grfwp_controller, 'rewrite_flush' ) );