<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @package    DDWC
 * @subpackage DDWC/includes
 * @author     Devio Digital <contact@deviodigital.com>
 * @license    GPL-2.0+ http://www.gnu.org/licenses/gpl-2.0.txt
 * @link       https://www.deviodigital.com
 * @since      1.0.0
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @package    DDWC
 * @subpackage DDWC/includes
 * @author     Devio Digital <contact@deviodigital.com>
 * @license    GPL-2.0+ http://www.gnu.org/licenses/gpl-2.0.txt
 * @link       https://www.deviodigital.com
 * @since      1.0.0
 */
class DDWC {

    /**
     * The loader that's responsible for maintaining and registering all
     * hooks that power the plugin.
     *
     * @since  1.0.0
     * @access protected
     * @var    Delivery_Drivers_Loader $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since  1.0.0
     * @access protected
     * @var    string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since  1.0.0
     * @access protected
     * @var    string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since 1.0.0
     */
    public function __construct() {
        $this->plugin_name = 'delivery-drivers-for-woocommerce';
        $this->version     = '3.4.0';

        if ( defined( 'DDWC_VERSION' ) ) {
            $this->version = DDWC_VERSION;
        }

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();

    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - Delivery_Drivers_Loader. Orchestrates the hooks of the plugin.
     * - Delivery_Drivers_i18n. Defines internationalization functionality.
     * - Delivery_Drivers_Admin. Defines all hooks for the admin area.
     * - Delivery_Drivers_Public. Defines all hooks for the public side of the site.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since  1.0.0
     * @access private
     * @return void
     */
    private function load_dependencies() {

        /**
         * The class responsible for orchestrating the actions and filters of the
         * core plugin.
         */
        require_once( plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-ddwc-loader.php' );

        /**
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */
        require_once( plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-ddwc-i18n.php' );

        /**
         * The class responsible for defining all actions that occur in the admin area.
         */
        require_once( plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-ddwc-admin.php' );

        /**
         * The file responsible for defining the DDWC helper functions
         *
         * @since 2.0
         */
        require_once( plugin_dir_path( dirname( __FILE__ ) ) . 'admin/ddwc-functions.php' );

        /**
         * The file responsible for defining the Delivery Driver Dashboard Shortcode.
         */
        require_once( plugin_dir_path( dirname( __FILE__ ) ) . 'admin/ddwc-dashboard-shortcode.php' );

        /**
         * The file responsible for defining the Delivery Driver Metaboxes.
         */
        require_once( plugin_dir_path( dirname( __FILE__ ) ) . 'admin/ddwc-metaboxes.php' );

        /**
         * The file responsible for defining the WooCommerce Orders functions.
         */
        require_once( plugin_dir_path( dirname( __FILE__ ) ) . 'admin/ddwc-woocommerce-orders.php' );

        /**
         * The file responsible for defining the WooCommerce My Account Drivers tab.
         */
        require_once( plugin_dir_path( dirname( __FILE__ ) ) . 'admin/ddwc-woocommerce-account-tab.php' );

        /**
         * The file responsible for defining the WooCommerce Settings page.
         */
        require_once( plugin_dir_path( dirname( __FILE__ ) ) . 'admin/ddwc-woocommerce-settings.php' );

        /**
         * The file responsible for defining the functions that add Delivery Driver details
         * to WooCommerce orders.
         */
        require_once( plugin_dir_path( dirname( __FILE__ ) ) . 'admin/ddwc-woocommerce-driver-ratings.php' );

        /**
         * The file responsible for defining the user profile settings.
         */
        require_once( plugin_dir_path( dirname( __FILE__ ) ) . 'admin/ddwc-user-profile-settings.php' );

        /**
         * The class responsible for defining all actions that occur in the public-facing
         * side of the site.
         */
        require_once( plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-ddwc-public.php' );

        $this->loader = new Delivery_Drivers_Loader();

    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the Delivery_Drivers_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since  1.0.0
     * @access private
     * @return void
     */
    private function set_locale() {

        $plugin_i18n = new Delivery_Drivers_i18n();

        $this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since  1.0.0
     * @access private
     * @return void
     */
    private function define_admin_hooks() {

        $plugin_admin = new Delivery_Drivers_Admin( $this->get_plugin_name(), $this->get_version() );

        $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
        $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since  1.0.0
     * @access private
     * @return void
     */
    private function define_public_hooks() {

        $plugin_public = new Delivery_Drivers_Public( $this->get_plugin_name(), $this->get_version() );

        $this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
        $this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since  1.0.0
     * @return void
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since  1.0.0
     * @return string
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since  1.0.0
     * @return Delivery_Drivers_Loader 
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since  1.0.0
     * @return string
     */
    public function get_version() {
        return $this->version;
    }

}
